# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliMode
import BasicCli

# -----------------------------------------------------------------------------------
# (config)# monitor twamp
# -----------------------------------------------------------------------------------
class MonitorTwampModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'monitor twamp'

   def __init__( self, param ):
      self.modeKey = 'monitor-twamp'
      self.longModeKey = 'monitor-twamp'
      CliMode.ConfigMode.__init__( self, param )

class MonitorTwampMode( MonitorTwampModeBase, BasicCli.ConfigModeBase ):
   name = "Monitor TWAMP configuration"

   def __init__( self, parent, session ):
      MonitorTwampModeBase.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

# -----------------------------------------------------------------------------------
# (config-monitor-twamp)# [ no | default ] twamp-light
# -----------------------------------------------------------------------------------
class TwampLightModeBase( CliMode.ConfigMode ):
   def enterCmd( self ):
      return 'twamp-light'

   def __init__( self, param ):
      self.modeKey = 'twamp-light'
      self.longModeKey = 'twamp-light'
      CliMode.ConfigMode.__init__( self, param )

class TwampLightMode( TwampLightModeBase, BasicCli.ConfigModeBase ):
   name = "TWAMP light configuration"

   def __init__( self, parent, session ):
      TwampLightModeBase.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

# -----------------------------------------------------------------------------------
# (config-twamp-light)# [ no | default ] ( reflector | sender ) profile NAME
# -----------------------------------------------------------------------------------
class TwampLightProfileMode( CliMode.ConfigMode ):
   def enterCmd( self ):
      return f'{self.profileType} profile {self.instanceName}'

   def __init__( self, param ):
      ( self.instanceName, self.profileType ) = param
      self.modeKey = f'twamp-light-{self.profileType}-profile-{self.instanceName}'
      self.longModeKey = f'twamp-light-{self.profileType}' +\
                         f'-profile-{self.instanceName}'
      CliMode.ConfigMode.__init__( self, param )

   def commentKey( self ):
      return f'{self.profileType} profile {self.instanceName}'

# -----------------------------------------------------------------------------------
# (config-monitor-twamp)# [ no | default ] reflector profile NAME
# -----------------------------------------------------------------------------------
class TwampLightReflectorProfileMode( TwampLightProfileMode,
                                      BasicCli.ConfigModeBase ):
   name = 'TWAMP light reflector profile configuration'

   def __init__( self, parent, session, instanceConfig ):
      TwampLightProfileMode.__init__( self, ( instanceConfig.name, 'reflector' ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.instanceName = instanceConfig.name

# -----------------------------------------------------------------------------------
# (config-monitor-twamp)# [ no | default ] sender profile NAME
# -----------------------------------------------------------------------------------
class TwampLightSenderProfileMode( TwampLightProfileMode, BasicCli.ConfigModeBase ):
   name = 'TWAMP light sender profile configuration'

   def __init__( self, parent, session, instanceConfig ):
      TwampLightProfileMode.__init__( self, ( instanceConfig.name, 'sender' ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.instanceName = instanceConfig.name
      self.config = instanceConfig

# -----------------------------------------------------------------------------------
# (config-twamp-light)# [ no | default ] ( reflector | sender ) defaults
# -----------------------------------------------------------------------------------
class TwampLightDefaultsMode( CliMode.ConfigMode ):
   def enterCmd( self ):
      return f'{self.profileType} defaults'

   def __init__( self, param ):
      self.profileType = param
      self.modeKey = f'twamp-light-{self.profileType}-defaults'
      self.longModeKey = f'twamp-light-{self.profileType}-defaults'
      CliMode.ConfigMode.__init__( self, param )

   def commentKey( self ):
      return f'{self.profileType} defaults'

# -----------------------------------------------------------------------------------
# (config-twamp-light)# [ no | default ] reflector defaults
# -----------------------------------------------------------------------------------
class TwampLightReflectorDefaultsMode( TwampLightDefaultsMode,
                                       BasicCli.ConfigModeBase ):
   name = 'TWAMP light reflector profile defaults'

   def __init__( self, parent, session ):
      TwampLightDefaultsMode.__init__( self, 'reflector' )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

# -----------------------------------------------------------------------------------
# (config-twamp-light)# [ no | default ] sender defaults
# -----------------------------------------------------------------------------------
class TwampLightSenderDefaultsMode( TwampLightDefaultsMode,
                                    BasicCli.ConfigModeBase ):
   name = 'TWAMP light sender profile defaults'

   def __init__( self, parent, session ):
      TwampLightDefaultsMode.__init__( self, 'sender' )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
