# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import os
import re
import shutil
import stat
import tempfile
from EosVersion import sidToOptimizationFile

def listFilesInZip( path ):
   try:
      output = Tac.run( [ 'unzip', '-Z1', path ], stdout=Tac.CAPTURE,
                       stderr=Tac.CAPTURE )
   except OSError:
      return None
   except Tac.SystemCommandError:
      return None
   if not output:
      return None
   files = output.split( '\n' )
   if files[ -1 ] == '':
      return files[ : -1 ]
   return files

def extractFileFromZip( tmpDir, zipPath, filename ):
   '''Extract file from zip and return path to it.
   Arguments:
      tmpDir: String: Path temporary directory where the file will be extracted
      zipPath: String: Path to zip file
      filename: String: Name of the file to extract
   Returns:
      Path to extracted filename or None in case of failure
   '''
   files = listFilesInZip( zipPath )
   if files is None:
      return None
   if filename not in files:
      return None
   try:
      Tac.run( [ 'unzip', zipPath, filename, '-d', tmpDir ], stdout=Tac.CAPTURE,
               stderr=Tac.CAPTURE )
   except Tac.SystemCommandError:
      return None
   return os.path.join( tmpDir, filename )

def disableSwiAdaptPresent( path='/mnt/flash/disable_swi_adapt' ):
   if os.path.exists( path ):
      return True
   return False

def hasImageMoreThanOneRootfs( path ):
   atLeastOneRootfs = False
   files = listFilesInZip( path )
   if files is None:
      return False
   for f in files:
      if '.rootfs-i386.sqsh' in f:
         if atLeastOneRootfs:
            return True
         atLeastOneRootfs = True
   return False

def getSidFrom( path ):
   if not os.path.isfile( path ):
      return None
   try:
      with open( path ) as f:
         readFile = f.read()
         sidMatch = re.search( r'(^|\s)sid(:|=)\s*(\S+)', readFile,
                               re.MULTILINE | re.IGNORECASE )
         if sidMatch:
            return sidMatch.group( 3 )
      return None
   except OSError:
      return None

def getSid():
   sid = getSidFrom( '/etc/prefdl' )
   if sid:
      return sid
   return getSidFrom( '/proc/cmdline' )

def getOptimization( sid, path ):
   try:
      with open( path ) as f:
         for line in f:
            # Because Capitola*:Strata-4GB
            entry, optimization = line.split( ':' )
            if entry[ -1 ] == '*':
               # Make that '*' regexp readable
               entry = entry[ : -1 ] + r'\S*'
            reStr = r'^%s$' % entry
            entryMatch = re.search( reStr, sid,
                                    re.IGNORECASE )
            if entryMatch:
               # Remove endlines
               return optimization.strip()
   except OSError:
      return None
   return None

def imageHasOptimization( path, optimization ):
   files = listFilesInZip( path )
   if files and optimization + '.rootfs-i386.sqsh' in files:
      return True
   return False

def adaptImage( mode, tmpDir, imagePath ):
   swadaptPath = extractFileFromZip( tmpDir, imagePath,
                                     'swadapt' )
   if swadaptPath is None:
      return
   # For some reason file permissions are lost when file is extracted
   st = os.stat( swadaptPath )
   os.chmod( swadaptPath, st.st_mode | stat.S_IEXEC )
   # This will adapt swi image in place
   command = [ swadaptPath, imagePath, imagePath, 'auto' ]
   mode.addMessage( 'Optimizing image for current system - '
                    'this may take a minute...' )
   try:
      # Adapt image 'in-place'
      Tac.run( command, asRoot=True )
   except ( OSError, Tac.SystemCommandError ):
      mode.addError( 'Failed to optimize boot-config image. '
                     'Image may be corrupted.' )
      return

def maybeOptimizeSwi( nextImagePath, mode ):
   tmpDir = tempfile.mkdtemp( dir='/tmp' )
   _maybeOptimizeSwi( nextImagePath, mode, tmpDir )
   shutil.rmtree( tmpDir )

def _maybeOptimizeSwi( nextImagePath, mode, tmpDir ):
   """Optimize swi image in place if possible. This function only makes sense
      when it's run on a dut.
   Arguments:
      nextImagePath: String: path to swi image
      mode: used to log messages and errors
      tmpDir: String: Path to temporary directory where some files
                      from the image will be extracted
   """
   if disableSwiAdaptPresent():
      return

   if not hasImageMoreThanOneRootfs( nextImagePath ):
      return

   sid = getSid()
   if not sid:
      return

   pathToSidMap = extractFileFromZip( tmpDir, nextImagePath,
                                      sidToOptimizationFile )
   if not pathToSidMap:
      return

   optimization = getOptimization( sid, pathToSidMap )
   if not optimization:
      return

   if not imageHasOptimization( nextImagePath, optimization ):
      return

   adaptImage( mode, tmpDir, nextImagePath )
