# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# Monkey patch Tac.Run to provide the correct environment/std(out|err) to subproceses

import os
import sys

import Tac
import TacUtils

origTacRun = None

# A SystemCommandError with an extra attribute about stdout conditions.
# If Tac.run's output is not visible to the user, it cannot go through a pipe (pagi-
# nation) that can cause a SIGPIPE which we need to silence, so mark it accordingly
# so that higher levels can determine how exactly to react (ignore or not).
class SystemCommandError( TacUtils.SystemCommandError ):
   def __init__( self, msg='Unknown Command Error', error=None, output=None ):
      super().__init__( msg=msg, error=error, output=output )
      self.outputRedirected = False

def newTacRun( *args, **kwargs ): # default for stdout/err is Tac.INHERIT
   # Tac.run has only 1 mandatory args; args[3]=stdout; args[4]=stderr
   # if no stdout/err provided, pass one explicitely to make sure we
   # ultimately use the correct fd (instead of some c code using fd 1 as default).
   assert len( args ) == 1, ( 'Tac.run within ConfigAgent doesn\'t take '
                              'more than 1 args. Please use kwargs instead. '
                              'Contact cli-dev@ for more info' )
   kwargs.setdefault( 'env', os.environ )
   kwargs.setdefault( 'stdout', sys.stdout )
   kwargs.setdefault( 'stderr', sys.stderr )
   try:
      return origTacRun( *args, **kwargs )
   except Tac.SystemCommandError as e:
      # Later we might have to silence the error if the cli cmd was going through a
      # pipe, unless the Tac.run itself was not outputing to the pipe, in which case
      # it is a real error, so protocol the output state into the exception for
      # later determination of its gravity.
      if kwargs.get( 'stdout' ) is not sys.stdout:
         e.outputRedirected = True
      raise e

def init():
   global origTacRun
   if origTacRun is not None:
      # some tests call init over and over and get into infinite recursion
      # because origTacRun = newTacRun by then.
      return

   TacUtils.SystemCommandError = SystemCommandError

   assert TacUtils.run == Tac.run
   origTacRun = TacUtils.run
   TacUtils.run = newTacRun
   Tac.run = newTacRun
