# Copyright (c) 2006-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

"""
Legacy network equipment allows admins to configure per-"line"
passwords, or accounts with usernames and passwds (and privilege
levels, etc.).

In the world of SSH sessions, configuring "lines" doesn't make much
sense, so we support only the username/password method with locally
configured passwords. We can fancy this up with RADIUS, TACACS, etc.
later.

We use industry-standard CLI:

[no] enable secret { [0] <cleartext-passwd> | 5 <encrypted-passwd> }
[no] username <name> [privilege <0-15>] secret { [0] <cleartext-passwd> |
   5 <encrypted-passwd> }

Note that cleartext passwds never leave the AaaCli module -- only
encrypted passwds are written into the TAC config objects.
"""

import Tac, Tracing, Intf.Log
import AaaCliLib
from AaaCliLib import aaaKwMatcher, groupKwMatcher
from AaaDefs import ( acctLogFileName, authzBuiltinRoles, roleNameRe,
                      roleRegexRe, SEQ_INC, MAX_SEQ, MAX_ROLES )
import Cell
import CliPlugin.TechSupportCli
from CliPlugin import AaaModel
import AaaPluginLib
import BasicCli
import BasicCliUtil
import CliAaa
import CliCommand
import CliMatcher
import CliParser
import CliToken.Clear
import ConfigMount
import LazyMount
import LocalUserLib
import MultiRangeRule
from CliMode.Aaa import RoleMode
from FileCliUtil import checkUrl
import SecretCli
from Url import UrlMatcher
import TimeRangeRule
import UtmpDump
import datetime
import errno
import os
import re
import subprocess
import sys
import tempfile
import time
import collections
from PasswordPolicyLib import applyMinChangedCharactersPolicy, \
                              applyDenyUsernamePolicy, \
                              applyDenyLastPolicy, clearPasswordHistory, \
                              clearEnablePasswordHistory
import hashlib
from AaaPluginLib import primarySshKeyId

aaaStatus = None
localConfig = None
acctLogFile = None
userLockoutConfig = None
accountsConfig = None
securityConfig = None
mgmtSshConfig = None

traceHandle_ = Tracing.Handle( "AaaCli" )
debug = traceHandle_.trace0

def _getAaaStatus( mode ):
   return aaaStatus

def _getLocalUserConfig( mode ):
   return localConfig

def _getSshConfig():
   return mgmtSshConfig

configMode = BasicCli.GlobalConfigMode

def _maybeCleanupSshUser( user, sshConfig ):
   '''
      Clear the SSH user ONLY if Aaa added it and there is no more SSH configuration
      present. If the configuration was added via "management ssh" then we will have
      the flag userModeEntered set for the user.
   '''
   userConfig = sshConfig.user.get( user )
   if userConfig:
      if not ( userConfig.userModeEntered or userConfig.sshAuthKeys
            or userConfig.sshAuthPrincipals ):
         del sshConfig.user[ user ]

def showSessions( mode, args ):
   status = _getAaaStatus( mode )
   ret1 = AaaModel.ShowAaaSessions()
   for sid in sorted( status.session ):
      session = status.session.get( sid )
      if not session or session.state == "uninitialized":
         # not fully setup, skip
         continue

      ret = ret1.Sessions()
      # for purpose of this command, treat "deleted" the same as "established"
      state = "pending" if session.state == 'pending' else 'established'

      sessionStartTime = int( session.startTime + Tac.utcNow() - Tac.now() )

      remoteUser = session.remoteUser
      if remoteUser:
         ret.remoteHost = remoteUser

      sessionData = session.property.get( session.authenMethod )
      roles = '<unknown>'
      if sessionData:
         if ( sessionData.attr.get( AaaPluginLib.secureMonitor, False ) and
              not mode.session.secureMonitor() ):
            continue
         if sessionData.attr.get( 'roles' ):
            attr = sessionData.attr.get( 'roles' )
            roles = eval( attr ) # pylint: disable=eval-used
            assert isinstance( roles, list )
            roles = ','.join( roles )

      ret.username = session.userName
      ret.role = roles
      ret.state = state
      ret.sessionStartTime = sessionStartTime
      ret.authMethod = session.authenMethod
      if session.remoteHost:
         ret.remoteAddress = session.remoteHost
      if session.tty:
         ret.terminal = session.tty
      if session.service:
         ret.service = session.service

      if AaaPluginLib.realTtyNameRe.match( session.tty ) or session.tty == 'ssh':
         if session.tty.startswith( 'con' ):
            ret1.serials[ session.id ] = ret
         else:
            ret1.vtys[ session.id ] = ret
      else:
         ret1.nonInteractives[ session.id ] = ret
   return ret1

principalKwMatcher = CliMatcher.KeywordMatcher( 'principal',
      helpdesc='Configure SSH principals for the user' )
roleMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode : _getLocalUserConfig( mode ).role, 'Role name',
      pattern=roleNameRe, helpname='WORD' )
sshKwMatcher = CliMatcher.KeywordMatcher( 'ssh',
      helpdesc='Configure SSH parameters for the user' )
usernameKwMatcher = CliMatcher.KeywordMatcher( 'username',
      helpdesc='Set up a user account' )
usernameMatcher = CliMatcher.PatternMatcher( LocalUserLib.usernameRe,
      # usernames consist of 1 or more lower-case letters or digits
      helpname='WORD',
      helpdesc='Account name string' )

#-------------------------------------------------------------------------------
# "show aaa accounting logs"
#-------------------------------------------------------------------------------
def showAccountLog( mode, args ):
   if not acctLogFile:
      mode.addError( f"Account log file at {acctLogFileName} does not exist." )
      return

   header = "{:<20} {:<8} {:<11} {:<15} Action Attributes"
   line = "{} {} {} {} {} ".format( '-' * 20,
                                    '-' * 8,
                                    '-' * 11,
                                    '-' * 15,
                                    '-' * len( 'Action' ) )
   heading = ''
   heading += header.format( "Time",
                           "User",
                           'TTY',
                           "Remote Addr" ) + "\n"
   heading += line.ljust( 80, '-' )

   # Get log
   logFileName = acctLogFileName.split( '/' )
   logdir = ( '/' ).join( logFileName[ :-1 ] )
   logname = logFileName[ -1 ]
   fetchLogs = [ 'FetchLogs', '-l', logdir, '-m', logname, '-n', 'dump' ]
   curProc = subprocess.Popen( fetchLogs, # pylint: disable=consider-using-with
                               stdout=subprocess.PIPE,
                               stderr=subprocess.PIPE,
                               universal_newlines=True )
   showLogOptions = []
   # The order of the two filters does not matter, so filter by user first,
   # since it's probably faster
   for k in ( 'USER', 'TIME' ):
      if k in args:
         showLogOptions.append( ( k, args[ k ] ) )

   for name, value in showLogOptions:
      if name == 'TIME':
         # Handle 'show aaa accounting logs time-range <begin> <end>'
         # <begin>,<end> is a tuple ((month,day,year),(hour,min,sec))
         beginDate = value[ 0 ][ 0 ]
         beginTime = value[ 0 ][ 1 ]
         endDate = value[ 1 ][ 0 ]
         endTime = value[ 1 ][ 1 ]

         # Convert them to string arguments for time-filter
         # By not converting the time to datetime objects here but in
         # time-filter, we allow human readable inputs
         # for the script as well
         beginRange = TimeRangeRule.datetimeToStr( mode, beginDate, beginTime )
         endRange = TimeRangeRule.datetimeToStr( mode, endDate, endTime )

         if beginRange == "" or endRange == "":
            return
         args = [ "time-filter", "-q", "-b", beginRange, "-e", endRange ]
      elif name == 'USER':
         # value is a list due to Iteration
         value = value[ 0 ]
         # username is the 5th element in log: "YYYY MM DD hh:mm:ss username etc"
         # If time-filter ran first and produced an error, keep the error message.
         # Use %r to escape the
         # username matches LocalUserLib.usernameRe, which doesn't allow single
         # or double quotes, so replacing single quote of %r with double quotes for
         # awk is fine.
         assert "'" not in value and '"' not in value
         awkArg = f'$5 == {repr(value)} || $1 == "Error:"'
         args = [ 'awk', awkArg.replace( '\'', '"' ) ]
      else:
         assert False

      nxtProc = subprocess.Popen( args, # pylint: disable=consider-using-with
                                  stdin=curProc.stdout,
                                  stdout=subprocess.PIPE,
                                  stderr=subprocess.PIPE,
                                  universal_newlines=True )
      curProc.stdout.close()
      curProc = nxtProc

   output, err = curProc.communicate()
   if err:
      mode.addError( err )
      return
   print( heading )
   if output:
      print( output.strip( "\n" ) )

#-------------------------------------------------------------------------------
# "show aaa counters" in enable mode
#-------------------------------------------------------------------------------
def showCounters( mode, args ):
   status = _getAaaStatus( mode )
   ret = AaaModel.AaaCounters()
   ret.authenticationSuccess = status.counters.authnSuccess
   ret.authenticationFail = status.counters.authnFail
   ret.authenticationUnavailable = status.counters.authnUnavailable
   ret.authorizationAllowed = status.counters.authzAllowed
   ret.authorizationDenied = status.counters.authzDenied
   ret.authorizationUnavailable = status.counters.authzUnavailable
   ret.accountingSuccess = status.counters.acctSuccess
   ret.accountingError = status.counters.acctError
   ret.pendingAccountingRequests = status.counters.numPendingAcctRequests
   if status.counters.lastClearTime:
      ret.counterResetTimestamp = ( status.counters.lastClearTime +
                                    Tac.utcNow() - Tac.now() )

   return ret

#-------------------------------------------------------------------------------
# "clear aaa counters" in enable mode
#-------------------------------------------------------------------------------
aaaAfterClearMatcher = CliMatcher.KeywordMatcher(
   'aaa',
   helpdesc="Clear AAA information" )
aaaCounterMatcher = CliMatcher.KeywordMatcher( 'counters',
                                               helpdesc="Clear AAA counters" )

def clearCounters( mode ):
   counterCfg = AaaCliLib.counterConfigAaa( mode )
   status = _getAaaStatus( mode )
   Intf.Log.logClearCounters( "Aaa" )
   counterCfg.clearCounterRequestTime = Tac.now()
   try:
      Tac.waitFor(
         lambda:
            status.counters.lastClearTime >= counterCfg.clearCounterRequestTime,
         description='Aaa clear counter request to complete',
         warnAfter=None, sleep=True, maxDelay=0.5, timeout=5 )
   except Tac.Timeout:
      mode.addWarning( "Aaa counters may not have been reset yet" )

class ClearAaaCounterCommand( CliCommand.CliCommandClass ):
   syntax = "clear aaa counters"
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'aaa' : aaaAfterClearMatcher,
      'counters' : aaaCounterMatcher,
      }
   # We set syncAcct=True for 'clear aaa counters' to make sure
   # any pending items in accounting queue are accounted and then
   # clear the counters.
   syncAcct = True
   @staticmethod
   def handler( mode, args ):
      clearCounters( mode )

BasicCli.EnableMode.addCommandClass( ClearAaaCounterCommand )

######### Code for Aaa's Cli AaaProvider #########
#
# Cli needs to authenticate the enable command and, perform authorization
# and accounting of commands before they are run. The code below is how
# Cli delegates to Aaa to perform these authn/authz/acct functions.
def handleAuthenMessage( mode, msg, username=None, password=None ):
   if msg.style == 'promptEchoOn':
      if username:
         return username
      mode.addMessage( msg.text )
      return sys.stdin.readline()[0:-1] # return entered text minus newline
   elif msg.style == 'promptEchoOff':
      if password:
         return password
      try:
         import getpass # pylint: disable=import-outside-toplevel
         return getpass.getpass( msg.text )
      except EOFError:
         # interrupted
         mode.addMessage( '' )
   elif msg.style == 'info':
      mode.addMessage( "%s" % ( msg.text ) )
   elif msg.style == 'error':
      if username or password:
         return None
      mode.addError( msg.text )
   else:
      raise Exception( "Unsupported message style: %s" % ( msg.style ) )
   return None

class AaaProvider( CliAaa.ProviderBase ):
   name = "Aaa"
   def __init__( self ):
      CliAaa.ProviderBase.__init__( self )

   def pyClient( self, mode ):
      sysname = mode.sysname
      pc = mode.session.sessionData( 'aaaPyClient' )
      if pc:
         return pc
      # create the PyClient
      import PyClient # pylint: disable=import-outside-toplevel
      retries = 0
      maxRetries = 5
      pc = None
      while retries < maxRetries:
         try:
            debug( "Connecting to Aaa" )
            pc = PyClient.PyClient(
               sysname, 'Aaa', reconnect=True,
               execMode=PyClient.Rpc.execModeThreadPerConnection,
               initConnectCmd="import AaaApi" )
            break
         except OSError as e:
            # Rarely when we try to connect to Aaa socket we get EWOULDBLOCK
            # which we handle by retrying the PyClient connection.
            # Please see BUG 107253 for more details.
            if e.errno == errno.EWOULDBLOCK and retries < maxRetries:
               retries += 1
               time.sleep( 1 )
            else:
               debug( "Exception occurred during connection" )
               raise
      mode.session.sessionDataIs( 'aaaPyClient', pc )
      return pc

   def processMessages( self, mode, s, username=None, password=None ):
      # This is pretty ugly, but tacc doesn't support collections in value
      # types, so I ended up with this message0..3 garbage.
      start = Tac.now()
      answers = []
      for i in range( 0, 4 ):
         attr = "message%d" % ( i )
         m = getattr( s, attr )
         if m.style == 'invalid':
            break
         # Impose a timeout, so we do not cause timeout on TACACS+ server
         # and fallback
         resp = handleAuthenMessage( mode, m, username, password )
         if resp is not None:
            answers.append( resp )
      if ( Tac.now() - start >=
           int( os.environ.get( "ENABLE_PASSWORD_TIMEOUT", "60" ) ) ):
         mode.addError( "Timeout" )
         raise CliParser.AlreadyHandledError()
      return answers

   def geteuid( self, mode ):
      debug( 'geteuid', mode.session_.aaaUser_ )
      if mode.session_.aaaUser_:
         return mode.session_.aaaUser_.uid
      return None

   def getSessionId( self, mode ):
      debug( 'getSessionId', mode.session_.aaaUser_ )
      if mode.session_.aaaUser_:
         return mode.session_.aaaUser_.sessionId
      return None

   def authenticateEnable( self, mode, privLevel ):
      r = False

      uid = self.geteuid( mode )
      sessionId = self.getSessionId( mode )
      try:
         service = '' if sessionId else 'Cli'
         debug( "RPC to Aaa: startEnableAuthenticate(service=",service,", uid=", uid,
                ", privLevel=", privLevel, ")" )
         pc = self.pyClient( mode )
         cmd = "AaaApi.startEnableAuthenticate(service=%r, uid=%r, " \
            "privLevel=%r, sessionId=%r)" % ( service, uid, privLevel, sessionId )
         result = pc.eval( cmd )
         s = Tac.strepToValue( result )

         # pylint: disable-msg=E1103
         while s.status == 'inProgress':
            answers = self.processMessages( mode, s )
            cmd = "AaaApi.continueAuthenticate(%d" % ( s.id )
            for a in answers:
               cmd += ", " + repr( a )
            cmd += ")"
            s = Tac.strepToValue( pc.eval( cmd ) )

         # Last status response might have some messages to display
         self.processMessages( mode, s )
         debug( "Final RPC status is", s.status )
         if s.status == 'success':
            r = True
      except:
         debug( "Exception occurred during authentication" )
         raise
      return r

   def authenticateTest( self, mode, groupName, userName, password ):
      r = False

      try:
         pc = self.pyClient( mode )
         cmd = "AaaApi.startLoginAuthenticate(service='Cli', method='"
         cmd += groupName
         cmd += "')"
         result = pc.eval( cmd )
         s = Tac.strepToValue( result )

         # pylint: disable-msg=E1103
         while s.status == 'inProgress':
            answers = self.processMessages( mode, s, userName, password )
            cmd = "AaaApi.continueAuthenticate(%d" % ( s.id )
            for a in answers:
               cmd += ", " + repr( a )
            cmd += ", groupName='"
            cmd += groupName
            cmd += "')"
            s = Tac.strepToValue( pc.eval( cmd ) )

         if s.status == 'success':
            print( "User was successfully authenticated." )
         else:
            print( "Authentication failed" )
         # Last status response might have some messages to display
         self.processMessages( mode, s, userName, password )
         debug( "Final RPC status is", s.status )
         if s.status == 'success':
            r = True
      except:
         debug( "Exception occurred during authentication" )
         raise
      return r

   # _skipCmdAuthz and _skipCmdAcct are optimizations to avoid
   # talking to Aaa when command authorization/accounting is not
   # enabled.
   def _skipCmdAuthz( self, mode, privLevel ):
      # restricted shell should always do Aaa
      if mode.session_.isStandalone():
         return False
      # we should not access the config mount; instead use the actual Sysdb object
      cfg = AaaCliLib.configAaa( mode ).sysdbObj()
      name = "command%02d" % privLevel
      return cfg.skipCmdAuthz( name, isinstance( mode, BasicCli.ConfigModeBase ) )

   def _skipCmdAcct( self, mode, privLevel, secureUser ):
      # restricted shell should always do Aaa
      if mode.session_.isStandalone():
         return False
      # we should not access the config mount; instead use the actual Sysdb object
      cfg = AaaCliLib.configAaa( mode ).sysdbObj()
      if secureUser and not cfg.acctCmdSecureMonitorEnabled:
         # Skip cmd accounting for secure user
         return True
      name = "command%02d" % privLevel
      status = AaaCliLib.statusAaa( mode )
      # No traditional Aaa accounting nor any extra accounting methods
      return cfg.skipCmdAcct( name ) and not status.extraAcctMethods

   def authorizeCommand( self, mode, privLevel, tokens ):
      if self._skipCmdAuthz( mode, privLevel ):
         return ( True, "" )
      authorized = False
      message = ""

      uid = self.geteuid( mode )

      modeKey = mode.modeKey if hasattr( mode, 'modeKey' ) else None
      longModeKey = mode.longModeKey if hasattr( mode, 'longModeKey' ) else None
      m = ( mode.name, modeKey, longModeKey )

      sessionId = self.getSessionId( mode )
      try:
         debug( "RPC to Aaa: authorizeShellCommandEx(", uid, ",", m, ",", privLevel,
                ",", tokens, ",", sessionId, ")" )
         pc = self.pyClient( mode )
         cmd = "AaaApi.authorizeShellCommandEx( uid=%r, mode=%r," \
             "privLevel=%r, tokens=%r, sessionId=%r, cmdType='cli' )" % (
            uid, m, privLevel, tokens, sessionId )
         result = pc.eval( cmd ).split('\x00')
         assert len(result) == 2
         status = result[0]
         message = result[1]
         debug( "RPC result is", result )
         # pylint: disable-msg=E1103
         if status == 'allowed':
            authorized = True
      except:
         debug( "Exception occurred during authorization" )
         raise
      return ( authorized, message )

   def getSession( self, mode ):
      sessionId = self.getSessionId( mode )
      if sessionId is None:
         return None
      status = _getAaaStatus( mode )
      return status.session.get( sessionId )

   def getTty( self, session ):
      if session and session.tty:
         return session.tty
      return ''

   def getRemoteAddr( self, session ):
      if session and session.remoteHost:
         return session.remoteHost
      return ''

   def getUsername( self, mode ):
      uid = self.geteuid( mode )
      if uid is not None:
         status = _getAaaStatus( mode )
         acct = status.userid.get( int( uid ) )
         if acct:
            return acct.userName
      return ''

   def logAccountCommand( self, mode, privLevel, tokens, secureUser ):
      if secureUser:
         # do not log commands typed by secure-monitor operators
         return

      aaaLogInfo = mode.session.sessionData( 'aaaLogInfo' )
      if not aaaLogInfo:
         session = self.getSession( mode )
         username = self.getUsername( mode )
         tty = self.getTty( session )
         remoteAddr = self.getRemoteAddr( session )
         # This part of the log won't change, so calculate it now.
         aaaLogInfo = "{:8} {:<11} {:<15} {:<6} service=shell".format( username,
                                                                       tty,
                                                                       remoteAddr,
                                                                       "stop" )
         mode.session.sessionDataIs( 'aaaLogInfo', aaaLogInfo )

      # log entry format:
      # ( Time stamp, User, TTY, Remote Addr, action, service, priv-level, cmd )
      # 2016 Apr 20 20:56:07 arastra tty1 172.17.18.47 stop
      # service=shell priv-lvl=15 cmd=show running-config <cr>
      timestamp = datetime.datetime.now().strftime( '%Y %b %d %H:%M:%S' )
      cmd = ' '.join( tokens )
      # Use widths of 20 for timestamp, 8 for user, 11 for possible 'command-api'
      # tty, 15 for remote IP addr, 6 for length of header "Action"
      log = '{:<20} {} priv-lvl={} cmd={}\n'
      logline = log.format( timestamp,
                            aaaLogInfo,
                            privLevel,
                            cmd )
      try:
         acctLogFile.write( logline )
         acctLogFile.flush()
      except OSError:
         pass

   def sendCommandAcct( self, mode, privLevel, tokens, waitTime=0 ):
      secureUser = mode.session.secureMonitor()

      if acctLogFile:
         self.logAccountCommand( mode, privLevel, tokens, secureUser )

      if self._skipCmdAcct( mode, privLevel, secureUser ):
         return

      uid = self.geteuid( mode )
      sessionId = self.getSessionId( mode )
      try:
         debug( "RPC to Aaa: sendCommandAcct(", uid, ",", privLevel,
                ",", tokens, ",", sessionId, ", cmdType='cli' )" )
         pc = self.pyClient( mode )
         cmd = "AaaApi.sendCommandAcct( uid=%r, " \
               "privLevel=%r, tokens=%r, sessionId=%r, cmdType='cli' )" % (
            uid, privLevel, tokens, sessionId )
         output = pc.eval( cmd )
         debug( "RPC output = ", output )
      except:
         debug( "Exception occurred during accounting", sys.exc_info()[1] )
         raise

   def flushAcctQueue( self, mode, waitTime ):
      AaaPluginLib.flushAcctQueue( mode.sysname, waitTime )

   def authenSessionData( self, mode ):
      status = _getAaaStatus( mode )
      sid = self.getSessionId( mode )
      if sid is not None:
         session = status.session.get( sid )
         if session:
            data = session.property.get( session.authenMethod, {} )
            if data:
               return data.attr
      return {}

CliAaa.registerAaaProvider( AaaProvider() )

######### "enable ..." config command #########
#
# Configure the password to get into 'enable' mode.
#
# Implemented
# -----------
#
# [no] enable password { [0] <cleartext-passwd> | 5 <encrypted-passwd> }

secretDeprecated = CliCommand.Node(
   CliMatcher.KeywordMatcher( 'secret',
                              helpdesc='Assign the enable password' ),
   deprecatedByCmd='enable password' )
def getPasswordPolicy():
   return securityConfig.passwordPolicies.get( accountsConfig.passwordPolicy )

class EnablePasswordCommand( CliCommand.CliCommandClass ):
   syntax = "enable password | secretDeprecated SECRET"
   noOrDefaultSyntax = "enable password ..."
   data = { 'enable' : 'Enable-privilege related configuration',
            'password' : 'Assign the enable password',
            'secretDeprecated' : secretDeprecated,
            'SECRET' : SecretCli.secretCliExpression( 'encryptedPasswd',
                                                      policyFn=getPasswordPolicy) }
   allowCache = False

   @staticmethod
   def handler( mode, args ):
      cfg = _getLocalUserConfig( mode )
      policy = getPasswordPolicy()
      secretValue = args.get( 'encryptedPasswd' )
      debug( "enable pasword handler" )
      applyDenyLastPolicy( mode, username=None, policy=policy, 
                           secretValue=secretValue, enable=True )
      cfg.encryptedEnablePasswd = args[ 'encryptedPasswd' ].hash()

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = _getLocalUserConfig( mode )
      cfg.encryptedEnablePasswd = ""
      clearEnablePasswordHistory()

configMode.addCommandClass( EnablePasswordCommand )

sshkeyDeprecated = CliCommand.Node(
   CliMatcher.KeywordMatcher(
      'sshkey',
      helpdesc='Configure an SSH public key for the user' ),
   deprecatedByCmd='username USERNAME ssh-key' )

sshkeyKwMatcher = CliMatcher.KeywordMatcher(
   'ssh-key',
   helpdesc='Configure an SSH public key for the user' )

def getKeysFromFile( mode, url ):
   try:
      checkUrl( url )
      authKeyFile = url.open()
      authKeys = [ authKey.rstrip( "\n" ) for authKey in authKeyFile ]
      authKeyFile.close()
      return authKeys
   except OSError as e:
      mode.addError( "Error reading keys from %s (%s)" %
                     ( url.url, e ) )
      raise CliParser.AlreadyHandledError()

sshkeyContentMatcher = CliCommand.Node(
            CliMatcher.StringMatcher( helpname='SSHKEY',
                                      helpdesc='SSH key string' ),
            sensitive=True )

class SshKeyExpression( CliCommand.CliExpression ):
   expression = "( file SSHKEY_URL ) | SSHKEY_CONTENT"
   data = { 'file' : 'Path to file containing an SSH public key',
            'SSHKEY_URL' : UrlMatcher(
               lambda fs: fs.realFileSystem() and fs.supportsRead(), 
               'File containing SSH public key' ),
            'SSHKEY_CONTENT' : sshkeyContentMatcher
   }

def addAuthorizedKey( mode, username, key, secondary=False ):
   # Write it to a tmp file and verify before continuing
   try:
      with tempfile.NamedTemporaryFile( mode='w' ) as f:
         f.write( key + '\n' )
         f.flush()
         try:
            Tac.run( [ '/usr/bin/ssh-keygen', '-l', '-f', f.name ],
                     stdout=Tac.DISCARD, stderr=Tac.DISCARD )
         except Tac.SystemCommandError:
            mode.addError( "Unrecognized ssh key" )
            return
   except OSError:
      mode.addError( "can't write to filesystem" )
      return

   cfg = _getLocalUserConfig( mode )

   # Both normal user and root user will have their keys stored in the
   # sshConfig user collection
   if username != 'root' and username not in cfg.acct:
      # create a default user with no login
      cfg.newAcct( username, '*' )

   sshConfig = _getSshConfig()

   # Create/fetch user configuration
   userConfig = sshConfig.newUser( username )

   # Name the primary key as "pri-key" and the rest as a function of keyContents
   # which is explained below
   if not secondary:
      userConfig.sshAuthKeys.newMember( primarySshKeyId ).keyContents = key

      if username != 'root':
         userConfig.publishedSshAuthKey = key
   else:
      # Multiple secondary keys can be specified, to uniquely
      # and deterministically identify them, they are named as
      # sec-key-<First 20 characters of sha256 hash of key>. Hopefully,
      # there is no conflict.
      keyName = f"sec-key-{hashlib.sha256(key.encode()).hexdigest()[:20]}"
      userConfig.sshAuthKeys.newMember( f"{keyName}" ).keyContents = key

def addAuthorizedKeysFromFile( mode, username, keys ):
   # Use the first key in file as the primary key
   for i, key in enumerate( keys ):
      addAuthorizedKey( mode, username, key, secondary=( i > 0 ) )

def removeSecondaryAuthorizedKey( mode, username, key ):
   cfg = _getLocalUserConfig( mode )
   sshConfig = _getSshConfig()

   def _deleteKey():
      # A key can be removed with prefix if there's unique match
      if username in sshConfig.user:
         candidateKeys = []
         for n, k in sshConfig.user[ username ].sshAuthKeys.items():
            if k.keyContents.startswith( key ):
               candidateKeys.append( n )
         if not candidateKeys:
            mode.addErrorAndStop( "SSH key not present" )
         elif len( candidateKeys ) > 1:
            mode.addErrorAndStop( "Ambiguous SSH key" )
         del sshConfig.user[ username ].sshAuthKeys[ candidateKeys[ 0 ] ]
         _maybeCleanupSshUser( username, sshConfig )

   if username in cfg.acct or username == 'root':
      _deleteKey()

######### "root ..." config command #########
#
# Configure the password for root.
#
# Implemented
# -----------
#
# aaa root secret { [0] <cleartext-passwd> | 5 <encrypted-passwd> }
# aaa root secret nopasswd
# no|default aaa root

class RootCommand( CliCommand.CliCommandClass ):
   syntax = """aaa root 
               nopassword 
               | ( secret SECRET ) 
               | ( ( ssh-key | sshkey ) SSHKEY )
               | ( ( ssh-key | sshkey ) secondary SSHKEY_CONTENT )"""
   noOrDefaultSyntax = """aaa root [ secret | 
                          ( ( ssh-key | sshkey )
                          [ secondary SSHKEY_CONTENT ] ) ] ..."""
   data = { 'aaa' : aaaKwMatcher,
            'root' : 'Modify root login attributes',
            'secret' : 'Assign the root password',
            'nopassword' : 'Allow root login with no password',
            'SECRET' : SecretCli.secretCliExpression( 'encryptedPasswd',
                                                      supportInvalidPassword=True
            ),
            'ssh-key' : sshkeyKwMatcher,
            'sshkey' : sshkeyDeprecated,
            'secondary' : 'Secondary root SSH keys',
            'SSHKEY' : SshKeyExpression,
            'SSHKEY_CONTENT' : sshkeyContentMatcher,
            }
   allowCache = False

   @staticmethod
   def handler( mode, args ):
      cfg = _getLocalUserConfig( mode )
      if 'nopassword' in args:
         debug( "Updating root password to default." )
         cfg.encryptedRootPasswd = ""
         mode.addWarning( "Root can only login through console without " \
                          "password." )
      elif 'secret' in args:
         cfg.encryptedRootPasswd = args[ 'encryptedPasswd' ].hash()
      else:
         filename = args.pop( 'SSHKEY_URL', None )
         if filename:
            keys = getKeysFromFile( mode, filename )
            addAuthorizedKeysFromFile( mode, 'root', keys )
         else:
            key = args[ 'SSHKEY_CONTENT' ]
            secondary = 'secondary' in args
            addAuthorizedKey( mode, 'root', key, secondary=secondary )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      def _clearRootSshKeys( clearAll=True ):
         sshConfig = _getSshConfig()
         if 'root' in sshConfig.user:
            if clearAll:
               sshConfig.user[ 'root' ].sshAuthKeys.clear()
            else:
               del sshConfig.user[ 'root' ].sshAuthKeys[ primarySshKeyId ]
            _maybeCleanupSshUser( 'root', sshConfig )

      cfg = _getLocalUserConfig( mode )
      # If nothing is specified, reset everything.
      if 'secret' not in args and 'sshkey' not in args and 'ssh-key' not in args:
         cfg.encryptedRootPasswd = '*'
         _clearRootSshKeys()
      else:
         if 'secret' in args:
            cfg.encryptedRootPasswd = '*'
         if 'secondary' in args:
            key = args[ 'SSHKEY_CONTENT' ]
            removeSecondaryAuthorizedKey( mode, 'root', key )
         elif 'sshkey' in args or 'ssh-key' in args:
            _clearRootSshKeys( clearAll=False )

configMode.addCommandClass( RootCommand )

######### [no] aaa authorization console ##########
def aaaAuthzSerialConsole( mode, no=False ):
   cfg = AaaCliLib.configAaa( mode )
   cfg.consoleAuthz = not no

authzKwMatcher = CliMatcher.KeywordMatcher( 'authorization',
   helpdesc='Configure authorization parameters' )

consoleDeprecated = CliCommand.Node(
   CliMatcher.KeywordMatcher( 'console',
                              helpdesc='Enable console authorization' ),
   deprecatedByCmd='aaa authorization serial-console' )

class AaaAuthzSerialConsoleCommand( CliCommand.CliCommandClass ):
   syntax = "aaa authorization serial-console | consoleDeprecated"
   noOrDefaultSyntax = syntax
   data = { 'aaa' : aaaKwMatcher,
            'authorization' : authzKwMatcher,
            'serial-console' : 'Enable console authorization',
            'consoleDeprecated' : consoleDeprecated
            }

   @staticmethod
   def handler( mode, args ):
      aaaAuthzSerialConsole( mode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      aaaAuthzSerialConsole( mode, no=True )

configMode.addCommandClass( AaaAuthzSerialConsoleCommand )

######### [no] aaa authorization policy local default-role ... ##########

class AaaAuthzPolicyDefaultRoleCommand( CliCommand.CliCommandClass ):
   syntax = 'aaa authorization policy local default-role ROLE'
   noOrDefaultSyntax = 'aaa authorization policy local default-role ...'
   data = {
      'aaa' : aaaKwMatcher,
      'authorization' : authzKwMatcher,
      'policy' : 'Set authorization policy',
      'local' : 'Configure policy for local authorization',
      'default-role' : 'Set the default role for users without a role',
      'ROLE' : roleMatcher
   }

   @staticmethod
   def handler( mode, args ):
      cfg = _getLocalUserConfig( mode )
      cfg.defaultRole = args.get( 'ROLE', cfg.defaultRoleDefault )

   noOrDefaultHandler = handler

configMode.addCommandClass( AaaAuthzPolicyDefaultRoleCommand )

######### "role ..." config command #########
#
# Implemented
# -----------
#
# role <name>
#    [ [no] | [seq]] <permit | deny> [mode mode-key] command <regex>
#    resequence <start> <inc>
#    <abort | exit>
#    show active
#
# no role <name>
class RoleContext:
   def __init__( self, config, role ):
      self.name = role
      self.role = Tac.newInstance( 'LocalUser::Role', role )
      self.config = config
      # Copy the old role if it exists
      if role in self.config.role:
         for seq, rule in self.config.role[ role ].rule.items():
            self.role.rule[ seq ] = Tac.newInstance( 'LocalUser::Rule',
                                                     rule.action,
                                                     rule.regex, rule.modeKey )

   def roleName( self ):
      return self.name

   def lastSeq( self ):
      lastSeq = 0
      for s in self.role.rule:
         lastSeq = s
      return lastSeq

   def checkBuiltinModified( self ):
      """ Check if any built-in role is being modified """
      if self.name in authzBuiltinRoles:
         oldRules = authzBuiltinRoles[ self.name ]
         if len( self.role.rule ) != len( oldRules ):
            return True
         else:
            for t, oldRule in zip( self.role.rule.items(), oldRules ):
               newRule = ( t[ 0 ], str( t[ 1 ].action ),
                           t[ 1 ].regex, t[ 1 ].modeKey )
               if newRule != oldRule:
                  return True
      return False

   def _syncRole( self, oldRole, newRole ):
      """Sync new role to the old role"""
      for k in oldRole.rule:
         if k not in newRole.rule:
            del oldRole.rule[ k ]
      for k, v in newRole.rule.items():
         oldRole.rule[ k ] = v

   def commit( self, mode ):
      """ Commit current role. Delete the old role if it exists. """
      # Report error if a built-in role is being modified
      if self.checkBuiltinModified():
         mode.addError( 'Built-in role %s cannot be changed' % self.name )
         return
      role = self.config.role.newMember( self.name )
      self._syncRole( role, self.role )

   def addRule( self, mode, seq, action, regex, modeKey ):
      def verifyRegex( exp, msg ):
         try:
            assert re.compile( exp )
         except Exception:  # pylint: disable-msg=W0703
            mode.addError( 'Invalid ' + msg + ': ' + exp )
            return False
         return True
      # Validate the regular expressions
      if not ( verifyRegex( regex, 'regular expression' ) and
               verifyRegex( modeKey, 'mode name or key' ) ):
         return
      # Warn on a duplicte rule
      for seq0, rule in self.role.rule.items():
         if rule.action == action and rule.regex == regex and \
               rule.modeKey == modeKey:
            mode.addWarning( 'Rule (#%d) already exists' % seq0 )
            return
      # Allocate a new sequence number
      if seq is None:
         seq = self.lastSeq() + SEQ_INC
      # Check sequence number against the limit
      if seq > MAX_SEQ:
         mode.addError( 'Sequence number out of range' )
         return
      self.role.rule[ seq ] = Tac.newInstance( 'LocalUser::Rule',
                                               action, regex, modeKey )

   def removeRule( self, action, regex, modeKey ):
      for seq in self.role.rule:
         rule = self.role.rule[ seq ]
         if rule.action == action and rule.regex == regex and \
               rule.modeKey == modeKey:
            del self.role.rule[ seq ]

   def removeRuleBySeq( self, seq ):
      del self.role.rule[ seq ]

   def resequence( self, mode, start, inc ):
      seq0 = int( start ) + int( inc ) * ( len( self.role.rule ) - 1 )
      if seq0 > MAX_SEQ:
         mode.addError( 'Sequence number out of range' )
         return
      # Create a new role and replace the old one
      role = Tac.newInstance( 'LocalUser::Role', self.name )
      for rule in self.role.rule.values():
         role.rule[ start ] = Tac.newInstance( 'LocalUser::Rule', rule.action,
               rule.regex, rule.modeKey )
         start += inc
      self.role = role

class RoleConfigMode( RoleMode, BasicCli.ConfigModeBase ):
   name = "Role Configuration"

   def __init__( self, parent, session, context, roleName ):
      self.context = context
      RoleMode.__init__( self, roleName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      if self.context is None:
         debug( 'RoleConfigMode has no context' )
         return
      else:
         debug( 'RoleConfigMode onExit...' )

      self.context.commit( self )
      self.context = None
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      self.context = None
      self.session_.gotoParentMode()

class GotoRoleModeCommand( CliCommand.CliCommandClass ):
   syntax = "role <ROLENAME>"
   noOrDefaultSyntax = syntax
   data = {
      "role" : "Role",
      "<ROLENAME>" : roleMatcher
   }

   @staticmethod
   def handler( mode, args ):
      role = args[ '<ROLENAME>' ]
      aaaConfig = _getLocalUserConfig( mode )
      numRoles = len( aaaConfig.role )

      # Limit maximum number of roles
      if role not in aaaConfig.role and numRoles >= MAX_ROLES:
         msg = 'Maximum number (%d) of roles have been configured' % numRoles
         mode.addError( msg )
         return

      context = RoleContext( aaaConfig, role )
      childMode = mode.childMode( RoleConfigMode, context=context,
                                  roleName=role )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      role = args[ '<ROLENAME>' ]
      # Do not delete built-in roles
      if role in authzBuiltinRoles:
         msg = 'Built-in role %s cannot be deleted' % role
         mode.addError( msg )
         return
      cfg = _getLocalUserConfig( mode )
      del cfg.role[ role ]
      if role == cfg.defaultRole:
         mode.addWarning( 'The default role has been deleted' )

configMode.addCommandClass( GotoRoleModeCommand )

builtInModes_ = { 'config': 'Global Configuration mode',
                  'config-all': 'All the Configuration modes',
                  'exec': 'EXEC mode' }
builtInModeMatcher = CliMatcher.EnumMatcher( builtInModes_ )
modeKeyMatcher = CliMatcher.PatternMatcher( roleRegexRe,
                                            helpname='WORD',
                                            helpdesc='Mode key' )

def _modeKey( args ):
   return args.get( '<BUILTIN_MODE>' ) or args.get( '<MODE>', '' )

class RoleRuleCommand( CliCommand.CliCommandClass ):
   syntax = '[ <SEQ> ] permit | deny [ mode <MODE> | <BUILTIN_MODE> ] ' \
            'command <COMMAND>'
   noOrDefaultSyntax = '( permit | deny [ mode <MODE> | <BUILTIN_MODE> ] ' \
            'command <COMMAND> ) | ( <SEQ> ... )'
   data = {
      '<SEQ>' : CliMatcher.IntegerMatcher( 1, MAX_SEQ,
                                           helpdesc='Index in the sequence' ),
      'permit' : 'Allow commands',
      'deny' : 'Deny commands',
      'mode' : 'Specify under which modes to apply the rule',
      '<MODE>' : modeKeyMatcher,
      '<BUILTIN_MODE>' : builtInModeMatcher,
      'command' : 'Specify commands by regular expression',
      '<COMMAND>' : CliMatcher.StringMatcher( helpname='COMMAND',
                                              helpdesc='Regular expression' )
   }
   @staticmethod
   def handler( mode, args ):
      t = Tac.Type( 'LocalUser::Action' )
      if 'permit' in args:
         act = t.permit
      else:
         act = t.deny

      seq = args.get( '<SEQ>' )
      regex = args[ '<COMMAND>' ]

      # add or modify a rule
      mode.context.addRule( mode, seq, act, regex, _modeKey( args ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      seq = args.get( '<SEQ>' )
      if seq:
         mode.context.removeRuleBySeq( seq )
         return

      t = Tac.Type( 'LocalUser::Action' )
      if 'permit' in args:
         act = t.permit
      else:
         act = t.deny

      regex = args[ '<COMMAND>' ]
      mode.context.removeRule( act, regex, _modeKey( args ) )

class RoleResequenceCommand( CliCommand.CliCommandClass ):
   syntax = 'resequence [ <START> [ <INC> ] ]'
   data = { 'resequence' : 'Resequence the rules',
            '<START>' :
            CliMatcher.IntegerMatcher( 1, MAX_SEQ,
                            helpdesc='Starting sequence number (default 10)' ),
            '<INC>' : CliMatcher.IntegerMatcher( 1, MAX_SEQ,
               helpdesc='Step to increment the sequence number (default 10)' )
   }
   @staticmethod
   def handler( mode, args ):
      start = args.get( '<START>', 10 )
      inc = args.get( '<INC>', 10 )
      mode.context.resequence( mode, start, inc )

class RoleAbortCommand( CliCommand.CliCommandClass ):
   syntax = 'abort'
   data = { 'abort' : 'Exit without committing changes' }
   @staticmethod
   def handler( mode, args ):
      mode.abort()

RoleConfigMode.addCommandClass( RoleRuleCommand )
RoleConfigMode.addCommandClass( RoleResequenceCommand )
RoleConfigMode.addCommandClass( RoleAbortCommand )

######### "username ..." #########
#
# Implemented
# -----------
#
# [no] username <name> [privilege <0-15>] [role <role-name>]
#    secret { [0] <cleartext-passwd> | 5 <encrypted-passwd> }
# [no] username <name> ssh-key { <key> | file <filename> }
# [no] username <name> ssh cert principal { principals }

# Deleting those accounts gives a warning.
warningAccounts_ = { 'admin' }

def clearUsername( mode, username, sshConfig ):
   cfg = _getLocalUserConfig( mode )

   # 'username' can be anything the user typed, so make sure it's
   # an actual account name before we try to delete it.
   if username not in cfg.acct:
      return

   # Certain account names are not valid (e.g., they're reserved by
   # other Linux packages). Accounts with invalid names should
   # never exist in our 'userAcct' collection, but someone might
   # have added them to the TAC collection manually. We don't allow
   # deleting these reserved accounts.
   if not LocalUserLib.isValidAccountName( username ):
      mode.addError(
         "'%s' is a system account, but managed by Eos?" % username )
      return

   if username in warningAccounts_:
      mode.addWarning( "Removing built-in account '%s'" % username )

   del cfg.acct[ username ]
   try:
      clearPasswordHistory( username )
   except OSError:
      mode.addError( f"Failed to clear password history for username:{ username }" )

   userConfig = sshConfig.user.get( username )
   if userConfig:
      # Delete the SSH keys if we are using the old CLI for keys
      if not sshConfig.useNewSshCliKey:
         # Using old CLI for keys, clear the SSH authorized keys
         userConfig.sshAuthKeys.clear()
         userConfig.publishedSshAuthKey = ""

      # Delete the SSH principals if we are using the old CLI for principals
      if not sshConfig.useNewSshCliPrincipal:
         # Using old CLI for principals, clear the SSH authorized principals
         userConfig.sshAuthPrincipals.clear()

      # Check if the SSH user can be deleted
      _maybeCleanupSshUser( username, sshConfig )

secretMatchObj = object()


class UsernameCommand( CliCommand.CliCommandClass ):
   syntax = """username USERNAME { ( privilege PRIV )
                                 | ( role ROLE )
                                 | ( secret SECRET ) | nopassword
                                 | ( shell SHELL ) }"""
   # note we do the "no sshkey/ssh principal" command here as well
   noOrDefaultSyntax = """username USERNAME [ { ( privilege [ PRIV ] )
                                              | ( role [ ROLE ] )
                                              | ( secret [ SECRET ] ) | nopassword
                                              | ( shell [ SHELL ] )
                                              | ( ssh principal [ { PRINCIPALS } ] )
                                              | ( ssh-key | sshkey ... ) } ]"""
   data = {
      'username' : usernameKwMatcher,
      'USERNAME' : usernameMatcher,
      'privilege' : CliCommand.singleKeyword( 'privilege',
         helpdesc='Initial privilege level (with local EXEC authorization)' ),
      'PRIV' : CliMatcher.IntegerMatcher( 0, 15,
         helpdesc='Initial privilege level (with local EXEC authorization)' ),
      'role' : CliCommand.singleKeyword( 'role',
         helpdesc='Specify a role for the user' ),
      'ROLE' : roleMatcher,
      'secret' : CliCommand.singleKeyword( 'secret',
         helpdesc='Configure login secret for the account',
         sharedMatchObj=secretMatchObj ),
      'SECRET' : SecretCli.secretCliExpression( 'encryptedPasswd',
         supportInvalidPassword=True,
         # this allows options after secret
         extraClearTextExclude=( 'privilege', 'role', 'shell' ),
         policyFn=getPasswordPolicy ),
      'nopassword' : CliCommand.singleKeyword( 'nopassword',
         helpdesc='Allow user login with no password',
         sharedMatchObj=secretMatchObj ),
      'shell' : CliCommand.singleKeyword( 'shell',
         helpdesc='Specify shell for the user' ),
      'SHELL' : CliMatcher.EnumMatcher( {
            shell: f'Specify {shell} as shell'
            for shell in ( '/bin/bash', '/bin/sh', '/sbin/nologin' ) } ),
      'ssh' : sshKwMatcher,
      'principal' : principalKwMatcher,
      'PRINCIPALS' : CliMatcher.StringMatcher( helpname='PRINCIPAL',
                                               helpdesc='Principal string' ),
      'sshkey' : sshkeyDeprecated,
      'ssh-key' : sshkeyKwMatcher,
   }
   allowCache = False

   @staticmethod
   def handler( mode, args ):
      username = args[ 'USERNAME' ]
      cfg = _getLocalUserConfig( mode )

      if not LocalUserLib.isValidAccountName( username ):
         mode.addErrorAndStop( f'Unable to create reserved username {username!r}. '
                               'Please try another.' )

      if 'nopassword' in args:
         secretValue = SecretCli.SecretValue( mode, '' )
      else:
         secretValue = args.get( 'encryptedPasswd' )

      acct = cfg.acct.get( username )
      policy = getPasswordPolicy()
      applyDenyUsernamePolicy( username, policy, secretValue )
      if acct:
         debug( "Updating username", repr( username ) )
         if not applyMinChangedCharactersPolicy( mode, username, policy, secretValue,
                                                 acct.encryptedPasswd ):
            return
      else: # New account.
         if secretValue is None:
            mode.addErrorAndStop( "'secret' or 'nopassword' option required" )
         debug( "Adding username", repr( username ) )
         acct = cfg.newAcct( username, secretValue.hash() )
      applyDenyLastPolicy( mode, username, policy, secretValue )

      if secretValue is not None:
         acct.encryptedPasswd = secretValue.hash()
         secretValue.newPass = None
      if priv := args.get( 'PRIV' ):
         acct.privilegeLevel = priv[ 0 ]
      if role := args.get( 'ROLE' ):
         acct.role = role[ 0 ]
      if shell := args.get( 'SHELL' ):
         acct.shell = shell[ 0 ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      username = args[ 'USERNAME' ]
      admin = username == 'admin'
      cfg = _getLocalUserConfig( mode )
      sshConfig = _getSshConfig()
      userSshConfig = sshConfig.user.get( username )

      options = (
            'privilege' in args,
            'role' in args,
            'secret' in args,
            'shell' in args,
            'principal' in args,
            'sshkey' in args or 'ssh-key' in args,
      )
      resetAll = not any( options )
      privilege, role, secret, shell, principal, sshkey = options

      if username == 'root' and principal:
         if userSshConfig:
            userSshConfig.sshAuthPrincipals.clear()
            _maybeCleanupSshUser( 'root', sshConfig )
      elif admin and CliCommand.isDefaultCmd( args ) and resetAll:
         # make sure we create the user
         cfg.newAcct( "admin", "" )
         # "no username admin" will default the config, meaning clearing
         # all ssh-keys/principals for the admin user if using old CLI.
         # The SSH user config might get deleted if the user was not
         # explicitly added by the operator, _maybeCleanUpSshUser later
         # does the cleanup.
         # If new CLI is used, ssh-keys/principals can be cleared from
         # the new CLI under "management ssh" for all users
         if userSshConfig:
            if not sshConfig.useNewSshCliKey:
               userSshConfig.sshAuthKeys.clear()
               userSshConfig.publishedSshAuthKey = ""

            if not sshConfig.useNewSshCliPrincipal:
               userSshConfig.sshAuthPrincipals.clear()
      elif resetAll:
         # delete user
         clearUsername( mode, username, sshConfig )
         return

      acct = cfg.acct.get( username )
      if not acct:
         return

      # reset to default
      if role or resetAll:
         acct.role = "network-admin" if admin else acct.roleDefault
      if ( sshkey or resetAll ) and userSshConfig:
         del userSshConfig.sshAuthKeys[ primarySshKeyId ]
         userSshConfig.publishedSshAuthKey = ""
         _maybeCleanupSshUser( username, sshConfig )
      if ( principal or resetAll ) and userSshConfig:
         userSshConfig.sshAuthPrincipals.clear()
         _maybeCleanupSshUser( username, sshConfig )
      if shell or resetAll:
         acct.shell = ''
      if secret or resetAll:
         acct.encryptedPasswd = ""
      if privilege or resetAll:
         acct.privilegeLevel = acct.privilegeLevelDefault

      # If everything's default, delete username.
      if ( acct.privilegeLevel == acct.privilegeLevelDefault and
           not acct.encryptedPasswd and
           not acct.shell and
           acct.role == acct.roleDefault ):
         clearUsername( mode, username, sshConfig )

configMode.addCommandClass( UsernameCommand )

class UsernameSshKeyCommand( CliCommand.CliCommandClass ):
   syntax = "username USERNAME ssh-key | sshkey SSHKEY"
   data = { 'username' : usernameKwMatcher,
            'USERNAME' : usernameMatcher,
            'ssh-key' : sshkeyKwMatcher,
            'sshkey' : sshkeyDeprecated,
            'SSHKEY' : SshKeyExpression
   }
   @staticmethod
   def handler( mode, args ):
      username = args[ 'USERNAME' ]
      if filename := args.get( 'SSHKEY_URL' ):
         keys = getKeysFromFile( mode, filename )
         addAuthorizedKeysFromFile( mode, username, keys )
      else:
         key = args[ 'SSHKEY_CONTENT' ]
         addAuthorizedKey( mode, username, key )

configMode.addCommandClass( UsernameSshKeyCommand )

class UsernameSecondarySshKeyCommand( CliCommand.CliCommandClass ):
   syntax = "username USERNAME ssh-key | sshkey secondary SSHKEY_CONTENT"
   noOrDefaultSyntax = syntax
   data = { 'username' : usernameKwMatcher,
            'USERNAME' : usernameMatcher,
            'ssh-key' : sshkeyKwMatcher,
            'sshkey' : sshkeyDeprecated,
            'secondary' : 'Secondary SSH keys',
            'SSHKEY_CONTENT' : sshkeyContentMatcher,
   }
   @staticmethod
   def handler( mode, args ):
      username = args[ 'USERNAME' ]
      key = args[ 'SSHKEY_CONTENT' ]
      addAuthorizedKey( mode, username, key, secondary=True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      username = args[ 'USERNAME' ]
      key = args[ 'SSHKEY_CONTENT' ]
      removeSecondaryAuthorizedKey( mode, username, key )

configMode.addCommandClass( UsernameSecondarySshKeyCommand )

#----------------------------------------------------------------------
# The "username <name> ssh principal { PRINCIPALS }" cmd in config mode
#----------------------------------------------------------------------
class UsernameSshPrincipalCommand( CliCommand.CliCommandClass ):
   syntax = "username USERNAME ssh principal PRINCIPALS"
   data = { 'username' : usernameKwMatcher,
            'USERNAME' : usernameMatcher,
            'ssh' : sshKwMatcher,
            'principal' : principalKwMatcher,
            'PRINCIPALS' : CliMatcher.StringMatcher( helpname='PRINCIPAL',
                                                     helpdesc='Principal string' )
          }
   @staticmethod
   def handler( mode, args ):
      username = args[ 'USERNAME' ]
      principals = args[ 'PRINCIPALS' ]

      sshConfig = _getSshConfig()

      # Create/fetch SSH user configuration
      userConfig = sshConfig.newUser( username )

      # Clear all existing principals, this CLI accepts a list of principals
      # as a whole. We cannot incrementally add principals. Everytime this
      # CLI is used to enter principals, it expects the entire list of principals
      userConfig.sshAuthPrincipals.clear()
      
      for principal in principals.split():
         userConfig.sshAuthPrincipals.newMember( principal )

      cfg = _getLocalUserConfig( mode )
      if username != 'root':
         acct = cfg.acct.get( username )
         if not acct:
            # create a new user with no login
            acct = cfg.newAcct( username, '*' )

configMode.addCommandClass( UsernameSshPrincipalCommand )

######### "show users ..." #########
#
# Implemented
# -----------
#
# show users
# show users detail
# show users accounts
# who

def getSecureMonitorUsers( mode ):
   # Get all known secure-monitor users
   users = set()
   status = _getAaaStatus( mode )
   for session in status.session.values():
      if session.account:
         data = session.property.get( session.authenMethod, {} )
         if data:
            data = data.attr
         if data.get( AaaPluginLib.secureMonitor, False ):
            users.add( session.account.name )
   return users

UTMP_FILE = None # Use default utmp file

UtmpEntry = collections.namedtuple( "UtmpEntry",
      [ "sanTty", "user", "ipAddr", "timestamp", "pid", "idle" ] )

def pidExists( pid ):
   return os.path.exists( "/proc/%d" % pid )

def getUtmpInfo():
   '''gets the utmpdump, discards multiple entries keeping the most recently logged
   in entry and returns the following dictionary
   utmpInfo[ fullTty ] = [ sanTty, user, ipAddr, timeLogged, entryPid ]
   fullTty - full tty path, for example /dev/pts/2
   sanTty - sanitized tty, for example vty2
   user - the user, for example admin
   ipAddr - ip address of the user
   timeLogged - login time. Note that login time may be invalid if utmp file is bad.
   entryPid - the pid of the user process'''
   utmpData = UtmpDump.getUtmpData( UTMP_FILE )
   utmpInfo = {}

   for entry in utmpData:
      if entry[ "type" ] == UtmpDump.UTMP_TYPE_USER_PROCESS:
         entryPid = int( entry[ "pid" ] )
         sanTty = entry[ "tty" ]
         fullTty = '/dev/' + sanTty.replace( 'con', 'ttyS' ).replace( 'vty', 'pts/' )
         entryIp = '-' if entry[ "ipAddr" ] == '0.0.0.0' else entry[ "ipAddr" ]
         # utmp may be out of sync, leading it to list pids that don't exist
         if pidExists( entryPid ):
            try:
               timeLogged = UtmpDump.parseTime( entry[ "time" ] )
            except ValueError:
               # utmp in rare cases does not store the login time. Still consider
               # the utmp entry as valid.
               timeLogged = None
            # Get the most recent entry, since utmp can be out of sync if users
            # do not properly log out. It looks like utmp is organized with the
            # most recent entry last. Thus, if two entries have the same
            # creation date, choose the entry at the end.
            prevData = utmpInfo.get( fullTty )
            if not prevData or prevData.timestamp is None or \
                   ( timeLogged is not None and prevData.timestamp <= timeLogged ):
               utmpInfo[ fullTty ] = UtmpEntry( sanTty, entry[ "user" ], entryIp,
                                       timeLogged, entryPid,
                                       entry[ "idle" ] if "idle" in entry else None )

   return utmpInfo

def validateUser( targetPid, startPids ):
   '''validates the user process by starting at the entries in startPids and walking
   up the parent tree looking for targetPid'''
   found = False
   for pid in startPids:
      curr = pid
      try:
         # pylint: disable-next=consider-using-in
         while ( curr != targetPid and curr != 1 ):
            with open( "/proc/%d/status" % curr ) as f:
               status = f.read().splitlines()
            # Parse out the pid from the procfile
            nextPid = None
            for line in status:
               match = re.match( r'^PPid:[ \t]*(\d+)', line )
               if match:
                  nextPid = int( match.group( 1 ) )
                  break
            assert nextPid is not None, "The pid is missing from the procfile"
            curr = nextPid
      except ( Tac.SystemCommandError, ValueError, OSError ):
         pass

      # compare to pid, break if true
      if curr == targetPid:
         found = True
         break

   return found

# The 'clear line' command is based on the output of the 'who' command.
# Share the way they retrieve the data necessary to perform their actions.
fuserRe = re.compile( r'(/dev/\S+):\s+' )
fuserPidRe = re.compile( r'\s+[a-zA-Z]*(\d+)[a-zA-Z]*' )

def getFuserData( utmpInfo ):
   # execute the fuser command
   terminals = [ 'sudo', 'fuser' ] + list( utmpInfo )
   # pylint: disable-next=consider-using-with
   sub = subprocess.Popen( terminals, stdout=subprocess.PIPE,
                           stderr=subprocess.STDOUT,
                           universal_newlines=True )
   flines = sub.communicate()[ 0 ].splitlines()

   # parse data from fuser command
   # Start with empty pid list for all ptys we care about
   fuserData = { k: [] for k in utmpInfo }
   for line in flines:
      fields = fuserRe.match( line )
      if not fields:
         continue

      pids = fuserPidRe.findall( line )
      fuserData[ fields.group( 1 ) ] = [ int( pid ) for pid in pids ]
   return fuserData

def getWhoData( mode ):
   # get data from utmpdump
   utmpInfo = getUtmpInfo()
   fuserData = getFuserData( utmpInfo )

   if mode.session.secureMonitor():
      skippedUsers = set()
   else:
      skippedUsers = getSecureMonitorUsers( mode )

   return sorted( utmp for tty, utmp in utmpInfo.items()
                  if utmp.user not in skippedUsers and
                  validateUser( utmp.pid, fuserData[ tty ] ) )

   # Output of 'w -h -s'.
   #jchl     tty1     -                11days /bin/sh /usr/X11R6/bin/startx
   #jchl     pts/0    :0.0              2:17  xemacs -nw
   #jchl     pts/2    :0.0              2:09  Cli [interactive]
   #jchl     pts/5    :0.0              2:02m bash
   #jchl     pts/8    :0.0              0.00s ssh obsidian
   #jchl     pts/9    :0.0             50:23  ../../usr/bin/gnome-terminal
   #jchl     pts/11   obsidian          0.00s w -h -s

   # Output of industry-standard 'show users'.
   #    Line       User       Host(s)              Idle       Location
   #*  1 vty 0                idle                 00:00:00 192.168.2.8
   #   2 vty 1                idle                     1w6d 192.168.2.4
   #   3 vty 2                idle                    1d23h 192.168.2.4
   #   4 vty 3                idle                    1d23h 192.168.2.4

def whoCmd( mode ):
   # Get the model and call render() manually
   model = showUsers( mode, None )
   model.render()

def showUsers( mode, args ):
   # First, find the terminal associated with the current process.  Note that you
   # can't do this using Tac.run(), as this detaches the child from the controlling
   # terminal.
   myTty = UtmpDump.getTtyName()
   users = getWhoData( mode )
   ret = AaaModel.ShowAaaUsers()
   for n, user in enumerate( users, start=1 ):
      ret1 = AaaModel.ShowAaaUserInfo()
      ret1.sessionTty = user.sanTty == myTty
      ret1.lineNumber = n
      tty = re.sub( r'(\d+)', r' \1', user.sanTty )
      r = tty.split( ' ' )[ -1 ]

      ret1.user = user.user
      ret1.idleTime = -1 if user.idle is None else int( user.idle )
      if "con" in tty:
         ret.serials[ r ] = ret1
      else:
         ret1.location = user.ipAddr
         ret.vtys[ r ] = ret1

   return ret

# Clear line functionality that resets a specified terminal.
# Can only be run in enabled mode
# syntax: clear line {vty|console} terminalNumber
#
maxVty = 101
maxCon = 1

def clearTerminal( mode, terminalName ):
   userData = getWhoData( mode )
   termPid = -1
   for data in userData:
      if data.sanTty == terminalName:
         termPid = data.pid
         break
   resetTerminalLine( mode, terminalName, termPid )

# wrapper for clearing by line number
def clearLine( mode, number ):
   userData = getWhoData( mode )
   if number > len( userData ) or number == 0:
      terminalName = ""
      termPid = -1
   else:
      terminalData = userData[ number - 1 ]
      terminalName = terminalData.sanTty
      termPid = terminalData.pid
   resetTerminalLine( mode, terminalName, termPid )

def resetTerminalLine( mode, terminalName, termPid ):
   """Clear the specified terminal. termPid is the process corresponding to the
   terminal that will be reset. termPid is -1 if no terminal was found, thus no
   action can be performed."""
   import signal # pylint: disable=import-outside-toplevel
   if UtmpDump.getTtyName() == terminalName:
      mode.addError( "Not allowed to clear current line" )
   else:
      if BasicCliUtil.confirm( mode, "[confirm]" ):
         # get login entry second
         if termPid != -1:
            try:
               Tac.run( [ "kill", "-s", "%d" % signal.SIGHUP, "%d" % termPid ],
                        asRoot=True, stdout=Tac.DISCARD, stderr=Tac.DISCARD )
            except Tac.SystemCommandError:
               # ignore errors. Have user retry
               pass
         print( "[OK]" )
      else:
         print()

def conNum():
   cmdOutput = Tac.run( [ 'cat', '/proc/cmdline' ],
                        stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )
   consoleRangeNum = 1 if 'ttyS1' in cmdOutput else 0
   return consoleRangeNum
consoleRangeNumber = conNum()

class ClearLineCommand( CliCommand.CliCommandClass ):
   syntax = "clear line LINE | ( vty VTY ) | ( console [ CON ] )"
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'line' : 'A terminal session',
      'LINE' : CliMatcher.IntegerMatcher( 0, maxVty + maxCon,
                                         helpdesc='Line number' ),
      'vty' : 'Terminal',
      'VTY' : CliMatcher.IntegerMatcher( 0, maxVty - 1,
                                         helpdesc='Vty terminal number' ),
      'console' : 'Console (primary) terminal',
      # this is just a range with one number
      'CON' : CliMatcher.IntegerMatcher( consoleRangeNumber, consoleRangeNumber,
                                         helpdesc='Console terminal number' )
      }

   @staticmethod
   def handler( mode, args ):
      line = args.get( 'LINE' )
      if line is not None:
         return clearLine( mode, line )

      vty = args.get( 'VTY' )
      if vty is not None:
         return clearTerminal( mode, "vty%d" % vty )

      return clearTerminal( mode, "con%d" % consoleRangeNumber )

BasicCli.EnableMode.addCommandClass( ClearLineCommand )   

def _idleWToDisplay( idle ):
   """Converts an idle time output by 'w' to a form suitable for display by 'show
   users'."""

   m = re.match( r'(\d+)days', idle )
   if m:
      days = int( m.group( 1 ) )
      if days > 7:
         return '%dw%dd' % ( ( days // 7 ), ( days % 7 ) )
      else:
         # Once the idle time is at least a day, 'w' only gives the time in a
         # resolution of days, so we'll just pretend the number of hours is zero.
         return '%dd%02dh' % ( days, 0 )

   m = re.match( r'(\d+):(\d+)m', idle )
   if m:
      hours = int( m.group( 1 ) )
      minutes = int( m.group( 2 ) )
      seconds = 0
   else:
      m = re.match( r'(\d+):(\d+)', idle )
      if m:
         hours = 0
         minutes = int( m.group( 1 ) )
         seconds = int( m.group( 2 ) )
      else:
         m = re.match( r'(\d+)\.\d+s', idle )
         if m:
            hours = 0
            minutes = 0
            seconds = int( m.group( 1 ) )
         else:
            # This isn't a format we recognize.  Just return the original string.
            return idle

   return '%02d:%02d:%02d' % ( hours, minutes, seconds )

showUsersDesc = 'Display information about terminal lines'

class WhoCommand( CliCommand.CliCommandClass ):
   syntax = "who"
   data = { 'who' : showUsersDesc }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      whoCmd( mode )

BasicCli.ExecMode.addCommandClass( WhoCommand )

#------------------------------------------------------------------------------
# aaa authentication login {default | <list-name>} method-1 [method-2] ...
#    [method-n]
#------------------------------------------------------------------------------
authnKwMatcher = CliMatcher.KeywordMatcher( 'authentication',
   helpdesc='Configure authentication parameters' )

servicenamePattern = ''.join( [ BasicCliUtil.notAPrefixOf( p )
                             for p in ( 'console', 'default' ) ] )
servicenameRegex = servicenamePattern + '[-A-Za-z0-9_]*'
servicenameMatcher = CliMatcher.PatternMatcher(
   servicenameRegex,
   helpname='WORD',
   helpdesc='Name of authentication list',
   partialPattern=servicenamePattern + '.*' )

# workaround not being able to use 'default' in syntax
defaultAuthnKwMatcher = CliMatcher.KeywordMatcher(
   'default',
   helpdesc='The default authentication list' )

class AuthnGroupMethodExpression( CliCommand.CliExpression ):
   expression = "group <groupType> | <groupName>"
   data = { 'group' : groupKwMatcher,
            '<groupType>' : AaaCliLib.authnMethodListGroupMatcher,
            '<groupName>' : AaaCliLib.serverGroupNameMatcher }

class AuthnMethodExpression( CliCommand.CliExpression ):
   expression = "<method> | <groupMethod>"
   data = { "<method>" : AaaCliLib.authnMethodListNonGroupMatcher,
            "<groupMethod>" : AuthnGroupMethodExpression }

mlParam = '<methodList>'
mcastParam = "multicast"

def methodListAdapter( mode, args, argsList ):
   # this converts <method>, <groupType>, <groupName> to <methodList>
   for replaced in ( '<method>', '<groupType>', '<groupName>', 'group', mcastParam ):
      args.pop( replaced, None )
   args[ mlParam ] = []
   args[ mcastParam ] = {}
   for arg in argsList:
      if arg[ 0 ] == '<groupType>' or arg[ 0 ] == '<groupName>':
         methodName = 'group ' + arg[ 1 ]
         args[ mlParam ].append( methodName )
      elif arg[ 0 ] == '<method>':
         args[ mlParam ].append( arg[ 1 ] )
      elif arg[ 0 ] == mcastParam:
         args[ mcastParam ][ methodName ] = ( arg[ 1 ] == mcastParam )

def _getAuthenServiceName( args ):
   if '_default_' in args:
      serviceName = 'default'
   elif 'console' in args:
      serviceName = 'login'
   else:
      serviceName = args[ '<serviceName>' ]
   return serviceName

class AaaAuthenLoginCommand( CliCommand.CliCommandClass ):

   syntax = "aaa authentication login _default_ | console | <serviceName> " + \
            "{ <methodExpr> }"
   noOrDefaultSyntax = "aaa authentication login _default_ | console | " \
                       "<serviceName> ..."
   data = {
      'aaa' : aaaKwMatcher,
      'authentication' : authnKwMatcher,
      'login' : 'Login related configuration',
      '_default_' : defaultAuthnKwMatcher,
      'console' : 'Console authentication list',
      '<serviceName>' : CliCommand.Node( servicenameMatcher, hidden=True ),
      '<methodExpr>' : AuthnMethodExpression }

   adapter = methodListAdapter

   @staticmethod
   def handler( mode, args ):
      serviceName = _getAuthenServiceName( args )
      methods = args[ mlParam ]
      debug( "Setting method list", serviceName, "to use methods", methods )
      cfg = AaaCliLib.configAaa( mode )
      if not _verifyGroups( mode, cfg, AaaCliLib.ActionType.AUTHN_LOGIN, methods ):
         return
      ml = getOrCreateMethodList( cfg, serviceName, "login" )
      ml.method.clear()
      for i in range( 0, len( methods ) ): # pylint: disable=consider-using-enumerate
         ml.method[ i ] = methods[ i ]
         if methods[ i ] == "none":
            mode.addWarning( "'none' allows anyone to login if previous methods"
                             " fall back (TACACS+/RADIUS server unavailable"
                             " or unknown local user provided)" )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      serviceName = _getAuthenServiceName( args )
      cfg = AaaCliLib.configAaa( mode )
      if serviceName == "default":
         # The default config of the default method list includes local
         # authentication only.
         cfg.defaultLoginMethodList.method.clear()
         cfg.defaultLoginMethodList.method[ 0 ] = "local"
      else:
         del cfg.loginMethodList[ serviceName ]

def getOrCreateMethodList( cfg, serviceName, authenType ):
   if serviceName == "default":
      methodList = ( cfg.defaultLoginMethodList 
                      if authenType == "login" else 
                      cfg.defaultEnableMethodList )
      return methodList
   methodList = ( cfg.loginMethodList 
                   if authenType == "login" 
                   else cfg.enableMethodList )
   debug( "Creating new method list if it doesnt exist", serviceName )
   ml = methodList.newMember( serviceName )
   return ml

def _verifyGroups( mode, cfg, actionType, methods ):
   """Some sanity check on the methods: some group methods do not support
   all kinds of features (as of this writing, RADIUS does not support
   command authorization, for example). This could avoid customers
   misconfiguring AAA and lock themselves out of the box.

   Currently two checks are added:
   1. for built-in groups (radius or tacacs+), the method has to support
      the feature.
   2. for user-created groups, the group has to exist.

   This should handle most of the cases, especially when the user tries to
   configure 'aaa authorization|accounting commands all default group radius'.

   There are other ways the users could shoot themselves with, for example:
   1. the group might be empty (including built-in groups when there are no
      such servers configured)
   2. the user can still modify/remove the groups after authn/authz/acct is
      configured."""
   for m in methods:
      if m.startswith( 'group ' ):
         groupName = m[ 6: ]
         groupType = AaaCliLib.getGroupTypeFromToken( groupName )

         if groupType == 'unknown':
            # not a built-in group, check if it exists, and if so,
            # get its group type
            if groupName in cfg.hostgroup:
               groupType = cfg.hostgroup[ groupName ].groupType
            else:
               mode.addError( "Group %s does not exist" % groupName )
               return False

         if actionType & ( AaaCliLib.ActionType.AUTHZ |
                           AaaCliLib.ActionType.ACCT |
                           AaaCliLib.ActionType.ACCT_DOT1X |
                           AaaCliLib.ActionType.AUTHN_DOT1X ):
            supported = AaaCliLib.groupTypeSupported( groupType )
            if not supported & actionType:
               if actionType & AaaCliLib.ActionType.AUTHZ_EXEC:
                  typeName = 'exec authorization'
               elif actionType & AaaCliLib.ActionType.AUTHZ_COMMAND:
                  typeName = 'command authorization'
               elif actionType & AaaCliLib.ActionType.ACCT_EXEC:
                  typeName = 'exec accounting'
               elif actionType & AaaCliLib.ActionType.ACCT_COMMAND:
                  typeName = 'command accounting'
               elif actionType & AaaCliLib.ActionType.ACCT_SYSTEM:
                  typeName = 'system accounting'
               elif actionType & AaaCliLib.ActionType.AUTHN_DOT1X:
                  typeName = 'dot1x authentication'
               elif actionType & AaaCliLib.ActionType.ACCT_DOT1X:
                  typeName = 'dot1x accounting'
               else:
                  assert False, 'unknown type %d' % actionType
               mode.addError( f"{m} does not support {typeName}" )
               return False
   return True

BasicCli.GlobalConfigMode.addCommandClass( AaaAuthenLoginCommand )

#------------------------------------------------------------------------------
# test aaa group <groupName> <username> <password>
#------------------------------------------------------------------------------
passwordMatcher = CliMatcher.StringMatcher( helpname='PASSWORD',
                                            helpdesc='Password' )
class AaaTestCommand( CliCommand.CliCommandClass ):

   syntax = "test aaa <groupMethodExpr> <username> { <password> }"
   data = {
      'test' : 'Test feature',
      'aaa' : aaaKwMatcher,
      '<username>' : usernameMatcher,
      '<password>' : CliCommand.Node( passwordMatcher,
                                      sensitive=True ),
      '<groupMethodExpr>' : AuthnGroupMethodExpression }
   allowCache = False

   @staticmethod
   def handler( mode, args ):
      groupname = args.get( '<groupType>', None ) or args[ '<groupName>' ]
      method = 'group ' + groupname
      username = args[ '<username>' ]
      password = ' '.join( args[ '<password>' ] )

      cfg = AaaCliLib.configAaa( mode )
      if not _verifyGroups( mode, cfg, AaaCliLib.ActionType.AUTHN_LOGIN,
                            [ method ] ):
         return
      provider = AaaProvider()
      provider.authenticateTest( mode, method, username, password )

BasicCli.EnableMode.addCommandClass( AaaTestCommand )

#---------------------------------------------------------------------------------
# aaa authentication enable { default|console } method-1 [method-2] ... [method-n]
#---------------------------------------------------------------------------------
class AaaAuthenEnableCommand( CliCommand.CliCommandClass ):

   syntax = "aaa authentication enable _default_ | console | <serviceName> " + \
            "{ <methodExpr> }"
   noOrDefaultSyntax = "aaa authentication enable _default_ | console | " \
                       "<serviceName> ..."
   data = {
      'aaa' : aaaKwMatcher,
      'authentication' : authnKwMatcher,
      'enable' : 'Set authentication lists for enable command',
      '_default_' : defaultAuthnKwMatcher,
      'console' : 'Console authentication list',
      '<serviceName>' : CliCommand.Node( servicenameMatcher, hidden=True ),
      '<methodExpr>' : AuthnMethodExpression }

   adapter = methodListAdapter

   @staticmethod
   def handler( mode, args ):
      methods = args[ mlParam ]
      serviceName = _getAuthenServiceName( args )
      debug( "Setting enable method list to", methods )
      cfg = AaaCliLib.configAaa( mode )
      if not _verifyGroups( mode, cfg, AaaCliLib.ActionType.AUTHN_ENABLE, methods ):
         return
      ml = getOrCreateMethodList( cfg, serviceName , "enable" )
      ml.method.clear()
      for i in range( 0, len( methods ) ): # pylint: disable=consider-using-enumerate
         ml.method[ i ] = methods[ i ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      serviceName = _getAuthenServiceName( args )
      cfg = AaaCliLib.configAaa( mode )
      # The default config of the default method list includes local
      # authentication only.
      if serviceName == "default":
         cfg.defaultEnableMethodList.method.clear()
         cfg.defaultEnableMethodList.method[ 0 ] = "local"
      else:
         del cfg.enableMethodList[ serviceName ]

configMode.addCommandClass( AaaAuthenEnableCommand )

#------------------------------------------------------------------------------
# aaa authentication dot1x default method-1 [method-2] ... [method-n]
#------------------------------------------------------------------------------
class AuthnDot1xMethodExpression( CliCommand.CliExpression ):
   expression = "<method> | ( group <groupType> | <groupName> )"
   data = { "<method>" : AaaCliLib.authnDot1xMethodListNonGroupMatcher,
            'group' : groupKwMatcher,
            '<groupType>' : AaaCliLib.authnDot1xMethodListGroupMatcher,
            '<groupName>' : AaaCliLib.serverGroupNameMatcher }

class AaaAuthenDot1xCommand( CliCommand.CliCommandClass ):

   syntax = "aaa authentication dot1x _default_ { <methodExpr> }"
   noOrDefaultSyntax = "aaa authentication dot1x _default_ ..."
   data = {
      'aaa' : aaaKwMatcher,
      'authentication' : authnKwMatcher,
      'dot1x' : 'Set authentication lists for IEEE 802.1X',
      '_default_' : defaultAuthnKwMatcher,
      '<methodExpr>' : AuthnDot1xMethodExpression }

   adapter = methodListAdapter

   @staticmethod
   def handler( mode, args ):
      methods = args[ mlParam ]
      debug( "Setting dot1x method list to", methods )
      cfg = AaaCliLib.configAaa( mode )
      if not _verifyGroups( mode, cfg, AaaCliLib.ActionType.AUTHN_DOT1X, methods ):
         return
      ml = cfg.defaultDot1xMethodList
      ml.method.clear()
      for i in range( 0, len( methods ) ): # pylint: disable=consider-using-enumerate
         ml.method[ i ] = methods[ i ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = AaaCliLib.configAaa( mode )
      cfg.defaultDot1xMethodList.method.clear()

configMode.addCommandClass( AaaAuthenDot1xCommand )

#------------------------------------------------------------------------------
# aaa authentication dot1x mba multi-group GROUP1 GROUP2
#------------------------------------------------------------------------------

class AaaDot1xMbaMultiGroupAuthCommand( CliCommand.CliCommandClass ):
   syntax = "aaa authentication dot1x mba multi-group" \
         " { ( <groupType> | <groupName> ) }"
   noOrDefaultSyntax = "aaa authentication dot1x mba multi-group ..."
   data = {
         'aaa' : aaaKwMatcher,
         'authentication' : authnKwMatcher,
         'dot1x' : 'Set authentication lists for IEEE 802.1X',
         'mba' : 'Set authentication lists for MAC based authentication',
         'multi-group' : 'Specify server groups',
         '<methodExpr>' : AuthnDot1xMethodExpression }

   hidden = True
   adapter = methodListAdapter

   @staticmethod
   def handler( mode, args ):
      groups = args[ mlParam ]
      cfg = AaaCliLib.configAaa( mode )
      if not _verifyGroups( mode, cfg, AaaCliLib.ActionType.AUTHN_DOT1X, groups ):
         return
      gl = cfg.mbaMultiGroupList
      gl.method.clear()
      gl.method.update( enumerate( groups ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = AaaCliLib.configAaa( mode )
      cfg.mbaMultiGroupList.method.clear()

configMode.addCommandClass( AaaDot1xMbaMultiGroupAuthCommand )

#------------------------------------------------------------------------------
# aaa authorization dynamic dot1x additional-groups method-1 [method-2 --- ]
#------------------------------------------------------------------------------

class AaaDynAuthzDot1xCommand( CliCommand.CliCommandClass ):

   syntax = "aaa authorization dynamic dot1x additional-groups" \
         " { ( group (<groupType> | <groupName> ) ) }"
   noOrDefaultSyntax = "aaa authorization dynamic dot1x additional-groups ..."
   data = {
         'aaa' : aaaKwMatcher,
         'authorization' : authzKwMatcher,
         'dynamic' : ' Configure dynamic authorization parameters',
         'dot1x' : 'IEEE 802.1X port authentication',
         'additional-groups' : 'Additional dynamic authorization list',
         'group' : groupKwMatcher,
         '<groupType>' : AaaCliLib.authnDot1xMethodListGroupMatcher,
         '<groupName>' : AaaCliLib.serverGroupNameMatcher }

   adapter = methodListAdapter

   @staticmethod
   def handler( mode, args ):
      groups = args[ mlParam ]
      cfg = AaaCliLib.configAaa( mode )
      if not _verifyGroups( mode, cfg, AaaCliLib.ActionType.AUTHN_DOT1X, groups ):
         return
      gl = cfg.coaOnlyGroupList
      gl.method.clear()
      gl.method.update( enumerate( groups ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = AaaCliLib.configAaa( mode )
      cfg.coaOnlyGroupList.method.clear()

configMode.addCommandClass( AaaDynAuthzDot1xCommand )

#------------------------------------------------------------------------------
# aaa authorization {exec | commands {<0-15> | all}} default method-1
#    [method-2] ...  [method-n]
#------------------------------------------------------------------------------

defaultAuthzKwMatcher = CliMatcher.KeywordMatcher(
   'default',
   helpdesc='The default authorization method list' )

def setAuthzMethodList( mode, authzType, methods, isConsole ):
   debug( "Setting authz method list for", authzType, "to", methods )
   def _setMethodList( ml, m ):
      ml.clear()
      for i in range( 0, len( m ) ): # pylint: disable=consider-using-enumerate
         ml[i] = m[i]
   cfg = AaaCliLib.configAaa( mode )
   ml = ( cfg.authzMethod[ authzType ].consoleMethod if isConsole else 
          cfg.authzMethod[ authzType ].defaultMethod )
   _setMethodList( ml, methods )

def noAuthzMethodList( mode, authzType, isConsole ):
   cfg = AaaCliLib.configAaa( mode )
   ml = ( cfg.authzMethod[ authzType ].consoleMethod if isConsole else 
          cfg.authzMethod[ authzType ].defaultMethod )
   ml.clear()
   if not isConsole:
      ml[0] = "none"


class AuthzExecMethodExpression( CliCommand.CliExpression ):
   expression = "<method> | ( group <groupType> | <groupName> )"
   data = { '<method>' : AaaCliLib.authzExecMethodListNonGroupMatcher,
            'group' : groupKwMatcher,
            '<groupType>' : AaaCliLib.authzExecMethodListGroupMatcher,
            '<groupName>' : AaaCliLib.serverGroupNameMatcher }

class AaaAuthzExecCommand( CliCommand.CliCommandClass ):

   syntax = "aaa authorization exec _default_ | console { <methodExpr> }"
   noOrDefaultSyntax = "aaa authorization exec _default_ | console ..."
   data = {
      'aaa' : aaaKwMatcher,
      'authorization' : authzKwMatcher,
      'exec' : 'Configure authorization for starting a shell',
      '_default_' : defaultAuthzKwMatcher,
      'console' : 'Console authorization method list',
      '<methodExpr>' : AuthzExecMethodExpression }

   adapter = methodListAdapter

   @staticmethod
   def handler( mode, args ):
      methods = args[ mlParam ]
      cfg = AaaCliLib.configAaa( mode )
      if _verifyGroups( mode, cfg, AaaCliLib.ActionType.AUTHZ_EXEC,
                            methods ):
         isConsole = 'console' in args
         setAuthzMethodList( mode, "exec", methods, isConsole )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      isConsole = 'console' in args
      noAuthzMethodList( mode, "exec", isConsole )

configMode.addCommandClass( AaaAuthzExecCommand )

class AuthzCommandMethodExpression( CliCommand.CliExpression ):
   expression = "<method> | ( group <groupType> | <groupName> )"
   data = { '<method>' : AaaCliLib.authzCommandMethodListNonGroupMatcher,
            'group' : groupKwMatcher,
            '<groupType>' : AaaCliLib.authzCommandMethodListGroupMatcher,
            '<groupName>' : AaaCliLib.serverGroupNameMatcher }

class CommandLevelExpression( CliCommand.CliExpression ):
   expression = "all | LEVELS"
   data = {
      'all' : 'All privilege levels',
      'LEVELS' : MultiRangeRule.MultiRangeMatcher(
         rangeFn=lambda: ( 0, 15 ),
         noSingletons=False,
         helpdesc='privilege level(s) or range(s) of privilege levels' )
      }
   @staticmethod
   def adapter( mode, args, argList ):
      if 'all' in args:
         args[ 'LEVELS' ] = range( 16 )
      elif 'LEVELS' in args:
         args[ 'LEVELS' ] = list( args[ 'LEVELS' ].values() )

class AaaAuthzCmdCommand( CliCommand.CliCommandClass ):
   syntax = "aaa authorization commands PRIV _default_ { <methodExpr> }"
   noOrDefaultSyntax = "aaa authorization commands PRIV _default_ ..."
   data = {
      'aaa' : aaaKwMatcher,
      'authorization' : authzKwMatcher,
      'commands' : 'Configure authorization for shell commands',
      'PRIV' : CommandLevelExpression,
      '_default_' : defaultAuthzKwMatcher,
      '<methodExpr>' : AuthzCommandMethodExpression }

   adapter = methodListAdapter

   @staticmethod
   def handler( mode, args ):
      methods = args[ mlParam ]
      cfg = AaaCliLib.configAaa( mode )
      if not _verifyGroups( mode, cfg, AaaCliLib.ActionType.AUTHZ_COMMAND, methods ):
         return
      isConsole = 'console' in args
      for i in args[ 'LEVELS' ]:
         setAuthzMethodList( mode, "command%02d" % i, methods, isConsole )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      isConsole = 'console' in args
      for i in args[ 'LEVELS' ]:
         noAuthzMethodList( mode, "command%02d" % i, isConsole )

configMode.addCommandClass( AaaAuthzCmdCommand )

#------------------------------------------------------------------------------
# [no|default] aaa authorization config-commands
#------------------------------------------------------------------------------
def setAuthzConfigCommands( mode, args ):
   cfg = AaaCliLib.configAaa( mode )
   if CliCommand.isDefaultCmd( args ):
      value = cfg.suppressConfigCommandAuthzDefault
   else:
      value = CliCommand.isNoCmd( args )
   cfg.suppressConfigCommandAuthz = value

class SetAuthzConfigCommand( CliCommand.CliCommandClass ):
   syntax = '''aaa authorization config-commands'''
   noOrDefaultSyntax = '''aaa authorization config-commands'''
   data = {
            'aaa' : aaaKwMatcher,
            'authorization' : authzKwMatcher,
            'config-commands' : 'Enable authorization for configuration commands'
          }
   handler = setAuthzConfigCommands
   noOrDefaultHandler = setAuthzConfigCommands

configMode.addCommandClass( SetAuthzConfigCommand )

#------------------------------------------------------------------------------
# aaa accounting {exec | commands {<0-15> | all}} {default|console}
#    {start-stop|stop-only} method-1 [method-2] ...  [method-n]
#------------------------------------------------------------------------------
acctKwMatcher = CliMatcher.KeywordMatcher( 'accounting',
   helpdesc='Configure accounting parameters' )

defaultAcctKwMatcher = CliMatcher.KeywordMatcher(
   'default',
   helpdesc='The default accounting list' )

def setAcctMethodList( mode, acctType, service, action, methods,
                       multicastMethods=None ):
   debug( "Setting acct method list for", acctType, "to", methods )
   multicastMethods = multicastMethods or {}
   cfg = AaaCliLib.configAaa( mode )
   acctMethod = cfg.acctMethod[ acctType ]
   setattr( acctMethod, service + 'Action', action )
   serviceMethodList = getattr( acctMethod, service + 'Method' )
   serviceMethodMulticastList = getattr( acctMethod, service + 'MethodMulticast' )

   # avoid spurious syslog messages by clearing the items in excess only
   for i in range( len( methods ), len( serviceMethodList ) ):
      method = serviceMethodList[ i ]
      del serviceMethodList[ i ]
      if method in serviceMethodMulticastList:
         del serviceMethodMulticastList[ method ]

   # overwrite the current items
   for i, method in enumerate( methods ):
      multicast = multicastMethods.get( method, False )
      if serviceMethodMulticastList.get( method, False ) != multicast:
         serviceMethodMulticastList[ method ] = multicast
      if serviceMethodList.get( i, None ) != method:
         serviceMethodList[ i ] = method

   if service == 'console':
      acctMethod.consoleUseOwnMethod = True

def noAcctMethodList( mode, acctType, service ):
   cfg = AaaCliLib.configAaa( mode )
   def _clearMethodList( ml, service ):
      method = '%sMethod' % service
      getattr( ml, method ).clear()
      setattr( ml, '%sAction' % service, 'none' )
      if service == 'console':
         ml.consoleUseOwnMethod = False

   ml = cfg.acctMethod[ acctType ]
   _clearMethodList( ml, service )

def _acctService( args ):
   return args.get( 'console', 'default' )

def _acctAction( args ):
   if 'start-stop' in args:
      return 'startStop'
   elif 'stop-only' in args:
      return 'stopOnly'
   else:
      assert 'none' in args
      return 'none'

class AcctExecMethodExpression( CliCommand.CliExpression ):
   expression = "<method> | ( group <groupType> | <groupName> )"
   data = { '<method>' : AaaCliLib.acctExecMethodListNonGroupMatcher,
            'group' : groupKwMatcher,
            '<groupType>' : AaaCliLib.acctExecMethodListGroupMatcher,
            '<groupName>' : AaaCliLib.serverGroupNameMatcher }

class AaaAcctExecCommand( CliCommand.CliCommandClass ):

   syntax = "aaa accounting exec _default_ | console " \
            "( start-stop | stop-only { <methodExpr> } ) | none"
   noOrDefaultSyntax = "aaa accounting exec _default_ | console " + \
                       "..."
   data = {
      'aaa' : aaaKwMatcher,
      'accounting' : acctKwMatcher,
      'exec' : 'Configure accounting for starting or stopping a shell',
      '_default_' : defaultAcctKwMatcher,
      'console' : 'Console accounting method list',
      'start-stop' : 'Record start and stop',
      'stop-only' : 'Record stop only',
      '<methodExpr>' : AcctExecMethodExpression,
      'none' : 'Disable accounting' }

   adapter = methodListAdapter

   @staticmethod
   def handler( mode, args ):
      service = _acctService( args )
      action = _acctAction( args )
      methods = args.get( mlParam, [] )
      cfg = AaaCliLib.configAaa( mode )
      if ( action == 'none' or
           _verifyGroups( mode, cfg, AaaCliLib.ActionType.ACCT_EXEC, methods ) ):
         setAcctMethodList( mode, "exec", service, action, methods )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      service = _acctService( args )
      noAcctMethodList( mode, "exec", service )

configMode.addCommandClass( AaaAcctExecCommand )

class AcctCommandMethodExpression( CliCommand.CliExpression ):
   expression = "<method> | ( group <groupType> | <groupName> )"
   data = { '<method>' : AaaCliLib.acctCommandMethodListNonGroupMatcher,
            'group' : groupKwMatcher,
            '<groupType>' : AaaCliLib.acctCommandMethodListGroupMatcher,
            '<groupName>' : AaaCliLib.serverGroupNameMatcher }

class AaaAcctCmdCommand( CliCommand.CliCommandClass ):
   syntax = "aaa accounting commands ( PRIV _default_ | console " \
            "none | ( start-stop | stop-only { <methodExpr> } ) ) " \
            "| ( all secure-monitor none )"
   noOrDefaultSyntax = "aaa accounting commands ( PRIV _default_ | console ) " \
                       "| ( all secure-monitor ) ..."
   data = {
      'aaa' : aaaKwMatcher,
      'accounting' : acctKwMatcher,
      'commands' : 'Configure authorization for shell commands',
      'PRIV' : CommandLevelExpression,
      '_default_' : defaultAcctKwMatcher,
      'console' : 'Console accounting method list',
      'secure-monitor' : 'Secure-monitor accounting setting',
      'start-stop' : 'Record start and stop',
      'stop-only' : 'Record stop only',
      '<methodExpr>' : AcctExecMethodExpression,
      'none' : 'Disable accounting'
   }

   adapter = methodListAdapter

   @staticmethod
   def handler( mode, args ):
      action = _acctAction( args )
      cfg = AaaCliLib.configAaa( mode )
      if args.get( 'secure-monitor' ):
         cfg.acctCmdSecureMonitorEnabled = False
      else:
         service = _acctService( args )
         if action == 'none':
            methods = []
         else:
            methods = args[ mlParam ]
         if not _verifyGroups( mode, cfg, AaaCliLib.ActionType.ACCT_COMMAND,
                               methods ):
            return
         for i in args[ 'LEVELS' ]:
            setAcctMethodList( mode, "command%02d" % i, service, action, methods )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if args.get( 'secure-monitor' ):
         cfg = AaaCliLib.configAaa( mode )
         cfg.acctCmdSecureMonitorEnabled = cfg.acctCmdSecureMonitorEnabledDefault
      else:
         service = _acctService( args )
         for i in args[ 'LEVELS' ]:
            noAcctMethodList( mode, "command%02d" % i, service )

configMode.addCommandClass( AaaAcctCmdCommand )

class AcctSystemMethodExpression( CliCommand.CliExpression ):
   expression = "<method> | ( group <groupType> | <groupName> )"
   data = { '<method>' : AaaCliLib.acctSystemMethodListNonGroupMatcher,
            'group' : groupKwMatcher,
            '<groupType>' : AaaCliLib.acctSystemMethodListGroupMatcher,
            '<groupName>' : AaaCliLib.serverGroupNameMatcher }

class AaaAcctSystemCommand( CliCommand.CliCommandClass ):

   syntax = "aaa accounting system _default_ " \
            "( start-stop | stop-only { <methodExpr> } ) | none"
   noOrDefaultSyntax = "aaa accounting system _default_ " + \
                       "..."
   data = {
      'aaa' : aaaKwMatcher,
      'accounting' : acctKwMatcher,
      'system' : 'Configure accounting for system events',
      '_default_' : defaultAcctKwMatcher,
      'start-stop' : 'Record start and stop',
      'stop-only' : 'Record stop only',
      '<methodExpr>' : AcctSystemMethodExpression,
      'none' : 'Disable accounting' }

   adapter = methodListAdapter

   @staticmethod
   def handler( mode, args ):
      service = _acctService( args )
      action = _acctAction( args )
      if action == 'none':
         methods = []
      else:
         methods = args[ mlParam ]
      cfg = AaaCliLib.configAaa( mode )
      if ( action == 'none' or
           _verifyGroups( mode, cfg, AaaCliLib.ActionType.ACCT_SYSTEM, methods ) ):
         setAcctMethodList( mode, "system", service, action, methods )

      cfg = AaaCliLib.configAaa( mode )
      if _verifyGroups( mode, cfg, AaaCliLib.ActionType.ACCT_SYSTEM, methods ):
         setAcctMethodList( mode, "system", service, action, methods )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      service = _acctService( args )
      noAcctMethodList( mode, "system", service )

configMode.addCommandClass( AaaAcctSystemCommand )

mulitcastKwMatcher = CliMatcher.KeywordMatcher( 'multicast',
      helpdesc='forward accounting packets to all servers in a group' )

# TODO : Unhide the token multicast when Bug455169 gets fixed.
class AcctDot1xMethodExpression( CliCommand.CliExpression ):
   expression = "<method> | ( ( group <groupType> | <groupName> ) [ multicast ] )"
   data = { '<method>' : AaaCliLib.acctDot1xMethodListNonGroupMatcher,
            'group' : groupKwMatcher,
            '<groupType>' : AaaCliLib.acctDot1xMethodListGroupMatcher,
            '<groupName>' : AaaCliLib.serverGroupNameMatcher,
            'multicast' : CliCommand.Node( matcher=mulitcastKwMatcher ) }

class AaaAcctDot1xCommand( CliCommand.CliCommandClass ):

   syntax = "aaa accounting dot1x _default_ " \
            "start-stop | stop-only { <methodExpr> }"
   noOrDefaultSyntax = "aaa accounting dot1x _default_ " + \
                       "..."
   data = {
      'aaa' : aaaKwMatcher,
      'accounting' : acctKwMatcher,
      'dot1x' : 'Configure accounting for IEEE 802.1X',
      '_default_' : defaultAcctKwMatcher,
      'start-stop' : 'Record start and stop',
      'stop-only' : 'Record stop only',
      '<methodExpr>' : AcctDot1xMethodExpression }

   adapter = methodListAdapter

   @staticmethod
   def handler( mode, args ):
      service = _acctService( args )
      action = _acctAction( args )
      methods = args[ mlParam ]
      multicastMethods = args.get( mcastParam, None )
      cfg = AaaCliLib.configAaa( mode )
      if _verifyGroups( mode, cfg, AaaCliLib.ActionType.ACCT_DOT1X, methods ):
         setAcctMethodList( mode, "dot1x", service, action, methods,
                            multicastMethods )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      service = _acctService( args )
      noAcctMethodList( mode, "dot1x", service )

configMode.addCommandClass( AaaAcctDot1xCommand )

#------------------------------------------------------------------------------
# [no] aaa authentication policy local allow-nopassword-remote-login
#------------------------------------------------------------------------------
authnPolicyKwMatcher = CliMatcher.KeywordMatcher(
   'policy',
   helpdesc='Set authentication policy' )

class AaaAuthenPolicyNoPassRemoteLoginCommand( CliCommand.CliCommandClass ):

   syntax = "aaa authentication policy local allow-nopassword-remote-login"
   noOrDefaultSyntax = syntax
   data = {
      'aaa' : aaaKwMatcher,
      'authentication' : authnKwMatcher,
      'policy' : authnPolicyKwMatcher,
      'local' : 'Configure policy for local authentication',
      'allow-nopassword-remote-login' :
      'Allow remote login for a local user with an empty password'
      }

   @staticmethod
   def handler( mode, args ):
      cfg = _getLocalUserConfig( mode )
      cfg.allowRemoteLoginWithEmptyPassword = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = _getLocalUserConfig( mode )
      cfg.allowRemoteLoginWithEmptyPassword = False

configMode.addCommandClass( AaaAuthenPolicyNoPassRemoteLoginCommand )

#------------------------------------------------------------------------------
# [no|default] aaa authentication policy on-success|on-failure log
#------------------------------------------------------------------------------

class AaaAuthenPolicyLogCommand( CliCommand.CliCommandClass ):

   syntax = "aaa authentication policy on-success | on-failure log"
   noOrDefaultSyntax = syntax
   data = {
      'aaa' : aaaKwMatcher,
      'authentication' : authnKwMatcher,
      'policy' : authnPolicyKwMatcher,
      'on-success' : 'Set options for successful logins',
      'on-failure' : 'Set options for failed logins',
      'log' : 'Generate syslogs on login events'
      }

   @staticmethod
   def handler( mode, args ):
      cfg = AaaCliLib.configAaa( mode )
      if 'on-success' in args: # pylint: disable=simplifiable-if-statement
         cfg.loggingOnSuccess = True
      else:
         cfg.loggingOnFailure = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'on-success' in args:
         AaaCliLib.configAaa( mode ).loggingOnSuccess = False
      else:
         AaaCliLib.configAaa( mode ).loggingOnFailure = False

configMode.addCommandClass( AaaAuthenPolicyLogCommand )

#-------------------------------------------------------------------------------
# [no|default] aaa authentication policy lockout failure < 1... >
# [ window < 0... > duration < 1-... >
#-------------------------------------------------------------------------------

secondsMatcher = CliMatcher.IntegerMatcher( 1, 2 ** 32 - 1,
                                            helpdesc='Number of seconds' )

class AaaAuthenPolicyLockoutCommand( CliCommand.CliCommandClass ):
   syntax = ( "aaa authentication policy lockout failure <failureCount>"
              " [ window <windowTime> ] duration <lockoutTime>" )
   noOrDefaultSyntax = "aaa authentication policy lockout ..."

   data = { "aaa": aaaKwMatcher,
            "authentication": authnKwMatcher,
            "policy": authnPolicyKwMatcher,
            "lockout": "Configure account lockout policy",
            "failure": "Number of failed consecutive logins before lockout",
            "<failureCount>": CliMatcher.IntegerMatcher( 1, 2**8 - 1,
                                        helpdesc='Number of failed logins allowed' ),
            "window": "Track failed logins within this duration, default 1 day",
            "<windowTime>": secondsMatcher,
            "duration": "Time in seconds to block account from login",
            "<lockoutTime>": secondsMatcher
          }

   @staticmethod
   def handler( mode, args ):
      cfg = AaaCliLib.configAaa( mode )
      cfg.maxLoginAttempts = args[ '<failureCount>' ]
      cfg.lockoutTime = args[ '<lockoutTime>' ]
      cfg.lockoutWindow = args.get( '<windowTime>' ) or cfg.lockoutWindowDefault

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfg = AaaCliLib.configAaa( mode )
      cfg.maxLoginAttempts = 0
      cfg.lockoutTime = cfg.lockoutDisabled
      cfg.lockoutWindow = cfg.lockoutWindowDefault

configMode.addCommandClass( AaaAuthenPolicyLockoutCommand )

#-------------------------------------------------------------------------------
# clear aaa authentication lockout [user <name>]
#-------------------------------------------------------------------------------
class AcctLockoutClear( CliCommand.CliCommandClass ):
   syntax = 'clear aaa authentication lockout [ user <name> ]'
   data = { 'clear' : CliToken.Clear.clearKwNode,
            'aaa' : aaaAfterClearMatcher,
            'authentication' : authnKwMatcher,
            'lockout' : 'Configure account lockout policy',
            'user' : 'Username to clear lockout information',
            '<name>' : usernameMatcher }

   @staticmethod
   def handler( mode, args ):
      user = args.get( '<name>' )
      currTime = Tac.now()
      if user:
         userLockoutConfig.clearLockoutRequest[ user ] = currTime
      else:
         userLockoutConfig.clearAllLockoutRequest = currTime
         # This is just for cleanup so clearLockoutRequest doesn't get too big.
         # We don't clean up clearLockoutRequest when we actually clear the
         # lockout in Aaa.py to avoid multiple writers.
         userLockoutConfig.clearLockoutRequest.clear()

BasicCli.EnableMode.addCommandClass( AcctLockoutClear )

#-------------------------------------------------------------------------------
# Register "show tech-support" commands
#-------------------------------------------------------------------------------

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2011-04-21 17:25:58',
   cmds=[ 'show aaa counters',
          'show users detail', 
          'show aaa accounting logs | tail -n 50' ],
   summaryCmds=[ 'show aaa counters',
                 'show users detail' ] )

#--------------------------------------------------
# Plugin method
# Mount the objects we need from Sysdb
#--------------------------------------------------
def Plugin( entityManager ):
   global aaaStatus, localConfig, acctLogFile, userLockoutConfig, accountsConfig, \
          securityConfig, mgmtSshConfig
   try:
      Tac.run( [ 'touch', acctLogFileName ], asRoot=True )
      Tac.run( [ 'chmod', '666', acctLogFileName ], asRoot=True )
      # pylint: disable-next=consider-using-with
      acctLogFile = open( acctLogFileName, 'a+' )
   except ( OSError, Tac.SystemCommandError ):
      pass

   aaaStatus = LazyMount.mount( entityManager,
                                Cell.path( "security/aaa/status" ),
                                "Aaa::Status", "r" )
   localConfig = ConfigMount.mount( entityManager, "security/aaa/local/config",
                                    "LocalUser::Config", "w" )
   mgmtSshConfig = ConfigMount.mount( entityManager, "mgmt/ssh/config",
                                  "Mgmt::Ssh::Config", "w" )
   userLockoutConfig = LazyMount.mount( entityManager,
                                        "security/aaa/userLockoutConfig",
                                        "Aaa::UserLockoutConfig", "w" )
   accountsConfig = LazyMount.mount( entityManager, "mgmt/acct/config",
                                     "Mgmt::Account::Config", "r" )
   securityConfig = LazyMount.mount( entityManager, "mgmt/security/config",
                                     "Mgmt::Security::Config", "r" )
   AaaCliLib.initLibrary( entityManager )
