#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AclLib
import BasicCliModes
import CliCommand
import CliMatcher
import CliPlugin.AclCli as AclCli # pylint: disable=consider-using-from-import
# pylint: disable-next=consider-using-from-import
import CliPlugin.AclCliRules as AclCliRules
import CliPlugin.DpAclCli as DpAclCli # pylint: disable=consider-using-from-import
from CliToken.Hardware import hardwareMatcherForConfig
from SysConstants.in_h import IPPROTO_ICMPV6

matcherAccessList = CliMatcher.KeywordMatcher( 'access-list',
      helpdesc='Hardware parameter related to access-list' )
matcherIn = CliMatcher.KeywordMatcher( 'in', helpdesc='Ingress' )
matcherKeyWidth = CliMatcher.KeywordMatcher( 'key-width', helpdesc='Key Width' )
matcherNarrow = CliMatcher.KeywordMatcher( 'narrow',
      helpdesc='Narrow mode. Allows only /64 IPv6' )
matcherPayload = CliMatcher.KeywordMatcher( 'payload',
      helpdesc='Payload parameters' )
matcherSharing = CliMatcher.KeywordMatcher( 'sharing',
      helpdesc='Sharing behaviour of hardware resources' )
matcherSkip = CliMatcher.KeywordMatcher( 'skip', helpdesc='Deep inspection offset' )
nodeDeepInspection = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'deep-inspection',
         helpdesc='Configure deep inspection parameters' ),
      guard=AclCli.dpAclDIParsingSupported )
nodeIpv6 = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'ipv6',
         helpdesc='IPV6 ACLS' ),
      guard=DpAclCli.dpIp6KeyWidthModeSupported )
matcherResource = CliMatcher.KeywordMatcher( 'resource',
      helpdesc='Hardware resources' )
nodeResource = CliCommand.Node( matcher=matcherResource,
      guard=DpAclCli.dpAclSharingSupported )
nodeVlan = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'vlan',
         helpdesc='Share hardware resources when access-list is attached to '
                  'VLAN interfaces' ),
      guard=DpAclCli.dpAclSharingSupported )
matcherDrop = CliMatcher.KeywordMatcher( 'drop',
      helpdesc='Drop' )

#--------------------------------------------------------------------------------
#  [ no | default ] deep-inspection payload l2 skip SKIP
#--------------------------------------------------------------------------------
class DeepInspectionPayloadL2SkipSkipCmd( CliCommand.CliCommandClass ):
   syntax = 'deep-inspection payload l2 skip SKIP'
   noOrDefaultSyntax = 'deep-inspection payload l2 skip ...'
   data = {
      'deep-inspection': nodeDeepInspection,
      'payload': matcherPayload,
      'l2': 'Deep inspection for L2 packets',
      'skip': matcherSkip,
      'SKIP': CliMatcher.DynamicIntegerMatcher( DpAclCli.deepInspMaxSkipL2Fn_,
         helpdesc='Words to skip ( in 32 bits )' ),
   }
   handler = DpAclCli.setDeepInspL2Offset
   noOrDefaultHandler = DpAclCli.noSetDeepInspL2Offset

BasicCliModes.GlobalConfigMode.addCommandClass( DeepInspectionPayloadL2SkipSkipCmd )

#--------------------------------------------------------------------------------
# [ no | default ] deep-inspection payload l4 skip SKIP
#--------------------------------------------------------------------------------
class DeepInspectionPayloadL4SkipSkipCmd( CliCommand.CliCommandClass ):
   syntax = 'deep-inspection payload l4 skip SKIP'
   noOrDefaultSyntax = 'deep-inspection payload l4 skip ...'
   data = {
      'deep-inspection': nodeDeepInspection,
      'payload': matcherPayload,
      'l4': 'Deep inspection for L4/unknown L4 packets',
      'skip': matcherSkip,
      'SKIP': CliMatcher.DynamicIntegerMatcher( DpAclCli.deepInspMaxSkipL4Fn_,
         helpdesc='Words to skip ( in 32 bits )' ),
   }
   handler = DpAclCli.setDeepInspL4Offset
   noOrDefaultHandler = DpAclCli.noSetDeepInspL4Offset

BasicCliModes.GlobalConfigMode.addCommandClass( DeepInspectionPayloadL4SkipSkipCmd )

#--------------------------------------------------------------------------------
# [ no | default ] hardware access-list ipv4 egress resource sharing
#                                                               routed-interfaces
#--------------------------------------------------------------------------------
class HwAclIpv4EgressSource( CliCommand.CliCommandClass ):
   syntax = 'hardware access-list ipv4 egress resource sharing routed-interfaces'
   noOrDefaultSyntax = syntax
   data = {
      'hardware': hardwareMatcherForConfig,
      'access-list': matcherAccessList,
      'ipv4': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'ipv4', helpdesc='IPv4' ),
         guard=DpAclCli.dpEgressRoutedInterfaceAclSharingSupported ),
      'egress': 'Egress',
      'resource': nodeResource,
      'sharing': matcherSharing,
      'routed-interfaces': 'Change behaviour on Routed Interfaces ACL',
   }
   hidden = True
   handler = DpAclCli.changeEgressRoutedInterfaceAclSharing
   noOrDefaultHandler = DpAclCli.changeNoEgressRoutedInterfaceAclSharing

BasicCliModes.GlobalConfigMode.addCommandClass( HwAclIpv4EgressSource )

#--------------------------------------------------------------------------------
# [ no | default ] hardware access-list ipv4-ipv6 mirroring-policy key-width
#                                                                           narrow in
#--------------------------------------------------------------------------------
class HwAclIpv4Ipv6MirroringPolicy( CliCommand.CliCommandClass ):
   syntax = 'hardware access-list ipv4-ipv6 mirroring-policy key-width narrow in'
   noOrDefaultSyntax = 'hardware access-list ipv4-ipv6 mirroring-policy key-width in'
   data = {
      'hardware': hardwareMatcherForConfig,
      'access-list': matcherAccessList,
      'ipv4-ipv6': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'ipv4-ipv6',
            helpdesc='Key width for Banks of IPv4 and IPv6 versions of feature' ),
         guard=DpAclCli.dpMirroringAclKeyWidthModeSupported ),
      'mirroring-policy': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'mirroring-policy',
            helpdesc='Mirroring Policy' ),
         guard=DpAclCli.dpMirroringAclKeyWidthModeSupported ),
      'in': matcherIn,
      'key-width': matcherKeyWidth,
      'narrow': matcherNarrow,
   }
   hidden = True
   handler = DpAclCli.gotoHardwareInMirroringAclKeyWidthModeNarrow
   noOrDefaultHandler = DpAclCli.gotoHardwareInMirroringAclKeyWidthModeNone

BasicCliModes.GlobalConfigMode.addCommandClass( HwAclIpv4Ipv6MirroringPolicy )

#--------------------------------------------------------------------------------
# ( no | default ) hardware access-list ipv6 implicit-permit
#                                 PROTOCOL ( all | neighbor-discovery | disabled)
#--------------------------------------------------------------------------------
class HwAclIpv6ImplicitPermitCmd( CliCommand.CliCommandClass ):
   syntax = ( 'hardware access-list ipv6 implicit-permit PROTOCOL '
              '( all | neighbor-discovery | disabled )' )
   noOrDefaultSyntax = ( 'hardware access-list ipv6 implicit-permit PROTOCOL ...' )
   data = {
      'hardware': hardwareMatcherForConfig,
      'access-list': matcherAccessList,
      'ipv6': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'ipv6', helpdesc='IPV6 ACLS' ),
         guard=DpAclCli.configurableIcmp6RulesSupportedGuard ),
      'implicit-permit': 'Add default icmpv6permit rules',
      'PROTOCOL': AclCliRules.ipProtocolMatcher( 'icmpv6', IPPROTO_ICMPV6,
         'Internet Control Message Protocol version 6' ),
      'all': CliCommand.guardedKeyword( 'all', helpdesc='Match on all ICMPv6 types',
                              guard=DpAclCli.permitOfAllIcmp6TypesSupportedGuard ),
      'neighbor-discovery': 'Match only on ICMPv6 Neighbor Discovery protocol types',
      'disabled': CliCommand.guardedKeyword( 'disabled', 
                        helpdesc='Disable implicit-permit of ICMPv6 protocol types',
                        guard=DpAclCli.disableImplictIcmp6RulesGuard ),
   }
   handler = DpAclCli.ipv6AclImplicitIcmp6RulesConfig
   noOrDefaultHandler = DpAclCli.noOrDefaultIpv6AclImplicitIcmp6RulesConfig 

BasicCliModes.GlobalConfigMode.addCommandClass( HwAclIpv6ImplicitPermitCmd )

#--------------------------------------------------------------------------------
# hardware access-list ipv6 ( qos-policy | security-acl ) key-width narrow in
# ( no | default ) hardware access-list ipv6 ( qos-policy | security-acl )
#                                                                        key-width in
#--------------------------------------------------------------------------------
class HwAclIpv6KeyWidth( CliCommand.CliCommandClass ):
   syntax = ( 'hardware access-list ipv6 ( qos-policy | security-acl ) key-width '
              'narrow in' )
   noOrDefaultSyntax = ( 'hardware access-list ipv6 ( qos-policy | security-acl ) '
                         'key-width in' )
   data = {
      'hardware': hardwareMatcherForConfig,
      'access-list': matcherAccessList,
      'ipv6': nodeIpv6,
      'qos-policy': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'qos-policy', helpdesc='Qos PolicyMap' ),
         guard=DpAclCli.dpIp6QosAclKeyWidthModeSupported ),
      'security-acl': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'security-acl', helpdesc='Port ACL' ),
         guard=DpAclCli.dpIp6SecurityAclKeyWidthModeSupported ),
      'key-width': matcherKeyWidth,
      'narrow': matcherNarrow,
      'in': matcherIn,
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      if 'qos-policy' in args:
         DpAclCli.gotoHardwareInIpv6QosAclKeyWidthMode( mode,
            sharingMode=AclLib.TcamBankSharingMode.bankSharingModeNarrow )
      elif 'security-acl' in args:
         DpAclCli.gotoHardwareInIpv6SecurityAclKeyWidthMode( mode,
            sharingMode=AclLib.TcamBankSharingMode.bankSharingModeNarrow )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'qos-policy' in args:
         DpAclCli.gotoHardwareInIpv6QosAclKeyWidthMode( mode,
            sharingMode=AclLib.TcamBankSharingMode.bankSharingModeNone )
      elif 'security-acl' in args:
         DpAclCli.gotoHardwareInIpv6SecurityAclKeyWidthMode( mode,
            sharingMode=AclLib.TcamBankSharingMode.bankSharingModeNone )

BasicCliModes.GlobalConfigMode.addCommandClass( HwAclIpv6KeyWidth )

#--------------------------------------------------------------------------------
# [ no | default ] hardware access-list resource sharing ipv4-ipv6
#                                 ( mirroring-policy | qos-policy | security-acl ) in
#--------------------------------------------------------------------------------
class HwAclResourceSharingIpv4Ipv6( CliCommand.CliCommandClass ):
   syntax = ( 'hardware access-list resource sharing ipv4-ipv6 '
              '( mirroring-policy | qos-policy | security-acl ) in' )
   noOrDefaultSyntax = syntax
   data = {
      'hardware': hardwareMatcherForConfig,
      'access-list': matcherAccessList,
      'resource': matcherResource,
      'sharing': matcherSharing,
      'ipv4-ipv6': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'ipv4-ipv6',
            helpdesc='Share TCAM Banks for IPv4 and IPv6 versions of feature' ),
         guard=DpAclCli.dpIpv4Ip6BankSharingSupported ),
      'qos-policy': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'qos-policy', helpdesc='Qos PolicyMap' ),
         guard=DpAclCli.dpIpv4Ip6QosAclBankSharingSupported ),
      'security-acl': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'security-acl', helpdesc='Port ACL' ),
         guard=DpAclCli.dpIpv4Ip6SecurityAclBankSharingSupported ),
      'mirroring-policy': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'mirroring-policy',
            helpdesc='Mirroring ACL' ),
         guard=DpAclCli.dpIpv4Ip6MirroringAclBankSharingSupported ),
      'in': matcherIn,
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      if 'mirroring-policy' in args:
         DpAclCli.gotoHardwareInIpv4Ipv6MirroringAclSharing( mode, True )
      elif 'qos-policy' in args:
         DpAclCli.gotoHardwareInIpv4Ipv6QosAclSharing( mode, True )
      elif 'security-acl' in args:
         DpAclCli.gotoHardwareInIpv4Ipv6SecurityAclSharing( mode, True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'mirroring-policy' in args:
         DpAclCli.gotoHardwareInIpv4Ipv6MirroringAclSharing( mode, False )
      elif 'qos-policy' in args:
         DpAclCli.gotoHardwareInIpv4Ipv6QosAclSharing( mode, False )
      elif 'security-acl' in args:
         DpAclCli.gotoHardwareInIpv4Ipv6SecurityAclSharing( mode, False )

BasicCliModes.GlobalConfigMode.addCommandClass( HwAclResourceSharingIpv4Ipv6 )

#--------------------------------------------------------------------------------
# [ no | default ] hardware access-list resource sharing vlan in
#--------------------------------------------------------------------------------
class HwAclResourceSharingVlanInCmd( CliCommand.CliCommandClass ):
   syntax = 'hardware access-list resource sharing vlan in'
   noOrDefaultSyntax = syntax
   data = {
      'hardware': hardwareMatcherForConfig,
      'access-list': matcherAccessList,
      'resource': matcherResource,
      'sharing': matcherSharing,
      'vlan': nodeVlan,
      'in': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'in', helpdesc='Inbound packets' ),
         guard=DpAclCli.dpIngressRaclSharingSupported ),
   }
   handler = DpAclCli.gotoHardwareInAclSharing
   noOrDefaultHandler = DpAclCli.gotoNoHardwareInAclSharing

BasicCliModes.GlobalConfigMode.addCommandClass( HwAclResourceSharingVlanInCmd )

#--------------------------------------------------------------------------------
# [ no | default ] hardware access-list resource sharing vlan ( ipv4 | ipv6 ) out
#--------------------------------------------------------------------------------
class HwAclResourceSharingVlanIpv4OutCmd( CliCommand.CliCommandClass ):
   syntax = 'hardware access-list resource sharing vlan AF out'
   noOrDefaultSyntax = syntax
   data = {
      'hardware': hardwareMatcherForConfig,
      'access-list': matcherAccessList,
      'resource': matcherResource,
      'sharing': matcherSharing,
      'vlan': nodeVlan,
      'AF': CliCommand.Node(
         matcher=CliMatcher.EnumMatcher( {
            'ipv4': 'IPv4',
            'ipv6': 'IPv6',
         } ),
         guard=DpAclCli.dpEgressRaclSharingSupported ),
      'out': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'out', helpdesc='Outbound packets' ),
         guard=DpAclCli.dpAclSharingSupported ),
   }

   @staticmethod
   def handler( mode, args ):
      if args[ 'AF' ] == 'ipv4':
         DpAclCli.gotoHardwareOutIpv4AclSharing( mode, True )
      else:
         DpAclCli.gotoHardwareOutIp6AclSharing( mode, True )

   @staticmethod
   def noHandler( mode, args ):
      if args[ 'AF' ] == 'ipv4':
         DpAclCli.gotoHardwareOutIpv4AclSharing( mode, False )
      else:
         DpAclCli.gotoHardwareOutIp6AclSharing( mode, False )

   @staticmethod
   def defaultHandler( mode, args ):
      if args[ 'AF' ] == 'ipv4':
         DpAclCli.gotoHardwareOutIpv4AclSharing( mode, True )
      else:
         DpAclCli.gotoHardwareOutIp6AclSharing( mode, False )

BasicCliModes.GlobalConfigMode.addCommandClass( HwAclResourceSharingVlanIpv4OutCmd )

#--------------------------------------------------------------------------------
# [ no | default ] hardware access-list router-acl exclude mlag peer-link
#--------------------------------------------------------------------------------
class HwAclRouterAcl( CliCommand.CliCommandClass ):
   syntax = 'hardware access-list router-acl exclude mlag peer-link'
   noOrDefaultSyntax = syntax
   data = {
      'hardware': hardwareMatcherForConfig,
      'access-list': matcherAccessList,
      'router-acl': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'router-acl',
            helpdesc='Change behaviour on Router ACL' ),
         guard=DpAclCli.dpRouterAclExcludeMlagPeerLink ),
      'exclude': 'Exclude application of access-list on particular interfaces',
      'mlag': 'MLAG interfaces',
      'peer-link': 'Peer-link interfaces',
   }
   handler = DpAclCli.excludeMlagPeerLink
   noOrDefaultHandler = DpAclCli.noExcludeMlagPeerLink

BasicCliModes.GlobalConfigMode.addCommandClass( HwAclRouterAcl )

#--------------------------------------------------------------------------------
# [ no | default ] hardware access-list logging egress ipv4 recirculate
#--------------------------------------------------------------------------------
class RecircEgIpAclLog( CliCommand.CliCommandClass ):
   syntax = 'hardware access-list logging egress ipv4 recirculate'
   noOrDefaultSyntax = syntax
   data = {
      'hardware': hardwareMatcherForConfig,
      'access-list': matcherAccessList,
      'logging': CliCommand.Node(
        matcher=CliMatcher.KeywordMatcher( 'logging',
        helpdesc='Access Control List logging' ),
        guard=DpAclCli.recycleEgressIpAclLogSupportedGuard ),
      'egress': CliCommand.Node(
        matcher=CliMatcher.KeywordMatcher( 'egress',
        helpdesc='Egress Interface' )),
      'ipv4': CliCommand.Node(
        matcher=CliMatcher.KeywordMatcher( 'ipv4',
        helpdesc='IPv4' )),
      'recirculate': CliCommand.Node(
        matcher=CliMatcher.KeywordMatcher( 'recirculate',
        helpdesc='Enable Recirculation' )),
   }
   handler = DpAclCli.recycleEgressIpAclLog
   noOrDefaultHandler = DpAclCli.noRecycleEgressIpAclLog

BasicCliModes.GlobalConfigMode.addCommandClass( RecircEgIpAclLog )

#--------------------------------------------------------------------------------
# [ no | default ] hardware access-list logging egress recirculate
# --------------------------------------------------------------------------------
class RecircEgAclLog( CliCommand.CliCommandClass ):
   syntax = 'hardware access-list logging egress recirculate'
   noOrDefaultSyntax = syntax
   data = {
      'hardware': hardwareMatcherForConfig,
      'access-list': matcherAccessList,
      'logging': CliCommand.Node(
        matcher=CliMatcher.KeywordMatcher( 'logging',
        helpdesc='Access Control List logging' ),
        guard=DpAclCli.recycleEgressAclLogSupportedGuard ),
      'egress': 'Egress Interface',
      'recirculate': 'Enable Recirculation',
   }
   handler = DpAclCli.recycleEgressAclLog
   noOrDefaultHandler = DpAclCli.noRecycleEgressAclLog

BasicCliModes.GlobalConfigMode.addCommandClass( RecircEgAclLog )

# --------------------------------------------------------------------------------
# [ no | default ] hardware access-list update default-result permit
#--------------------------------------------------------------------------------
class HwAclUpdate( CliCommand.CliCommandClass ):
   syntax = 'hardware access-list update default-result permit'
   noOrDefaultSyntax = syntax
   data = {
      'hardware': hardwareMatcherForConfig,
      'access-list': matcherAccessList,
      'update': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'update',
            helpdesc='Hardware behavior while access-list is being updated' ),
         guard=DpAclCli.dpPermitDuringAclUpdate ),
      'default-result': ( 'Default action on packets while access-list is '
                          'being updated' ),
      'permit': 'Accept the packets when access-list is being updated',
   }
   handler = DpAclCli.gotoHardwareAclUpdate
   noHandler = DpAclCli.gotoNoHardwareAclUpdate
   defaultHandler = DpAclCli.gotoDefaultHardwareAclUpdate

BasicCliModes.GlobalConfigMode.addCommandClass( HwAclUpdate )
