# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string
# pylint: disable=singleton-comparison

from Intf.IntfRange import intfListToCanonical
import AclLib
import Ark
import Tac
import Toggles.AclToggleLib
import Tracing
import re
t0 = Tracing.trace0
t8 = Tracing.trace8

RangeOperator = Tac.Type( "Acl::RangeOperator" )

# ACl CLI model class instances delegate to per-class renderers.
# The renderer methods are class methods to allow polymorphism.

class PhoneAclBypassRenderer:
   @classmethod
   def render( cls, aclList ):
      if Toggles.AclToggleLib.toggleEnhancedAclSummaryEnabled():
         if aclList.phoneAclBypass:
            print( "Phone ACL bypass: enabled" )
         else:
            print( "Phone ACL bypass: disabled" )

class ListElementRenderer:
   """ This is the base class for list element renderers (remarks,
   ip rules and and mac rules). Subclasses must implement the render
   method with the signature below. """

   @classmethod
   def render( cls, elem ):
      """ Overridden to render the given list element. """
      pass # pylint: disable=unnecessary-pass

class RuleRenderer( ListElementRenderer ):
   """ Base renderer for all rules (mac, ip4 and ip6). """

   @classmethod
   def render( cls, rule, indent=2 ): # pylint: disable=arguments-renamed
      """ Formats the rule, prepending with the sequence number
      and appending the counters. """
      print( ' ' * 4 * indent + "%s %s%s" % ( rule.getSequenceNumberString(),
                                              rule.text,
                                              cls._getRuleCounterString( 
                                                 rule.counterData ) ) )

   @classmethod
   def _getRuleCounterString( cls, counterData ):
      """ Formats the counter presentation. """
      if counterData != None and counterData.lastChangedTime != None:
         pktStr = ''
         connStr = ''
         if counterData.packetCount != None and \
               counterData.checkpointPacketCount != None:
            if counterData.packetCount > counterData.checkpointPacketCount:
               if counterData.byteCount and counterData.checkpointByteCount != None \
                  and Toggles.AclToggleLib.togglePrintByteCountersEnabled():
                  # assumption that byteCount > checkpointByteCount given that
                  # packetCount > checkpointPacketCount
                  pktStr = '%ld bytes in %ld packets,' % (
                     counterData.byteCount - counterData.checkpointByteCount,
                     counterData.packetCount - counterData.checkpointPacketCount )
               else:
                  pktStr = '%ld packets,' % ( counterData.packetCount - \
                                              counterData.checkpointPacketCount )

         if counterData.connCount != None and \
               counterData.checkpointPacketCount != None:
            if counterData.connCount > counterData.checkpointConnCount:
               space = ' ' if pktStr else ''
               connStr = '%s%ld connections,' % ( space, counterData.connCount - \
                                 counterData.checkpointConnCount )
         if pktStr or connStr:
            return " [match %s%s %s]" % ( pktStr, connStr,
                 Ark.timestampToStr( counterData.lastChangedTime ) )
      return ""

class AclBaseRenderer:
   """ Base class for two presentations of ACLs, full and summary.
   Both use the utility function _getAnnotationsString. """

   @classmethod
   def _getAnnotationsString( cls, acl ):
      """ Formats the ACL annotation text. Copied from
      original. """

      annotations = []
      if acl.readonly:
         annotations.append( "readonly" )
      if acl.countersEnabled and acl.countersIncomplete:
         annotations.append( "incomplete statistics" )
      if acl.dynamic:
         annotations.append( "dynamic" )
      if annotations:
         return " [" + ", ".join( annotations ) + "]"
      else:
         return ""

class AclRenderer( AclBaseRenderer ):
   """ This is the renderer for full display of ACLs. It is invoked
   from the ACLList renderer through the Acl render method. It in turn
   calls the renderer for each element in the list. """

   @classmethod
   def render( cls, acl, header=True ):
      """ Presents the overall ACL properties, then invokes
      the renderer for each element in the list. """

      if header:
         print( "%s%s Access List %s%s" %
             ( 'Standard ' if acl.standard else '',
               cls._aclTypeEnumToDisplay( acl.type ).upper(),
               acl.name,
               cls._getAnnotationsString( acl ) ) )
          
      if acl.countersEnabled and acl.staleCounters:
         # somehow we timed out, but we'll show whatever we have
         print( "Warning: displaying stale counters" )
      
      if acl.chipName:
         # print chipName for 'detail' option
         print( "%s" % acl.chipName )

      if acl.countersEnabled:
         print( "        counters per-entry" )

      for elem in acl.sequence:
         elem.render( 2 )

      if acl.implicitRule:
         for elem in acl.implicitRule:
            elem.render( 2 )

      if acl.permitResponse == AclLib.AclPermitResponseType.natTraffic:
         print( ' ' * 4 * 2 + 'permit response traffic nat' )

      # pylint: disable-msg=W0212
      counterStr = RuleRenderer._getRuleCounterString( acl.noMatchCounter )
      if counterStr:
         print( "        implicit deny for no rules match%s" % counterStr )

      if acl.summary:
         acl.summary.renderAfter()

      print( "" )

   @classmethod
   def _aclTypeEnumToDisplay( cls, aclTypeEnum ):
      """ Converts the ACL type to the original display text. """

      if aclTypeEnum == 'Ip4Acl':
         return 'ip'
      if aclTypeEnum == 'Ip6Acl':
         return 'ipv6'
      if aclTypeEnum == 'MacAcl':
         return 'mac'
      return 'other' # future-proofing

class ServiceAclSummaryRenderer( AclBaseRenderer ):
   """ This is the renderer for the service ACL summary display. It is invoked
   from the ACLList renderer through the ServiceAclSummary render method. """

   @classmethod
   def render( cls, acl ):
      if acl.type == 'Ip4Acl':
         print( "%sIPv4 ACL %s%s" % ( 'Standard ' if acl.standard else '',
                                     acl.name, cls._getAnnotationsString( acl ) ) )
      elif acl.type == 'Ip6Acl':
         print( "%sIPv6 ACL %s%s" % ( 'Standard ' if acl.standard else '',
                                     acl.name, cls._getAnnotationsString( acl ) ) )
      sequenceLength = acl.sequenceLength
      if acl.permitResponse == AclLib.AclPermitResponseType.natTraffic:
         sequenceLength = acl.sequenceLength + 1
      print( "        Total rules configured:", sequenceLength )
      cls._printVrfs( "Configured on VRFs:", acl.configuredVrfs )
      cls._printVrfs( "Active on VRFs:", acl.activeVrfs )
      print( "" )

   @classmethod
   def renderAfter( cls, acl ):
      sequenceLength = acl.sequenceLength
      if acl.permitResponse == AclLib.AclPermitResponseType.natTraffic:
         sequenceLength = acl.sequenceLength + 1
      print( "Total rules configured:", sequenceLength )
      cls._printVrfs( "Configured on VRFs:", acl.configuredVrfs )
      cls._printVrfs( "Active on VRFs:", acl.activeVrfs )
      print( "" )

   @classmethod
   def _printVrfs( cls, title, vrfs ):
      if not vrfs:
         return
      indent = 8
      indent2 = indent + len( title )
      firstLine = True
      for vrf in sorted( vrfs ):
         if firstLine:
            print( '{0:{1}}{2} {3} VRF'.format( '', indent, title, vrf ) )
            firstLine = False
         else:
            print( '{0:{1}} {2} VRF'.format( '', indent2, vrf ) )

class AclSummaryRenderer( AclBaseRenderer ):
   """ This is the renderer for the ACL summary display. It is invoked
   from the ACLList renderer through the AclSummary render method. """

   @classmethod
   def render( cls, acl, onlySummary=True ):
      """ The render code is taken from the original implementation. """
      if onlySummary:
         # The onlySummary flag enables printing the ACL headers if summary
         # token is passed. It prohibits multiple headers for the same ACL to be
         # printed when the summary token is not passed as the output contains
         # both the rules and the configuration and status.
         if acl.type == 'Ip4Acl':
            print( "%sIPV4 ACL %s%s" % ( 'Standard ' if acl.standard else '',
                                       acl.name, cls._getAnnotationsString( acl ) ) )
         elif acl.type == 'Ip6Acl':
            print( "%sIPV6 ACL %s%s" % ( 'Standard ' if acl.standard else '',
                                       acl.name, cls._getAnnotationsString( acl ) ) )
         else:
            print( "MAC ACL %s%s" % ( acl.name, cls._getAnnotationsString( acl ) ) )
      sequenceLength = acl.sequenceLength
      if acl.permitResponse == AclLib.AclPermitResponseType.natTraffic:
         sequenceLength = acl.sequenceLength + 1
      print( "        Total rules configured:", sequenceLength )
      cls._printIntfs( "Configured on",
                       acl.configuredIngressIntfs,
                       acl.configuredEgressIntfs )
      cls._printIntfs( "Active on",
                       acl.activeIngressIntfs,
                       acl.activeEgressIntfs )

      if Toggles.AclToggleLib.toggleEnhancedAclSummaryEnabled():
         if ( Toggles.AclToggleLib.togglePerMacAclSummaryEnabled() and
              acl.activeFieldset ):
            print( "        Active field set:", *acl.activeFieldset )
         if acl.dot1x:
            cls._printDot1x( acl.dot1x )

      if acl.notFullySupportedByHardware:
         print( "      * Some rules are not fully supported by hardware" )

      print( "" )

   @classmethod
   def _parseMacAddr( cls, macAddr ):
      return '.'.join( re.findall( '....', ''.join( macAddr.split( ':' ) ) ) )

   @classmethod
   def _getSupplicants( cls, supplicantData ):
      supplicants = []
      for supplicant in supplicantData:
         if supplicant[ "macAddr" ] == supplicant[ "username" ]:
            supplicants.append( cls._parseMacAddr( str( supplicant[ "macAddr" ] ) ) )
         else:
            supplicantWithUsername = cls._parseMacAddr(
               str( supplicant[ "macAddr" ] ) ) + " (" + str(
                  supplicant[ "username" ] ) + ")"
            supplicants.append( supplicantWithUsername )
      return supplicants

   @classmethod
   def _printDot1xIntfs( cls, dot1xAclMode ):
      if not dot1xAclMode:
         return
      indent = 12
      for intf in dot1xAclMode:
         intfDict = intf.toDict()
         supplicants = ', '.join( cls._getSupplicants( intfDict[ "supplicants" ] ) )
         print( '{0:{1}}{2} {3}'.format( '', indent,
                                         "Interface:", intfDict[ "name" ] ) )
         if supplicants:
            print( '{0:{1}}{2} {3}'.format( '', indent + 4, "Supplicants:",
                                            supplicants ) )

   @classmethod
   def _printDot1x( cls, dot1x ):
      indent = 8
      if dot1x.perMacAclMode:
         print( '{0:{1}}{2}'.format( '', indent,
                                         "Dot1x Per-Mac ACL Mode" ) )
         cls._printDot1xIntfs( dot1x.perMacAclMode )
      if dot1x.interfaceAclMode:
         print( '{0:{1}}{2}'.format( '', indent,
                                         "Dot1x Interface ACL Mode" ) )
         cls._printDot1xIntfs( dot1x.interfaceAclMode )

   @classmethod
   def _printIntfs( cls, title, intfsIn, intfsOut ):
      """ Re-uses the Intf.IntfRange method to format. """

      if not intfsIn and not intfsOut : 
         return
      intfs = { 'Ingress' : cls._appendTarget( intfsIn ),
                'Egress' : cls._appendTarget( intfsOut ) }
      for direction, intfRangeList in intfs.items():
         firstLine = True
         for intf in intfRangeList:
            if firstLine:
               print( "%8s%-13s %s: %s" % ( '', title, direction, intf ) )
               firstLine = False
            else:
               print( "%23s%s" % ( '', intf ) )

   @classmethod
   def _appendTarget( cls, intfs ):
      """  Return list of targets where the acl is used. """

      aclInUse = {}
      summary = []
      for intf in intfs:
         if intf.vrf:
            summary.append( '%s(%s VRF)' % ( intf.name, intf.vrf ) )
         else:
            if intf.target in aclInUse:
               aclInUse[ intf.target ].append( intf.name )
            else:
               aclInUse[ intf.target ] = [ intf.name ]

      for target, intfList in aclInUse.items():
         if target:
            summary.append( '%s(%s)' % \
               ( target, ','.join( intfListToCanonical( intfList, strLimit=50 ) ) ) )
         else:
            summary.append(
                     ','.join( intfListToCanonical( intfList, strLimit=50 ) ) )

      return sorted( summary )


class AclListRenderer:
   """ This is the renderer for the top-level CLI model returned
   by the value function. It calls the renderer for each item in the
   list, which may be either Acl or AclSummary models. """

   @classmethod
   def render( cls, aclList ):
      for acl in aclList.aclList:
         acl.render()

class DpiRenderfn:
   @classmethod
   def DpiRender( cls, skipval ):
      print( "deep-inspection payload skipped value for L2", skipval.l2SkipValue )
      print( "deep-inspection payload skipped value for L4", skipval.l4SkipValue )

class UdfAliasRenderfn:
   """ This is the renderer for udf aliases and their payloads"""
   @classmethod
   def _payloadFromSpec( cls, payloadSpec ):
      cmd = ''
      if not payloadSpec.payload:
         return cmd
      cmd = 'payload'

      if payloadSpec.headerType == 'l4Header':
         direction = 'start' if payloadSpec.headerStart else 'end'
         cmd = cmd + ' header layer-4 ' + direction
      elif payloadSpec.headerStart:
         cmd = cmd + ' header start'

      for payload in payloadSpec.payload:
         offset = payload.offset
         pattern = payload.pattern
         mask = payload.patternMask
         # Mask should be displayed as an inverse mask
         displayMask = 0xFFFFFFFF ^ mask
         alias = payload.alias
         patternOverride = payload.patternOverride
         maskOverride = payload.maskOverride

         if alias:
            cmd += ' alias %s' % alias
            if patternOverride:
               cmd += ' pattern 0x%08x' % pattern
            if maskOverride:
               cmd += ' mask 0x%08x' % displayMask
         else:
            cmd = cmd + ' offset %d pattern 0x%08x mask 0x%08x' % (
                                 offset, pattern, displayMask )
         return cmd

   @classmethod
   def render( cls, udfAlias ):
      udfAliases = udfAlias.udfAliases
      if udfAliases:
         print( '%s %20s' % ( 'Name', 'Alias'  ) )
      for name, value in udfAliases.items():
         payload = cls._payloadFromSpec( value )
         print( '%-19s %s' % ( name, payload ) )
