#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AgentCommandRequest
import ArPyUtils
import BasicCli
import Cell
import CliCommand
import CliMatcher
import LazyMount
import ShowCommand
from CliPlugin.TrafficPolicyCli import guardAegisInterface
from CliPlugin.AegisCliModel import PlatformStageModel
from CliToken.Platform import platformMatcherForShow
from Toggles.AegisToggleLib import toggleAegisDumpLabelsEnabled
import Tac
import sys

policiesStatus = None
progressLogStatus = None

def chipNamesFunc( _ ):
   chipNames = {}
   for tpStatus in policiesStatus.values():
      for chipName in tpStatus.chipNameToId:
         assert chipName not in chipNames
         chipNames[ chipName ] = ''
   return chipNames

def chipNameToId( chipName ):
   for tpStatus in policiesStatus.values():
      if chipName in tpStatus.chipNameToId:
         return tpStatus.chipNameToId[ chipName ]
   return None

# cmds = [ ( cmdTemplate, asyncCommand ) ]
def agentCommandRequest( mode, args, cmds ):
   if 'CHIP' in args:
      chips = [ args[ 'CHIP' ] ]
   else:
      chips = ArPyUtils.naturalsorted( chipNamesFunc( None ) )

   for chipName in chips:
      print( chipName )
      chipId = chipNameToId( chipName )
      if chipId is None:
         # pylint: disable-next=consider-using-f-string
         print( 'warning: %s could not be mapped to a chipId' % chipName )
         continue

      for cmdTemplate, asyncCommand in cmds:
         AgentCommandRequest.runSocketCommand( mode.entityManager,
                                               "aegis",
                                               "Aegis",
                                               cmdTemplate % chipId,
                                               asyncCommand=asyncCommand )

def doShowPlatformAegisAlgoXform( mode, args ):
   cmds = [ ( 'dumpPdConfigPrefixXform chipId=%d', True ) ]
   cmds += [ ( 'dumpPdConfigPortXform chipId=%d', False ) ]
   cmds += [ ( 'dumpPdConfigGtpv1TeidXform chipId=%d', True ) ]
   agentCommandRequest( mode, args, cmds )

def doShowPlatformAegisAlgoXformPrefix( mode, args ):
   cmds = [ ( 'dumpPdConfigPrefixXform chipId=%d', True ) ]
   agentCommandRequest( mode, args, cmds )

def doShowPlatformAegisAlgoXformPort( mode, args ):
   cmds = [ ( 'dumpPdConfigPortXform chipId=%d', False ) ]
   agentCommandRequest( mode, args, cmds )

def doShowPlatformAegisAlgoXformTeid( mode, args ):
   cmds = [ ( 'dumpPdConfigGtpv1TeidXform chipId=%d', True ) ]
   agentCommandRequest( mode, args, cmds )

def doShowPlatformAegisAlgoRules( mode, args ):
   cmds = [ ( 'dumpPdConfigRules chipId=%d', True ) ]
   agentCommandRequest( mode, args, cmds )

def doShowPlatformAegisAlgoSummary( mode, args ):
   cmds = [ ( 'summary chipId=%d', False ) ]
   agentCommandRequest( mode, args, cmds )

def doShowPlatformAegisAlgoLabels( mode, args ):
   detail = 1 if 'detail' in args else 0
   # pylint: disable-next=consider-using-f-string
   cmds = [ ( 'dumpLabels chipId=%d ' + 'detail=%d' % detail, False ) ]
   agentCommandRequest( mode, args, cmds )

aegisalgo = CliCommand.Node(
   matcher=CliMatcher.KeywordMatcher( 'aegisalgo', helpdesc='Aegis ACL' ),
   guard=guardAegisInterface )

#----------------------------------------------------------------------------
# show platform aegisalgo [ chip CHIP ] transforms
#----------------------------------------------------------------------------
class ShowPlatformAegisAlgoXform( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform aegisalgo [ chip CHIP ] transforms'
   data = {
      'platform' : platformMatcherForShow,
      'aegisalgo' : aegisalgo,
      'chip' : 'select a chip',
      'CHIP' : CliMatcher.DynamicKeywordMatcher( chipNamesFunc ),
      'transforms' : 'show transforms',
   }

   handler = doShowPlatformAegisAlgoXform
   privileged = True

BasicCli.addShowCommandClass( ShowPlatformAegisAlgoXform )

#----------------------------------------------------------------------------
# show platform aegisalgo [ chip CHIP ] transforms prefix
#----------------------------------------------------------------------------
class ShowPlatformAegisAlgoXformPrefix( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform aegisalgo [ chip CHIP ] transforms prefix'
   data = {
      'platform' : platformMatcherForShow,
      'aegisalgo' : aegisalgo,
      'chip' : 'select a chip',
      'CHIP' : CliMatcher.DynamicKeywordMatcher( chipNamesFunc ),
      'transforms' : 'show transforms',
      'prefix' : 'show prefix transforms',
   }

   handler = doShowPlatformAegisAlgoXformPrefix
   privileged = True

BasicCli.addShowCommandClass( ShowPlatformAegisAlgoXformPrefix )

#----------------------------------------------------------------------------
# show platform aegisalgo [ chip CHIP ] transforms port
#----------------------------------------------------------------------------
class ShowPlatformAegisAlgoXformPort( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform aegisalgo [ chip CHIP ] transforms port'
   data = {
      'platform' : platformMatcherForShow,
      'aegisalgo' : aegisalgo,
      'chip' : 'select a chip',
      'CHIP' : CliMatcher.DynamicKeywordMatcher( chipNamesFunc ),
      'transforms' : 'show transforms',
      'port' : 'show port transforms',
   }

   handler = doShowPlatformAegisAlgoXformPort
   privileged = True

BasicCli.addShowCommandClass( ShowPlatformAegisAlgoXformPort )

#----------------------------------------------------------------------------
# show platform aegisalgo [ chip CHIP ] transforms gtpv1 teid
#----------------------------------------------------------------------------
class ShowPlatformAegisAlgoXformTeid( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform aegisalgo [ chip CHIP ] transforms gtpv1 teid'
   data = {
      'platform' : platformMatcherForShow,
      'aegisalgo' : aegisalgo,
      'chip' : 'select a chip',
      'CHIP' : CliMatcher.DynamicKeywordMatcher( chipNamesFunc ),
      'transforms' : 'show transforms',
      'gtpv1' : 'show gtpv1 transforms',
      'teid' : 'show teid transforms',
   }

   handler = doShowPlatformAegisAlgoXformTeid
   privileged = True

BasicCli.addShowCommandClass( ShowPlatformAegisAlgoXformTeid )

#----------------------------------------------------------------------------
# show platform aegisalgo [ chip CHIP ] rules
#----------------------------------------------------------------------------
class ShowPlatformAegisAlgoRules( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform aegisalgo [ chip CHIP ] rules'
   data = {
      'platform' : platformMatcherForShow,
      'aegisalgo' : aegisalgo,
      'chip' : 'select a chip',
      'CHIP' : CliMatcher.DynamicKeywordMatcher( chipNamesFunc ),
      'rules' : 'show rules',
   }

   handler = doShowPlatformAegisAlgoRules
   privileged = True

BasicCli.addShowCommandClass( ShowPlatformAegisAlgoRules )

#----------------------------------------------------------------------------
# show platform aegisalgo [ chip CHIP ] summary
#----------------------------------------------------------------------------
class ShowPlatformAegisAlgoSummary( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform aegisalgo [ chip CHIP ] summary'
   data = {
      'platform' : platformMatcherForShow,
      'aegisalgo' : aegisalgo,
      'chip' : 'select a chip',
      'CHIP' : CliMatcher.DynamicKeywordMatcher( chipNamesFunc ),
      'summary' : 'show summary',
   }

   handler = doShowPlatformAegisAlgoSummary
   privileged = True

BasicCli.addShowCommandClass( ShowPlatformAegisAlgoSummary )

# ----------------------------------------------------------------------------
# show platform aegisalgo stage [ last [ n ] ]
# ----------------------------------------------------------------------------
MAX_BATCHES_NUM = 3
class ShowPlatformAegisProgressLog( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform aegisalgo stage [ last [ N ] ]'
   data = {
      'platform' : platformMatcherForShow,
      'aegisalgo' : aegisalgo,
      'stage' : 'show traffic-policy progress log',
      'last' : 'show latest n batch(es) from newest to oldest',
      'N' : CliMatcher.IntegerMatcher( 1, MAX_BATCHES_NUM,
                                       helpdesc='batch number' )
   }
   privileged = True
   cliModel = PlatformStageModel

   @staticmethod
   def handler( mode, args ):
      if 'last' in args:
         batchNumRequested = args.get( 'N', 1 )
      else:
         batchNumRequested = MAX_BATCHES_NUM
      LazyMount.force( progressLogStatus )
      chipNameToIdMap = Tac.newInstance( 'AegisCli::ChipNameToId' )
      chipNames = ArPyUtils.naturalsorted( chipNamesFunc( None ) )
      for chipName in chipNames:
         chipNameToIdMap.chipName.enq( chipName )
         chipNameToIdMap.chipId[ chipName ] = chipNameToId( chipName )
      helper = Tac.newInstance( 'AegisCli::ShowAegisProgressLog',
                                progressLogStatus, batchNumRequested,
                                chipNameToIdMap )
      fd = sys.stdout.fileno()
      fmt = mode.session_.outputFormat()
      helper.render( fd, fmt )

      return PlatformStageModel

BasicCli.addShowCommandClass( ShowPlatformAegisProgressLog )

# ----------------------------------------------------------------------------
# show platform aegisalgo [ chip CHIP ] labels [ detail ]
# ----------------------------------------------------------------------------
class ShowPlatformAegisAlgoLabels( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform aegisalgo [ chip CHIP ] labels [ detail ]'
   data = {
      'platform' : platformMatcherForShow,
      'aegisalgo' : aegisalgo,
      'chip' : 'select a chip',
      'CHIP' : CliMatcher.DynamicKeywordMatcher( chipNamesFunc ),
      'labels' : 'show labels',
      'detail' : 'show detail',
   }

   handler = doShowPlatformAegisAlgoLabels
   privileged = True

if toggleAegisDumpLabelsEnabled():
   BasicCli.addShowCommandClass( ShowPlatformAegisAlgoLabels )

#-------------------------------------------------------------------------------
# Mount the needed sysdb state
#-------------------------------------------------------------------------------
def Plugin( em ):
   global policiesStatus
   # pylint: disable-next=consider-using-f-string
   policiesCellStatusPath = 'cell/%d/trafficPolicies/status' % Cell.cellId()
   policiesStatusType = 'Tac::Dir'
   policiesStatus = LazyMount.mount( em, policiesCellStatusPath, policiesStatusType,
                                     'ri' )
   global progressLogStatus
   progressLogStatus = LazyMount.mount( em, 'trafficPolicies/status/aegis',
                                        'Aegis::ProgressLogStatus', 'r' )
