#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from datetime import timedelta

from Ark import timestampToStr
from CliModel import Bool, Dict, Int, Float, Model, Str
from TableOutput import createTable, Format
from Tac import utcNow

AGENT_NOT_RUNNING_REPR = '---'

class AgentThreadCpuEntry( Model ):
   name = Str( help="Thread name" )
   lastCpuPct = Int( help="Thread current CPU usage (percent)" )
   maxCpuPct = Int( help="Thread peak CPU usage (percent)" )
   maxCpuDuration = Int( help="Duration of thread peak CPU usage (seconds)" )
   totalCpuTime = Float( help="Thread total CPU usage (seconds)" )

class AgentCpuEntry( Model ):
   pid = Int( help="Agent process ID" )
   lastCpuPct = Int( help="Agent current CPU usage (percent)" )
   maxCpuPct = Int( help="Agent peak CPU usage (percent)" )
   maxCpuDuration = Int( help="Duration of agent peak CPU usage (seconds)" )
   totalCpuTime = Float( help="Agent total CPU usage (seconds)" )
   threads = Dict( keyType=int, valueType=AgentThreadCpuEntry,
                   help="Mapping from thread ID to thread CPU statistics" )

def formatCpuTime( stats ):
   mins, secs = divmod( stats.totalCpuTime, 60 )
   hours, mins = divmod( mins, 60 )
   return "%d:%02d:%05.2f" % ( hours, mins, secs )

def formatMaxCpu( stats ):
   return "%d/%d" % ( stats.maxCpuPct, stats.maxCpuDuration )

class AgentCpu( Model ):
   agents = Dict( keyType=str, valueType=AgentCpuEntry,
                  help="Mapping from Agent name to Agent CPU statistics" )
   _summary = Bool( help="Whether to show information in detail or as a summary" )

   def renderDetail( self ):
      headings = ( "Agent Name", "Thread ID", "Thread Name", "CPU (%)",
                   "Max CPU (%/secs)", "Total CPU (h:m:s)" )
      fl = Format( justify='left' )
      fl.noPadLeftIs( True )
      fl.padLimitIs( True )
      fr = Format( justify='right' )
      fr.noPadLeftIs( True )
      fr.padLimitIs( True )
      table = createTable( headings )
      table.formatColumns( fl, fr, fl, fr, fr, fr )
      for agent, agentStats in sorted( self.agents.items(),
                                       key=lambda item: item[ 1 ].maxCpuPct,
                                       reverse=True ):
         table.newRow( agent, agentStats.pid, "", agentStats.lastCpuPct,
                       formatMaxCpu( agentStats ), formatCpuTime( agentStats ) )
         for threadId, stats in sorted( agentStats.threads.items(),
                                        key=lambda item: item[ 1 ].maxCpuPct,
                                        reverse=True ):
            table.newRow( "", threadId, stats.name, stats.lastCpuPct,
                          formatMaxCpu( stats ), formatCpuTime( stats ) )
      print( table.output() )

   def renderSummary( self ):
      headings = ( "Agent Name", "Agent PID", "CPU (%)", "Max CPU (%/secs)",
                   "Total CPU (h:m:s)" )
      fl = Format( justify='left' )
      fl.noPadLeftIs( True )
      fl.padLimitIs( True )
      fr = Format( justify='right' )
      table = createTable( headings )
      fr.noPadLeftIs( True )
      fr.padLimitIs( True )
      table.formatColumns( fl, fr, fr, fr, fr )
      for agent, stats in sorted( self.agents.items(),
                                  key=lambda item: item[ 1 ].maxCpuPct,
                                  reverse=True ):
         table.newRow( agent, stats.pid, stats.lastCpuPct, formatMaxCpu( stats ),
                       formatCpuTime( stats ) )
      print( table.output() )

   def render( self ):
      if self._summary:
         self.renderSummary()
      else:
         self.renderDetail()

class AgentMemoryEntry( Model ):
   lastVmRss = Int( help="Agent current resident set size (kB)" )
   maxVmRss = Int( help="Agent peak resident set size (kB)" )
   lastVmSize = Int( help="Agent current virtual memory usage (kB)" )
   maxVmSize = Int( help="Agent peak virtual memory usage (kB)" )

class AgentMemory( Model ):
   agents = Dict( keyType=str, valueType=AgentMemoryEntry,
                  help="Mapping from Agent name to Agent memory statistics" )

   def render( self ):
      headings = ( "Agent Name",
         "VmRss current/max (kB)", "VmSize current/max (kB)" )
      fl = Format( justify='left' )
      fr = Format( justify='right' )
      table = createTable( headings )
      table.formatColumns( fl, fr, fr )
      for agent, stats in sorted( self.agents.items(),
                                  key=lambda k: k[ 1 ].maxVmRss,
                                  reverse=True ):
         table.newRow( agent,
                       "%d/%d" % ( stats.lastVmRss, stats.maxVmRss ),
                       "%d/%d" % ( stats.lastVmSize, stats.maxVmSize ) )

      print( table.output() )

class AgentPingEntry( Model ):
   lastPingRtt = Float( optional=True,
      help="RTT in seconds of the most recent agent ping event" )
   maxPingRtt = Float( optional=True,
      help="Max RTT in seconds observed for agent ping event" )
   lastPingTimestamp = Int( optional=True,
      help="Epoch timestamp in seconds of the most recent agent ping event" )
   maxPingTimestamp = Int( optional=True,
      help="Epoch timestamp in seconds when max RTT value was recorded" )

class AgentPings( Model ):
   agents = Dict( keyType=str, valueType=AgentPingEntry,
                  help="Mapping from Agent name to Agent ping statistics" )

   def render( self ):

      def _rttStr( ping ):
         return '%8.3f ms' % ( ping * 1000 ) if ping else 'unknown'

      def _timestampStr( utcTimestamp ):
         return timestampToStr( utcTimestamp, relative=False, now=utcNow() )

      headings = ( "Agent Name", "Last Ping", "Max Ping", "Max Ping Response Seen",
                   "Last Ping Response Seen" )
      fl = Format( justify='left' )
      fr = Format( justify='right' )
      table = createTable( headings )
      table.formatColumns( fl, fr, fr, fr, fr )
      for agent, stats in sorted( self.agents.items() ):
         table.newRow( agent,
                       _rttStr( stats.lastPingRtt ),
                       _rttStr( stats.maxPingRtt ),
                       _timestampStr( stats.maxPingTimestamp ),
                       _timestampStr( stats.lastPingTimestamp ) )

      print( table.output() )

class AgentUptimeEntry( Model ):
   restartCount = Int( help="Number of agent restarts" )
   agentStartTime = Int( help="Agent start time in UTC seconds since epoch" )
   agentRunning = Bool( help="Whether agent is currently running" )

class AgentUptime( Model ):
   agents = Dict( keyType=str, valueType=AgentUptimeEntry,
                  help="Mapping from Agent name to Agent restart statistics" )

   def render( self ):
      headings = ( "Agent Name", "Restarts", "Uptime" )
      fl = Format( justify='left' )
      fr = Format( justify='right' )
      table = createTable( headings )
      table.formatColumns( fl, fr, fr )
      # Secondary key: sort by the agent name
      agentList = sorted( self.agents.items(), key=lambda k: k[ 0 ] )
      # Primary key: reverse sort by the number of restarts
      agentList = sorted( agentList, key=lambda k: k[ 1 ].restartCount,
                          reverse=True )
      for agent, stats in agentList:
         uptime = timedelta( seconds=int( utcNow() ) - stats.agentStartTime ) \
            if stats.agentRunning else AGENT_NOT_RUNNING_REPR
         table.newRow( agent, stats.restartCount, uptime )

      print( table.output() )

class AgentOomScoresEntry( Model ):
   rssAnon = Int( help="Agent resident anonymous memory size" )
   vmSize = Int( help="Agent virtual memory size" )
   maxVmSize = Int( help="Agent peak virtual memory size" )
   oomScore = Int( help="Out-of-memory score of agent: -1000 (least likely to be a "
                        "victim when system needs to find memory) to 1000" )
   oomScoreAdj = Int( help="Adjustment up or down (1000 to -1000) to apply to the "
                           "computed OOM-score of the agent in order to make it more"
                           " or less likely to become a target/victim of "
                           "out-of-memory killer)" )

# the optimal unit e.g. _humanReadableFormat( 168963795964 ) ---> '157.4GB'
# Input num should be in KB
def _humanReadableFormat( num, suffix='B' ):
   # Convert to bytes first
   num = num * 1024
   for unit in [ '', 'K', 'M', 'G', 'T', 'P', 'E', 'Z' ]:
      if abs( num ) < 1024.0:
         return f"{num:3.1f}{unit}{suffix}"
      num /= 1024.0
   return "{:.1f}{}{}".format( num, 'Y', suffix )

class AgentOomScores( Model ):
   agents = Dict( keyType=str, valueType=AgentOomScoresEntry,
                  help="Mapping from Agent name to Agent OOM scores information" )

   def render( self ):
      headings = ( "Agent Name", "RssAnon", "VmSize", "Max VmSize", "oom_score",
                   "oom_score_adj" )
      fl = Format( justify='left' )
      fr = Format( justify='right' )
      table = createTable( headings )
      table.formatColumns( fl, fr, fr )
      # Secondary key: sort by the agent name
      agentList = sorted( self.agents.items(), key=lambda k: k[ 0 ] )
      # Primary key: reverse sort by oom_score
      agentList = sorted( agentList, key=lambda k: k[ 1 ].oomScore,
                          reverse=True )
      for agent, stats in agentList:
         rssAnonDisp = _humanReadableFormat( stats.rssAnon )
         vmSizeDisp = _humanReadableFormat( stats.vmSize )
         maxVmSizeDisp = _humanReadableFormat( stats.maxVmSize )
         table.newRow( agent, rssAnonDisp, vmSizeDisp, maxVmSizeDisp, stats.oomScore,
                       stats.oomScoreAdj )

      print( table.output() )
