#!/usr/bin/env python3
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Ark
import BasicCli
import Cell
import CliCommand
import CliMatcher
import CliPlugin.AgentCli as AgentCli # pylint: disable=consider-using-from-import
from CliPlugin.AgentMonitorModel import AgentPings, AgentPingEntry
import CliToken.Agent
import CliToken.Clear
import LazyMount
import ShowCommand
import Tac

agentHealthConfig = None
agentHealthStatus = None

#--------------------------------------------------------------------------------
# agent monitor ping-interval INTERVAL
#--------------------------------------------------------------------------------
class SetPingIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'agent monitor ping-interval INTERVAL'
   data = {
      'agent': CliToken.Agent.agentKwForConfig,
      'monitor': 'Configure the agent monitor settings',
      'ping-interval': 'Set the agent monitor ping interval',
      'INTERVAL': CliMatcher.IntegerMatcher( 0, 3600,
         helpdesc='Ping interval in seconds' ),
   }

   @staticmethod
   def handler( mode, args ):
      agentHealthConfig.pingInterval = args[ 'INTERVAL' ]

BasicCli.EnableMode.addCommandClass( SetPingIntervalCmd )

#--------------------------------------------------------------------------------
# show agent [ { AGENTS } ] ping
#--------------------------------------------------------------------------------
class ShowAgentPingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show agent [ { AGENTS } ] ping'
   data = {
      'agent': CliToken.Agent.agentKwForShow,
      'AGENTS': AgentCli.agentNameNewMatcher,
      'ping': 'Ping statistics for the agent',
   }
   cliModel = AgentPings

   @staticmethod
   def handler( mode, args ):

      def _roundRtt( rtt ):
         return round( rtt, 6 ) if rtt and rtt != Tac.endOfTime else None

      def _convertTs( timestamp ):
         ts = int( Ark.switchTimeToUtc( timestamp ) )
         return ts if ts else None

      agentPings = AgentPings()
      agentsRequested = args.get( 'AGENTS' )
      if not agentsRequested:
         agentHealthCollection = agentHealthStatus.agent
      else:
         agentHealthCollection = {}
         for agent in agentsRequested:
            agentRequestedStatus = agentHealthStatus.agent.get( agent )
            if agentRequestedStatus:
               agentHealthCollection[ agent ] = agentRequestedStatus

      for agentName, agentStatus in iter( agentHealthCollection.items() ):
         if not agentStatus.active:
            continue

         agentPings.agents[ agentName ] = AgentPingEntry(
            lastPingRtt=_roundRtt( agentStatus.lastPing ),
            maxPingRtt=_roundRtt( agentStatus.maxPing ),
            lastPingTimestamp=_convertTs( agentStatus.lastPingResponseAt ),
            maxPingTimestamp=_convertTs( agentStatus.maxPingResponseAt ),
         )

      return agentPings

BasicCli.addShowCommandClass( ShowAgentPingCmd )

#--------------------------------------------------------------------------------
# clear agent ping
#--------------------------------------------------------------------------------
class ClearAgentPingCmd( CliCommand.CliCommandClass ):
   syntax = 'clear agent ping'
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'agent': CliToken.Agent.agentKwForClear,
      'ping': 'Clear agent ping statistics',
   }

   @staticmethod
   def handler( mode, args ):
      clearRequestTime = Tac.now()
      agentHealthConfig.clearPingRequest = clearRequestTime
      # Wait for the clear to be acknowledged
      try:
         Tac.waitFor( lambda: ( agentHealthStatus.clearPingResponse >=
                                clearRequestTime ),
                      timeout=10,
                      warnAfter=5,
                      maxDelay=0.1,
                      sleep=True,
                      description='agent ping statistics to clear' )
      except Tac.Timeout:
         mode.addWarning( 'Failed to clear agent ping statistics' )

BasicCli.EnableMode.addCommandClass( ClearAgentPingCmd )

def Plugin( entityManager ):
   global agentHealthConfig, agentHealthStatus
   agentHealthConfig = LazyMount.mount( entityManager,
                                        Cell.path( 'agent/health/config' ),
                                        'Agent::Health::Config', 'wc' )
   agentHealthStatus = LazyMount.mount( entityManager,
                                        Cell.path( 'agent/health/status' ),
                                        'Agent::Health::Status', 'r' )
