#!/usr/bin/env python3
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import ast
import BasicCli
import Cell
import CliCommand
import CliPlugin.AgentCli as AgentCli # pylint: disable=consider-using-from-import
from CliPlugin.AgentLibShowCommands import matcherMemory
from CliPlugin.AgentMonitorModel import (
   AgentCpu,
   AgentCpuEntry,
   AgentMemory,
   AgentMemoryEntry,
   AgentThreadCpuEntry,
   AgentUptime,
   AgentUptimeEntry,
   AgentOomScores,
   AgentOomScoresEntry,
)
from CliPlugin.CliCliModel import AgentMemoryUsage
import CliPlugin.TechSupportCli
import CliToken.Agent
import CliToken.Clear
import LazyMount
import PyClient
import PyClientBase
import ShowCommand
import Tac

# pkgdeps: rpm AgentMonitor-lib

def agentHealthConfigSysdbPath():
   return Cell.path( 'agent/health/config' )

def agentHealthStatusSysdbPath():
   return Cell.path( 'agent/health/status' )

def agentStatusSysdbPath():
   return Cell.path( 'agent/status' )

ERR_NOT_RUNNING_PREFIX = 'Could not connect to '
ERR_NOT_RUNNING_TEMPLATE = ERR_NOT_RUNNING_PREFIX + '%s. Is it running?'
PYCLIENT_TIMEOUT = 10
HIDDEN_AGENTS = [ 'AgentMonitor', 'SysdbMonitor' ]

def hiddenAgents():
   return HIDDEN_AGENTS

agentHealthConfig = None
agentHealthStatus = None
agentBaseStatus = None
launchedAgents = None

#--------------------------------------------------------------------------------
# show agent [ { AGENTS } ] memory
#--------------------------------------------------------------------------------
class AgentMemoryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show agent [ { AGENTS } ] memory'
   data = {
      'agent': CliToken.Agent.agentKwForShow,
      'AGENTS': AgentCli.agentNameNewMatcher,
      'memory': matcherMemory,
   }
   cliModel = AgentMemory

   @staticmethod
   def handler( mode, args ):
      agentMemories = AgentMemory()
      agentsRequested = args.get( 'AGENTS' )
      if not agentsRequested:
         agentHealthCollection = agentHealthStatus.agent
      else:
         agentHealthCollection = {}
         for agent in agentsRequested:
            agentRequestedStatus = agentHealthStatus.agent.get( agent )
            if agentRequestedStatus:
               agentHealthCollection[ agent ] = agentRequestedStatus

      for agentName, agentStatus in agentHealthCollection.items():
         if not agentStatus.active:
            continue

         agentMemories.agents[ agentName ] = AgentMemoryEntry(
            lastVmRss=agentStatus.lastVmRss,
            maxVmRss=agentStatus.maxVmRss,
            lastVmSize=agentStatus.lastVmSize,
            maxVmSize=agentStatus.maxVmSize,
         )

      return agentMemories

BasicCli.addShowCommandClass( AgentMemoryCmd )

#--------------------------------------------------------------------------------
# show agent AGENT memory detail
#--------------------------------------------------------------------------------
class AgentAgentMemoryDetailCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show agent AGENT memory detail'
   data = {
      'agent': CliToken.Agent.agentKwForShow,
      'AGENT': AgentCli.agentNameNewMatcher,
      'memory': matcherMemory,
      'detail':
         'Python private memory count, thread count, and 100 most common objects',
   }
   cliModel = AgentMemoryUsage

   @staticmethod
   def handler( mode, args ):
      agent = args[ 'AGENT' ]
      status = agentHealthStatus.agent.get( agent )
      if status is None or not status.active:
         mode.addError( ERR_NOT_RUNNING_TEMPLATE % agent )
         return None
      agentStatus = agentBaseStatus.get( agent )
      if agentStatus is None:
         mode.addError( ERR_NOT_RUNNING_TEMPLATE % agent )
         return None
      if not agentStatus.libPythonLoaded:
         # pylint: disable-next=consider-using-f-string
         mode.addError( 'Agent %s does not have python loaded' % agent )
         return None

      try:
         pyC = PyClient.PyClient( mode.sysname, agent,
               connectTimeout=PYCLIENT_TIMEOUT )
      except PyClientBase.RpcConnectionError:
         mode.addError( ERR_NOT_RUNNING_TEMPLATE % agent )
         return None

      raw = pyC.execute( 'import Agent; print( Agent.getDataForModel() )' )
      return AgentMemoryUsage( **ast.literal_eval( raw ) )

BasicCli.addShowCommandClass( AgentAgentMemoryDetailCmd )

#-----------------------------------------------------------------------------
#
# Agent uptime cli commands:
#    show agent [ <agentName> ] uptime
#
#----------------------------------------------------------------------------
class ShowAgentUptimeCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show agent [ { AGENTS } ] uptime"
   data = {
      'agent': CliToken.Agent.agentKwForShow,
      'AGENTS': AgentCli.agentNameNewMatcher,
      'uptime': 'Uptime and restart statistics for the agent',
   }
   cliModel = AgentUptime

   @staticmethod
   def handler( mode, args ):
      agentUptimeEntries = AgentUptime()
      agentsRequested = args.get( 'AGENTS' )
      launchedAgentsList = [ a for a in list( launchedAgents )
                             if a not in HIDDEN_AGENTS ]
      if not agentsRequested:
         agentHealthCollection = agentHealthStatus.agent

         # We can already extend agentUptimeEntries here
         for agent in launchedAgentsList:
            if agent in agentHealthCollection:
               continue
            agentUptimeEntries.agents[ agent ] = AgentUptimeEntry(
                  restartCount=0, agentStartTime=0, agentRunning=False )

      else:
         agentHealthCollection = {}
         for agent in agentsRequested:
            agentRequestedStatus = agentHealthStatus.agent.get( agent )
            if agentRequestedStatus:
               agentHealthCollection[ agent ] = agentRequestedStatus
            elif agent in launchedAgentsList:
               agentUptimeEntries.agents[ agent ] = AgentUptimeEntry(
                     restartCount=0, agentStartTime=0, agentRunning=False )

      for agentName, healthStatus in iter( agentHealthCollection.items() ):
         if agentName in HIDDEN_AGENTS:
            # Skip agents marked as hidden even if they are running
            continue
         status = agentBaseStatus.get( agentName )
         # Gracefully skip agents with incomplete status
         if status is None:
            continue
         # Some agents might not have started at least once,
         # Make sure their restartCount is not negative.
         restartCount = status.startCount - 1 if status.startCount > 0 else 0
         startTimeUTC = 0

         if healthStatus.active:
            agentUptime = Tac.now() - status.startTime
            startTimeUTC = int( Tac.utcNow() - agentUptime )
         elif agentName not in launchedAgentsList:
            # Skip inactive agents that should not be running
            continue

         agentUptimeEntries.agents[ agentName ] = AgentUptimeEntry(
            restartCount=restartCount, agentStartTime=startTimeUTC,
            agentRunning=healthStatus.active )

      return agentUptimeEntries

BasicCli.addShowCommandClass( ShowAgentUptimeCmd )

#--------------------------------------------------------------------------------
# show agent [ { AGENTS } ] cpu [ detail ]
#--------------------------------------------------------------------------------
class AgentCpuCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show agent [ { AGENTS } ] cpu [ detail ]'
   data = {
      'agent': CliToken.Agent.agentKwForShow,
      'AGENTS': AgentCli.agentNameNewMatcher,
      'cpu': 'CPU statistics for the agent',
      'detail': 'CPU statistics for individual threads',
   }

   cliModel = AgentCpu

   @staticmethod
   def handler( mode, args ):
      agentCpu = AgentCpu( _summary='detail' not in args )
      agentsRequested = args.get( 'AGENTS' )
      if not agentsRequested:
         agentHealthCollection = agentHealthStatus.agent
      else:
         agentHealthCollection = {}
         for agent in agentsRequested:
            agentRequestedStatus = agentHealthStatus.agent.get( agent )
            if agentRequestedStatus:
               agentHealthCollection[ agent ] = agentRequestedStatus

      for agentName, agentStatus in iter( agentHealthCollection.items() ):
         if agentStatus.active and agentStatus.cpuStatsValid():
            agentCpuEntry = AgentCpuEntry(
               pid=agentStatus.pid,
               lastCpuPct=agentStatus.lastCpu,
               maxCpuPct=agentStatus.maxCpu,
               maxCpuDuration=agentStatus.secsAtMaxCpu,
               totalCpuTime=agentStatus.totalCpuTime,
            )
            for threadStatus in agentStatus.threadStatus.values():
               agentCpuEntry.threads[ threadStatus.tid ] = AgentThreadCpuEntry(
                  # sanitize thread name: '1234:(FlowWatcher)' -> 'FlowWatcher'
                  name=threadStatus.name.split( ":", 1 )[ -1 ].strip( "()" ),
                  lastCpuPct=threadStatus.lastCpu,
                  maxCpuPct=threadStatus.maxCpu,
                  maxCpuDuration=threadStatus.secsAtMaxCpu,
                  totalCpuTime=threadStatus.totalCpuTime )
            agentCpu.agents[ agentName ] = agentCpuEntry

      return agentCpu

BasicCli.addShowCommandClass( AgentCpuCmd )

#--------------------------------------------------------------------------------
# clear agent cpu
#--------------------------------------------------------------------------------
class ClearAgentCpuCmd( CliCommand.CliCommandClass ):
   syntax = 'clear agent cpu'
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'agent': CliToken.Agent.agentKwForClear,
      'cpu': 'Clear agent CPU statistics',
   }

   @staticmethod
   def handler( mode, args ):
      clearRequestTime = Tac.now()
      agentHealthConfig.clearCpuRequest = clearRequestTime
      # Wait for the clear to be acknowledged
      try:
         Tac.waitFor( lambda: agentHealthStatus.clearCpuResponse >= clearRequestTime,
                      timeout=10,
                      warnAfter=5,
                      maxDelay=0.1,
                      sleep=True,
                      description="agent cpu statistics to clear" )
      except Tac.Timeout:
         mode.addWarning( "Failed to clear agent cpu statistics" )

BasicCli.EnableMode.addCommandClass( ClearAgentCpuCmd )

#-----------------------------------------------------------------------------
#
# Agent oom score cli commands:
#    show agent [ { <agentName> } ] oom scores
#
#----------------------------------------------------------------------------
class ShowAgentOomScores( ShowCommand.ShowCliCommandClass ):
   syntax = "show agent [ { AGENTS } ] oom scores"
   data = {
      'agent': CliToken.Agent.agentKwForShow,
      'AGENTS': AgentCli.agentNameNewMatcher,
      'oom': 'Show agent oom parameters',
      'scores': 'Show agent oom score related parameters',
   }
   cliModel = AgentOomScores

   @staticmethod
   def handler( mode, args ):
      agentOomScoresEntries = AgentOomScores()
      agentsRequested = args.get( 'AGENTS' )
      if not agentsRequested:
         agentHealthCollection = agentHealthStatus.agent
      else:
         agentHealthCollection = {}
         for agent in agentsRequested:
            agentRequestedStatus = agentHealthStatus.agent.get( agent )
            if agentRequestedStatus:
               agentHealthCollection[ agent ] = agentRequestedStatus

      for agentName, healthStatus in iter( agentHealthCollection.items() ):
         if not healthStatus.active:
            continue
         status = agentBaseStatus.get( agentName )
         if status is None or status.startCount == 0:
            continue
         agentOomScoresEntries.agents[ agentName ] = AgentOomScoresEntry(
               rssAnon=healthStatus.rssAnon, vmSize=healthStatus.lastVmSize,
               maxVmSize=healthStatus.maxVmSize, oomScore=healthStatus.oomScore,
               oomScoreAdj=healthStatus.oomScoreAdj )

      return agentOomScoresEntries

BasicCli.addShowCommandClass( ShowAgentOomScores )

#--------------------------------------------------------------
# show tech support
#--------------------------------------------------------------
CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2019-04-12 13:58:49',
   cmds=[ 'show agent oom scores' ] )

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2023-08-17 16:16:16',
   cmds=[ 'show agent cpu' ] )

def Plugin( entityManager ):
   global agentHealthConfig, agentHealthStatus, agentBaseStatus
   global launchedAgents
   agentHealthConfig = LazyMount.mount( entityManager,
                                        agentHealthConfigSysdbPath(),
                                        "Agent::Health::Config", "wc" )
   agentHealthStatus = LazyMount.mount( entityManager,
                                        agentHealthStatusSysdbPath(),
                                        "Agent::Health::Status", "r" )
   agentBaseStatus = LazyMount.mount( entityManager,
                                      agentStatusSysdbPath(),
                                      "Tac::Dir", "ri" )
   launchedAgents = LazyMount.mount( entityManager,
                                     Cell.path( 'launcher/LaunchedAgents' ),
                                     "Tac::Dir", 'ri' )
