# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import TableOutput
from CliModel import Model
from CliModel import List
from CliModel import Dict
from CliModel import Str
from CliModel import Int

class TableEntry( Model ):
   table = Str( help="Name of hardware resource / table" )
   feature = Str( help="Name of feature using the given hardware table. When this "
                       "column is empty, it is an aggregate of the usage by all "
                       "features using this table" )
   chip = Str( help="Name of chip. Populated only for divergent ( per-chip ) "
                    "features" )
   used = Int( help="Number of entries currently in use for the table by "
                    "given feature" )
   usedPercent = Int( help="Used percent ( % ) calculated as 'Number of entries in "
                           "use by the given feature / ( Number of entries in use "
                           "by the given feature + Number of free entries ) * 100'" )
   free = Int( help="Number of free entries: 'Max entries available now - Used "
                    "entries', where maximum available entries exclude entries used "
                    "by and committed to other features" )
   committed = Int( help="Block of entries dedicated to this feature, but may or "
                         "may not be in use now. This value is applicable only to "
                         "those tables which allocate a block of resources ( like "
                         "Bank ) to a given feature" )
   maxLimit = Int( help="Best case maximum entries available for this feature, "
                        "considering no other feature is using them. This is the "
                        "theoretical maximum resource utilization possible for this "
                        "feature" )
   highWatermark = Int( help="The maximum number of entries used by this feature so "
                             "far. This value may get reset after agent restart or "
                             "SSO switchover" )
   sharedFeatures = List( valueType=str, help="Feature Groups programmed in the "
                                              "TCAM Bank" )

class TableThreshold( Model ):
   configThreshold = Int( help="Configured Threshold in percent ( % )" )
   configClearThreshold = Int( help="Configured Clear Threshold in percent ( % )" )
   defaultThreshold = Int( help="Default Threshold in percent ( % )" )

class CapacityUsage( Model ):
   tables = List( valueType=TableEntry, help="A list of tables and utilization" )

   def render( self ):
      print( 'Forwarding Resources Usage' )
      print()
      table = TableOutput.createTable( [ "Table",
                                         "Feature",
                                         "Chip",
                                         "Used Entries",
                                         "Used (%)",
                                         "Free Entries",
                                         "Committed Entries",
                                         "Best Case Max Entries",
                                         "High Watermark",
                                         "Shared Features" ],
                                       tableWidth=200 )

      # The maxWidth for "Used Entries" and "Free Entries" has been set to 7 as today
      # the number of table entries do not exceed 7 digits.
      # The maxWidth for "Committed Entries", "Best Case Max Entries" and "High
      # Watermark" has been set to 9 to accommodate the length of the header strings.
      # The maxWidth for "Used (%)" has been set to 3 to accommodate the maximum
      # length for "100"
      # The maxWidth for "Table", "Feature" and "Chip" has not been set so that they
      # can adjust based on the terminal width available to the user. However, it is
      # upto the engineer who whould fill the names of the table, feature or chip to
      # make sure they keep the names small
      fTable = TableOutput.Format( justify='left', wrap=True )
      fTable.noPadLeftIs( True )
      fFeature = TableOutput.Format( justify='left', wrap=True )
      fFeature.noPadLeftIs( True )
      fChip = TableOutput.Format( justify='left', wrap=True )
      fChip.noPadLeftIs( True )
      fUsed = TableOutput.Format( maxWidth=7, wrap=True )
      fUsedPercent = TableOutput.Format( maxWidth=4, wrap=True )
      fFree = TableOutput.Format( maxWidth=7, wrap=True )
      fCommitted = TableOutput.Format( maxWidth=9, wrap=True )
      fMax = TableOutput.Format( maxWidth=9, wrap=True )
      fHigh = TableOutput.Format( maxWidth=9, wrap=True )
      fSharedFeatures = TableOutput.Format( justify='left', wrap=True )

      table.formatColumns( fTable, fFeature, fChip, fUsed, fUsedPercent, fFree,
                           fCommitted, fMax, fHigh, fSharedFeatures )

      for tableEntry in sorted( self.tables,
                                key=lambda t: ( t.table, t.feature, t.chip ) ):
         
         table.newRow( tableEntry.table,
                       tableEntry.feature,
                       tableEntry.chip,
                       tableEntry.used,
                       str( tableEntry.usedPercent ) + '%',
                       tableEntry.free,
                       tableEntry.committed,
                       tableEntry.maxLimit,
                       tableEntry.highWatermark,
                       "\n".join( tableEntry.sharedFeatures ) )

      print( table.output() )

class CapacityThreshold( Model ):
   thresholds = Dict( keyType=str, valueType=TableThreshold,
                              help="A mapping from table name to thresholds" )

   def render( self ):
      print( 'Configured Hardware Capacity Thresholds' )
      print()
      table = TableOutput.createTable( [ "Table",
                                         "Feature",
                                         "Configured Threshold",
                                         "Configured Clear Threshold",
                                         "Default Threshold" ] )

      fTable = TableOutput.Format( justify='left', wrap=True )
      fTable.noPadLeftIs( True )
      fFeature = TableOutput.Format( justify='left', wrap=True )
      fFeature.noPadLeftIs( True )
      fConfigThreshold = TableOutput.Format( maxWidth=10, wrap=True )
      fClearThreshold = TableOutput.Format( maxWidth=10, wrap=True )
      fDefaultThreshold = TableOutput.Format( maxWidth=10, wrap=True )

      table.formatColumns( fTable, fFeature, fConfigThreshold, fClearThreshold,
                           fDefaultThreshold )

      for rowName in sorted( self.thresholds ):
         name = rowName.split( '-', 1 )
         table.newRow( name[ 0 ],
                       '' if len( name ) == 1 else name[ 1 ],
                       self.thresholds[ rowName ].configThreshold,
                       self.thresholds[ rowName ].configClearThreshold,
                       self.thresholds[ rowName ].defaultThreshold )

      print( table.output() )

