#!/usr/bin/env python3
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
import CliCommand
from CliMatcher import (
   KeywordMatcher
)
import CliParser
from CliPlugin import (
   IpAddrMatcher,
   Ip6AddrMatcher,
)
from IpLibConsts import DEFAULT_VRF
import Shark
import SharkLazyMount
import SmashLazyMount
import Smash
import LazyMount
import Tac
import Tracing

__defaultTraceHandle__ = Tracing.Handle( 'StrataCliLib' )

t0 = Tracing.trace0

aleCliConfig = None
entityManager = None
resilientEcmpStatus = None
resilientEcmpStatus6 = None
routeMonitorStatus = None
routeMonitorSharkStatus = None
routingHwConfig = None
routingHwStatusCommon = None
routingHwStatus = None
routing6HwConfig = None
routing6HwStatus = None
routingHardwareRouteStatus = None
routing6HardwareRouteStatus = None
smashReaderInfo = None
vrfIdMap = None
vrfTableStatus = None
l3Config = None
unifiedForwardingStatus = None

routeStatus = {}
route6Status = {}
programmedRouteStatus = {}
programmedRoute6Status = {}
forwardingStatus = None
programmedFecStatus = None
fecModeStatus = None
fecModeSm = None
tunnelFib = None
tunnelProgrammingStatus = None
nhgSmashStatus = None
nhgHwStatus = None

FecMode = Tac.Type( 'Smash::Fib::FecMode' )

def getRouteStatus( af, vrfName ):
   if af == 'ipv4':
      if vrfName not in routeStatus:
         routeStatus[ vrfName ] = SmashLazyMount.mount( entityManager,
            'routing/vrf/status/' + vrfName,
            'Smash::Fib::RouteStatus',
            smashReaderInfo, autoUnmount=True )

      returnRouteStatus = routeStatus[ vrfName ]
   else:
      if vrfName not in route6Status:
         route6Status[ vrfName ] = SmashLazyMount.mount( entityManager,
            'routing6/vrf/status/' + vrfName,
            'Smash::Fib6::RouteStatus',
            smashReaderInfo, autoUnmount=True )

      returnRouteStatus = route6Status[ vrfName ]

   return returnRouteStatus

def getForwardingStatus( af ):
   assert fecModeStatus
   if fecModeStatus.fecMode == FecMode.fecModeUnified:
      returnForwardingStatus = unifiedForwardingStatus
   elif fecModeStatus.fecMode == FecMode.fecModePerVrf:
      returnForwardingStatus = forwardingStatus
   else:
      returnForwardingStatus = None

   return returnForwardingStatus

def getProgrammedRouteStatus( af, vrfName ):
   if af == 'ipv4':
      if vrfName not in programmedRouteStatus:
         programmedRouteStatus[ vrfName ] = SmashLazyMount.mount(
            entityManager,
            'routing/vrf/routeVersion/' + vrfName,
            'Smash::Fib::ProgrammedRouteStatus',
            smashReaderInfo )

      returnProgrammedRouteStatus = programmedRouteStatus[ vrfName ]
   else:
      if vrfName not in programmedRoute6Status:
         programmedRoute6Status[ vrfName ] = SmashLazyMount.mount(
            entityManager,
            'routing6/vrf/routeVersion/' + vrfName,
            'Smash::Fib6::ProgrammedRouteStatus',
            smashReaderInfo )

      returnProgrammedRouteStatus = programmedRoute6Status[ vrfName ]

   return returnProgrammedRouteStatus

def vrfRootSmGuard( mode, token ):
   if not vrfTableStatus.aleL3UnicastRunning:
      return CliParser.guardNotThisPlatform
   return None

v4RoutesMatcher = CliCommand.Node(
   matcher=KeywordMatcher( "routes", helpdesc="Display IPv4 routes" ),
   guard=vrfRootSmGuard )

class IpPrefixOrAddrMatcher( CliCommand.CliExpression ):
   expression = 'ADDR | PREFIX | ADDR6 | PREFIX6'
   data = { 'ADDR': IpAddrMatcher.IpAddrMatcher( 'Match this IP address' ),
            'PREFIX': IpAddrMatcher.ipPrefixExpr(
                        'Match this IP address',
                        'Match this subnet mask',
                        'Match this IP prefix',
                        partial=False,
                        overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO ),
            'ADDR6': Ip6AddrMatcher.Ip6AddrMatcher( 'Match this IP address' ),
            'PREFIX6': Ip6AddrMatcher.ip6PrefixExpr(
                        'Match this IP address',
                        'Match this subnet mask',
                        'Match this IP prefix',
                        overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO ),
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      '''Consistently provides PREFIX as an Arnet::Prefix or Arnet::Ip6Prefix'''
      af = 'ipv4' if 'ip' in args else 'ipv6'

      if af == 'ipv4':
         # If an address is provided, convert it to /32 prefix
         if 'ADDR' in args:
            args[ 'PREFIX' ] = Arnet.Prefix( args.pop( 'ADDR' ) )
         elif 'PREFIX' in args:
            args[ 'PREFIX' ] = Arnet.Prefix( args[ 'PREFIX' ] )
      else:
         # If an address is provided, convert it to /128 prefix
         if 'ADDR6' in args:
            args[ 'PREFIX' ] = Arnet.Ip6Prefix( args.pop( 'ADDR6' ) )
         elif 'PREFIX6' in args:
            args[ 'PREFIX' ] = Arnet.Ip6Prefix( args[ 'PREFIX6' ] )

def Plugin( em ):
   def doMountsComplete():
      global forwardingStatus
      global unifiedForwardingStatus
      global programmedFecStatus
      global tunnelFib
      global tunnelProgrammingStatus
      global nhgSmashStatus

      fsType = "Smash::FibGen::ForwardingStatus"
      pfsGenType = "Smash::Fib::ProgrammedFecStatus"
      tunFibGenType = "Tunnel::TunnelFib::TunnelFib"
      tunStatusGenType = "Tunnel::Hardware::TunnelProgrammingStatus"
      nhgSmashStatusType = "NexthopGroup::EntryStatus"
      readerInfo = Smash.mountInfo( 'reader' )

      unifiedForwardingStatus = SmashLazyMount.mount( em,
                                                      "forwardingGen/unifiedStatus",
                                                      fsType, readerInfo )
      forwardingStatus = SmashLazyMount.mount( em, "forwardingGen/status",
                                               fsType, readerInfo )

      programmedFecStatus = SmashLazyMount.mount( em,
                                             'forwarding/fecVersion',
                                             pfsGenType,
                                             smashReaderInfo )

      tunnelFib = SmashLazyMount.mount( em,
                                        'tunnel/tunnelFib',
                                        tunFibGenType,
                                        smashReaderInfo )

      # mount tunnelFib and tunnelProgrammingStatus to dump tunnel fec ack/nack
      tunnelProgrammingStatus = SmashLazyMount.mount( em,
                                "routing/hardware/tunnelProgramming/status",
                                tunStatusGenType,
                                smashReaderInfo )

      # mount nexthop-group smash status to dump nhg fec ack
      nhgSmashStatus = SmashLazyMount.mount( em,
                                             "routing/nexthopgroup/entrystatus",
                                             nhgSmashStatusType,
                                             readerInfo )

      # Set up the FecModeSm
      global fecModeSm
      global fecModeStatus
      fecModeStatus = Tac.newInstance( 'Smash::Fib::FecModeStatus', 'fms' )
      fecModeSm = Tac.newInstance( 'Ira::FecModeSm', l3Config, fecModeStatus )

   global entityManager
   entityManager = em

   global aleCliConfig
   aleCliConfig = LazyMount.mount( em, 'hardware/ale/cliconfig',
                                   'Ale::HwCliConfig', 'w' )

   global routingHwConfig
   routingHwConfig = LazyMount.mount( em, "routing/hardware/config",
                                      "Routing::Hardware::Config", "w" )

   global routingHwStatusCommon
   routingHwStatusCommon = LazyMount.mount( em, "routing/hardware/statuscommon",
                                            "Routing::Hardware::StatusCommon", "r" )
   global routingHwStatus
   routingHwStatus = LazyMount.mount( em, "routing/hardware/status",
                                      "Routing::Hardware::Status", "r" )

   global routing6HardwareRouteStatus
   routing6HardwareRouteStatus = LazyMount.mount( em,
                                                  "routing6/hardware/route/status",
                                                  "Routing6::Hardware::RouteStatus",
                                                  'r' )
   global routing6HwConfig
   routing6HwConfig = LazyMount.mount( em, "routing6/hardware/config",
                                       "Routing6::Hardware::Config", "w" )
   global routing6HwStatus
   routing6HwStatus = LazyMount.mount( em, "routing6/hardware/status",
                                      "Routing6::Hardware::Status", "r" )

   global routingHardwareRouteStatus
   routingHardwareRouteStatus = LazyMount.mount( em,
                                                 "routing/hardware/route/status",
                                                 "Routing::Hardware::RouteStatus",
                                                 'r' )

   global vrfTableStatus
   vrfTableStatus = LazyMount.mount( em, 'ale/vrfTable/status',
                                     'Ale::VrfTableStatus', 'r' )

   global routeMonitorStatus
   routeMonitorStatus = SmashLazyMount.mount( em,
                                      'routing/routeMonitorStatus',
                                      'Smash::RouteMonitor::RouteMonitorStatus',
                                      Smash.mountInfo( 'reader' ) )

   global routeMonitorSharkStatus
   routeMonitorSharkStatus = SharkLazyMount.mount( em,
                                      'routing/routeMonitorSharkStatus',
                                      'Shark::RouteMonitor::RouteProgrammingState',
                                      Shark.mountInfo( 'shadow' ), True )

   global vrfIdMap
   vrfIdMap = SmashLazyMount.mount( entityManager, 'vrf/vrfIdMapStatus',
                                    'Vrf::VrfIdMap::Status',
                                    Smash.mountInfo( 'reader' ),
                                    autoUnmount=True )

   global smashReaderInfo
   smashReaderInfo = Smash.mountInfo( 'reader' )

   global resilientEcmpStatus
   resilientEcmpStatus = SmashLazyMount.mount( em,
                                               'routing/resilientEcmpStatus',
                                               'Ale::ResilientEcmpStatus',
                                               smashReaderInfo )

   global resilientEcmpStatus6
   resilientEcmpStatus6 = SmashLazyMount.mount( em,
                                                'routing6/resilientEcmpStatus',
                                                'Ale::ResilientEcmpStatus6',
                                                 smashReaderInfo )

   routeStatus[ DEFAULT_VRF ] = SmashLazyMount.mount( em,
                                       'routing/status',
                                       'Smash::Fib::RouteStatus',
                                       smashReaderInfo )

   route6Status[ DEFAULT_VRF ] = SmashLazyMount.mount( em,
                                        'routing6/status',
                                        'Smash::Fib6::RouteStatus',
                                        smashReaderInfo )

   programmedRouteStatus[ DEFAULT_VRF ] = SmashLazyMount.mount( em,
                                                 'routing/routeVersion',
                                                 'Smash::Fib::ProgrammedRouteStatus',
                                                 smashReaderInfo )

   programmedRoute6Status[ DEFAULT_VRF ] = SmashLazyMount.mount( em,
                                                'routing6/routeVersion',
                                                'Smash::Fib6::ProgrammedRouteStatus',
                                                smashReaderInfo )

   global nhgHwStatus
   nhgHwStatus = LazyMount.mount( em, 'routing/hardware/nexthopgroup/status',
                                  'Routing::Hardware::NexthopGroupStatus', 'r' )

   mg = em.mountGroup()
   global l3Config
   l3Config = mg.mount( "l3/config", "L3::Config", "ri" )
   mg.close( doMountsComplete )
