# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import ConfigMount
import LazyMount
import Toggles.TrafficPolicyToggleLib

import CliCommand
import CliMatcher
import CliParser
import CliGlobal
from CliPlugin import IntfCli
from CliPlugin.ClassificationCliLib import CommitAbortModelet
from CliPlugin.TrafficPolicyCli import trafficPolicyNameMatcher, \
      TrafficPoliciesConfigCmd
from CliMode.AleCpuPolicy import CpuTrafficPolicyEnforcementConfigMode
from CliMode.TrafficPolicy import ( TrafficPoliciesConfigMode,
                                   TrafficPoliciesVrfConfigMode )
import ShowCommand
from Toggles.TrafficPolicyToggleLib import (
   toggleCpuTrafficPolicyFragmentRuleDisableEnabled,
   toggleCpuTrafficPolicyEnforcementIpTtlExpiredEnabled )

gv = CliGlobal.CliGlobal(
   policiesVrfConfig=None,
   policiesIntfConfig=None,
   cpuPolicyGlobalConfig=None,
   trafficPolicyHwStatus=None
)

def cpuTrafficPolicySupportedGuard( mode, token ):
   if gv.trafficPolicyHwStatus.cpuTrafficPolicySupported:
      return None
   return CliParser.guardNotThisPlatform

def intfCpuPolicySupportedGuard( mode, token ):
   if gv.trafficPolicyHwStatus.intfCpuTrafficPolicySupported:
      return None
   return CliParser.guardNotThisPlatform

def perVrfCpuPolicySupportedGuard( mode, token ):
   if Toggles.TrafficPolicyToggleLib.toggleCpuTrafficPolicyPerVrfEnabled():
      return cpuTrafficPolicySupportedGuard( mode, token )
   return CliParser.guardNotThisPlatform

CpuTrafficPolicyEnforcementConfigMode.addModelet( CommitAbortModelet )

cpuNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'cpu',
            helpdesc='Configure CPU traffic policy' ),
      guard=cpuTrafficPolicySupportedGuard )

#---------------------------------------------------------------
# The "[no|default] cpu traffic-policy <name> vrf all" command
#---------------------------------------------------------------
class CpuTrafficPolicyConfigCmd( CliCommand.CliCommandClass ):
   """
   Apply CPU traffic policy to all vrfs.
   """
   syntax = 'cpu traffic-policy POLICY vrf all'
   noOrDefaultSyntax = 'cpu traffic-policy POLICY vrf ...'
   data = {
      'cpu': cpuNode,
      'traffic-policy': 'Configure traffic policy',
      'POLICY': trafficPolicyNameMatcher,
      'vrf': 'Configure VRF for CPU traffic policy',
      'all': 'Configure policy for all VRFs'
   }

   handler = 'AleCpuPolicyCliHandler.handleCpuTrafficPolicyConfig'
   noOrDefaultHandler = 'AleCpuPolicyCliHandler.handleNoOrDefaultTrafficPolicyConfig'

# ---------------------------------------------------------------
# "[no|default] cpu traffic-policy POLICY fallback traffic-policy none" command
# ---------------------------------------------------------------
class PerVrfCpuPolicyVrfModeConfigCmd( CliCommand.CliCommandClass ):
   """
   Apply CPU traffic policy to a specific vrf.
   """
   syntax = "CPU traffic-policy POLICY fallback FALLBACK_TYPE none"
   noOrDefaultSyntax = 'CPU traffic-policy ...'
   data = {
      'CPU': CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
         'cpu',
         helpdesc='Configure per-VRF CPU traffic policy' ),
         guard=perVrfCpuPolicySupportedGuard ),
      'traffic-policy': 'Configure traffic policy',
      'POLICY': trafficPolicyNameMatcher,
      'fallback': 'Configure the fallback behavior for this vrf CPU '
      'traffic-policy',
      'FALLBACK_TYPE': CliMatcher.KeywordMatcher(
         'traffic-policy', helpdesc='Fallback to a traffic-policy' ),
      'none': 'Do not fallback to any traffic-policy'
   }

   handler = 'AleCpuPolicyCliHandler.handllePerVrfCpuPolicyConfig'
   noOrDefaultHandler = 'AleCpuPolicyCliHandler.handleNoOrDefaultVrfCpuPolicyConfig'

TrafficPoliciesVrfConfigMode.addCommandClass( PerVrfCpuPolicyVrfModeConfigCmd )

# ---------------------------------------------------------------
# "[no] cpu traffic-policy fragment implicit-permit disabled" command
# ---------------------------------------------------------------
class CpuTrafficPolicyFragmentPermitConfigCmd( CliCommand.CliCommandClass ):
   """
   Disable programming of implict permit fragment rules in TCAM for L4 rules with
   IP address match
   """
   syntax = 'cpu traffic-policy fragment implicit-permit disabled'
   noOrDefaultSyntax = syntax
   data = {
      'cpu': cpuNode,
      'traffic-policy': 'Configure traffic policy',
      'fragment': 'Configure handling of fragmented packets',
      'implicit-permit': 'Add rules to allow fragmented packets for Layer 4 rules',
      'disabled': 'Implicit permit-fragment rules will not be added'
   }

   handler = 'AleCpuPolicyCliHandler.handleCpuPolicyPermitFragmentConfig'
   noOrDefaultHandler =\
         'AleCpuPolicyCliHandler.handleNoOrDefaultCpuPolicyPermitFragmentConfig'

# ---------------------------------------------------------------
# "[no] cpu traffic-policy enforcement ip ttl expired
# ---------------------------------------------------------------
class CpuTrafficPolicyEnforcementIpTtlExpiredConfigCmd( CliCommand.CliCommandClass ):
   """
   Allow CPU traffic-policy to process expired TTL (TTL=1) packets
   """
   syntax = 'cpu traffic-policy enforcement ip ttl expired'
   noOrDefaultSyntax = syntax
   data = {
      'cpu': cpuNode,
      'traffic-policy': 'Configure traffic policy',
      'enforcement': 'CPU traffic-policy enforcement options',
      'ip': 'IPv4 and IPv6 packets',
      'ttl': 'Time to live',
      'expired': 'Expired (TTL=1)',
   }

   handler = 'AleCpuPolicyCliHandler.handleCpuPolicyEnforcementIpTtlExpiredConfig'
   noOrDefaultHandler = \
         'AleCpuPolicyCliHandler.handleNoOrDefaultCpuPolicy' \
         'EnforcementIpTtlExpiredConfig'

# ----------------------------------------------------------------------
# Modelet under Intf mode for cpu traffic-policy command
# ----------------------------------------------------------------------
class IntfCpuTrafficPolicyModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      supportedIntfs = ( 'Ethernet', 'Port-Channel' )
      return mode.intf.name.startswith( supportedIntfs )

IntfCli.IntfConfigMode.addModelet( IntfCpuTrafficPolicyModelet )

# ---------------------------------------------------------------
# "[no|default] cpu traffic-policy POLICY fallback traffic-policy vrf" command
# ---------------------------------------------------------------
class IntfCpuTrafficPolicyConfigCmd( CliCommand.CliCommandClass ):
   """
   Apply CPU traffic policy to specific interfaces.
   """
   syntax = 'CPU traffic-policy POLICY fallback FALLBACK_TYPE vrf'
   noOrDefaultSyntax = 'CPU traffic-policy ...'
   data = {
      'CPU': CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
         'cpu',
         helpdesc='Configure interface-based CPU traffic policy' ),
         guard=intfCpuPolicySupportedGuard ),
      'traffic-policy': 'Configure traffic policy',
      'POLICY': trafficPolicyNameMatcher,
      'fallback': 'Configure the fallback behavior for this interface CPU '
      'traffic-policy',
      'FALLBACK_TYPE': CliMatcher.KeywordMatcher(
         'traffic-policy', helpdesc='Fallback to a traffic-policy' ),
      'vrf': 'Fallback to the CPU traffic policy applied to the VRF in which this '
      'interface is configured',
   }

   handler = 'AleCpuPolicyCliHandler.handleIntfCpuTrafficPolicyConfig'
   noOrDefaultHandler = (
         'AleCpuPolicyCliHandler.handleNoOrDefaultIntfCpuTrafficPolicyConfig' )

# ----------------------------------------------------------------------
# Support for default interface command
#
# NOTE: This is tested as part of CliSaveTest
# ----------------------------------------------------------------------
class IntfCpuTrafficPolicyJanitor( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      del gv.policiesIntfConfig.intf[ self.intf_.name ]

def noTrafficPolicies( mode, args ):
   gv.policiesVrfConfig.trafficPolicies.clear()
   gv.cpuPolicyGlobalConfig.installPermitFragment = None
   gv.cpuPolicyGlobalConfig.enforcementIpTtlExpired = False

class ShowPendingCpuPolicyEnforcement( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ pending ]'
   data = {
      'pending': 'Show pending CPU traffic-policy enforcement options'
   }

   handler = 'AleCpuPolicyCliHandler.handleShowPendingCpuPolicyEnforcement'

CpuTrafficPolicyEnforcementConfigMode.addShowCommandClass(
      ShowPendingCpuPolicyEnforcement )

class CpuTrafficPolicyEnforceManagementCmd( CliCommand.CliCommandClass ):
   syntax = 'enforcement management'
   noOrDefaultSyntax = syntax

   data = {
      'enforcement' : 'CPU traffic-policy enforcement options',
      'management' : 'Enforce CPU traffic-policy on management ports',
   }

   handler = 'AleCpuPolicyCliHandler.handleCpuTrafficPolicyEnforceManagementCmd'
   noOrDefaultHandler = handler

IntfCpuTrafficPolicyModelet.addCommandClass( IntfCpuTrafficPolicyConfigCmd )
IntfCli.Intf.registerDependentClass( IntfCpuTrafficPolicyJanitor )

# pylint: disable=protected-access
TrafficPoliciesConfigCmd._registerNoHandler( noTrafficPolicies )
# pylint: enable=protected-access

TrafficPoliciesConfigMode.addCommandClass( CpuTrafficPolicyConfigCmd )
if toggleCpuTrafficPolicyFragmentRuleDisableEnabled():
   TrafficPoliciesConfigMode.addCommandClass(
                             CpuTrafficPolicyFragmentPermitConfigCmd )
if toggleCpuTrafficPolicyEnforcementIpTtlExpiredEnabled():
   TrafficPoliciesConfigMode.addCommandClass(
                             CpuTrafficPolicyEnforcementIpTtlExpiredConfigCmd )
CpuTrafficPolicyEnforcementConfigMode.addCommandClass(
      CpuTrafficPolicyEnforceManagementCmd )

def Plugin( em ):
   policiesRootNode = 'trafficPolicies'
   policiesVrfConfigNode = 'cpu/vrf'
   policiesVrfConfigPath = policiesRootNode + '/' + policiesVrfConfigNode
   policiesVrfConfigType = 'PolicyMap::VrfConfig'
   policiesIntfConfigNode = 'cpu/intf'
   policiesIntfConfigPath = policiesRootNode + '/' + policiesIntfConfigNode
   policiesIntfConfigType = 'PolicyMap::IntfConfig'
   cpuPolicyGlobalConfigPath = policiesRootNode + '/param/config/cpu'
   cpuPolicyGlobalConfigType = 'TrafficPolicy::CpuPolicyGlobalConfig'
   entityManager = em

   gv.policiesVrfConfig = ConfigMount.mount( entityManager, policiesVrfConfigPath,
                                             policiesVrfConfigType, 'wi' )
   gv.policiesIntfConfig = ConfigMount.mount( entityManager, policiesIntfConfigPath,
                                              policiesIntfConfigType, 'wi' )
   gv.cpuPolicyGlobalConfig = ConfigMount.mount( entityManager,
                                              cpuPolicyGlobalConfigPath,
                                              cpuPolicyGlobalConfigType, 'wi' )
   gv.trafficPolicyHwStatus = LazyMount.mount( entityManager,
                                               'trafficPolicies/hardware/status/cpu',
                                               'TrafficPolicy::HwStatus', 'r' )
