#!/usr/bin/env python3
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AgentCommandRequest
import Arnet
import BasicCli
import CliCommand
from CliMatcher import (
   IntegerMatcher,
   KeywordMatcher,
   SlashedIntegerMatcher,
)
import CliModel
import CliParser
from CliPlugin import VrfCli
from CliPlugin import IpAddrMatcher
from CliPlugin import Ip6AddrMatcher
from CliPlugin import IraCommonCli
from CliPlugin import TechSupportCli
from CliPlugin import AleCliLib
from CliPlugin.AleFibCliModel import (
      AleRoute,
      AleL2Adj,
      AleAdj,
      AllAleRoutes,
)
import CliToken.Hardware
import CliToken.Ip
import CliToken.Ipv6
import CommonGuards
import ShowCommand
import TableOutput
import Tac
import Tracing
from CliDynamicSymbol import CliDynamicPlugin

__defaultTraceHandle__ = Tracing.Handle( 'AleFibCli' )

#------------------------------------------------------------------------------------
# Common tokens used in multiple commands
#------------------------------------------------------------------------------------

VRFNODE_HELPDESC = 'Specify the VRF'

ipMatcherForShow = CliToken.Ip.ipMatcherForShow
ipv6MatcherForShow = CliToken.Ipv6.ipv6MatcherForShow

hardwareMatcherForShow = CliToken.Hardware.hardwareMatcherForShow

aleBindingsKwMatcher = KeywordMatcher( 'bindings',
      helpdesc="Show counter-related bindings" )
aleCountersKwMatcher = KeywordMatcher( 'counters',
      helpdesc="Show counter-related information" )
aleEgressKwMatcher = KeywordMatcher( 'egress',
      helpdesc="Show egress counter-related information" )
aleKwForShow = CliCommand.guardedKeyword( 'ale',
      helpdesc="Ale-specific information",
      guard=CommonGuards.standbyGuard )
alePlatformKeyMatcher = KeywordMatcher( 'platform-key',
      helpdesc="Show counter-related bindings for platform-key" )
ecmpMatcherForShow = KeywordMatcher( 'ecmp',
      helpdesc='Show ECMP routes' )
resilienceMatcherForShow = KeywordMatcher( 'resilience',
      helpdesc='Show Resilience in ECMP routes' )
summaryMatcher = KeywordMatcher( 'summary',
      helpdesc="configuration parameters summary" )
vrfExprFactory = VrfCli.VrfExprFactory( helpdesc=VRFNODE_HELPDESC, inclAllVrf=False )

slashedIntegerRangeMatcher = SlashedIntegerMatcher(
    0, 18446744073709551615,
    helpname='<0-18446744073709551615>/<0-18446744073709551615>',
    helpdesc='PDCK values in slash notation',
    subLbound=0, subUbound=18446744073709551615 )

AleFibCliDynamicModel = CliDynamicPlugin( "AleFibCliDynamicModel" )

#------------------------------------------------------------------------------------
# "show ip hardware fib summary
# "show ipv6 hardware fib summary
#------------------------------------------------------------------------------------

class ShowIpIpv6HardwareFibSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ( ip | ipv6 ) hardware fib summary'
   data = { 'ip': ipMatcherForShow,
            'ipv6': ipv6MatcherForShow,
            'hardware': hardwareMatcherForShow,
            'fib': IraCommonCli.fibMatcher,
            'summary': summaryMatcher,
          }
   cliModel = "AleFibCliDynamicModel.FibSummaryStatus"
   privileged = True

   handler = "AleFibCliHandler.handleShowIpIpv6HardwareFibSummary"

BasicCli.addShowCommandClass( ShowIpIpv6HardwareFibSummaryCmd )

#------------------------------------------------------------------------------------
# "show ip hardware fib diff
#------------------------------------------------------------------------------------


diffMatcher = KeywordMatcher(
      "diff",
      helpdesc="Display routing table differences between fib and platform" )

class ShowIpHardwareFibDiffCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip hardware fib diff'
   data = { 'ip': ipMatcherForShow,
            'hardware': hardwareMatcherForShow,
            'fib': IraCommonCli.fibMatcher,
            'diff': diffMatcher,
          }
   privileged = True
   handler = "AleFibCliHandler.doShowL3FibRoutesV4Diff"

BasicCli.addShowCommandClass( ShowIpHardwareFibDiffCmd )

#------------------------------------------------------------------------------------
# "show ip hardware fib routes [vrf vrfName] [prefix]
#------------------------------------------------------------------------------------

class ShowIpHardwareFibRoutesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip hardware fib routes [ VRF ] [ PREFIX ]'
   data = { 'ip': ipMatcherForShow,
            'hardware': hardwareMatcherForShow,
            'fib': IraCommonCli.fibMatcher,
            'routes': AleCliLib.v4RoutesMatcher,
            'VRF': vrfExprFactory,
            'PREFIX': AleCliLib.IpPrefixOrAddrMatcher,
          }
   cliModel = "AleFibCliDynamicModel.FibRoute"
   privileged = True
   handler = "AleFibCliHandler.doShowL3FibRoutesV4"

BasicCli.addShowCommandClass( ShowIpHardwareFibRoutesCmd )

#------------------------------------------------------------------------------------
# "show ipv6 hardware fib routes [vrf vrfName] [prefix]
#------------------------------------------------------------------------------------

v6RoutesMatcher = CliCommand.Node(
   matcher=KeywordMatcher( "routes", helpdesc="Display IPv6 routes" ),
   guard=AleCliLib.vrfRootSmGuard )

class Ip6PrefixOrAddrMatcher( CliCommand.CliExpression ):
   expression = 'ADDR | PREFIX'
   data = { 'ADDR': Ip6AddrMatcher.Ip6AddrMatcher( 'Match this IPv6 address' ),
            'PREFIX': Ip6AddrMatcher.ip6PrefixExpr(
               'Match this IPv6 address',
               'Match this subnet mask',
               'Match this IPv6 prefix',
               overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO,
               fullMaskValid=False ),
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      '''Consistently provides PREFIX as an Ip6AddrWithMask'''
      # If an address is provided, convert it to /128 prefix
      if 'ADDR' in args:
         args[ 'PREFIX' ] = Arnet.Ip6AddrWithMask( args.pop( 'ADDR' ) )
      elif 'PREFIX' in args:
         # Even though fullMaskValid is False, we still get back a
         # Ip6AddrWithFullMask. Convert, since the mask is required to be contiguous,
         # and it is easier to return a consistent type.
         pfx = args[ 'PREFIX' ]
         args[ 'PREFIX' ] = Arnet.Ip6AddrWithMask( pfx.address,
                                                   mask=pfx.prefixLen() )

class ShowIp6HardwareFibRoutesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 hardware fib routes [ VRF ] [ PREFIX ]'
   data = { 'ipv6': ipv6MatcherForShow,
            'hardware': hardwareMatcherForShow,
            'fib': IraCommonCli.fibMatcher,
            'routes': v6RoutesMatcher,
            'VRF': vrfExprFactory,
            'PREFIX': Ip6PrefixOrAddrMatcher,
          }
   cliModel = "AleFibCliDynamicModel.FibRoute"
   privileged = True
   handler = "AleFibCliHandler.doShowL3FibRoutesV6"

BasicCli.addShowCommandClass( ShowIp6HardwareFibRoutesCmd )

fecIdMatcher = IntegerMatcher( 1, 0xFFFFFFFFFFFFFFFF, helpdesc='fec identifier' )

def fecIdValid( fecId ):
   tacFecId = Tac.Value( "Smash::Fib::FecId", fecId )
   if not tacFecId.isValid():
      return False
   return True

#------------------------------------------------------------------------------------
# "show ip hardware fib fec [vrf vrfName] [fecId]
# "show ipv6 hardware fib fec [vrf vrfName] [fecId]
#------------------------------------------------------------------------------------

fecMatcher = CliCommand.Node(
   matcher=KeywordMatcher( "fec",
                           helpdesc="Display adjacencies in this address family" ),
   guard=AleCliLib.vrfRootSmGuard )

class ShowIpIp6HardwareFibFecCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ( ip | ipv6 ) hardware fib fec [ VRF ] [ FECID ]'
   data = { 'ip': ipMatcherForShow,
            'ipv6': ipv6MatcherForShow,
            'hardware': hardwareMatcherForShow,
            'fib': IraCommonCli.fibMatcher,
            'fec': fecMatcher,
            'VRF': vrfExprFactory,
            'FECID': fecIdMatcher,
          }
   cliModel = "AleFibCliDynamicModel.FibFec"
   privileged = True
   handler = "AleFibCliHandler.doShowL3FibFec"

BasicCli.addShowCommandClass( ShowIpIp6HardwareFibFecCmd )

#------------------------------------------------------------------------------------
# "show ip hardware fib ecmp resilience [vrf vrfname] summary"
#------------------------------------------------------------------------------------

def resilientEcmpGuard( mode, token ):
   if AleCliLib.routingHwStatus.resilientEcmpSupported:
      return None
   return CliParser.guardNotThisPlatform

class ShowIpEcmpResilienceSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip hardware fib ecmp resilience [ VRF ] summary'
   data = { "ip": ipMatcherForShow,
            "hardware": hardwareMatcherForShow,
            "fib": IraCommonCli.fibMatcher,
            "ecmp": CliCommand.Node( matcher=ecmpMatcherForShow,
                                     guard=resilientEcmpGuard ),
            "resilience": resilienceMatcherForShow,
            "VRF": vrfExprFactory,
            "summary": summaryMatcher,
            }
   cliModel = "AleFibCliDynamicModel.AleResEcmpSummary"
   privileged = True
   handler = "AleFibCliHandler.doShowResilientEcmpSummary"

BasicCli.addShowCommandClass( ShowIpEcmpResilienceSummaryCmd )

#------------------------------------------------------------------------------------
# "show ip hardware fib ecmp resilience [vrf vrfname] [prefix]"
#------------------------------------------------------------------------------------

class ShowIpEcmpResilienceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip hardware fib ecmp resilience [ VRF ] [ PREFIX ]'
   data = { "ip": ipMatcherForShow,
            "hardware": hardwareMatcherForShow,
            "fib": IraCommonCli.fibMatcher,
            "ecmp": CliCommand.Node( matcher=ecmpMatcherForShow,
                                     guard=resilientEcmpGuard ),
            "resilience": resilienceMatcherForShow,
            "VRF": vrfExprFactory,
            "PREFIX": AleCliLib.IpPrefixOrAddrMatcher,
            }
   cliModel = "AleFibCliDynamicModel.AleResEcmpAllOrSpecific"
   privileged = True
   handler = "AleFibCliHandler.doShowResilientEcmpStatus"

BasicCli.addShowCommandClass( ShowIpEcmpResilienceCmd )

#------------------------------------------------------------------------------------
# "show ipv6 hardware fib ecmp resilience [vrf vrfname] summary"
#------------------------------------------------------------------------------------

def resilientEcmpIp6Guard( mode, token ):
   if AleCliLib.routing6HwStatus.resilientEcmpSupported:
      return None
   return CliParser.guardNotThisPlatform

class ShowIpv6EcmpResilienceSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 hardware fib ecmp resilience [ VRF ] summary'
   data = { "ipv6": ipv6MatcherForShow,
            "hardware": hardwareMatcherForShow,
            "fib": IraCommonCli.fibMatcher,
            "ecmp": CliCommand.Node( matcher=ecmpMatcherForShow,
                                     guard=resilientEcmpIp6Guard ),
            "resilience": resilienceMatcherForShow,
            "VRF": vrfExprFactory,
            "summary": summaryMatcher,
            }
   cliModel = "AleFibCliDynamicModel.AleResEcmpSummary"
   privileged = True
   handler = "AleFibCliHandler.doShowResilientEcmpSummary"

BasicCli.addShowCommandClass( ShowIpv6EcmpResilienceSummaryCmd )

#------------------------------------------------------------------------------------
# "show ipv6 hardware fib ecmp resilience [vrf vrfname] [prefix]"
#------------------------------------------------------------------------------------

class ShowIpv6EcmpResilienceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 hardware fib ecmp resilience [ VRF ] [ PREFIX ]'
   data = { "ipv6": ipv6MatcherForShow,
            "hardware": hardwareMatcherForShow,
            "fib": IraCommonCli.fibMatcher,
            "ecmp": CliCommand.Node( matcher=ecmpMatcherForShow,
                                     guard=resilientEcmpIp6Guard ),
            "resilience": resilienceMatcherForShow,
            "VRF": vrfExprFactory,
            "PREFIX": Ip6PrefixOrAddrMatcher,
            }
   cliModel = "AleFibCliDynamicModel.AleResEcmpAllOrSpecific"
   privileged = True
   handler = "AleFibCliHandler.doShowResilientEcmpStatus"

BasicCli.addShowCommandClass( ShowIpv6EcmpResilienceCmd )

#------------------------------------------------------------------------------------
# "show ip hardware ale routes [vrf vrfname] inconsistent
# "show ipv6 hardware ale routes [vrf vrfname] inconsistent
#------------------------------------------------------------------------------------

inconsistentMatcher = KeywordMatcher(
      "inconsistent", "Show the misforwarding routes" )

class ShowIpHardwareAleInconsistentRoutesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ( ip | ipv6 ) hardware ale routes [ VRF ] inconsistent'
   data = { 'ip': ipMatcherForShow,
            'ipv6': ipv6MatcherForShow,
            'hardware': hardwareMatcherForShow,
            'ale': aleKwForShow,
            'routes': AleCliLib.v4RoutesMatcher,
            'VRF': vrfExprFactory,
            'inconsistent': inconsistentMatcher,
          }
   cliModel = "AleFibCliDynamicModel.InconsistentRouteStatus"
   privileged = True
   handler = "AleFibCliHandler.doShowL3AleInconsistentRoutes"

BasicCli.addShowCommandClass( ShowIpHardwareAleInconsistentRoutesCmd )

#------------------------------------------------------------------------------------
# "show ip hardware ale routes [vrf vrfname] drop
# "show ipv6 hardware ale routes [vrf vrfname] drop
#------------------------------------------------------------------------------------

dropMatcher = KeywordMatcher( "drop", "Show the routes pointing to drop fec" )

class ShowIpHardwareAleDropRoutesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ( ip | ipv6 ) hardware ale routes [ VRF ] drop'
   data = { 'ip': ipMatcherForShow,
            'ipv6': ipv6MatcherForShow,
            'hardware': hardwareMatcherForShow,
            'ale': aleKwForShow,
            'routes': AleCliLib.v4RoutesMatcher,
            'VRF': vrfExprFactory,
            'drop': dropMatcher,
          }
   cliModel = "AleFibCliDynamicModel.DropRouteStatus"
   privileged = True
   handler = "AleFibCliHandler.doShowL3AleDropRoutes"

BasicCli.addShowCommandClass( ShowIpHardwareAleDropRoutesCmd )

#------------------------------------------------------------------------------------
# "show ip hardware ale routes [vrf vrfname | all] [unprogrammed] [prefix]"
#------------------------------------------------------------------------------------

def doShowL3AleRoutesV4( mode, args ):
   cmd = 'AR4#'
   vrfName = args.get( 'VRF' )
   if vrfName:
      cmd += vrfName
   cmd += '#'
   if 'unprogrammed' in args:
      cmd += 'u'
   cmd += '#'
   prefix = args.get( 'PREFIX' )
   if prefix is not None:
      cmd += prefix.stringValue
   cmd += '#'

   return AgentCommandRequest.runCliPrintSocketCommand(
         AleCliLib.entityManager, "ale", "VrfRootSmCliCallbackWithFormat",
         cmd, mode=mode, asyncCommand=True, model=AleRoute )

unprogrammedMatcher = KeywordMatcher( "unprogrammed", "Show unprogrammed routes" )

class ShowIpHardwareAleRoutesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip hardware ale routes [ VRF ] [ unprogrammed ] [ PREFIX ]'
   data = { 'ip': ipMatcherForShow,
            'hardware': hardwareMatcherForShow,
            'ale': aleKwForShow,
            'routes': AleCliLib.v4RoutesMatcher,
            'VRF': vrfExprFactory,
            'unprogrammed': unprogrammedMatcher,
            'PREFIX': AleCliLib.IpPrefixOrAddrMatcher,
          }
   cliModel = AleRoute
   privileged = True
   handler = doShowL3AleRoutesV4

BasicCli.addShowCommandClass( ShowIpHardwareAleRoutesCmd )

def doShowAllL3AleRoutesV4( mode, args ):
   cmd = 'AR4#all#'
   if 'unprogrammed' in args:
      cmd += 'u'
   cmd += '#'
   prefix = args.get( 'PREFIX' )
   if prefix is not None:
      cmd += prefix.stringValue
   cmd += '#'

   return AgentCommandRequest.runCliPrintSocketCommand(
         AleCliLib.entityManager, "ale", "VrfRootSmCliCallbackWithFormat",
         cmd, mode=mode, asyncCommand=True, model=AllAleRoutes )

vrfMatcher = KeywordMatcher( "vrf", VRFNODE_HELPDESC )
allMatcher = KeywordMatcher( "all", "Show routes in all VRFs" )

class ShowIpHardwareAleAllVrfRoutesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip hardware ale routes vrf all [ unprogrammed ] [ PREFIX ]'
   data = { 'ip': ipMatcherForShow,
            'hardware': hardwareMatcherForShow,
            'ale': aleKwForShow,
            'routes': AleCliLib.v4RoutesMatcher,
            'vrf': vrfMatcher,
            'all': allMatcher,
            'unprogrammed': unprogrammedMatcher,
            'PREFIX': AleCliLib.IpPrefixOrAddrMatcher,
          }
   cliModel = AllAleRoutes
   privileged = True
   handler = doShowAllL3AleRoutesV4

BasicCli.addShowCommandClass( ShowIpHardwareAleAllVrfRoutesCmd )

#------------------------------------------------------------------------------------
# "show ipv6 hardware ale routes [vrf vrfName | all] [unprogrammed] [prefix]"
#------------------------------------------------------------------------------------

def doShowL3AleRoutesV6( mode, args ):
   cmd = "AR6#"
   vrfName = args.get( 'VRF' )
   if vrfName:
      cmd += vrfName
   cmd += "#"
   if 'unprogrammed' in args:
      cmd += "u"
   cmd += "#"
   prefix6 = args.get( 'PREFIX' )
   if prefix6 is not None:
      cmd += f"{ prefix6.stringValue }"
   cmd += "#"

   return AgentCommandRequest.runCliPrintSocketCommand(
         AleCliLib.entityManager, "ale", "VrfRootSmCliCallbackWithFormat",
         cmd, mode=mode, asyncCommand=True, model=AleRoute )

class ShowIp6HardwareAleRoutesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 hardware ale routes [ VRF ] [ unprogrammed ] [ PREFIX ]'
   data = { 'ipv6': ipv6MatcherForShow,
            'hardware': hardwareMatcherForShow,
            'ale': aleKwForShow,
            'routes': v6RoutesMatcher,
            'VRF': vrfExprFactory,
            'unprogrammed': unprogrammedMatcher,
            'PREFIX': Ip6PrefixOrAddrMatcher,
          }
   cliModel = AleRoute
   privileged = True
   handler = doShowL3AleRoutesV6

BasicCli.addShowCommandClass( ShowIp6HardwareAleRoutesCmd )

def doShowAllL3AleRoutesV6( mode, args ):
   cmd = "AR6#all#"
   if 'unprogrammed' in args:
      cmd += "u"
   cmd += "#"
   prefix6 = args.get( 'PREFIX' )
   if prefix6 is not None:
      cmd += f"{ prefix6.stringValue }"
   cmd += "#"

   return AgentCommandRequest.runCliPrintSocketCommand(
         AleCliLib.entityManager, "ale", "VrfRootSmCliCallbackWithFormat",
         cmd, mode=mode, asyncCommand=True, model=AllAleRoutes )

class ShowIp6HardwareAleAllVrfRoutesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 hardware ale routes vrf all [ unprogrammed ] [ PREFIX ]'
   data = { 'ipv6': ipv6MatcherForShow,
            'hardware': hardwareMatcherForShow,
            'ale': aleKwForShow,
            'routes': v6RoutesMatcher,
            'vrf': vrfMatcher,
            'all': allMatcher,
            'unprogrammed': unprogrammedMatcher,
            'PREFIX': Ip6PrefixOrAddrMatcher,
          }
   cliModel = AllAleRoutes
   privileged = True
   handler = doShowAllL3AleRoutesV6

BasicCli.addShowCommandClass( ShowIp6HardwareAleAllVrfRoutesCmd )

#------------------------------------------------------------------------------------
# "show ip hardware ale adj [vrf vrfName] [fecId]
#------------------------------------------------------------------------------------

def doShowL3AleAdj( mode, args ):
   cmd = 'AA3#'
   vrfName = args.get( 'VRF' )
   if vrfName:
      cmd += vrfName
   cmd += '#'
   fecId = args.get( 'FECID' )
   if fecId:
      if not fecIdValid( fecId ):
         mode.addError( "fecId not valid" )
         return None
      cmd += str( fecId )
   cmd += '#'
   return AgentCommandRequest.runCliPrintSocketCommand(
         AleCliLib.entityManager, "ale", "VrfRootSmCliCallbackWithFormat",
         cmd, mode=mode, asyncCommand=True, model=AleAdj )

adjMatcher = CliCommand.Node(
   matcher=KeywordMatcher( "adj", helpdesc="Display ale adjacencies" ),
   guard=AleCliLib.vrfRootSmGuard )

class ShowIpHardwareAleAdjCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip hardware ale adj [ VRF ] [ FECID ]'
   data = { 'ip': ipMatcherForShow,
            'hardware': hardwareMatcherForShow,
            'ale': aleKwForShow,
            'adj': adjMatcher,
            'VRF': vrfExprFactory,
            'FECID': fecIdMatcher,
          }
   cliModel = AleAdj
   privileged = True
   handler = doShowL3AleAdj

BasicCli.addShowCommandClass( ShowIpHardwareAleAdjCmd )

#------------------------------------------------------------------------------------
# "show ip hardware ale adj proxy
#------------------------------------------------------------------------------------

class ShowIpHardwareAleAdjProxyCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip hardware ale adj proxy [ VRF ] [ FECID ]'
   data = { 'ip': ipMatcherForShow,
            'hardware': hardwareMatcherForShow,
            'ale': aleKwForShow,
            'adj': adjMatcher,
            'proxy': 'Display ale FEC aliases',
            'VRF': vrfExprFactory,
            'FECID': fecIdMatcher,
          }
   cliModel = "AleFibCliDynamicModel.FecProxyInfoList"
   privileged = True
   handler = "AleFibCliHandler.doShowL3AleAdjProxy"

BasicCli.addShowCommandClass( ShowIpHardwareAleAdjProxyCmd )

#------------------------------------------------------------------------------------
# "show ip hardware ale l2-adj [adjIndex]
#------------------------------------------------------------------------------------

def doShowL3AleL2Adj( mode, args ):

   cmd = "AA2#"
   adjIndex = args.get( 'ADJ_INDEX' )
   if adjIndex:
      cmd += str( adjIndex )
   cmd += '#'

   return AgentCommandRequest.runCliPrintSocketCommand(
         AleCliLib.entityManager, "ale", "VrfRootSmCliCallbackWithFormat",
         cmd, mode=mode, asyncCommand=True, model=AleL2Adj )

l2AdjMatcher = CliCommand.Node(
   matcher=KeywordMatcher( "l2-adj", helpdesc="Display ale layer2 adjacencies" ),
   guard=AleCliLib.vrfRootSmGuard )

l2AdjIndexMatcher = IntegerMatcher( 1, 0xFFFFFFFF, helpdesc='Adjacency index' )

class ShowIpHardwareAleL2AdjCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip hardware ale l2-adj [ ADJ_INDEX ]'
   data = { 'ip': ipMatcherForShow,
            'hardware': hardwareMatcherForShow,
            'ale': aleKwForShow,
            'l2-adj': l2AdjMatcher,
            'ADJ_INDEX': l2AdjIndexMatcher,
          }
   cliModel = AleL2Adj
   privileged = True
   handler = doShowL3AleL2Adj

BasicCli.addShowCommandClass( ShowIpHardwareAleL2AdjCmd )

# -----------------------------------------------------------------------------------
# "show ip hardware ale counters egress bindings l2-adj [ADJ_INDEX]
# -----------------------------------------------------------------------------------

class ShowIpHardwareAleL2AdjCounterBindings( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip hardware ale counters egress bindings l2-adj [ ADJ_INDEX ]'
   data = { 'ip': ipMatcherForShow,
            'hardware': hardwareMatcherForShow,
            'ale': aleKwForShow,
            'counters': aleCountersKwMatcher,
            'egress': aleEgressKwMatcher,
            'bindings': aleBindingsKwMatcher,
            'l2-adj': l2AdjMatcher,
            'ADJ_INDEX': l2AdjIndexMatcher,
          }
   cliModel = "AleFibCliDynamicModel.AleL2AdjCounterBindings"
   privileged = True
   handler = "AleFibCliHandler.doShowAleL2AdjCounterBindings"

BasicCli.addShowCommandClass( ShowIpHardwareAleL2AdjCounterBindings )

# -----------------------------------------------------------------------------------
# "show ip hardware ale counters egress bindings platform-key [ PDCK ]
# -----------------------------------------------------------------------------------

class ShowIpHardwareAlePdckCounterBindings( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip hardware ale counters egress bindings platform-key [ PDCK ]'
   data = { 'ip': ipMatcherForShow,
            'hardware': hardwareMatcherForShow,
            'ale': aleKwForShow,
            'counters': aleCountersKwMatcher,
            'egress': aleEgressKwMatcher,
            'bindings': aleBindingsKwMatcher,
            'platform-key': alePlatformKeyMatcher,
            'PDCK': slashedIntegerRangeMatcher,
          }
   cliModel = "AleFibCliDynamicModel.AlePdckCounterBindings"
   privileged = True
   handler = "AleFibCliHandler.doShowAlePdckCounterBindings"

BasicCli.addShowCommandClass( ShowIpHardwareAlePdckCounterBindings )

#------------------------------------------------------------------------------------
# "show ip hardware ale vrf
#------------------------------------------------------------------------------------

def doShowL3AleVrf( mode, args ):
   vrfStatus = {}
   for vs in AleCliLib.routingHardwareRouteStatus.vrfStatus.values():
      vrfStatus[ vs.vrfName ] = VrfStatus( vrfName=vs.vrfName, vrfId=vs.vrfId,
                                           tableId=0 )
   return RouteStatus( vrfStatus=vrfStatus )

class VrfStatus( CliModel.Model ):
   vrfName = CliModel.Str( help="Name of a VRF" )
   vrfId = CliModel.Int( help="ID used on hardware for the VRF" )
   tableId = CliModel.Int( help="ID used in Ale for the FIB table" )

class RouteStatus( CliModel.Model ):
   vrfStatus = CliModel.Dict( keyType=str, valueType=VrfStatus,
                              help="List of all Vrf names and IDs" )

   def render( self ):
      if not self.vrfStatus:
         return
      table = TableOutput.createTable( ( "VRF Name", "VRF ID", "Table ID" ) )
      tableMaxWidth = table.printableWidth()
      tableMaxWidth = max( tableMaxWidth, 80 )

      f1 = TableOutput.Format( justify="left", maxWidth=64, minWidth=20 )
      f1.noPadLeftIs( True )
      f1.padLimitIs( True )

      f2 = TableOutput.Format( justify="left", maxWidth=10, minWidth=6 )
      f2.noPadLeftIs( True )
      f2.padLimitIs( True )

      f3 = TableOutput.Format( justify="left", maxWidth=10, minWidth=6 )
      f3.noPadLeftIs( True )
      f3.padLimitIs( True )

      table.formatColumns( f1, f2 )

      for _, vrfStatus in sorted( self.vrfStatus.items() ):
         table.newRow( vrfStatus.vrfName, vrfStatus.vrfId, vrfStatus.tableId )
      print( table.output() )

class ShowIpHardwareAleVrfCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip hardware ale vrf'
   data = { 'ip': ipMatcherForShow,
            'hardware': hardwareMatcherForShow,
            'ale': aleKwForShow,
            'vrf': KeywordMatcher( "vrf", helpdesc="Displays the vrf table" ),
          }
   cliModel = RouteStatus
   handler = doShowL3AleVrf

BasicCli.addShowCommandClass( ShowIpHardwareAleVrfCmd )

#--------------------------------------------------------------------------
# register show tech-support to include Ale command outputs
#--------------------------------------------------------------------------
def _showTechGuard():
   return AleCliLib.vrfRootSmGuard( mode=None, token=None ) is None

TechSupportCli.registerShowTechSupportCmd(
   '2019-07-19 14:24:51',
   cmds=[ 'show ip hardware ale vrf' ] )

TechSupportCli.registerShowTechSupportCmd(
   '2019-07-19 14:24:51',
   cmds=[ 'show ip hardware ale routes vrf all',
            'show ipv6 hardware ale routes vrf all',
            'show ip hardware ale adj',
            'show ip hardware ale l2-adj' ],
   cmdsGuard=_showTechGuard )
