#!/usr/bin/env python3
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
import CliParser
import CliParserCommon
import CliToken.Hardware
import ConfigMount
import LazyMount
import ShowCommand
import Tracing

__defaultTraceHandle__ = Tracing.Handle( 'PortGroupCli' )
t0 = __defaultTraceHandle__.trace0

portGroupDir = None
portGroupCliConfig = None

def portGroupSupported( mode, token ):
   for portGroup in portGroupDir.portGroup.values():
      if portGroup.defaultMode:
         return None
   return CliParser.guardNotThisPlatform

nodePortGroup = CliCommand.guardedKeyword( 'port-group',
      helpdesc='Port-group of QSFP or SFP ports', guard=portGroupSupported )

def isStartupConfig( mode ):
   return mode.session_.startupConfig()

class PortGroupMatcher( CliMatcher.Matcher ):
   def match( self, mode, context, token ):
      if not isStartupConfig( mode ):
         if token in portGroupDir.portGroup:
            return CliParserCommon.MatchResult( token, token )
         return CliParserCommon.noMatch
      return CliParserCommon.MatchResult( token, token )

   def completions( self, mode, context, token ):
      if portGroupDir.portGroup:
         ret = []
         if token == '':
            for portGroup in sorted( portGroupDir.portGroup ):
               ret.append( CliParser.Completion( portGroup,
                  f'Set mode for port group { portGroup }' ) )
         else:
            if token in portGroupDir.portGroup:
               ret.append( CliParser.Completion( token,
                  f'Set mode for port group { token }' ) )
         return ret
      return [ CliParser.Completion( 'WORD', 'Portgroup name' ) ]

class PortModeMatcher( CliMatcher.Matcher ):
   def match( self, mode, context, token ):
      if not isStartupConfig( mode ):
         portGroup = context.sharedResult[ 'PORT_GROUP' ]
         portGroupConfig = portGroupDir.portGroup.get( portGroup )
         portModes = None
         if portGroupConfig:
            portModes = portGroupConfig.mode
         if portModes and token in portModes:
            return CliParserCommon.MatchResult( token, token )
         return CliParserCommon.noMatch
      return CliParserCommon.MatchResult( token, token )

   def completions( self, mode, context, token ):
      if portGroupDir.portGroup:
         ret = []
         portGroup = context.sharedResult[ 'PORT_GROUP' ]
         portGroupConfig = portGroupDir.portGroup.get( portGroup )
         portModes = None
         if portGroupConfig:
            portModes = portGroupConfig.mode
         if portModes:
            if token == '':
               for portMode, portModeData in portModes.items():
                  if not portModeData.hiddenFromCli:
                     ret.append( CliParser.Completion( portMode,
                        print( f'Activate ports {portMode}' ) ) )
            else:
               matchingPortModes = [ portMode
                                     for portMode, portModeData in portModes.items()
                                     if token in portMode and
                                     not portModeData.hiddenFromCli ]
               if matchingPortModes:
                  for portMode in matchingPortModes:
                     ret.append( CliParser.Completion( portMode,
                        print( f'Activate ports {portMode}' ) ) )
            return ret
         return []
      return [ CliParser.Completion( 'WORD', 'Portgroup mode' ) ]

#--------------------------------------------------------------------------------
# [ no | default ] hardware port-group PORT_GROUP select PORT_MODE
#--------------------------------------------------------------------------------
class ConfigPortGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'hardware port-group PORT_GROUP select PORT_MODE'
   noOrDefaultSyntax = 'hardware port-group PORT_GROUP ...'
   data = {
      'hardware' : CliToken.Hardware.hardwareMatcherForConfig,
      'port-group' : nodePortGroup,
      'PORT_GROUP' : CliCommand.Node( matcher=PortGroupMatcher(),
         storeSharedResult=True ),
      'select' : 'Set the operational mode for port group',
      'PORT_MODE' : PortModeMatcher(),
   }

   handler = "AlePortGroupCliHandler.handleConfigPortGroup"
   noOrDefaultHandler = "AlePortGroupCliHandler.handleNoConfigPortGroup"

BasicCli.GlobalConfigMode.addCommandClass( ConfigPortGroupCmd )

#--------------------------------------------------------------------------------
# show hardware port-group
#--------------------------------------------------------------------------------
class HardwarePortGroupCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show hardware port-group'
   data = {
      'hardware' : CliToken.Hardware.hardwareMatcherForShow,
      'port-group' : nodePortGroup,
   }

   handler = "AlePortGroupCliHandler.handleHardwarePortGroup"

BasicCli.addShowCommandClass( HardwarePortGroupCmd )

def Plugin( em ):
   global portGroupCliConfig, portGroupDir
   portGroupDir = LazyMount.mount( # all slices configs
      em, 'hardware/ale/portGroup', 'Ale::PortGroupDir', 'ri' )

   # there are two config sources for the port groups.
   # 1 - the user, with the Cli and the Cli's startup config
   # 2 - default group, set by fdl and Fru.
   # the forwaring agent creats a portGroupConfig with those two as input

   # here we mount portGroupCliConfig where we set the CLI's portgroup config
   portGroupCliConfig = ConfigMount.mount(
      em, 'hardware/ale/portGroupCliConfig', 'Ale::PortGroupConfig', 'rw' )
