# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from AleFlexCounterTableMounter import tableMountPath
from Arnet import IpGenAddr, IpAddr, Ip6Addr
from CliPlugin.AleCountersCli import ( checkCounterFeatureEnabled,
                                       checkCounterFeatureSupported,
                                       getCurrentCounter )
from CliPlugin.AleCountersCliLib import vtepCounter, vxlanCounter
from CliPlugin.AleVxlanCliLib import vnisInSystem, vtepsInSystem
from CliPlugin.VxlanCli import showVtepCountersHook
import LazyMount
import SmashLazyMount
import Tac
from TypeFuture import TacLazyType

FapId = TacLazyType( 'FlexCounters::FapId' )
FeatureIdEnum = TacLazyType( 'FlexCounters::FeatureId' )
FeatureIdEnumVal = TacLazyType( 'FlexCounters::FeatureIdEnumVal' )
TunnelId = TacLazyType( 'Tunnel::TunnelTable::TunnelId' )
TunnelType = TacLazyType( 'Tunnel::TunnelTable::TunnelType' )

class VxlanPlugin:
   def __init__( self, pathPrefix, agentName, useHelperForEncapCounters=False,
                 isSupported=lambda: True ):
      self.vtepHwStatusDir = None
      self.vtiStatusDir = None
      self.vtepDecapCounterTable = None
      self.vtepDecapSnapshotTable = None
      self.vtepEncapCounterTable = None
      self.vtepEncapSnapshotTable = None
      self.vniCounterTable = {}
      self.vniSnapshotTable = {}
      self.unlearntVtepsIpAddr = "0.0.0.0"
      self.pathPrefix = pathPrefix
      self.agentName = agentName
      self.useHelperForEncapCounters = useHelperForEncapCounters
      self.isSupported = isSupported

   def mountVtepCountersPaths( self, em ):
      self.vtepHwStatusDir = LazyMount.mount(
         em, 'vxlan/vtepHwStatus', 'Vxlan::VtepHwStatusDir', 'r' )

      info = SmashLazyMount.mountInfo( 'reader' )

      # Mount tables for VTEP Decap counters.
      mountPath = tableMountPath(
         em, FeatureIdEnum.VtepDecap, FapId.allFapsId, False,
         pathPrefix=self.pathPrefix )
      self.vtepDecapCounterTable = SmashLazyMount.mount(
         em, mountPath, "Ale::FlexCounter::VtepDecapCounterTable", info )

      mountPath = tableMountPath(
         em, FeatureIdEnum.VtepDecap, FapId.allFapsId, True,
         pathPrefix=self.pathPrefix )
      self.vtepDecapSnapshotTable = SmashLazyMount.mount(
         em, mountPath, "Ale::FlexCounter::VtepDecapCounterTable", info )

      # Note that currently the helper is for distinguishing VTEP encap counters on
      # Sand and Strata. encapDropExcptPkts is present on Strata only, namely when
      # the helper is being used.
      if self.useHelperForEncapCounters:
         encapFeatureId = FeatureIdEnum.VtepEncap
         encapPrefix = self.pathPrefix
         encapTableTypeName = "Ale::FlexCounter::VtepEncapCounterTable"
      else:
         # Reference to Nexthop feature ID is deliberate here as we are
         # reusing Nexthop counter smashes by design.
         encapFeatureId = FeatureIdEnum.Nexthop
         encapPrefix = None
         encapTableTypeName = "FlexCounters::CounterTable"
      mountPath = tableMountPath(
         em, encapFeatureId, FapId.allFapsId, False, pathPrefix=encapPrefix )
      self.vtepEncapCounterTable = SmashLazyMount.mount(
         em, mountPath, encapTableTypeName, info )

      mountPath = tableMountPath(
         em, encapFeatureId, FapId.allFapsId, True, pathPrefix=encapPrefix )
      self.vtepEncapSnapshotTable = SmashLazyMount.mount(
         em, mountPath, encapTableTypeName, info )

   def registerShowVtepCountersHook( self ):
      # Register hook for showing VTEP decap/encap counters.
      showVtepCountersHook.addExtension( self.showVtepCountersHook )

   def mountVniCountersPaths( self, em ):
      self.vtiStatusDir = LazyMount.mount( em, 'interface/status/eth/vxlan',
                                           'Vxlan::VtiStatusDir', 'r' )

      info = SmashLazyMount.mountInfo( 'reader' )

      # Mount tables for VNI Decap counters.
      mountPath = tableMountPath( em, FeatureIdEnum.VniDecap,
                                  FapId.allFapsId, False,
                                  pathPrefix=self.pathPrefix )
      self.vniCounterTable[ 'decap' ] = SmashLazyMount.mount(
         em, mountPath, "Ale::FlexCounters::VniCountersTable", info )

      mountPath = tableMountPath( em, FeatureIdEnum.VniDecap,
                                  FapId.allFapsId, True,
                                  pathPrefix=self.pathPrefix )
      self.vniSnapshotTable[ 'decap' ] = SmashLazyMount.mount(
         em, mountPath, "Ale::FlexCounters::VniCountersTable", info )

      # Mount tables for VNI Encap counters.
      mountPath = tableMountPath( em, FeatureIdEnum.VniEncap,
                                  FapId.allFapsId, False,
                                  pathPrefix=self.pathPrefix )
      self.vniCounterTable[ 'encap' ] = SmashLazyMount.mount(
         em, mountPath, "Ale::FlexCounters::VniCountersTable", info )

      mountPath = tableMountPath( em, FeatureIdEnum.VniEncap,
                                  FapId.allFapsId, True,
                                  pathPrefix=self.pathPrefix )
      self.vniSnapshotTable[ 'encap' ] = SmashLazyMount.mount(
         em, mountPath, "Ale::FlexCounters::VniCountersTable", info )

   # -------------------------------------------------------------------------------
   #  "show vxlan counters vtep [<vtep>] [encap|decap]" command, in "enable" mode.
   # -------------------------------------------------------------------------------
   def showVtepCountersHook( self, mode, inVtep, direction, vtepCounters ):
      if not self.isSupported():
         return

      def addVtepDecapModel( vtep ):
         vtepCtr = vtepCounters.addVtep( vtep )
         updateVtepDecapModel( vtep, vtepCtr )

      def addUnlearntVtepsModel():
         unlearntVtepsCtr = vtepCounters.addUnlearntVteps()
         updateVtepDecapModel( self.unlearntVtepsIpAddr, unlearntVtepsCtr )

      vtepDecapSupported = checkCounterFeatureSupported( FeatureIdEnum.VtepDecap )
      vtepEncapSupported = checkCounterFeatureSupported( FeatureIdEnum.VtepEncap )

      def updateVtepDecapModel( vtep, vtepCtr ):
         knownUcastPkts, knownUcastBytes, bumPkts, bumBytes, dropExcptPkts, _ = \
            vtepCounter( IpGenAddr( str( vtep ) ),
                         self.vtepDecapCounterTable,
                         self.vtepDecapSnapshotTable )

         # Update counter attributes.
         vtepCtr.addToAttr( "decapBytes", knownUcastBytes + bumBytes )
         vtepCtr.addToAttr( "decapKnownUcastPkts", knownUcastPkts )
         vtepCtr.addToAttr( "decapBUMPkts", bumPkts )
         vtepCtr.addToAttr( "decapDropExcptPkts", dropExcptPkts )

      def updateVtepEncapModel( vtep ):
         if not isinstance( vtep, Tac.Type( "Arnet::IpGenAddr" ) ):
            vtep = IpGenAddr( str( vtep ) )

         vtepCtr = vtepCounters.addVtep( vtep )
         if self.useHelperForEncapCounters:
            encapPkts, encapBytes, _, _, encapDropExcptPkts, _ = vxlanCounter(
               vtep,
               self.vtepEncapCounterTable.counter,
               self.vtepEncapSnapshotTable.counter )
         else:
            if vtep.af == "ipv6":
               tunnelIndex = Ip6Addr( vtep ).hash
            else:
               tunnelIndex = IpAddr( vtep ).hash
            tunnelId = TunnelId.convertToTunnelValue(
               TunnelType.vxlanVtepEncapTunnel, tunnelIndex )
            encapPkts, encapBytes = getCurrentCounter(
               tunnelId, self.vtepEncapCounterTable, self.vtepEncapSnapshotTable )

         # Update counter attributes.
         vtepCtr.addToAttr( "encapBytes", encapBytes )
         vtepCtr.addToAttr( "encapPkts", encapPkts )
         if self.useHelperForEncapCounters:
            vtepCtr.addToAttr( "encapDropExcptPkts", encapDropExcptPkts )

      vteps = vtepsInSystem( None, None, vtepHwStatusDir=self.vtepHwStatusDir )

      if vtepDecapSupported and ( not direction or direction == "decap" ):
         # Update the model passed to us. We update only decap counters here.
         vtepCounters.decapAttrsToDisplay( [ "decapBytes",
                                             "decapKnownUcastPkts",
                                             "decapBUMPkts",
                                             "decapDropExcptPkts" ] )
         featureEnabled = checkCounterFeatureEnabled( FeatureIdEnum.VtepDecap )
         vtepCounters.decapFeatureEnabledIs( featureEnabled )

         if featureEnabled:
            if inVtep:
               if inVtep == 'unlearnt':
                  addUnlearntVtepsModel()
               elif IpGenAddr( inVtep ) in vteps:
                  addVtepDecapModel( inVtep )
               # else inVtep is invalid, and so leave the model empty
            else:
               for vtep in vteps:
                  addVtepDecapModel( vtep )
               addUnlearntVtepsModel()

      if vtepEncapSupported and ( not direction or direction == "encap" ):
         # Update the model passed to us. We update only encap counters here.
         vtepCounters.encapAttrsToDisplay( [ "encapBytes",
                                             "encapPkts" ] )
         if self.useHelperForEncapCounters:
            vtepCounters.encapAttrsToDisplay( [ "encapDropExcptPkts" ] )
         featureEnabled = checkCounterFeatureEnabled( FeatureIdEnum.VtepEncap )
         vtepCounters.encapFeatureEnabledIs( featureEnabled )

         if featureEnabled:
            # Populate counters only if feature is enabled.
            if isinstance( inVtep, str ) and inVtep != 'unlearnt':
               inVtep = IpGenAddr( inVtep )

            if inVtep:
               if inVtep in vteps:
                  updateVtepEncapModel( inVtep )
               # Else inVtep is invalid, and so leave the model empty.
            else:
               for vtep in vteps:
                  updateVtepEncapModel( vtep )

   # -------------------------------------------------------------------------------
   # "show vxlan counters vni [<vni>] [encap|decap]" command, in "enable" mode.
   # -------------------------------------------------------------------------------
   def showVniCountersHook( self, mode, inVni, direction, vniCounters ):
      def updateVniModel( vni, direction ):
         vniCtr = vniCounters.addVni( vni )

         numPkts, numBytes = getCurrentCounter( vni,
                                                self.vniCounterTable[ direction ],
                                                self.vniSnapshotTable[ direction ] )

         # Update counter attributes.
         vniCtr.addToAttr( f"{direction}Bytes", numBytes )
         vniCtr.addToAttr( f"{direction}Pkts", numPkts )

      vniDecapSupported = checkCounterFeatureSupported( FeatureIdEnum.VniDecap )
      vniEncapSupported = checkCounterFeatureSupported( FeatureIdEnum.VniEncap )
      systemVnis = vnisInSystem( self.vtiStatusDir )
      vnis = set( systemVnis ) & set( [ inVni ] ) if inVni else systemVnis

      if vniDecapSupported and ( not direction or direction == "decap" ):
         # Update the model passed to us. We update only decap counters here.
         vniCounters.decapAttrsToDisplay( [ "decapBytes", "decapPkts" ] )
         featureEnabled = checkCounterFeatureEnabled( FeatureIdEnum.VniDecap )
         vniCounters.decapFeatureEnabledIs( featureEnabled )

         if featureEnabled:
            for vni in vnis:
               updateVniModel( vni, "decap" )

      if vniEncapSupported and ( not direction or direction == "encap" ):
         # Update the model passed to us. We update only encap counters here.
         vniCounters.encapAttrsToDisplay( [ "encapBytes", "encapPkts" ] )
         featureEnabled = checkCounterFeatureEnabled( FeatureIdEnum.VniEncap )
         vniCounters.encapFeatureEnabledIs( featureEnabled )

         if featureEnabled:
            for vni in vnis:
               updateVniModel( vni, "encap" )
