#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Arnet import IpGenAddr
from AleFlexCounterTableMounter import tableMountPath
from ArfaCountersUtils import etbaFeatureStatusDir
from CliPlugin.AleCountersCli import ( checkCounterFeatureEnabled,
                                       checkCounterFeatureSupported,
                                       getCurrentCounter )
from CliPlugin.AleVxlanCliLib import vtepsInSystem, vnisInSystem
from CliPlugin.VxlanCli import showVtepCountersHook, showVniCountersHook
from CliPlugin.MplsTunnelCountersCli import mplsTunnelCounterActiveHook
from CliPlugin.AleCountersCli import (
   counterFeatureSettingHook,
   counterFeatureSupportedHook,
)
import LazyMount
import SmashLazyMount
from TypeFuture import TacLazyType
import Tac

etbaStatus = None

FeatureId = TacLazyType( 'FlexCounters::FeatureId' )
FeatureIdEnumVal = TacLazyType( 'FlexCounters::FeatureIdEnumVal' )
AllFapsId = TacLazyType( 'FlexCounters::FapId' ).allFapsId
ArfaModeType = TacLazyType( "Arfa::ArfaMode" )
VtepVniDecapCounterTypes = TacLazyType( "Arfa::Counter::VtepVniDecapCounterTypes" )
VtepVniEncapCounterTypes = TacLazyType( "Arfa::Counter::VtepVniEncapCounterTypes" )

vtepHwStatusDir = None
unlearntVtepsIpAddr = "0.0.0.0"

vtepCounterStatus = None
vtepDecapCounterTable = None
vtepDecapSnapshotTable = None
vtepEncapCounterTable = None
vtepEncapSnapshotTable = None
vniCounterStatus = None
vniDecapCounterTable = None
vniDecapSnapshotTable = None
vniEncapCounterTable = None
vniEncapSnapshotTable = None
vtiStatusDir = None

def checkArfaTunnelCounterActive( tunnelId ):
   return etbaStatus.arfaMode == ArfaModeType.arfaOnlyMode

mplsTunnelCounterActiveHook.addExtension( checkArfaTunnelCounterActive )

# -------------------------------------------------------------------------------
# "show vxlan counters vtep [<vtep>] [encap|decap]" command, in "enable" mode.
# -------------------------------------------------------------------------------
def showArfaVtepCountersHook( mode, inVtep, direction, vtepCounters ):
   if etbaStatus.arfaMode != ArfaModeType.arfaOnlyMode:
      return

   def addVtepDecapModel( vtep ):
      vtepCtr = vtepCounters.addVtep( vtep )
      updateVtepDecapModel( vtep, vtepCtr )

   def addUnlearntVtepsModel():
      unlearntVtepsCtr = vtepCounters.addUnlearntVteps()
      updateVtepDecapModel( unlearntVtepsIpAddr, unlearntVtepsCtr )

   vtepDecapSupported = checkCounterFeatureSupported( FeatureId.VtepDecap )
   vtepEncapSupported = checkCounterFeatureSupported( FeatureId.VtepEncap )

   def updateVtepDecapModel( vtep, vtepCtr ):
      try:
         counterInfo = vtepCounterStatus.vtepCounterInfo[ IpGenAddr(
            str( vtep ) ) ]
      except KeyError:
         return
      baseIdx = counterInfo.counterIndex
      decapDropExcptPkts, _ = getCurrentCounter(
         baseIdx, vtepDecapCounterTable, vtepDecapSnapshotTable )
      decapKnownUcastPkts, decapKnownUcastOctets = getCurrentCounter(
         baseIdx + Tac.enumValue( "Arfa::Counter::VtepVniDecapCounterTypes",
                                  VtepVniDecapCounterTypes.ucastPkts ),
         vtepDecapCounterTable,
         vtepDecapSnapshotTable )
      decapBUMPkts, decapBUMOctets = getCurrentCounter(
         baseIdx + Tac.enumValue( "Arfa::Counter::VtepVniDecapCounterTypes",
                                  VtepVniDecapCounterTypes.bumPkts ),
         vtepDecapCounterTable,
         vtepDecapSnapshotTable )

      # update counter attributes
      decapBytes = decapKnownUcastOctets + decapBUMOctets

      # Update counter attributes.
      vtepCtr.addToAttr( "decapBytes", decapBytes )
      vtepCtr.addToAttr( "decapKnownUcastPkts", decapKnownUcastPkts )
      vtepCtr.addToAttr( "decapBUMPkts", decapBUMPkts )
      vtepCtr.addToAttr( "decapDropExcptPkts", decapDropExcptPkts )

   def updateVtepEncapModel( vtep ):
      if not isinstance( vtep, Tac.Type( "Arnet::IpGenAddr" ) ):
         vtep = IpGenAddr( str( vtep ) )

      vtepCtr = vtepCounters.addVtep( vtep )
      try:
         counterInfo = vtepCounterStatus.vtepCounterInfo[ IpGenAddr(
            str( vtep ) ) ]
      except KeyError:
         return
      baseIdx = counterInfo.counterIndex
      encapDropPkts, _ = getCurrentCounter(
         baseIdx, vtepEncapCounterTable, vtepEncapSnapshotTable )
      encapUcastPkts, encapUcastOctets = getCurrentCounter(
         baseIdx + Tac.enumValue( "Arfa::Counter::VtepVniEncapCounterTypes",
                                  VtepVniEncapCounterTypes.ucastPkts ),
         vtepEncapCounterTable,
         vtepEncapSnapshotTable )

      # Update counter attributes.
      vtepCtr.addToAttr( "encapBytes", encapUcastOctets )
      vtepCtr.addToAttr( "encapPkts", encapUcastPkts )
      vtepCtr.addToAttr( "encapDropPkts", encapDropPkts )

   vteps = vtepsInSystem( None, None, vtepHwStatusDir=vtepHwStatusDir )

   if vtepDecapSupported and ( not direction or direction == "decap" ):
      # Update the model passed to us. We update only decap counters here.
      vtepCounters.decapAttrsToDisplay( [ "decapBytes",
                                          "decapKnownUcastPkts",
                                          "decapBUMPkts",
                                          "decapDropExcptPkts" ] )
      featureEnabled = checkCounterFeatureEnabled( FeatureId.VtepDecap )
      vtepCounters.decapFeatureEnabledIs( featureEnabled )

      if featureEnabled:
         if inVtep:
            if inVtep == 'unlearnt':
               addUnlearntVtepsModel()
            elif IpGenAddr( inVtep ) in vteps:
               addVtepDecapModel( inVtep )
            # else inVtep is invalid, and so leave the model empty
         else:
            for vtep in vteps:
               addVtepDecapModel( vtep )
            addUnlearntVtepsModel()

   if vtepEncapSupported and ( not direction or direction == "encap" ):
      # Update the model passed to us. We update only encap counters here.
      vtepCounters.encapAttrsToDisplay( [ "encapBytes",
                                          "encapPkts",
                                          "encapDropPkts" ] )
      featureEnabled = checkCounterFeatureEnabled( FeatureId.VtepEncap )
      vtepCounters.encapFeatureEnabledIs( featureEnabled )

      if featureEnabled:
         # Populate counters only if feature is enabled.
         if isinstance( inVtep, str ) and inVtep != 'unlearnt':
            inVtep = IpGenAddr( inVtep )

         if inVtep:
            if inVtep in vteps:
               updateVtepEncapModel( inVtep )
            # Else inVtep is invalid, and so leave the model empty.
         else:
            for vtep in vteps:
               updateVtepEncapModel( vtep )

# -------------------------------------------------------------------------------
# "show vxlan counters vni [<vni>] [encap|decap]" command, in "enable" mode.
# -------------------------------------------------------------------------------
def showArfaVniCountersHook( mode, inVni, direction, vniCounters ):
   if etbaStatus.arfaMode != ArfaModeType.arfaOnlyMode:
      return

   def updateDecapCtrsInModel( vni ):
      # Add vni counter model irrespective of whether we find vni counterInfo.
      vniCtr = vniCounters.addVni( vni )

      if vni not in vniCounterStatus.vniCounterInfo:
         # Since we don't have counterInfo for this vni leave the model empty.
         # Empty will indicate that vni is valid, but there are no counters
         # for it.
         return

      counterInfo = vniCounterStatus.vniCounterInfo[ vni ]
      baseIdx = counterInfo.counterIndex

      decapDropExcptPkts, _ = getCurrentCounter(
         baseIdx, vniDecapCounterTable, vniDecapSnapshotTable )
      decapKnownUcastPkts, decapKnownUcastOctets = getCurrentCounter(
         baseIdx + Tac.enumValue( "Arfa::Counter::VtepVniDecapCounterTypes",
                                  VtepVniDecapCounterTypes.ucastPkts ),
         vniDecapCounterTable,
         vniDecapSnapshotTable )
      decapBUMPkts, decapBUMOctets = getCurrentCounter(
         baseIdx + Tac.enumValue( "Arfa::Counter::VtepVniDecapCounterTypes",
                                  VtepVniDecapCounterTypes.bumPkts ),
         vniDecapCounterTable,
         vniDecapSnapshotTable )

      # update counter attributes
      decapBytes = decapKnownUcastOctets + decapBUMOctets
      vniCtr.addToAttr( "decapBytes", decapBytes )
      vniCtr.addToAttr( "decapKnownUcastPkts", decapKnownUcastPkts )
      vniCtr.addToAttr( "decapBUMPkts", decapBUMPkts )
      vniCtr.addToAttr( "decapDropExcptPkts", decapDropExcptPkts )

   def updateEncapCtrsInModel( vni ):
      # Add vni counter model irrespective of whether we find vni counterInfo.
      vniCtr = vniCounters.addVni( vni )

      if vni not in vniCounterStatus.vniCounterInfo:
         # Since we don't have counterInfo for this vni leave the model empty.
         # Empty will indicate that vni is valid, but there are no counters
         # for it.
         return

      counterInfo = vniCounterStatus.vniCounterInfo[ vni ]
      baseIdx = counterInfo.counterIndex

      dropPkts, _ = getCurrentCounter(
         baseIdx, vniEncapCounterTable, vniEncapSnapshotTable )
      encapUcastPkts, encapUcastOctets = getCurrentCounter(
         baseIdx + Tac.enumValue( "Arfa::Counter::VtepVniEncapCounterTypes",
                                  VtepVniEncapCounterTypes.ucastPkts ),
         vniEncapCounterTable,
         vniEncapSnapshotTable )

      # update counter attributes
      vniCtr.addToAttr( "encapBytes", encapUcastOctets )
      vniCtr.addToAttr( "encapPkts", encapUcastPkts )
      vniCtr.addToAttr( "dropPkts", dropPkts )

   allVnis = vnisInSystem( vtiStatusDir )

   if not direction or direction == "decap":
      # update the model passed to us. We update only decap counters here.
      vniCounters.decapAttrsToDisplay( [ "decapBytes",
                                         "decapKnownUcastPkts",
                                         "decapBUMPkts",
                                         "decapDropExcptPkts" ] )
      featureEnabled = checkCounterFeatureEnabled( FeatureId.VniDecap )
      vniCounters.decapFeatureEnabledIs( featureEnabled )
      if featureEnabled:
         # populating counters only if feature is enabled
         if inVni:
            if inVni in allVnis:
               updateDecapCtrsInModel( inVni )
         else:
            for vni in allVnis:
               updateDecapCtrsInModel( vni )

   if not direction or direction == "encap":
      # update the model passed to us. We update only encap counters here.
      vniCounters.encapAttrsToDisplay( [ "encapBytes",
                                         "encapPkts" ] )
      featureEnabled = checkCounterFeatureEnabled( FeatureId.VniEncap )
      vniCounters.encapFeatureEnabledIs( featureEnabled )
      if featureEnabled:
         if inVni:
            if inVni in allVnis:
               updateEncapCtrsInModel( inVni )
         else:
            for vni in allVnis:
               updateEncapCtrsInModel( vni )

# -------------------------------------------------------------------------------
# Guard for "hardware counter feature [<counterFeature>]" command
# -------------------------------------------------------------------------------
def etbaCounterFeatureSupported( mode, token ):
   if etbaStatus.arfaMode == ArfaModeType.arfaOnlyMode:
      featureInfo = Tac.newInstance( "Arfa::ArfaCounterFeatureInfo", "" )
      return featureInfo.isSupportedFeature.keys()
   return []

def Plugin( entityManager ):
   global etbaStatus
   global vtepHwStatusDir, vtiStatusDir
   global vtepCounterStatus
   global vtepDecapCounterTable, vtepDecapSnapshotTable
   global vtepEncapCounterTable, vtepEncapSnapshotTable
   global vniCounterStatus
   global vniDecapCounterTable, vniDecapSnapshotTable
   global vniEncapCounterTable, vniEncapSnapshotTable

   vtepHwStatusDir = LazyMount.mount( entityManager,
         'vxlan/vtepHwStatus', 'Vxlan::VtepHwStatusDir', 'r' )

   etbaStatus = LazyMount.mount(
      entityManager, "bridging/etba/status", "Bridging::Etba::Status", "r" )

   smir = SmashLazyMount.mountInfo( 'reader' )

   mountPath = tableMountPath( entityManager, FeatureId.VtepDecap, AllFapsId, False )
   vtepDecapCounterTable = SmashLazyMount.mount( entityManager, mountPath,
         "FlexCounters::CounterTable", smir )

   mountPath = tableMountPath( entityManager, FeatureId.VtepDecap, AllFapsId, True )
   vtepDecapSnapshotTable = SmashLazyMount.mount( entityManager, mountPath,
         "FlexCounters::CounterTable", smir )

   mountPath = tableMountPath( entityManager, FeatureId.VtepEncap, AllFapsId, False )
   vtepEncapCounterTable = SmashLazyMount.mount( entityManager, mountPath,
         "FlexCounters::CounterTable", smir )

   mountPath = tableMountPath( entityManager, FeatureId.VtepEncap, AllFapsId, True )
   vtepEncapSnapshotTable = SmashLazyMount.mount( entityManager, mountPath,
         "FlexCounters::CounterTable", smir )

   vtepCounterStatus = LazyMount.mount( entityManager,
         'flexCounter/etba/vtepFlexCounterStatus',
         "Ale::FlexCounter::VtepFlexCounterStatus", 'r' )

   mountPath = tableMountPath( entityManager, FeatureId.VniDecap, AllFapsId, False )
   vniDecapCounterTable = SmashLazyMount.mount( entityManager, mountPath,
         "FlexCounters::CounterTable", smir )

   mountPath = tableMountPath( entityManager, FeatureId.VniDecap, AllFapsId, True )
   vniDecapSnapshotTable = SmashLazyMount.mount( entityManager, mountPath,
         "FlexCounters::CounterTable", smir )

   mountPath = tableMountPath( entityManager, FeatureId.VniEncap, AllFapsId, False )
   vniEncapCounterTable = SmashLazyMount.mount( entityManager, mountPath,
         "FlexCounters::CounterTable", smir )

   mountPath = tableMountPath( entityManager, FeatureId.VniEncap, AllFapsId, True )
   vniEncapSnapshotTable = SmashLazyMount.mount( entityManager, mountPath,
         "FlexCounters::CounterTable", smir )

   vniCounterStatus = LazyMount.mount( entityManager,
         'flexCounter/etba/vniFlexCounterStatus',
         "Ale::FlexCounter::VniFlexCounterStatus", 'r' )

   vtiStatusDir = LazyMount.mount( entityManager,
         "interface/status/eth/vxlan",
         "Vxlan::VtiStatusDir", "r" )

   # Register hook for platform-specific counter feature supported.
   counterFeatureSupportedHook.addExtension( etbaCounterFeatureSupported )
   # Register hook for platform-specific counter feature setting.
   counterFeatureSettingHook.addExtension( etbaFeatureStatusDir )

   # Register hook for showing VTEP decap/encap counters.
   showVtepCountersHook.addExtension( showArfaVtepCountersHook )

   # Register hook for showing VNI decap/encap counters.
   showVniCountersHook.addExtension( showArfaVniCountersHook )
