#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCliModes
import CliCommand
import CliMatcher
from CliPlugin import (
   ArpIpCli,
   ArpIp6Cli,
   VrfCli
)
from CliPlugin.EthIntfCli import EthPhyIntf
from CliPlugin.IntfCli import Intf
from CliPlugin.Ip6AddrMatcher import Ip6AddrMatcher
from CliPlugin.IpAddrMatcher import IpAddrMatcher
from CliPlugin.SubIntfCli import subMatcher
from CliPlugin.VirtualIntfRule import IntfMatcher
from CliToken.Clear import clearKwNode
from CliToken.Ip import ipMatcherForRefresh, ipMatcherForClear
from CliToken.Ipv6 import ipv6MatcherForRefresh, ipv6MatcherForClear
from CliToken.Refresh import refreshMatcherForExec

matcherArpClear = CliMatcher.KeywordMatcher(
   'arp', helpdesc='ARP entries to clear' )
matcherArpCacheClear = CliMatcher.KeywordMatcher(
   'arp-cache', helpdesc='Clear dynamic entries from the ARP table' )
matcherArpCacheRefresh = CliMatcher.KeywordMatcher(
   'arp-cache', helpdesc='Refresh dynamic entries in the ARP table' )
matcherNeighborsRefresh = CliMatcher.KeywordMatcher(
   'neighbors', helpdesc='Neighbor entries to refresh' )
matcherNeighborsClear = CliMatcher.KeywordMatcher(
   'neighbors', helpdesc='Neighbor entries to clear' )
vrfExprFactoryClearArp = VrfCli.VrfExprFactory(
   helpdesc='Clear dynamic ARP entries in a VRF' )
vrfExprFactoryClearNbr = VrfCli.VrfExprFactory(
   helpdesc='Clear dynamic Neighbor entries in a VRF' )
vrfExprFactoryRefreshArp = VrfCli.VrfExprFactory(
   helpdesc='Refresh dynamic ARP entries in a VRF' )
vrfExprFactoryRefreshNbr = VrfCli.VrfExprFactory(
   helpdesc='Refresh dynamic Neighbor entries in a VRF' )

#--------------------------------------------------------------------------------
# clear arp-cache [ VRF ]
#--------------------------------------------------------------------------------
class ClearArpCacheCmd( CliCommand.CliCommandClass ):
   syntax = 'clear arp-cache [ VRF ]'
   data = {
      'clear': clearKwNode,
      'arp-cache': matcherArpCacheClear,
      'VRF': vrfExprFactoryClearArp,
   }
   handler = ArpIpCli.clearArpCache

BasicCliModes.EnableMode.addCommandClass( ClearArpCacheCmd )

#--------------------------------------------------------------------------------
# clear arp-cache [ VRF ] interface INTF
#--------------------------------------------------------------------------------
class ClearArpCacheInterfaceIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'clear arp-cache [ VRF ] interface INTF'
   data = {
      'clear': clearKwNode,
      'arp-cache': matcherArpCacheClear,
      'VRF': vrfExprFactoryClearArp,
      'interface': 'Clear dynamic entries from the ARP table by interface',
      'INTF': Intf.matcher,
   }
   handler = ArpIpCli.clearArpInterface

BasicCliModes.EnableMode.addCommandClass( ClearArpCacheInterfaceIntfCmd )

#--------------------------------------------------------------------------------
# clear arp [ VRF ] IPADDR
#--------------------------------------------------------------------------------
class ClearArpCmd( CliCommand.CliCommandClass ):
   syntax = 'clear arp [ VRF ] IPADDR'
   data = {
      'clear': clearKwNode,
      'arp': matcherArpClear,
      'VRF': vrfExprFactoryClearArp,
      'IPADDR': IpAddrMatcher( helpdesc='Address to be cleared from ARP table' ),
   }
   handler = ArpIpCli.clearIpArp

BasicCliModes.EnableMode.addCommandClass( ClearArpCmd )

#--------------------------------------------------------------------------------
# clear arp counters hardware ingress [ interface INTF ]
#--------------------------------------------------------------------------------
subIntfMatcher = IntfMatcher()
subIntfMatcher |= EthPhyIntf.ethMatcher
subIntfMatcher |= subMatcher

class ClearArpCountersHardwareCmd( CliCommand.CliCommandClass ):
   syntax = 'clear arp counters hardware ingress [ interface INTF ]'
   data = {
      'clear': clearKwNode,
      'arp': matcherArpClear,
      'counters': CliCommand.guardedKeyword( 'counters',
                                             helpdesc='Clear ARP counters',
                                             guard=ArpIpCli.arpCountersSupported ),
      'hardware': 'ARP hardware counters',
      'ingress': 'Ingress direction',
      'interface': 'Interface to clear ARP counters for',
      'INTF': subIntfMatcher,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'cmd' ] = ' '.join( str( arg[ 1 ] ) for arg in argsList )

   handler = ArpIpCli.clearArpCountersHardware

BasicCliModes.EnableMode.addCommandClass( ClearArpCountersHardwareCmd )

#--------------------------------------------------------------------------------
# clear ip arp [ VRF ] IPADDR
#--------------------------------------------------------------------------------
class ClearIpArpCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ip arp [ VRF ] IPADDR'
   data = {
      'clear': clearKwNode,
      'ip': ipMatcherForClear,
      'arp': CliCommand.Node( matcher=matcherArpClear,
                              deprecatedByCmd='clear arp' ),
      'VRF': vrfExprFactoryClearArp,
      'IPADDR': IpAddrMatcher( helpdesc='Address to be cleared from ARP table' ),
   }
   handler = ArpIpCli.clearIpArp

BasicCliModes.EnableMode.addCommandClass( ClearIpArpCmd )

#--------------------------------------------------------------------------------
# clear ipv6 neighbors [ VRF ]
#--------------------------------------------------------------------------------
class ClearIpv6NeighborsCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ipv6 neighbors [ VRF ]'
   data = {
      'clear': clearKwNode,
      'ipv6': ipv6MatcherForClear,
      'neighbors': matcherNeighborsClear,
      'VRF': vrfExprFactoryClearNbr,
   }
   handler = ArpIp6Cli.clearAllIpv6Neighbors

BasicCliModes.EnableMode.addCommandClass( ClearIpv6NeighborsCmd )

#--------------------------------------------------------------------------------
# clear ipv6 neighbors [ VRF ] INTF [ INTF ]
#--------------------------------------------------------------------------------
class ClearIpv6NeighborsIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ipv6 neighbors [ VRF ] INTF [ IPV6ADDR ]'
   data = {
      'clear': clearKwNode,
      'ipv6': ipv6MatcherForClear,
      'neighbors': matcherNeighborsClear,
      'VRF': vrfExprFactoryClearNbr,
      'INTF': Intf.matcher,
      'IPV6ADDR': Ip6AddrMatcher( helpdesc='IPv6 address of neighbor' ),
   }
   handler = ArpIp6Cli.clearIntfIpv6Neighbors

BasicCliModes.EnableMode.addCommandClass( ClearIpv6NeighborsIntfCmd )

#--------------------------------------------------------------------------------
# refresh arp-cache [ VRF ]
#--------------------------------------------------------------------------------
class RefreshArpCacheCmd( CliCommand.CliCommandClass ):
   syntax = 'refresh arp-cache [ VRF ]'
   data = {
      'refresh': refreshMatcherForExec,
      'arp-cache': matcherArpCacheRefresh,
      'VRF': vrfExprFactoryRefreshArp,
   }
   handler = ArpIpCli.refreshArpByVrf

BasicCliModes.EnableMode.addCommandClass( RefreshArpCacheCmd )

#--------------------------------------------------------------------------------
# refresh arp-cache [ VRF ] interface INTF
#--------------------------------------------------------------------------------
class RefreshArpCacheInterfaceIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'refresh arp-cache [ VRF ] interface INTF'
   data = {
      'refresh': refreshMatcherForExec,
      'arp-cache': matcherArpCacheRefresh,
      'VRF': vrfExprFactoryRefreshArp,
      'interface': 'Refresh dynamic entries from the ARP table by interface',
      'INTF': Intf.matcher,
   }
   handler = ArpIpCli.refreshArpByIntf

BasicCliModes.EnableMode.addCommandClass( RefreshArpCacheInterfaceIntfCmd )

#--------------------------------------------------------------------------------
# refresh ip arp [ VRF ] IPADDR
#--------------------------------------------------------------------------------
class RefreshIpArpCmd( CliCommand.CliCommandClass ):
   syntax = 'refresh ip arp [ VRF ] IPADDR'
   data = {
      'refresh': refreshMatcherForExec,
      'ip': ipMatcherForRefresh,
      'arp': 'Refresh dynamic ARP entry by IP',
      'VRF': vrfExprFactoryRefreshArp,
      'IPADDR': IpAddrMatcher( helpdesc='Address to be refreshed in ARP table' ),
   }
   handler = ArpIpCli.refreshIpArp

BasicCliModes.EnableMode.addCommandClass( RefreshIpArpCmd )

#--------------------------------------------------------------------------------
# refresh ipv6 neighbors [ VRF ]
#--------------------------------------------------------------------------------
class RefreshIpv6NeighborsCmd( CliCommand.CliCommandClass ):
   syntax = 'refresh ipv6 neighbors [ VRF ]'
   data = {
      'refresh': refreshMatcherForExec,
      'ipv6': ipv6MatcherForRefresh,
      'neighbors': matcherNeighborsRefresh,
      'VRF': vrfExprFactoryRefreshNbr,
   }
   handler = ArpIp6Cli.refreshIpv6NeighborsByVrf

BasicCliModes.EnableMode.addCommandClass( RefreshIpv6NeighborsCmd )

#--------------------------------------------------------------------------------
# refresh ipv6 neighbors [ VRF ] INTF [ IPV6ADDR ]
#--------------------------------------------------------------------------------
class RefreshIpv6NeighborsIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'refresh ipv6 neighbors [ VRF ] INTF [ IPV6ADDR ]'
   data = {
      'refresh': refreshMatcherForExec,
      'ipv6': ipv6MatcherForRefresh,
      'neighbors': matcherNeighborsRefresh,
      'VRF': vrfExprFactoryRefreshNbr,
      'INTF': Intf.matcher,
      'IPV6ADDR': Ip6AddrMatcher( helpdesc='IPv6 address of neighbor' ),
   }
   handler = ArpIp6Cli.refreshIpv6NeighborsByIntfOrAddr

BasicCliModes.EnableMode.addCommandClass( RefreshIpv6NeighborsIntfCmd )
