#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCliModes
import CliCommand
import CliMatcher
from Intf import (
   IntfRange
)
from CliPlugin import (
   ArpIp6Cli,
   ArpIpCli
)
from CliPlugin.VrfCli import VrfExprFactory
from CliPlugin.Ip6AddrMatcher import Ip6AddrMatcher
from CliPlugin.IpAddrMatcher import IpAddrMatcher
from CliPlugin.MacAddr import MacAddrMatcher
from CliToken.Ipv6 import ipv6MatcherForConfig

matcherArp = CliMatcher.KeywordMatcher( 'arp',
   helpdesc='ARP config commands' )
matcherRefreshDelay = CliMatcher.KeywordMatcher( 'refresh-delay',
   helpdesc='Time to wait before refresh' )

#--------------------------------------------------------------------------------
# [ no | default ] arp aging timeout default SECONDS
#--------------------------------------------------------------------------------
class ArpAgingTimeoutDefaultCmd( CliCommand.CliCommandClass ):
   syntax = 'arp aging timeout default SECONDS'
   noOrDefaultSyntax = 'arp aging timeout default ...'
   data = {
      'arp': matcherArp,
      'aging': 'Set timeout for dynamic ARP entries',
      'timeout': 'Set timeout for dynamic ARP entries',
      'default': 'Default value',
      'SECONDS': CliMatcher.IntegerMatcher( 60, 65535, helpdesc='Seconds' ),
   }
   handler = ArpIpCli.setGlobalArpTimeOut
   noOrDefaultHandler = ArpIpCli.noSetGlobalArpTimeOut

BasicCliModes.GlobalConfigMode.addCommandClass( ArpAgingTimeoutDefaultCmd )

#--------------------------------------------------------------------------------
# [ no | default ] arp [ VRF ] IPADDR MACADDR arpa
#--------------------------------------------------------------------------------
class ArpArpaCmd( CliCommand.CliCommandClass ):
   syntax = 'arp [ VRF ] IPADDR MACADDR arpa'
   noOrDefaultSyntax = 'arp [ VRF ] IPADDR ...'
   data = {
      'arp': matcherArp,
      'VRF': VrfExprFactory( helpdesc='Configure an ARP entry for a VRF' ),
      'IPADDR': IpAddrMatcher( helpdesc="ARP entry's IPv4 address" ),
      'MACADDR': MacAddrMatcher( helpdesc='ARP entry MAC address' ),
      'arpa': 'ARPA-type ARP entry',
   }
   handler = ArpIpCli.addStaticArp
   noOrDefaultHandler = ArpIpCli.delStaticArp

BasicCliModes.GlobalConfigMode.addCommandClass( ArpArpaCmd )

#--------------------------------------------------------------------------------
# [ no | default ] arp cache persistent
#--------------------------------------------------------------------------------
class ArpCachePersistentCmd( CliCommand.CliCommandClass ):
   syntax = 'arp cache persistent'
   noOrDefaultSyntax = 'arp cache persistent ...'
   data = {
      'arp': matcherArp,
      'cache': 'Manage the ARP cache',
      'persistent': 'Restore the ARP cache after reboot',
   }
   handler = ArpIpCli.enableArpPersistent
   noOrDefaultHandler = ArpIpCli.disableArpPersistent

BasicCliModes.GlobalConfigMode.addCommandClass( ArpCachePersistentCmd )

#--------------------------------------------------------------------------------
# [ no | default ] arp monitor mac-address
#--------------------------------------------------------------------------------
class ArpMonitorMacAddressCmd( CliCommand.CliCommandClass ):
   syntax = 'arp monitor mac-address'
   noOrDefaultSyntax = 'arp monitor mac-address ...'
   data = {
      'arp': matcherArp,
      'monitor': 'Set ARP monitor events',
      'mac-address': 'Monitor ARPs MAC address',
   }

   handler = ArpIpCli.enableMonitorMac
   noOrDefaultHandler = ArpIpCli.disableMonitorMac

BasicCliModes.GlobalConfigMode.addCommandClass( ArpMonitorMacAddressCmd )

#--------------------------------------------------------------------------------
# [ no | default ] arp persistent [ refresh-delay SECONDS ]
#--------------------------------------------------------------------------------
class ArpPersistentCmd( CliCommand.CliCommandClass ):
   syntax = 'arp persistent [ refresh-delay SECONDS ]'
   noOrDefaultSyntax = 'arp persistent ...'
   data = {
      'arp': matcherArp,
      'persistent': 'Restore the ARP cache after reboot',
      'refresh-delay': matcherRefreshDelay,
      'SECONDS': CliMatcher.IntegerMatcher(
         600, 3600, helpdesc='Refresh delay time in seconds (default 600)' ),
   }
   handler = ArpIpCli.enableIpv4Persistent
   noOrDefaultHandler = ArpIpCli.disableIpv4Persistent

BasicCliModes.GlobalConfigMode.addCommandClass( ArpPersistentCmd )

#--------------------------------------------------------------------------------
# [ no | default ] arp proxy max-delay MAXDELAY
#--------------------------------------------------------------------------------
class ArpProxyMaxDelayCmd( CliCommand.CliCommandClass ):
   syntax = 'arp proxy max-delay MILLISECONDS'
   noOrDefaultSyntax = 'arp proxy max-delay ...'
   data = {
      'arp': matcherArp,
      'proxy': 'Proxy ARP',
      'max-delay': 'Max delay in milliseconds before replying proxy ARPs',
      'MILLISECONDS': CliMatcher.IntegerMatcher(
         0, 1000, helpdesc='Max delay time (default 800ms)' ),
   }
   handler = ArpIpCli.setProxyMaxDelay
   noOrDefaultHandler = ArpIpCli.unsetProxyMaxDelay

BasicCliModes.GlobalConfigMode.addCommandClass( ArpProxyMaxDelayCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 neighbor cache persistent
#--------------------------------------------------------------------------------
class Ipv6NeighborCachePersistentCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 neighbor cache persistent'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6': ipv6MatcherForConfig,
      'neighbor': 'Manage static IPv6 neighbors',
      'cache': 'Manage the IPv6 neighbor cache',
      'persistent': 'Restore the IPv6 neighbor cache after reboot',
   }
   handler = ArpIp6Cli.enableNCPersistent
   noOrDefaultHandler = ArpIp6Cli.disableNCPersistent

BasicCliModes.GlobalConfigMode.addCommandClass( Ipv6NeighborCachePersistentCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 neighbor [ VRF ] IP6ADDR INTFS MACADDR
#--------------------------------------------------------------------------------
class Ipv6NeighborIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 neighbor [ VRF ] IP6ADDR INTFS MACADDR'
   noOrDefaultSyntax = 'ipv6 neighbor [ VRF ] IP6ADDR INTFS ...'
   data = {
      'ipv6': ipv6MatcherForConfig,
      'neighbor': 'Manage static IPv6 neighbors',
      'VRF': VrfExprFactory( helpdesc='Configure a neighbor entry for a VRF' ),
      'IP6ADDR': Ip6AddrMatcher( helpdesc='IPv6 address of neighbor' ),
      'INTFS': IntfRange.IntfRangeMatcher( helpdesc='Interface type',
         explicitIntfTypes=IntfRange.intfTypesWithArpSupport ),
      'MACADDR': MacAddrMatcher( helpdesc='48-bit hardware address of neighbor' ),
   }
   handler = ArpIp6Cli.doipv6Neighbor
   noOrDefaultHandler = ArpIp6Cli.noipv6Neighbor

BasicCliModes.GlobalConfigMode.addCommandClass( Ipv6NeighborIntfCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 neighbor persistent [ refresh-delay SECONDS ]
#--------------------------------------------------------------------------------
class Ipv6NeighborPersistentCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 neighbor persistent [ refresh-delay SECONDS ]'
   noOrDefaultSyntax = 'ipv6 neighbor persistent ...'
   data = {
      'ipv6': ipv6MatcherForConfig,
      'neighbor': 'Manage static IPv6 neighbors',
      'persistent': 'Restore the IPv6 neighbor cache after reboot',
      'refresh-delay': matcherRefreshDelay,
      'SECONDS': CliMatcher.IntegerMatcher(
         600, 3600, helpdesc='Refresh delay time in seconds (default 600)' ),
   }
   handler = ArpIp6Cli.enableIpv6Persistency
   noOrDefaultHandler = ArpIp6Cli.disableIpv6Persistency

BasicCliModes.GlobalConfigMode.addCommandClass( Ipv6NeighborPersistentCmd )
