#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import LazyMount
from CliPlugin.ArpInputModel import ArpInputEntry
from CliPlugin.ArpInputModel import ArpInputEntryList
from CliPlugin.ArpInputModel import ArpInputEntriesModel
from CliPlugin import VrfCli
from CliPlugin.VrfCli import getAllVrfNames, vrfExists, ALL_VRF_NAME
from ArpLibTypes import (
   permanent,
   arpSourceCli,
   arpSourceNeighborResolution,
   arpSourceEosSdk,
   arpSourceDfwService,
   arpSourceDmf,
   arpSourceEvpnVxlan,
   arpSourceVxlan,
   arpSourceArpSuppression,
   arpSourceProxy,
   arpSourceFhrpArfa,
   arpSourceFhrp,
   arpSourceNat,
   )

#-------------------------------------------------------------------------------
# Aliases for some useful token rules.
#-------------------------------------------------------------------------------
arpInputConfigDir = None
arpInputStatusDir = None

class ArpInputSources:
   def __init__( self ):
      self._commandToken = []
      self._prettyName = []
      self._sourceDirName = []
      self._sourceName = []
      self._mapping = {}
      self._sourceNameMapping = {}

   def entryIs( self, commandToken, prettyName, sourceDirName, sourceName ):
      '''set entry for a source. '''
      if commandToken:
         self._commandToken.append( commandToken )
         self._mapping.update( { commandToken : [ prettyName, sourceDirName ] } )
      self._prettyName.append( prettyName )
      self._sourceDirName.append( sourceDirName )
      self._sourceName.append( sourceName )
      self._sourceNameMapping.update( { sourceName : prettyName } )

   def commandToken( self ):
      '''returns command token of all sources '''
      return self._commandToken

   def prettyName( self, token ):
      '''returns display name associated with command tokens'''
      info = self._mapping.get( token )
      return info[ 0 ]

   def sourceDirName( self, token ):
      '''returns display name associated with command tokens'''
      info = self._mapping.get( token )
      return info[ 1 ]

   def prettyNameFromSource( self, name ):
      '''returns display name associated with command tokens'''
      return self._sourceNameMapping.get( name )

# Populate lookup tables
arpInputSources = ArpInputSources()
arpInputSources.entryIs( 'cli', 'CLI', 'cli', arpSourceCli )
arpInputSources.entryIs( 'neighbor-resolution', 'NeighborResolution',
                         'neighborResolution', arpSourceNeighborResolution )
arpInputSources.entryIs( 'eos-sdk', 'EosSdk', 'eosSdk', arpSourceEosSdk )
arpInputSources.entryIs( 'dfw-service', 'DfwService',
                         'dfwService', arpSourceDfwService )
arpInputSources.entryIs( 'dmf', 'Dmf', 'dmf', arpSourceDmf )
arpInputSources.entryIs( 'evpn-vxlan', 'EvpnVxlan', 'evpnVxlan', arpSourceEvpnVxlan )
arpInputSources.entryIs( 'vxlan', 'Vxlan', 'vxlan', arpSourceVxlan )
arpInputSources.entryIs( 'arp-suppression', 'ArpSuppression',
                         'arpSuppression', arpSourceArpSuppression )
arpInputSources.entryIs( 'proxy', 'Proxy', 'proxy', arpSourceProxy )
arpInputSources.entryIs( 'fhrp', 'Fhrp', 'fhrp', arpSourceFhrp )
arpInputSources.entryIs( 'fhrp-arfa', 'FhrpArfa', 'fhrpArfa', arpSourceFhrpArfa )
arpInputSources.entryIs( 'nat', 'Nat', 'nat', arpSourceNat )

# maps entry type to refresh time and gen id.
def entryTypeToInfo( entryInfo ):
   if entryInfo.type == permanent:
      return "static", "NA", "NA"
   else:
      return "dynamic", str( int( entryInfo.refreshTime ) ), str( entryInfo.genId )

def getArpInputConfigEntries ( version, source, model, vrfList ):
   if vrfList is None or source is None:
      return

   # function to get all the entry information belonging to a
   # particular source in a VRF
   def getEntryList( vrfName, version, dirName, prettyName ):
      if dirName not in arpInputConfigDir:
         return None
      if vrfName in arpInputConfigDir[ dirName ].vrf:
         entriesPerVrf = arpInputConfigDir[ dirName ].vrf[ vrfName ]
         versionEntries = ArpInputEntryList()
         ethAddr = Tac.newInstance( "Arnet::EthAddr" )
         if version == 'ipv4':
            for i in entriesPerVrf.ipv4:
               info = entriesPerVrf.ipv4[ i ]
               entryType, refreshTime, genId = entryTypeToInfo( info )
               ethAddr.stringValue = info.ethAddr
               entry = ArpInputEntry( ipAddr=info.addr.stringValue,
                                      intfId=info.intfId,
                                      ethAddr=ethAddr.displayString,
                                      entryType=entryType, refreshTime=refreshTime,
                                      genId=genId, source=prettyName )
               versionEntries.entrys.append( entry )
         else:
            for i in entriesPerVrf.ipv6:
               info = entriesPerVrf.ipv6[ i ]
               entryType, refreshTime, genId = entryTypeToInfo( info )
               ethAddr.stringValue = info.ethAddr
               entry = ArpInputEntry( ipAddr=info.addr.stringValue,
                                      intfId=info.intfId,
                                      ethAddr=ethAddr.displayString,
                                      entryType=entryType, refreshTime=refreshTime,
                                      genId=genId, source=prettyName )
               versionEntries.entrys.append( entry )
         return versionEntries
      return None

   model.entryMap = { vrf : ArpInputEntryList() for vrf in vrfList }
   if source == 'all':
      # get information from every ARP source
      for tok in arpInputSources.commandToken():
         dirName = arpInputSources.sourceDirName( tok )
         prettyName = arpInputSources.prettyName( tok )
         for vrfName in model.entryMap:
            entryList = getEntryList( vrfName, version, dirName, prettyName )
            if entryList is not None and entryList.entrys:
               model.entryMap[ vrfName ].entrys.extend( entryList.entrys )
   else:
      for vrfName in model.entryMap:
         model.entryMap[ vrfName ] = ArpInputEntryList()
         dirName = arpInputSources.sourceDirName( source )
         prettyName = arpInputSources.prettyName( source )
         entryList = getEntryList( vrfName, version, dirName, prettyName )
         if entryList is not None and entryList.entrys:
            model.entryMap[ vrfName ] = entryList
   return

def showArpInputConfigEntries( mode, args ):
   if 'ipv6' in args:
      filterAf = 'ipv6'
   else:
      filterAf = 'ipv4'

   vrfName = args.get( 'VRF' )
   assert vrfName != ''
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   source = args.get( 'SOURCE' )
   if not source:
      source = "all"

   if vrfName == ALL_VRF_NAME:
      # Note that this function returns a chain, which functions as a generator under
      # the hood. The return value must be converted to a list if it is to be
      # iterated over more than once.
      vrfList = getAllVrfNames( mode )
   elif vrfExists( vrfName ):
      vrfList = [ vrfName ]
   else:
      mode.addError( f"Vrf {vrfName} does not exist" )
      return None

   model = ArpInputEntriesModel()
   getArpInputConfigEntries( filterAf, source, model, vrfList )
   return model

def getArpInputStatusEntries ( version, intf, model, vrfList ):
   if vrfList is None:
      return

   # function to get all the entry information belonging to a
   # VRF and interface
   def getEntryList( vrfName, version, intf ):
      if vrfName in arpInputStatusDir.vrf:
         entriesPerVrf = arpInputStatusDir.vrf[ vrfName ]
         versionEntries = ArpInputEntryList()
         ethAddr = Tac.newInstance( "Arnet::EthAddr" )
         if version == 'ipv4':
            for intfId in entriesPerVrf.ipv4:
               # filter based on interface if given
               if intf is None or intfId == intf.name:
                  for i in entriesPerVrf.ipv4[ intfId ].entry:
                     info = entriesPerVrf.ipv4[ intfId ].entry[ i ]
                     entryType, refreshTime, genId = entryTypeToInfo( info )
                     prettyName = arpInputSources.prettyNameFromSource(
                                                 info.source )
                     ethAddr.stringValue = info.ethAddr
                     entry = ArpInputEntry( ipAddr=info.addr.stringValue,
                                            intfId=info.intfId,
                                            ethAddr=ethAddr.displayString,
                                            entryType=entryType,
                                            refreshTime=refreshTime,
                                            genId=genId, source=prettyName )
                     versionEntries.entrys.append( entry )
         else:
            for intfId in entriesPerVrf.ipv6:
               # filter based on interface if given
               if intf is None or intfId == intf.name:
                  for i in entriesPerVrf.ipv6[ intfId ].entry:
                     info = entriesPerVrf.ipv6[ intfId ].entry[ i ]
                     entryType, refreshTime, genId = entryTypeToInfo( info )
                     prettyName = arpInputSources.prettyNameFromSource(
                                                 info.source )
                     ethAddr.stringValue = info.ethAddr
                     entry = ArpInputEntry( ipAddr=info.addr.stringValue,
                                            intfId=info.intfId,
                                            ethAddr=ethAddr.displayString,
                                            entryType=entryType,
                                            refreshTime=refreshTime,
                                            genId=genId, source=prettyName )
                     versionEntries.entrys.append( entry )
         return versionEntries
      return None

   model.entryMap = { vrf : ArpInputEntryList() for vrf in vrfList }
   for vrfName in model.entryMap:
      if vrfName in arpInputStatusDir.vrf:
         entryList = getEntryList( vrfName, version, intf )
         if entryList is not None and entryList.entrys:
            model.entryMap[ vrfName ] = entryList
   return

def showArpInputStatusEntries( mode, args ):
   if 'ipv6' in args:
      filterAf = 'ipv6'
   else:
      filterAf = 'ipv4'
   intf = args.get( 'INTF' )
   vrfName = args.get( 'VRF' )
   assert vrfName != ''
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )

   if vrfName == ALL_VRF_NAME:
      # Note that this function returns a chain, which functions as a generator under
      # the hood. The return value must be converted to a list if it is to be
      # iterated over more than once.
      vrfList = getAllVrfNames( mode )
   elif vrfExists( vrfName ):
      vrfList = [ vrfName ]
   else:
      mode.addError( f"VRF {vrfName} does not exist" )
      return None

   model = ArpInputEntriesModel()
   getArpInputStatusEntries( filterAf, intf, model, vrfList )
   return model

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global arpInputConfigDir
   global arpInputStatusDir

   arpInputConfigDir = LazyMount.mount( entityManager, "arp/input/config",
                                        "Tac::Dir", "riS" )
   arpInputStatusDir = LazyMount.mount( entityManager, "arp/input/status",
                                        "Arp::InputStatus", "rS" )
