#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliMatcher
from CliPlugin import (
   ArpIpIntfCli
)
from CliPlugin.ArpCommon import (
   arpCacheCapacity,
)
from CliPlugin.ArpIpIntfCli import setMonitorMac
from CliPlugin.ArpIpIntfCli import noMonitorMac
from CliPlugin.ArpIpIntfCli import defaultMonitorMac
from CliPlugin.IraIp6IntfCli import ndMatcher
from CliPlugin.Ip6AddrMatcher import Ip6PrefixMatcher
from CliPlugin.SubIntfCli import SubIntfVlanModelet
from CliPlugin.EthIntfCli import EthIntfModelet
from CliPlugin.SwitchIntfCli import SwitchIntfModelet
from CliPlugin.LagIntfCli import LagIntfConfigModelet
from CliPlugin.VlanIntfCli import VlanIntf
from CliPlugin.VlanIntfCli import VlanIntfModelet
from CliPlugin.ArpCommon import minArpTimeout, maxArpTimeout
from CliToken.Ipv6 import ipv6MatcherForConfigIf

from Toggles.ArpToggleLib import toggleIpv6DadProxyEnabled
from Toggles.ArpToggleLib import toggleForcedPeriodicArpRefreshEnabled
from Toggles.ArpToggleLib import toggleArpCacheCapacityLinkLocalExcludedEnabled

# --------------------------------------------------------------------------------
# common tokens
# --------------------------------------------------------------------------------
matcherArp = CliMatcher.KeywordMatcher(
   'arp', helpdesc='ARP config commands' )
matcherCacheCapacity = CliMatcher.IntegerMatcher(
   arpCacheCapacity.min,
   arpCacheCapacity.max,
   helpdesc='Capacity of dynamic cache entries' )
matcherSeconds = CliMatcher.IntegerMatcher(
   minArpTimeout, maxArpTimeout, helpdesc='Seconds' )
matcherTimeout = CliMatcher.KeywordMatcher(
   'timeout', helpdesc='Set timeout for dynamic ARP entries' )
matcherAging = CliMatcher.KeywordMatcher(
   'aging', helpdesc='Set timeout for dynamic ARP entries' )

def guardNdProxySupported( mode, args ):
   if isinstance( mode.intf, VlanIntf ):
      return ArpIpIntfCli.ndProxySupportedGuard( mode, args )
   else:
      return ArpIpIntfCli.ndProxyIntfSupportedGuard( mode, args )

nodeProxy = CliCommand.guardedKeyword( 'proxy',
   helpdesc='proxy',
   guard=guardNdProxySupported )

# --------------------------------------------------------------------------------
# arp aging timeout SECONDS
# ( no | default ) arp aging timeout
# --------------------------------------------------------------------------------
class ArpAgingTimeoutArptimeoutCmd( CliCommand.CliCommandClass ):
   syntax = 'arp aging timeout SECONDS'
   noOrDefaultSyntax = 'arp aging timeout  ...'
   data = {
      'arp': matcherArp,
      'aging': matcherAging,
      'timeout': "timeout",
      'SECONDS': matcherSeconds,
   }
   handler = ArpIpIntfCli.setArpTimeout
   noOrDefaultHandler = ArpIpIntfCli.noArpTimeout

ArpIpIntfCli.ArpIpIntfConfigModelet.addCommandClass( ArpAgingTimeoutArptimeoutCmd )

# --------------------------------------------------------------------------------
# arp aging refresh always
# ( no | default ) arp aging refresh always
# --------------------------------------------------------------------------------
class ArpAgingRefreshAlwaysCmd( CliCommand.CliCommandClass ):
   syntax = 'arp aging refresh always'
   noOrDefaultSyntax = syntax
   data = {
      'arp': matcherArp,
      'aging': matcherAging,
      'refresh': "Force refresh on ARP timeout",
      'always': "Force refresh on ARP timeout",
   }
   handler = ArpIpIntfCli.setForceArpRequestOnTimeout
   noOrDefaultHandler = ArpIpIntfCli.noForceArpRequestOnTimeout

if toggleForcedPeriodicArpRefreshEnabled():
   ArpIpIntfCli.ArpIpIntfConfigModelet.addCommandClass( ArpAgingRefreshAlwaysCmd )

# --------------------------------------------------------------------------------
# arp timeout SECONDS
# ( no | default ) arp timeout
# --------------------------------------------------------------------------------
class ArpTimeoutArptimeoutCmd( CliCommand.CliCommandClass ):
   syntax = 'arp timeout SECONDS'
   noOrDefaultSyntax = 'arp timeout ...'
   data = {
      'arp': matcherArp,
      'timeout': CliCommand.Node( matcher=matcherTimeout,
                                  deprecatedByCmd='arp aging timeout' ),
      'SECONDS': matcherSeconds,
   }
   handler = ArpIpIntfCli.setArpTimeout
   noOrDefaultHandler = ArpIpIntfCli.noArpTimeout
   hidden = True

ArpIpIntfCli.ArpIpIntfConfigModelet.addCommandClass( ArpTimeoutArptimeoutCmd )

# --------------------------------------------------------------------------------
# arp cache dynamic capacity NUMBER
# ( no | default ) arp cache dynamic capacity
# --------------------------------------------------------------------------------
class ArpCacheCapacityCmd( CliCommand.CliCommandClass ):
   syntax = 'arp cache dynamic capacity CAPACITY'
   noOrDefaultSyntax = 'arp cache dynamic capacity  ...'
   data = {
      'arp': matcherArp,
      'cache': 'Manage the ARP cache',
      'dynamic': 'Cache parameters for dynamic neighbor resolutions',
      'capacity': 'Capacity of the ARP cache',
      'CAPACITY': matcherCacheCapacity,
   }
   handler = ArpIpIntfCli.setArpCacheCapacity
   noOrDefaultHandler = ArpIpIntfCli.noArpCacheCapacity

ArpIpIntfCli.ArpIpIntfConfigModelet.addCommandClass( ArpCacheCapacityCmd )

# --------------------------------------------------------------------------------
# Expression: ipv6 nd cache ...
# --------------------------------------------------------------------------------
class Ipv6NdCacheExpression( CliCommand.CliExpression ):
   expression = 'ipv6 nd cache'
   data = {
      'ipv6': ipv6MatcherForConfigIf,
      'nd': ndMatcher,
      'cache': 'Manage the IPv6 neighbor cache',
   }

# --------------------------------------------------------------------------------
# ipv6 nd cache expire SECONDS
# ( no | default ) ipv6 nd cache expire
# --------------------------------------------------------------------------------
class Ipv6NdCacheExpireNdtimeoutCmd( CliCommand.CliCommandClass ):
   syntax = 'IPV6_ND_CACHE expire SECONDS'
   noOrDefaultSyntax = 'IPV6_ND_CACHE expire ...'
   data = {
      'IPV6_ND_CACHE': Ipv6NdCacheExpression,
      'expire': 'Expiry time for ND entries',
      'SECONDS': matcherSeconds,
   }
   handler = ArpIpIntfCli.setNeighborCacheExpire
   noOrDefaultHandler = ArpIpIntfCli.noNeighborCacheExpire

ArpIpIntfCli.ArpIpIntfConfigModelet.addCommandClass( Ipv6NdCacheExpireNdtimeoutCmd )

# --------------------------------------------------------------------------------
# ipv6 nd cache refresh always
# ( no | default ) ipv6 nd cache refresh always
# --------------------------------------------------------------------------------
class Ipv6NdCacheRefreshAlwaysCmd( CliCommand.CliCommandClass ):
   syntax = 'IPV6_ND_CACHE refresh always'
   noOrDefaultSyntax = syntax
   data = {
      'IPV6_ND_CACHE': Ipv6NdCacheExpression,
      'refresh': "Force refresh on cache expiry",
      'always': "Force refresh on cache expiry",
   }
   handler = ArpIpIntfCli.setForceNsOnTimeout
   noOrDefaultHandler = ArpIpIntfCli.noForceNsOnTimeout

if toggleForcedPeriodicArpRefreshEnabled():
   ArpIpIntfCli.ArpIpIntfConfigModelet.addCommandClass( Ipv6NdCacheRefreshAlwaysCmd )

# --------------------------------------------------------------------------------
# ipv6 nd cache dynamic capacity NUMBER [ link-local excluded ]
# ( no | default ) ipv6 nd cache dynamic capacity
# --------------------------------------------------------------------------------
class Ipv6NdCacheCapacityCmd( CliCommand.CliCommandClass ):
   if toggleArpCacheCapacityLinkLocalExcludedEnabled():
      syntax = 'IPV6_ND_CACHE dynamic capacity CAPACITY [ link-local excluded ] '
   else:
      syntax = 'IPV6_ND_CACHE dynamic capacity CAPACITY'
   noOrDefaultSyntax = 'IPV6_ND_CACHE dynamic capacity ...'
   data = {
      'IPV6_ND_CACHE': Ipv6NdCacheExpression,
      'dynamic': 'Cache parameters for dynamic neighbor resolutions',
      'capacity': 'Capacity of the IPv6 neighbor cache',
      'CAPACITY': matcherCacheCapacity,
      'link-local': 'Link local address options',
      'excluded': 'Exclude link local from Cache Capacity'
   }
   handler = ArpIpIntfCli.setNeighborCacheCapacity
   noOrDefaultHandler = ArpIpIntfCli.noNeighborCacheCapacity

ArpIpIntfCli.ArpIpIntfConfigModelet.addCommandClass( Ipv6NdCacheCapacityCmd )

# --------------------------------------------------------------------------------
# [ no | default ] arp monitor mac-address
# --------------------------------------------------------------------------------
class ArpMonitorMacAddressCmd( CliCommand.CliCommandClass ):
   syntax = 'arp monitor mac-address'
   noOrDefaultSyntax = 'arp monitor mac-address'
   data = {
      'arp': matcherArp,
      'monitor': "monitor ARP events",
      'mac-address' : "monitor ARP's MAC address"
   }
   handler = setMonitorMac
   noHandler = noMonitorMac
   defaultHandler = defaultMonitorMac

VlanIntfModelet.addCommandClass( ArpMonitorMacAddressCmd )

# --------------------------------------------------------------------------------
# [ no | default ] ipv6 nd proxy prefix ( connected | IPV6PREFIX )
# --------------------------------------------------------------------------------
class NdProxyCmdForPrefix( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd proxy prefix ( connected | IPV6PREFIX )'
   noOrDefaultSyntax = 'ipv6 nd proxy prefix ( connected | IPV6PREFIX )'
   data = {
      'ipv6': ipv6MatcherForConfigIf,
      'nd': ndMatcher,
      'proxy': nodeProxy,
      'prefix' : "prefix",
      'connected' : "connected",
      'IPV6PREFIX': Ip6PrefixMatcher( helpdesc="ipv6 prefix for proxies" )
   }
   handler = ArpIpIntfCli.setNdProxyForPrefix
   noOrDefaultHandler = ArpIpIntfCli.noNdProxyForPrefix

VlanIntfModelet.addCommandClass( NdProxyCmdForPrefix )
SubIntfVlanModelet.addCommandClass( NdProxyCmdForPrefix )
SwitchIntfModelet.addCommandClass( NdProxyCmdForPrefix )
LagIntfConfigModelet.addCommandClass( NdProxyCmdForPrefix )
EthIntfModelet.addCommandClass( NdProxyCmdForPrefix )

# --------------------------------------------------------------------------------
# [ no | default ] ipv6 nd proxy
# --------------------------------------------------------------------------------
class NdProxyCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'ipv6 nd proxy ...'
   data = {
      'ipv6': ipv6MatcherForConfigIf,
      'nd': ndMatcher,
      'proxy': nodeProxy,
   }
   noOrDefaultHandler = ArpIpIntfCli.noNdProxy

VlanIntfModelet.addCommandClass( NdProxyCmd )
SubIntfVlanModelet.addCommandClass( NdProxyCmd )
SwitchIntfModelet.addCommandClass( NdProxyCmd )
LagIntfConfigModelet.addCommandClass( NdProxyCmd )
EthIntfModelet.addCommandClass( NdProxyCmd )

# --------------------------------------------------------------------------------
# [ no | default ] ipv6 nd dad proxy
# --------------------------------------------------------------------------------
class NdDadProxyCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd dad proxy'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6': ipv6MatcherForConfigIf,
      'nd': ndMatcher,
      'dad': 'Configure parameters for duplicate address detection mechanism',
      'proxy': 'Proxy'
   }
   handler = ArpIpIntfCli.setNdDadProxy
   noOrDefaultHandler = ArpIpIntfCli.noNdDadProxy

if toggleIpv6DadProxyEnabled():
   VlanIntfModelet.addCommandClass( NdDadProxyCmd )
   SubIntfVlanModelet.addCommandClass( NdDadProxyCmd )
   SwitchIntfModelet.addCommandClass( NdDadProxyCmd )
   LagIntfConfigModelet.addCommandClass( NdDadProxyCmd )
   EthIntfModelet.addCommandClass( NdDadProxyCmd )
