# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements non-interface-specific IP6 configuration.
#-------------------------------------------------------------------------------
'''Configuration commands supported for IP'''

import Logging
import UtmpDump
from CliPlugin import ArpCommon
from CliPlugin.ArpCommon import qtraceException
from CliPlugin.ArpCommon import sendArpRefreshReqMsg
from CliPlugin import VrfCli
from CliPlugin.VrfCli import getAllVrfNames, vrfExists, ALL_VRF_NAME
import LazyMount
import ArpLogMsgs
from IpLibConsts import DEFAULT_VRF
import Tac
import Cell
import Arnet
from ArpLibTypes import (
   tacInputConfigKey,
   )

arp = ArpCommon.Arp()
arpConfig = None
ip6Status = None
allVrfStatusLocal = None
routingHardwareStatus = None
noNeighborTableForVrfMsg = "Neighbor table for VRF %s does not exist."
ipv6Af = Tac.Type( 'Arnet::AddressFamily' ).ipv6

def noipv6Neighbor( mode, args ):
   ipv6Addr = args[ 'IP6ADDR' ]
   ipv6IntfNames = args[ 'INTFS' ]
   vrfName = args.get( 'VRF' )
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   assert vrfName and vrfName != ''
   if not vrfExists( vrfName ):
      mode.addError( noNeighborTableForVrfMsg % vrfName )
      return

   key = tacInputConfigKey( str( ipv6Addr ), ipv6IntfNames.strWithLongTag() )
   if key not in arp.arpInputConfigVrf( vrfName ).ipv6:
      mode.addError( "No matching neighbor to delete" )
   else:
      del arp.arpInputConfigVrf( vrfName ).ipv6[ key ]

def doipv6Neighbor( mode, args ):
   ipv6Addr = str( args[ 'IP6ADDR' ] )
   ipv6IntfNames = args[ 'INTFS' ]
   macAddr = args[ 'MACADDR' ]
   vrfName = args.get( 'VRF' )
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   assert vrfName and vrfName != ''
   if not vrfExists( vrfName ):
      mode.addError( noNeighborTableForVrfMsg % vrfName )
      return

   config = arp.arpInputConfigVrf( vrfName )
   source = arp.cliSource()
   for name in ipv6IntfNames:
      configKey = tacInputConfigKey( ipv6Addr, name )
      config.afEntryPermanentIs( configKey, macAddr, source )

def enableNCPersistent( mode, args ):
   arpConfig.ip6RestoreOnReboot = True

def disableNCPersistent( mode, args ):
   arpConfig.ip6RestoreOnReboot = False

def enableIpv6Persistency( mode, args ):
   arpConfig.ip6PersistanceEnabled = True
   arpConfig.refreshDelay = args.get( 'SECONDS', arpConfig.refreshDelayDefault )

def disableIpv6Persistency( mode, args ):
   arpConfig.ip6PersistanceEnabled = False
   arpConfig.refreshDelay = arpConfig.refreshDelayDefault

def runClearIp6NeighborsCmd( cmd, vrfName=DEFAULT_VRF ):
   if not vrfName:
      vrfName = DEFAULT_VRF
   assert vrfName != ''
   try:
      if vrfName != DEFAULT_VRF:
         vrf = allVrfStatusLocal.vrf.get( vrfName )
         if vrf and vrf.state == 'active':
            Arnet.NsLib.runMaybeInNetNs( vrf.networkNamespace,
                                         cmd,
                                         stdout=Tac.CAPTURE,
                                         stderr=Tac.CAPTURE,
                                         asRoot=True )
            return True
         else:
            return False
      else:
         Tac.run( cmd, asRoot=True, stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )
         return True
   except Tac.SystemCommandError as e:
      qtraceException( e )
   return True

def clearAllIpv6Neighbors( mode, args ):
   vrfName = args.get( 'VRF' )
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   uInfo = UtmpDump.getUserInfo()
   cmd =  [ "ip", "-6", "neigh", "flush", "to", "0::/0" ]
   # No state needs to be saved, and we never run Cli remotely, so
   # we'll just do this:
   if vrfName == ALL_VRF_NAME:
      Logging.log( ArpLogMsgs.ETH_IPV6_NEIGHBOR_CLEAR,
            vrfName, uInfo[ 'user' ], uInfo[ 'tty' ], uInfo[ 'ipAddr' ] )
      vrfList = getAllVrfNames( mode )
      for vrf in vrfList:
         vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrf )
         runClearIp6NeighborsCmd( cmd, vrfName )
   else:
      if runClearIp6NeighborsCmd( cmd, vrfName ):
         Logging.log( ArpLogMsgs.ETH_IPV6_NEIGHBOR_CLEAR,
               vrfName, uInfo[ 'user' ], uInfo[ 'tty' ], uInfo[ 'ipAddr' ] )

def clearIntfIpv6Neighbors( mode, args ):
   ipv6IntfName = args[ 'INTF' ]
   vrfName = args.get( 'VRF' )
   ipv6Addr = args.get( 'IPV6ADDR' )
   if not ipv6IntfName.lookup():
      mode.addError( "Interface does not exist" )
      return
   ip6IntfStatus = ip6Status.ipIntfStatus( ipv6IntfName.name_ )
   if not ip6IntfStatus:
      mode.addError( f"{ipv6IntfName.name_} is not a layer 3 interface" )
      return
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   if vrfName != ip6IntfStatus.vrf:
      mode.addError( "Interface vrf does not match" )
      return

   # Build the minimum command to execute
   tacCmd = [ "ip", "-6", "neigh", "flush", "dev", ipv6IntfName.status().deviceName ]

   # if the optional ipaddress is specified append it to the CLI
   if ipv6Addr:
      tacCmd.append( ipv6Addr.stringValue )

   if runClearIp6NeighborsCmd( tacCmd, vrfName ):
      uInfo = UtmpDump.getUserInfo()
      if ipv6Addr:
         Logging.log( ArpLogMsgs.ETH_IPV6_NEIGHBOR_INTERFACE_IP_CLEAR,
               ipv6Addr, vrfName, ipv6IntfName.name_, uInfo[ 'user' ],
               uInfo[ 'tty' ], uInfo[ 'ipAddr' ] )
      else:
         Logging.log( ArpLogMsgs.ETH_IPV6_NEIGHBOR_INTERFACE_CLEAR,
               vrfName, ipv6IntfName.name_, uInfo[ 'user' ], uInfo[ 'tty' ],
               uInfo[ 'ipAddr' ] )

def refreshIpv6NeighborsByVrf( mode, args ):
   vrfName = args.get( 'VRF' )
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   sendArpRefreshReqMsg( mode, ipv6Af, vrfName, None, None )

def refreshIpv6NeighborsByIntfOrAddr( mode, args ):
   ipv6IntfName = args[ 'INTF' ]
   vrfName = args.get( 'VRF' )
   ipv6Addr = args.get( 'IPV6ADDR' )
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   if not ipv6IntfName.lookup():
      mode.addError( "Interface does not exist" )
      return
   sendArpRefreshReqMsg( mode, ipv6Af, vrfName, ipv6IntfName.name,
                         ipv6Addr.stringValue if ipv6Addr else None )

myEntManager = None
#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global arpConfig
   global ip6Status
   global allVrfStatusLocal
   global routingHardwareStatus
   global myEntManager

   myEntManager = entityManager

   arp.plugin( entityManager )
   arpConfig = LazyMount.mount( entityManager, "arp/config", "Arp::Config", "w" )

   ip6Status = LazyMount.mount( entityManager, "ip6/status", "Ip6::Status", "r" )
   allVrfStatusLocal = LazyMount.mount( entityManager,
                                        Cell.path( "ip/vrf/status/local" ),
                                        "Ip::AllVrfStatusLocal", "r" )
   routingHardwareStatus = LazyMount.mount( entityManager,
                                            "routing/hardware/status",
                                            "Routing::Hardware::Status", "r" )
