# Copyright (c) 2005-2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements interface-specific IP configuration.
#-------------------------------------------------------------------------------
'''Commands supported in interface config mode'''

import CliParser
from CliPlugin import IntfCli
import LazyMount, Tac
import ConfigMount
from TypeFuture import TacLazyType

ipStatus = None
arpConfig = None
allIntfStatusDir = None
bridgingHwCapabilities = None

tristateBool = Tac.Type( "Ip::TristateBool" )
CacheCapacity = TacLazyType( 'Arp::ArpCacheCapacity' )

class ArpIpIntf ( IntfCli.IntfDependentBase ):
   def __init__( self, intf, sysdbRoot, createIfMissing=True ):
      self.arpIntfConfigs = arpConfig.arpIntfConfig
      IntfCli.IntfDependentBase.__init__( self, intf, sysdbRoot )

   def setDefault( self ):
      del self.arpIntfConfigs[ self.intf_.name ]

class ArpIpIntfConfigModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.routingSupported()

IntfCli.IntfConfigMode.addModelet( ArpIpIntfConfigModelet )
modelet = ArpIpIntfConfigModelet

def intfConfig( mode ):
   intfName = mode.intf.name
   arpIntfConfig = arpConfig.arpIntfConfig.get( intfName, None )
   if arpIntfConfig is None:
      arpIntfConfig = arpConfig.arpIntfConfig.newMember( intfName )
   return arpIntfConfig

def isIntfConfigDefault( arpIntfConfig ):
   # With introduction of global arp aging timeout, default value for timeout is 0.
   return ( arpIntfConfig.timeoutV4 == 0 and
            arpIntfConfig.timeoutV6 == 0 and
            arpIntfConfig.monitorMac == tristateBool.isInvalid and
            not arpIntfConfig.checkProxyNdEnabled() and
            arpIntfConfig.arpCacheCapacity is None and
            arpIntfConfig.nbrCacheCapacity is None and
            not arpIntfConfig.proxyNdDad and
            not arpIntfConfig.forceArpRequestOnTimeout and
            not arpIntfConfig.forceNsOnTimeout )

def setArpTimeout( mode, args ):
   arpIntfConfig = intfConfig( mode )
   arpIntfConfig.timeoutV4 = args[ 'SECONDS' ]

def noArpTimeout( mode, args ):
   arpIntfConfig = intfConfig( mode )
   # When the user does not explicitly configure a timeout, ie. if the user does
   # "[no|default] arp aging timeout", the timeout in the config is set to zero.
   arpIntfConfig.timeoutV4 = 0
   if isIntfConfigDefault( arpIntfConfig ):
      del arpConfig.arpIntfConfig[ mode.intf.name ]

def setArpCacheCapacity( mode, args ):
   arpIntfConfig = intfConfig( mode )
   arpIntfConfig.arpCacheCapacity = CacheCapacity( args[ 'CAPACITY' ], False )

def noArpCacheCapacity( mode, args ):
   arpIntfConfig = intfConfig( mode )
   arpIntfConfig.arpCacheCapacity = None
   if isIntfConfigDefault( arpIntfConfig ):
      del arpConfig.arpIntfConfig[ mode.intf.name ]

def setMonitorMac( mode, args ):
   arpIntfConfig = intfConfig( mode )
   arpIntfConfig.monitorMac = tristateBool.isTrue
   if isIntfConfigDefault( arpIntfConfig ):
      del arpConfig.arpIntfConfig[ mode.intf.name ]

def noMonitorMac( mode, args ):
   arpIntfConfig = intfConfig( mode )
   # no means stop monitoring
   arpIntfConfig.monitorMac = tristateBool.isFalse
   if isIntfConfigDefault( arpIntfConfig ):
      del arpConfig.arpIntfConfig[ mode.intf.name ]

def defaultMonitorMac( mode, args ):
   arpIntfConfig = intfConfig( mode )
   arpIntfConfig.monitorMac = tristateBool.isInvalid
   if isIntfConfigDefault( arpIntfConfig ):
      del arpConfig.arpIntfConfig[ mode.intf.name ]

def setNeighborCacheExpire( mode, args ):
   intfConfig( mode ).timeoutV6 = args[ 'SECONDS' ]

def noNeighborCacheExpire( mode, args ):
   arpIntfConfig = intfConfig( mode )
   # When the user does not explicitly configure a timeout, ie. if the user does
   # "[no|default] nd cache expire", the timeout in the config is set to zero.
   arpIntfConfig.timeoutV6 = 0
   if isIntfConfigDefault( arpIntfConfig ):
      del arpConfig.arpIntfConfig[ mode.intf.name ]

def setNeighborCacheCapacity( mode, args ):
   arpIntfConfig = intfConfig( mode )
   arpIntfConfig.nbrCacheCapacity = CacheCapacity( args[ 'CAPACITY' ],
                                                   'excluded' in args )

def noNeighborCacheCapacity( mode, args ):
   arpIntfConfig = intfConfig( mode )
   arpIntfConfig.nbrCacheCapacity = None
   if isIntfConfigDefault( arpIntfConfig ):
      del arpConfig.arpIntfConfig[ mode.intf.name ]

def ndProxySupportedGuard( mode, args ):
   return ( None if bridgingHwCapabilities.ndProxySupported
            else CliParser.guardNotThisPlatform )

def ndProxyIntfSupportedGuard( mode, args ):
   return ( None if bridgingHwCapabilities.ndProxyIntfSupported
            else CliParser.guardNotThisPlatform )

def setNdProxyForPrefix( mode, args ):
   arpIntfConfig = intfConfig( mode )
   if 'connected' in args:
      arpIntfConfig.proxyNdPrefixConnected = True
   else:
      arpIntfConfig.proxyNdPrefix[ args[ 'IPV6PREFIX' ] ] = True

def setNdDadProxy( mode, args ):
   arpIntfConfig = intfConfig( mode )
   arpIntfConfig.proxyNdDad = True

def noNdProxyForPrefix( mode, args ):
   arpIntfConfig = intfConfig( mode )
   if 'connected' in args:
      arpIntfConfig.proxyNdPrefixConnected = False
   else:
      del arpIntfConfig.proxyNdPrefix[ args[ 'IPV6PREFIX' ] ]
   if isIntfConfigDefault( arpIntfConfig ):
      del arpConfig.arpIntfConfig[ mode.intf.name ]

def noNdDadProxy( mode, args ):
   arpIntfConfig = intfConfig( mode )
   arpIntfConfig.proxyNdDad = arpIntfConfig.proxyNdDadDefault
   if isIntfConfigDefault( arpIntfConfig ):
      del arpConfig.arpIntfConfig[ mode.intf.name ]

def noNdProxy( mode, args ):
   arpIntfConfig = intfConfig( mode )
   arpIntfConfig.proxyNdPrefixConnected = False
   arpIntfConfig.proxyNdPrefix.clear()
   if isIntfConfigDefault( arpIntfConfig ):
      del arpConfig.arpIntfConfig[ mode.intf.name ]

def setForceArpRequestOnTimeout( mode, args ):
   arpIntfConfig = intfConfig( mode )
   arpIntfConfig.forceArpRequestOnTimeout = True

def noForceArpRequestOnTimeout( mode, args ):
   arpIntfConfig = intfConfig( mode )
   arpIntfConfig.forceArpRequestOnTimeout = False
   if isIntfConfigDefault( arpIntfConfig ):
      del arpConfig.arpIntfConfig[ mode.intf.name ]

def setForceNsOnTimeout( mode, args ):
   arpIntfConfig = intfConfig( mode )
   arpIntfConfig.forceNsOnTimeout = True

def noForceNsOnTimeout( mode, args ):
   arpIntfConfig = intfConfig( mode )
   arpIntfConfig.forceNsOnTimeout = False
   if isIntfConfigDefault( arpIntfConfig ):
      del arpConfig.arpIntfConfig[ mode.intf.name ]

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global ipStatus, arpConfig, allIntfStatusDir, bridgingHwCapabilities
   ipStatus = LazyMount.mount( entityManager, "ip/status", "Ip::Status", "r" )
   arpConfig = ConfigMount.mount( entityManager, "arp/config", "Arp::Config", "w" )
   allIntfStatusDir = LazyMount.mount( entityManager, "interface/status/all",
                                       "Interface::AllIntfStatusDir", "r" )
   bridgingHwCapabilities = LazyMount.mount( entityManager,
                                             "bridging/hwcapabilities",
                                             "Bridging::HwCapabilities", "r" )
   IntfCli.Intf.registerDependentClass( ArpIpIntf, priority=10 )
