#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliToken.Ipv4
import CliToken.Ipv6
import CliCommand
import CliMatcher
import ShowCommand
from CliPlugin import VrfCli
from CliPlugin.IntfCli import Intf
from CliPlugin import MonitoredNexthopCli
from CliPlugin import NdProxyPrefixModel
from CliPlugin.MonitoredNexthopModel import MonitoredNexthopCollection
from CliPlugin import ArpInputCli
from CliPlugin import ArpInputModel
import CliPlugin.ArpAgentCacheCapacityModels as CacheCapacityModels
import CliPlugin.ArpAgentCacheCapacityCli as CacheCapacityCli

matcherArp = CliMatcher.KeywordMatcher(
   'arp', helpdesc='ARP' )
matcherMonitor = CliMatcher.KeywordMatcher(
   'monitor', helpdesc='Monitored IP/IPv6 addresses' )
vrfExprFactory = VrfCli.VrfExprFactory(
   helpdesc='Information for specific VRF' )

#--------------------------------------------------------------------------------
# show arp monitor [ VRF ]
#--------------------------------------------------------------------------------
class ArpMonitorCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show arp monitor [ VRF ]'
   data = {
      'arp': matcherArp,
      'monitor': matcherMonitor,
      'VRF': VrfCli.VrfExprFactory( helpdesc='Show monitored IP/IPv6 '
                                             'addresses in a VRF' ),
   }
   handler = MonitoredNexthopCli.showMonitoredNexthop
   cliModel = MonitoredNexthopCollection

BasicCli.addShowCommandClass( ArpMonitorCmd )

#--------------------------------------------------------------------------------
# show arp monitor summary
#--------------------------------------------------------------------------------
class ArpMonitorSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show arp monitor summary'
   data = {
      'arp': matcherArp,
      'monitor': matcherMonitor,
      'summary': 'Display a summary of monitored IP/IPv6 addresses',
   }
   handler = MonitoredNexthopCli.showMonitoredNexthopSummary
   cliModel = MonitoredNexthopCollection

BasicCli.addShowCommandClass( ArpMonitorSummaryCmd )

#--------------------------------------------------------------------------------
# show ipv6 nd-proxy [ interface INTF ]
#--------------------------------------------------------------------------------
class Ipv6NdProxyCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 nd proxy [ interface INTF ]'
   data = {
      'ipv6' : CliToken.Ipv6.ipv6MatcherForShow,
      'nd': 'Neighbor Discovery / Router Advertisement',
      'proxy': 'Proxy',
      'interface': 'Specific interface to be displayed',
      'INTF' : Intf.matcher,
   }
   handler = NdProxyPrefixModel.showNdProxyPrefix
   cliModel = NdProxyPrefixModel.NdProxyPrefixCollection

BasicCli.addShowCommandClass( Ipv6NdProxyCmd )

# ----------------------------------------------------------------------------------
# sources with helptext
#--------------------------------------------------------------------------------
sourceMatchConf = {}
for tk in ArpInputCli.arpInputSources.commandToken():
   sourceHelpDesc = f'{ArpInputCli.arpInputSources.prettyName(tk)} ARP input entries'
   sourceMatchConf[ tk ] = sourceHelpDesc

class SourceMatcher( CliCommand.CliExpression ):
   expression = 'SOURCE'
   data = { 'SOURCE' : CliMatcher.EnumMatcher( sourceMatchConf ) }

#--------------------------------------------------------------------------------
# show arp agent ( ipv4 | ipv6 ) input [ SOURCE ] [ VRF ]
#--------------------------------------------------------------------------------
class ShowArpInput( ShowCommand.ShowCliCommandClass ):
   syntax = "show arp agent ( ipv4 | ipv6 ) input [ SOURCE ] [ VRF ]"
   data = {
      'arp' : matcherArp,
      'agent' : "ARP Agent information",
      'ipv4' : CliToken.Ipv4.ipv4MatcherForShow,
      'ipv6' : CliToken.Ipv6.ipv6MatcherForShow,
      'input' : "ARP clients input entries",
      'SOURCE' : SourceMatcher,
      'VRF' : vrfExprFactory,
   }
   cliModel = ArpInputModel.ArpInputEntriesModel

   handler = ArpInputCli.showArpInputConfigEntries

BasicCli.addShowCommandClass( ShowArpInput )

#--------------------------------------------------------------------------------
# show arp agent ( ipv4 | ipv6 ) output [ VRF ] [ interface INTF ]
#--------------------------------------------------------------------------------
class ShowArpOutput( ShowCommand.ShowCliCommandClass ):
   syntax = "show arp agent ( ipv4 | ipv6 ) output [ VRF ] [ interface INTF ]"
   data = {
      'arp' : matcherArp,
      'agent' : "ARP Agent information",
      'ipv4' : CliToken.Ipv4.ipv4MatcherForShow,
      'ipv6' : CliToken.Ipv6.ipv6MatcherForShow,
      'output' : "ARP clients output entries",
      'VRF' : vrfExprFactory,
      'interface' : 'Interface selector',
      'INTF' : Intf.matcher,
   }
   cliModel = ArpInputModel.ArpInputEntriesModel

   handler = ArpInputCli.showArpInputStatusEntries

BasicCli.addShowCommandClass( ShowArpOutput )

#--------------------------------------------------------------------------------
# show arp agent ( ipv4 | ipv6 ) cache dynamic capacity interface INTF [ summary ]
#--------------------------------------------------------------------------------
class ShowCacheCapacity( ShowCommand.ShowCliCommandClass ):
   syntax = """show arp agent ( ipv4 | ipv6 ) cache [ dynamic ] capacity
               [ interface INTF ] [ summary ]"""
   data = {
      'arp' : matcherArp,
      'agent' : "ARP Agent information",
      'ipv4' : CliToken.Ipv4.ipv4MatcherForShow,
      'ipv6' : CliToken.Ipv6.ipv6MatcherForShow,
      'cache' : "Neighbor cache information",
      'dynamic' : "Capacity of the dynamic neighbor resolutions",
      'capacity' : "Capacity information",
      'interface' : "Capacity information for an interface",
      'INTF' : Intf.matcher,
      'summary' : "Summarized capacity information",
   }
   cliModel = CacheCapacityModels.ArpAgentCacheCapacities
   handler = CacheCapacityCli.showArpAgentCacheCapacity

BasicCli.addShowCommandClass( ShowCacheCapacity )
