# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import Arnet
from CliModel import Model, Str, Dict
import TableOutput
import six

HEADER_PRESENT = ( 'Component', 'Serial Number', 'Asset Tag' )
HEADER_MISSING = ( 'Missing', ) + HEADER_PRESENT[ 1: ]

def createTable( headings ):
   header = tuple( ( h, 'l' ) for h in headings )
   return TableOutput.createTable( header )

class BaseAsset( Model ):
   tag = Str( help='Customer-supplied asset tag' )

class Asset( BaseAsset ):
   serial = Str( help='Factory-set serial number' )

class Missing( BaseAsset ):
   fruType = Str( help='Type of the missing FRU' )

class Assets( Model ):
   __revision__ = 2
   assets = Dict( keyType=str, valueType=Asset, help='Assets for this system' )
   missing = Dict( keyType=str, valueType=Missing, help='Missing assets' )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         if len( self.missing ):
            newMissing = {}
            for serial, asset in six.iteritems( dictRepr[ 'missing' ] ):
               loc = asset[ 'fruType' ]
               newMissing[ loc ] = { 'serial': serial, 'tag': asset[ 'tag' ] }
            dictRepr[ 'missing' ] = newMissing
      return dictRepr

   def render( self ):
      table = createTable( HEADER_PRESENT )
      for slot in Arnet.sortIntf( self.assets ):
         asset = self.assets[ slot ]
         table.newRow( slot, asset.serial, asset.tag )
      print( table.output() )

      if len( self.missing ):
         table = createTable( HEADER_MISSING )
         for serial, asset in sorted( self.missing.items() ):
            table.newRow( asset.fruType, serial, asset.tag )
         print( table.output() )
