#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from CliModel import List
from CliModel import Enum
from CliModel import Model
from CliModel import Str
import os
import EosVersion

blockingReasonsDict = {
   "IPv4uRPF": "IPv4 uRPF is in ErrorDisabled state",
   "IPv6uRPF": "IPv6 uRPF is in ErrorDisabled state",
   "directFlowEnabled": "Directflow is enabled",
   "vrrpEnabled": "VRRP is enabled",
   "lacpRateFastLocalConfigured": "lacp fast rate configured locally",
   "lacpRateFastRemoteConfigured": "lacp fast rate configured remotely",
   "openFlowEnabled": "Openflow is enabled",
   "pbrConfigured": "Pbr is configured",
   "stpMode": "Spanning-tree mode is not supported",
   "edgePort": "Spanning-tree portfast is not enabled for one or more ports",
   "portRole": "One or more non-edge ports are in spanning-tree"
               " designated/backup role",
   "transmitActive": "Transmit active is enabled for one or more ports",
   "mstInstances": "Spanning-tree is enabled with more than one MST instance",
   "stpBpduguard": "Spanning-tree BPDU guard is not enabled for one or more ports",
   "maintenanceModeConfigured": "Maintenance Mode is configured",
   "macsec": "MACsec is enabled",
   "lagModeInconsistent": "Lag Modes differ between hardware and configuration",
   "mlagStateUnstable": "Mlag state is neither primary nor secondary",
   "mlagNotEstablished": "Mlag peer negotiation is not done yet",
   "vxlanConfigured": "Vxlan is configured",
   "tunnelIntfConfigured": "Tunnel Interface is configured",
}

def mlagVersionCompatibilityWarning():
   return "If you are performing an upgrade, and the "\
          "Release Notes for the new version of EOS "\
          "indicate that MLAG is not backwards"\
          "-compatible with the currently installed "\
          "version (%s), the upgrade will result in "\
          "packet loss." % EosVersion.VersionInfo(
          sysdbRoot=None ).version()

def igmpSnoopingQuerierVersionWarning():
   return "If you are performing an upgrade, the "\
          "Igmp Snooping elected Querier version needs to be set to 3, " \
          "else the upgrade will result in packet loss"

warningReasonsDict = {
   "bgpNoGrace": "BGP is configured without graceful restart",
   "ospfNoGrace": "OSPF is configured without graceful restart",
   "isisNoGrace": "IS-IS is configured without graceful restart",
   "ospf3NoGrace": "OSPF3 is configured without graceful restart",
   "pimEnabled": "PIM is enabled",
   "ripEnabled": "RIP is enabled",
   "varpEnabled": "VARP is enabled",
   "igmpSnoopingEnabled": "IGMP Snooping is enabled",
   "igmpSnoopingQuerierVersion": igmpSnoopingQuerierVersionWarning(),
   "stpErrDisabled": "One or more ports is errDisabled due to BPDU guard",
   "stpState": "STP is not currently restartable",
   "mlagConfigured": "Mlag is configured",
   "runtimeErrdisabledIntfs":
   "Runtime errdisabled interface(s) found. Please note that they will be re-enabled"
   " after hitless reload. If same conditions persist, interface(s) will be"
   " errdisabled again after the reload.",
   "ptpEnabled": "PTP is enabled. PTP slaves will lose connection to the"
   " grandmaster and lose time sync.",
   "pfcWatchdogEnabled": "PFC Watchdog is enabled. The current PFC Watchdog state"
   " may be lost on hitless reload. Monitoring will restart after the reload.",
   "mlagVersionCompatibility": mlagVersionCompatibilityWarning(),
   "mlagPortsActivePartial": "",
   "reloadDelayConfigured": "Mlag is configured with non-zero mlag/non-mlag "
   "reload-delay. This may result in packet loss",
   "pimBidirEnabled": "PIM bidirectional is enabled",
   "pimHoldTime": "Pim interface has join/prune holdtime that is less than two "
   "times default holdtime. This may result in packet loss",
   "macSecFipsRestrictions": "MACsec FIPS restrictions are enabled. This may "
   "result in an extended traffic outage to perform FIPS POST during the reload.",
   "insufficientUptime": "",
}

def _generateHelpStr( reasonsDict ):
   helpStr = ""
   for key, val in reasonsDict.items():
      helpStr += f'{key} : {val}' + os.linesep
   return helpStr

def modifyValueOfWarningReasonsDict( reason, reasonStr ):
   warningReasonsDict[ reason ] = reasonStr

class ReloadHitlessBlockingReason( Model ):
   reason = Enum( values=list( blockingReasonsDict ),
                  help=_generateHelpStr( blockingReasonsDict ) )

   def render( self ):
      print( blockingReasonsDict[ self.reason ] )

class ReloadHitlessWarningReason( Model ):
   reason = Enum( values=list( warningReasonsDict ),
                  help=_generateHelpStr( warningReasonsDict ) )

   def render( self ):
      print( warningReasonsDict[ self.reason ] )

def render( blockingList, warningList, reloadCmd, scheduledReloadInfo ):
   if scheduledReloadInfo:
      print( scheduledReloadInfo )
   if blockingList:
      print( "'%s' cannot proceed due to the following:" % reloadCmd )
      for n in blockingList:
         print( " ", end=' ' )
         n.render()
   if warningList:
      print( "Warnings in the current configuration detected:" )
      for n in warningList:
         print( " ", end=' ' )
         n.render()
   if not blockingList and not warningList:
      print( "No warnings or unsupported configuration found." )

class ReloadHitlessDisruption( Model ):
   blockingList = List( valueType=ReloadHitlessBlockingReason,
                        help="List blocking configuration for 'reload hitless'" )
   warningList = List( valueType=ReloadHitlessWarningReason,
                       help="List of warnings for 'reload hitless'" )

   def render( self ):
      render( self.blockingList, self.warningList, 'reload hitless', None )

class ReloadFastbootDisruption( Model ):
   blockingList = List( valueType=ReloadHitlessBlockingReason,
                        help="List blocking configuration for 'reload fast-boot'" )
   warningList = List( valueType=ReloadHitlessWarningReason,
                       help="List of warnings for 'reload fast-boot'" )
   inCmd = Str( help='Command to schedule reload', optional=True )
   scheduledReloadInfo = Str( help='Scheduled reload info', optional=True )

   def render( self ):
      render( self.blockingList, self.warningList,
              'reload fast-boot' + ( f' {self.inCmd}' if self.inCmd else '' ),
              self.scheduledReloadInfo )
