# Copyright (c) 2007, 2008, 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

# pkgdeps: library SecureBoot
# pkgdeps: library CliSchedulerMgr
# pkgdeps: rpmwith %{_datadir}/asuRecovery.swi

import Tac
import CliParser
import LazyMount, Cell
import ConfigMount
import TacSigint
import BasicCliUtil
import Tracing
import glob
import re, os, sys, time
import sysconfig
import syslog
import Url, subprocess, SimpleConfigFile
import FileUrl
import AsuUtil
import importlib
import EosVersion
import PersistentRestartLogUtils
import FpgaUtil
import datetime
import EventHistoryUtil
import HwEpochPolicy
import pickle
import ReloadPolicy
import AsuPatchBase
import TpmGeneric.Defs as TpmDefs
from TpmGeneric.Tpm import TpmGeneric

from CliOptimizeSwiLib import maybeOptimizeSwi
from CliPlugin.IntfCli import Intf
from CliPlugin.ReloadCli import answerMatches, savePreReloadLogs
from CliPlugin.ReloadConfigSaveCli import ( doConfigSaveBeforeReload,
                                            _unsavedChangesExist,
                                            UCE_NO_UNSAVED_CHANGES )
from CliPlugin.AsuPStoreModel import ReloadHitlessDisruption
from CliPlugin.AsuPStoreModel import ReloadFastbootDisruption
from StageMgr import defaultStageInstance
from ProductAttributes import productAttributes
from CEosHelper import isCeos

__defaultTraceHandle__ = Tracing.Handle( "AsuReloadCli" )
t0 = Tracing.trace0
t1 = Tracing.trace1
t8 = Tracing.trace8

KERNEL_PARAMS_PATH = '/mnt/flash/kernel-params'

class AsuReloadCliGlobals:
   def __init__( self ):
      self.abootSbStatus = None
      self.asuHwStatus = None
      self.asuCliConfig = None
      self.asuShutDownEnabledConfig = None
      self.asuShutDownStatus = None
      self.fatalError = None
      self.asuDmaCancelStatus = None
      self.asuDebugConfig = None
      self.eventHistoryPersistentDir = None
      self.dotBootReloadSwi = "/mnt/flash/.boot-reload.swi"
      self.AsuMode = Tac.Type( "Asu::AsuMode" )
      # setup AsuPatch persistent message support
      self.ehAsuPatchDir = None
      self.ehAsuPatchWriter = None
      self.asuPatchThreads = []
      self.redundancyStatus = None
      self.bootCompletionStatusDir = None
      self.asuFastbootReloadReactor = None
      self.reloadStatus = None
      self.reloadConfig = None
      self.schedConfig = None
      self.mode = None
      self.ipStatus = None

_asuData = AsuReloadCliGlobals()
_extractPath = '/usr/share/patches/'
_pythonPath = sysconfig.get_path( 'purelib' ) + '/'
_asuPatchSwiData = [ 'AsuPatchDb.py', 'AsuPatchBase.py', 'AsuPatchPkgs.tar.gz' ]
_asuPatchFsData = [ 'AsuPatchPkgs', 'AsuPatchDb.pyc', 'AsuPatchBase.pyc' ]

dmaDeviceShutdownHook = []

def cleanupAsuPatchData():
   cmd = [ 'rm', '-rf' ] + [ _extractPath + name for name in _asuPatchSwiData ]
   if not _asuData.asuDebugConfig.testAutoPatch:
      cmd = cmd + [ _extractPath + name for name in _asuPatchFsData ]
   Tac.run( cmd, stdout=Tac.DISCARD, stderr=Tac.CAPTURE, asRoot=True )
   TacSigint.check()

# Some SSU patches might start their own threads when getting applied, but those
# threads will not exit gracefully when SSU is aborted (e.g. due to unsaved config)
# So we need to run this method to check for pending AsuPatch threads and terminate
# them gracefully before aborting SSU
def cleanupPendingAsuPatchThreads():
   for t in _asuData.asuPatchThreads:
      if t.is_alive():
         t.stopSSUPatchThread()
         _asuReloadCliHelper.logAsuPatch( 'Stopping patch thread %s' % t,
               'Warn' )
         t.join()

def doCopy( fromFile, toFile ):
   cpCmd = f'cp {fromFile} {toFile}'
   cmd = os.environ.get( 'TEMP_IMAGE_CMD', cpCmd )
   Tac.run( cmd.split( ' ' ), asRoot=True )

def extractAsuPatchFromTar():
   tarFile = f'{_extractPath}AsuPatchPkgs.tar.gz'
   untarCmd = [ 'tar', '-xzf', tarFile, '-C', _extractPath ]
   Tac.run( untarCmd, stdout=Tac.DISCARD, stderr=Tac.CAPTURE, asRoot=True )
   TacSigint.check()

def extractAsuPatchFromSwi( swiFilename ):
   cmd = os.environ.get( 'UNZIP_SQUASHFS_CMD' )
   if cmd is None:
      cmd = [ 'unzip', '-o', swiFilename ] + _asuPatchSwiData + [ '-d',
              _extractPath ]
   else:
      cmd = cmd.split( ' ' )
   Tac.run( cmd, stdout=Tac.DISCARD, stderr=Tac.CAPTURE, asRoot=True )
   TacSigint.check()

   # move the AsuPatchBase to usr/lib/python
   doCopy( _extractPath + 'AsuPatchBase.py', _pythonPath + 'AsuPatchBase.py' )

   # extract the tar file
   extractAsuPatchFromTar()

class ReturnCode:
   SUCCESS = 0
   FAILURE = 1

class CleanupStages:
   NoCleanup = 0
   FpgaCleanup = 1
   FailedCmdCleanup = 2

class AsuReloadContext:
   def __init__( self, mode, now, url, fs, asuReloadCliGlobals ):
      self.mode = mode
      self.now = now
      self.url = url
      self.fs = fs
      self.asuReloadCliGlobals = asuReloadCliGlobals

def logFuncAsuPatch( message ):
   if _asuData.ehAsuPatchWriter:
      _asuData.ehAsuPatchWriter.eventIs( message )

def asuReloadCliCheckSwi():
   _asuReloadCliHelper.logAsuPatch( 'Calling AsuReloadCli:checkSwi()' )
   return _asuReloadCliHelper.checkSwi()

asuInProgressMsg = \
      "Cannot reload while previous asu boot is still in progress."

def asuInProgress():
   # Check whether previous asu reload is still in progress, prevent
   # unexpected error such as bug 866697
   acd = Tac.newInstance( "Stage::AsuCompletionDetector", _asuData.asuHwStatus,
                          _asuData.bootCompletionStatusDir,
                          _asuData.redundancyStatus )
   return acd.isHitlessReloadInProgress()

def isNamespaceDut():
   return not os.path.exists( "/etc/swi-version" )

class AsuReloadCliHelper( AsuPatchBase.AsuPatchBase ):
   def __init__( self, initName ):
      self.qualifiedFpgas = []
      self.skipFpgaUpgrade = False
      self.doDelayedSwiCopyBack = False
      self.url = None
      self.fs = lambda:None
      self.mode = None
      self.now = False
      self.swiCa = None
      self.cmdInvoked = _asuData.AsuMode.fastboot
      self.cleanupStage = CleanupStages.NoCleanup
      AsuPatchBase.AsuPatchBase.__init__( self, initName )

   def cleanup( self ):
      if self.cleanupStage == CleanupStages.FpgaCleanup:
         self.fpgaRecovery()
      if self.cleanupStage != CleanupStages.NoCleanup:
         doFailedCmdCleanup()

   def logAsuPatch( self, message, lvl=None ):
      if lvl:
         if lvl == 'Msg':
            self.mode.addMessage( message )
         elif lvl == 'Warn':
            self.mode.addWarning( message )
         elif lvl == 'Err':
            self.mode.addError( message )
      self.log( message )
      logFuncAsuPatch( message )

   # Install the required patches
   def doPatch( self ):
      self.cleanupStage = CleanupStages.FailedCmdCleanup
      self.cleanup()
      namespaceDut = isNamespaceDut()
      if namespaceDut:
         currVersion = EosVersion.swiVersion( '/images/EOS.swi' )
      else:
         currVersion = EosVersion.VersionInfo(
            sysdbRoot=self.mode.entityManager.root() )
      try:
         if namespaceDut:
            doCopy( _pythonPath + 'AsuPatchDb.py', _extractPath +
                    'AsuPatchDb.py' )
            extractAsuPatchFromTar()
         else:
            extractAsuPatchFromSwi( self.fs.realFilename_ )
         if _asuData.ehAsuPatchDir and not _asuData.ehAsuPatchWriter:
            ehAsuPatchWConfig = Tac.newInstance( 'EventHistory::WriterConfig' )
            ehAsuPatchWConfig.size = 50
            _asuData.ehAsuPatchWriter = _asuData.ehAsuPatchDir.newWriter(
                  'AsuPatch-%d' % Cell.cellId(), ehAsuPatchWConfig )
         spec = importlib.util.spec_from_file_location( "AsuPatchDb",
                f'{_extractPath}AsuPatchDb.py' )
         AsuPatchDb = importlib.util.module_from_spec( spec )
         spec.loader.exec_module( AsuPatchDb )
         # pylint: disable=protected-access
         if ( not hasattr( AsuPatchDb, '_asuPatchData' ) or
              AsuPatchDb._asuPatchData.version < 2 ):
            self.logAsuPatch( 'Destination AsuPatch version is less than 2.' )
            # Going to a version 1 means we need to explicitly run checkSwi()
            rc = self.checkSwi()
            if rc != ReturnCode.SUCCESS:
               return rc
         fromVersion = 'test' if _asuData.asuDebugConfig.testAutoPatch else \
               currVersion.internalVersion()
         self.logAsuPatch( 'Running AsuPatchDb:doPatch( version=%s, model=%s )' %
                           ( fromVersion, getPlatformModel() ), 'Msg' )
         asuReloadContext = AsuReloadContext( mode=self.mode, now=self.now,
                                url=self.url, fs=self.fs,
                                asuReloadCliGlobals=_asuData )
         asuPatchContext = AsuPatchBase.AsuPatchData()
         asuPatchContext.logFunc = logFuncAsuPatch
         asuPatchContext.checkSwi = asuReloadCliCheckSwi
         patchesPath = f'{_extractPath}AsuPatchPkgs/'
         err = AsuPatchDb.doPatch( version=fromVersion,
                                   model=getPlatformModel(),
                                   patchPath=patchesPath,
                                   asuReloadData=asuReloadContext,
                                   asuPatchData=asuPatchContext )
         if err != 0:
            self.logAsuPatch( 'AsuPatch failed error=%s' % err, 'Err' )
            return ReturnCode.FAILURE
      except Tac.SystemCommandError as e:
         self.logAsuPatch( 'AsuPatch failed swi file missing data %s' %
                          str( e ), 'Err' )
         # To allow upgrade to a release that does not support AsuPatch
         return self.checkSwi()
      except KeyboardInterrupt:
         self.logAsuPatch( 'AsuPatch failed. Command aborted by user.', 'Warn' )
         return ReturnCode.FAILURE
      except SyntaxError:
         self.logAsuPatch( 'AsuPatch failed. Corrupted file(s) in swi.', 'Warn' )
         return ReturnCode.FAILURE
      except OSError:
         self.logAsuPatch( 'AsuPatch failed. Missing file(s) in swi.', 'Warn' )
         return ReturnCode.FAILURE

      self.cleanup()
      self.cleanupStage = CleanupStages.NoCleanup
      self.logAsuPatch( 'Done AsuPatchDb:doPatch' )
      return ReturnCode.SUCCESS

   def evalDelayedSwiCopyBack( self, url ):
      doDelayedSwiCopyBack = False
      if "FASTBOOT_SIM" in os.environ:
         if 'DELAYED_COPY_BACK' in os.environ:
            doDelayedSwiCopyBack = True
      else:
         minFlashPartitionSize = 4 * 1024 * 1024 * 1024  # 4GB
         flashPartitionSize, _ = url.fs.stat()
         doDelayedSwiCopyBack = ( flashPartitionSize <= minFlashPartitionSize )
      return doDelayedSwiCopyBack

   def initArgs( self, *args, **kwargs ):
      if 'asuReloadData' not in kwargs:
         self.logAsuPatch( 'AsuPatch failed missing args %s' % kwargs, 'Warn' )
         return ReturnCode.FAILURE
      for argName, argVal in kwargs.items():
         if argName == 'asuReloadData':
            asuReloadContext = argVal
            self.mode = asuReloadContext.mode
            self.now = asuReloadContext.now
            self.url = asuReloadContext.url
            self.fs = asuReloadContext.fs
            self.doDelayedSwiCopyBack = self.evalDelayedSwiCopyBack( self.url )
            global _asuData
            _asuData = asuReloadContext.asuReloadCliGlobals
      return ReturnCode.SUCCESS

   # We only check that none of the args were modified after initArgs (as other
   # stages may not see the change). If this is desired we can store the data in
   # persistent store and recover it in initArgs.
   def saveArgs( self, *args, **kwargs ):
      assert 'asuReloadData' in kwargs
      for argName, argVal in kwargs.items():
         if argName == 'asuReloadData':
            asuReloadContext = argVal
            assert self.mode == asuReloadContext.mode
            assert self.now == asuReloadContext.now
            assert self.url == asuReloadContext.url
            assert self.fs == asuReloadContext.fs
   
   # We open the new swi and do any validation from the swi
   def checkSwi( self ):
      # We want to optimize swi image as soon as possible. This way any checks
      # are done against optimized image, which is the one that will be booted.
      maybeOptimizeSwi( self.fs.realFilename_, self.mode )
      blockReload, self.swiCa = doAsuReloadPolicyChecks( self.mode,
                                                         self.fs.realFilename_ )
      if blockReload and not _asuData.asuDebugConfig.skipReloadPolicyBlock:
         self.logAsuPatch( 'ReloadPolicyChecks failed', 'Err' )
         return ReturnCode.FAILURE

      if "FASTBOOT_SIM" in os.environ:
         self.logAsuPatch( 'Delayed Copy Back = %s' % self.doDelayedSwiCopyBack,
                           'Msg' )
      else:
         if not self.doDelayedSwiCopyBack:
            spaceOk = doSpaceCheck( self.url, self.fs, self.mode )
            if not spaceOk:
               self.logAsuPatch( 'Space check failed', 'Err' )
               return ReturnCode.FAILURE
         allowed, errorStr = HwEpochPolicy.checkEOSImageCompatible(
               self.fs.realFilename_ )
         if not allowed:
            self.logAsuPatch( errorStr, 'Err' )
            return ReturnCode.FAILURE

      return ReturnCode.SUCCESS

   # Get the boot image FS object
   def doGetBootImageFs( self ):
      '''Get the boot image FS. Return True if the boot image exists, otherwise
      return False and add error message.
      '''
      validFs = True
      if "FASTBOOT_SIM" in os.environ:
         setattr( self.fs, 'realFilename_', os.environ[ "FASTBOOT_SIM" ] )
         self.doDelayedSwiCopyBack = self.evalDelayedSwiCopyBack( self.url )
         return validFs

      self.url = FileUrl.localBootConfig( self.mode.entityManager,
                                          self.mode.session_.disableAaa_ )
      bootConfig = SimpleConfigFile.SimpleConfigFileDict( self.url.realFilename_,
            createIfMissing=True, autoSync = True )

      if not bootConfig.get( 'SWI' ):
         self.mode.addError( "Boot image does not exist." )
         return False

      self.fs = Url.parseUrl( bootConfig[ 'SWI' ],
                  Url.Context( *Url.urlArgsFromMode( self.mode ) ),
                  lambda fs: fs.fsType == 'flash' )

      if not self.fs.exists():
         self.mode.addError( "Boot image %s does not exist."
                             % self.fs.realFilename_ )
         validFs = False
      else:
         self.doDelayedSwiCopyBack = self.evalDelayedSwiCopyBack( self.url )
      return validFs

   def cleanupAsuDataFiles( self ):
      # /mnt/flash/asu-persistent-store.json 
      # /mnt/flash/ssu/*
      Tac.run( [ "rm", "-f", "/mnt/flash/asu-persistent-store.json" ], asRoot=True )
      Tac.run( [ "rm", "-fr", "/mnt/flash/ssu" ], asRoot=True )

   # Note: Any code within this function would NOT allow the AsuPatch to extend
   # it in future. Consider adding your code within checkSwi() instead.
   def doFastBootCommon( self, mode, now, cmdInvoked ):
      self.mode = mode
      self.now = now
      self.cmdInvoked = cmdInvoked
      namespaceDut = isNamespaceDut()

      # dont bother initiate asu while the previous one is still in progress
      if asuInProgress():
         self.mode.addErrorAndStop( asuInProgressMsg )

      dirPath = '/mnt/flash/persist'
      if os.path.exists( dirPath ):
         filePath = os.path.join( dirPath, 'AsuEvent.log' )
         with open( filePath, 'a' ) as f:
            f.write( '\n******************SSU Reload Begins********************\n' )

      self.cleanupAsuDataFiles()

      if not namespaceDut or "FASTBOOT_SIM" in os.environ:
         if not self.doGetBootImageFs():
            return

      if "FASTBOOT_SIM" in os.environ:
         ret = self.checkSwi()
      else:
         # We call checkSwi from doPatch to validate the swi
         ret = self.doPatch()

      if ( ret == ReturnCode.SUCCESS and self.check() == ReturnCode.SUCCESS ):
         self.reboot()
      cleanupPendingAsuPatchThreads()
      self.cleanup()

   def check( self ):
      # FASTBOOT_SIM condition is added to prevent AsuCli btests from using namespace
      # DUT conditional code blocks, as btest DUTs do not contain the directory
      # "/etc/swi-version", and btests DUTs are not namespace DUTs
      namespaceDut = isNamespaceDut() and not "FASTBOOT_SIM" in os.environ
      ( warningList, blockingList ) = \
            AsuUtil.doCheckHitlessReloadSupported( self.mode.entityManager )
      if self.cmdInvoked == _asuData.AsuMode.fastboot:
         ReloadDisruption = ReloadFastbootDisruption
      else:
         ReloadDisruption = ReloadHitlessDisruption

      ReloadDisruption( blockingList=blockingList, 
                        warningList=warningList ).render()
      sys.stdout.flush()
      if blockingList:
         log( "features blocking asu boot" )
         return ReturnCode.FAILURE

      # Check if there is any unsaved config
      if not doConfigSaveBeforeReload( self.mode, power=False, now=self.now,
            noMeansFailure=True ):
         self.mode.addError( "Cannot reload without saving the configuration. "
                        "Aborting the command." )
         return ReturnCode.FAILURE

      self.cleanupStage = CleanupStages.FailedCmdCleanup
      try:
         if namespaceDut:
            self.log( 'Skipping kernel file extraction and saving prefdl data for ' \
                      'namespace DUTs' )
         else:
            # Extract kernel files from the SWI image
            doExtractBoot0FromSwi( self.fs.realFilename_ )
            log( f"Kernel Files {self.fs.realFilename_} extracted from SWI" )

            # Save prefdl data for use on way back up
            prefdl = savePrefdlData()

         if namespaceDut:
            self.log( 'Skipping FPGA upgrade since namespace DUTs have no FPGAs.' )
         else:
            # Extract FPGA image here to avoid the .boot-reboot.swi and extracted
            # file system to be present at same time
            self.skipFpgaUpgrade = doCheckSkipFpgaUpgrade( self.mode )
            if not self.skipFpgaUpgrade:
               self.qualifiedFpgas = doPrepareFpgaUpgrade( self.mode, prefdl, self.fs
                                                         )
            else:
               self.log( 'Skipping FPGA upgrade' )
         
         if namespaceDut:
            self.log( 'Skipping kernel boot for namespace DUTs' )
         else:
            # Generate the commandline to be passed to the kernel
            procOutput = generateProcOutput( self.fs,
                  doDelayedSwiCopyBack=self.doDelayedSwiCopyBack )
            log( f"ProcOutput passed to Kernel {procOutput}" )
            setupBootScript( self.fs, procOutput )
            TacSigint.check()
            doMaybeCopyDotBootSWI( self.fs, self.doDelayedSwiCopyBack )
            TacSigint.check()
      except ( Tac.SystemCommandError, FpgaUtil.UpdateFailed,
               FpgaUtil.UnprogrammedFpga ) as e:
         self.mode.addError( str( e ) )
         log( "FPGA upgrade failed" )
         return ReturnCode.FAILURE
      except KeyboardInterrupt:
         self.mode.addWarning( "Command aborted by user" )
         return ReturnCode.FAILURE
      except OSError as e:
         print( 'unexpected IO error: ' + str( e ) )
         return ReturnCode.FAILURE

      # issue a warning if the recommendend amount of free disk space is not
      # available to persist debug info in case of a fatal error. we do this check
      # after completing the file processing above, as these steps may change the
      # available disk space on persistent store.
      doFatalErrorSpaceCheck( self.mode )

      # Get confirmation from user
      if not self.now:
         answer = BasicCliUtil.getSingleChar( "Proceed with reload? [confirm]" )
         print( answer )
         sys.stdout.flush()
         if answer.lower() not in 'y\r\n':
            return ReturnCode.FAILURE

      self.mode.addMessage( 'Proceeding with reload' )
      sys.stdout.flush()

      # Reload Check passed
      self.cleanupStage = CleanupStages.NoCleanup
      return ReturnCode.SUCCESS

   def fpgaRecovery( self ):
      # Restore original FPGA image files
      for fpga in self.qualifiedFpgas:
         self.mode.addMessage( "Recovering image in flash for %s FPGA"
                               % fpga.name() )
         if os.path.isfile( fpga.imageFilePath() + '.old' ):
            doMove( fpga.imageFilePath() + '.old', fpga.imageFilePath() )
         if os.path.isfile( fpga.spiImageFilePath() + '.old' ):
            doMove( fpga.spiImageFilePath() + '.old', fpga.spiImageFilePath() )
         try:
            fpga.spiProgram()
         except FpgaUtil.UpdateFailed:
            self.mode.addError( "Recovering for %s FPGA failed" % fpga.name() )
         else:
            self.mode.addMessage( "Recovering for %s FPGA succeeded"
                                  % fpga.name() )

   def _doExtendSwiPcr( self, tpm ):
      import zipfile # pylint: disable=import-outside-toplevel
      swiPath = self.fs.realFilename_
      version = ''
      with zipfile.ZipFile( swiPath ) as swiZip:
         if 'version' in swiZip.namelist():
            version = swiZip.read( 'version' )
      logs = [ 'ASU' ]
      logs += [ f for f in version.decode( "utf-8" ).splitlines()
                if f.split( '=' )[ 0 ] in TpmDefs.SWI_FIELDS_TO_LOG ]

      tpm.pcrExtendFromFile( TpmDefs.PcrRegisters.SWI,
                             TpmDefs.PcrEventType.ABOOT_SWI,
                             swiPath, log=logs )

   def _doExtendCmdlinePcr( self, tpm ):
      kernelParamsContent = TpmDefs.DefaultPcrExtend.NO_KERNEL_PARAMS.value
      kernelParamsLog = [ 'no %s' % KERNEL_PARAMS_PATH ]
      if os.path.exists( KERNEL_PARAMS_PATH ):
         with open( KERNEL_PARAMS_PATH ) as f:
            kernelParamsContent = f.read()
         kernelParamsLog = kernelParamsContent.splitlines()
         kernelParamsLog = [ ' '.join( kernelParamsLog ) ]

      tpm.pcrExtendFromData( TpmDefs.PcrRegisters.EOS_EARLY_BOOT_CONF,
                             TpmDefs.PcrEventType.EOS_KERNEL_PARAMS,
                             kernelParamsContent, log=kernelParamsLog )

   def _doExtendSwiCertificate( self, tpm ):
      if not self.swiCa:
         data = TpmDefs.DefaultPcrExtend.NO_SWI_VERIF.value
         logs = [ 'no CA succesfully validated the SWI' ]
      else:
         data = self.swiCa.as_pem()
         logs = [ self.swiCa.get_subject().as_text() ]

      tpm.pcrExtendFromData( TpmDefs.PcrRegisters.SB_CONFIG,
                             TpmDefs.PcrEventType.ABOOT_SB_CERT,
                             data, log=logs )

   def extendPcr( self ):
      # The extension of TPM PCRs is a security feature that allows post boot
      # attestation. We precisely measure every step of EOS that can load code. We
      # record hashes that can be extracted and validated by an external controller
      # in a secure manner. This can used as part of a security model to verify that
      # no tampering of any kind has happened on a device.
      #
      # As ASU is a way to execute a new SWI without going through Aboot again,
      # we need to make sure that this change in running software is accounted for
      # in the PCRs
      try:
         tpm = TpmGeneric()
         if not tpm.isToggleBitSet( TpmDefs.SBToggleBit.MEASUREDBOOT ):
            return

         self._doExtendCmdlinePcr( tpm )
         self._doExtendSwiPcr( tpm )
         self._doExtendSwiCertificate( tpm )
      except ( TpmDefs.NoTpmDevice, TpmDefs.NoTpmImpl, TpmDefs.NoSBToggle ):
         pass
      except TpmDefs.Error as e:
         self.log( 'Failed to extend PCR for the SWI: %s' % str( e ) )

   def flushToDisk( self ):
      """Flush all caches to disk to ensure we're in a good state before going down
      for kexec."""
      # For euphrates-only, to minimize the impact, only do this on Catalina. See
      # BUG830632.
      if Cell.product() != 'catalinaDD':
         return
      self.log( 'Flushing cache to disk' )
      Tac.run( [ "sync" ], asRoot=True )
      # Find the drive that /mnt/flash is mounted on.
      out = Tac.run( [ "mount" ], asRoot=True, stdout=Tac.CAPTURE,
                     stderr=Tac.CAPTURE )
      m = re.search( r'(\S+) on /mnt/flash', out )
      if not m:
         warnStr = 'Failed to flush data to disk: unable to find device'
         self.mode.addWarning( warnStr )
         self.log( warnStr )
         return
      dev = m.group( 1 )
      Tac.run( [ "hdparm", "-F", dev ], asRoot=True, stdout=Tac.CAPTURE )
      self.log( f'Done flushing disk cache: {dev}' )

   def reboot( self ):
      # FASTBOOT_SIM condition is added to prevent AsuCli btests from using namespace
      # DUT conditional code blocks, as btest DUTs do not contain the directory
      # "/etc/swi-version", and btests DUTs are not namespace DUTs
      namespaceDut = isNamespaceDut() and not "FASTBOOT_SIM" in os.environ
      self.cleanupStage = CleanupStages.FpgaCleanup

      # save logs before ASU in case it fails
      savePreReloadLogs()

      # Mount the EventHistory path now so that there is no delay while saving it
      # after Dma is shutdown
      # Since C++ expects a fully mounted directory, mounts must be forced.
      _asuData.eventHistoryPersistentDir.force()

      # Programming the FPGA flash for upgrade
      try:
         if not self.skipFpgaUpgrade:
            self.log( 'Programming the FPGA flash for upgrade' )
            if self.qualifiedFpgas:
               self.mode.addMessage( 'Upgrading FPGAs' )
               self.log( 'Upgrading FPGAs' )
               programFpgaFlash( self.mode, self.qualifiedFpgas )
            else:
               self.mode.addMessage( 'No qualified FPGAs to upgrade' )
               self.log( 'No qualified FPGAs to upgrade' )
      except ( Tac.SystemCommandError, FpgaUtil.UnprogrammedFpga,
               FpgaUtil.UpdateFailed, KeyboardInterrupt ) as e:
         if e is KeyboardInterrupt:
            self.mode.addWarning( "Command aborted by user" )
         else:
            self.mode.addError( 'FPGA upgrade failed: ' + str( e ) )
         return ReturnCode.FAILURE

      if namespaceDut:
         self.log( 'Skipping shutdown of ProcMgr for namespace DUTs' )
      else:
         procMgrCmd = os.environ.get( "PROCMGR_CMD", "/usr/bin/ProcMgr stoppm" )
         Tac.run( procMgrCmd.split( " " ), asRoot=True )

      if namespaceDut:
         self.log( 'Skipping PCR extension for namespace DUTs' )
      else:
         self.extendPcr()

      if namespaceDut:
         self.log( 'Skipping shutdown of Watchdog for namespace DUTs' )
      else:
         shutdownWatchdog( self.mode )

      # Do preprocessing work for ASU
      # E.g. send protocol packets for protocols that support ASU.
      sys.stdout.flush()
      if not "FASTBOOT_SIM" in os.environ:
         if not doASUPreProcessing( self.mode ):
            return ReturnCode.FAILURE

      self.log( 'Shutting down packet drivers' )
      self.mode.addMessage( 'Shutting down packet drivers' )
      killForwardingAgents( self.mode )

      if not "FASTBOOT_SIM" in os.environ:
         dmaCancelSuccess = cancelDma()
         if not dmaCancelSuccess:
            fatalErrorResetMode = Tac.Type( "Stage::FatalError::ResetMode" )
            self.mode.addError(
               'DMA did not shut down properly. Forcing full reboot.' )
            self.log( 'DMA did not shut down properly. Forcing full reboot.' )
            _asuData.fatalError.rebootReason = 'DMA cancellation failure'
            _asuData.fatalError.requestReboot = fatalErrorResetMode.resetLocal
            return ReturnCode.FAILURE
         self.log( 'DMA cancellation completed' )

         # Call the platform specific hooks for shutting down DMA devices.
         for hook in dmaDeviceShutdownHook:
            hook( self )

      EventHistoryUtil.doSaveEventHistory( _asuData.eventHistoryPersistentDir )

      Tac.waitFor( lambda:( not _asuData.asuDebugConfig.holdHitlessShutdown ),
                   description="debug hold shutdown release",
                   maxDelay=0.1, sleep=True, timeout=180.0 )

      self.mode.addMessage( 'reloading %s' % self.fs.realFilename_ )
      sys.stdout.flush()

      if not "FASTBOOT_SIM" in os.environ:
         # Shut all the mgmt interface
         shutMgmtIntfs( self.mode )

      if not "FASTBOOT_SIM" in os.environ:
         self.flushToDisk()

      self.log( 'The system is about to kexec soon' )
      # Move the EOS swi to .boot-reload.swi. This is done for DUTs
      # where we don't have enough space to keep two copies of the
      # swi. In case a fatalError is triggered before this call, we
      # don't need to take care of moving the .boot-reload.swi back to
      # EOS.swi
      doMaybeMoveDotBootSWI( self.fs, self.doDelayedSwiCopyBack )
      # Don't put any code inbetween these two lines.
      # Move should happen immediately before kexec
      runBootScriptCmd = os.environ.get( "BOOTSCRIPT_CMD",
                                         "bash -c /tmp/boot-script" )
      Tac.run( runBootScriptCmd.split( " " ), stdin=sys.stdin, asRoot=True )

      self.cleanupStage = CleanupStages.NoCleanup
      return ReturnCode.SUCCESS

_asuReloadCliHelper = AsuReloadCliHelper( "AsuReloadCli" )

def execute( stageVal, *args, **kwargs ):
   return _asuReloadCliHelper.execute( stageVal, *args, **kwargs )

# A simple script that sets up environment to run boot0

#  For ASU+/ASU2, most of kernel args are inherited from previous
#  boot/aboot ( from /proc/cmdline ), functions used by boot0 in aboot
#  context are defined as dummy. kernel args per EOS image are setup by
#  boot0 which is invoked by Aboot or ASU boot.
# 
bootScriptTemplate = """
#!/bin/sh
parseconfig() {
   return
}

ifget() {
   return
}

writebootconfig() {
   return
}

export arch=i386
export swipath=%s

. /tmp/boot0
"""

def log( msg ):
   date = datetime.datetime.now()
   print( date, msg )

def printWatchdogValue( mode ):
   watchDogValueCmd = os.environ.get( "WATCHDOGVALUE_CMD", 
      "pciread32 4:0 0 0x120" )
   nowStr = time.strftime("%d %b %H:%M:%S")
   output = Tac.run( watchDogValueCmd.split( " " ), asRoot=True,
      stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )
   mode.addMessage( f'{nowStr} : Watchdog output {output}' )

# format the give number in appropriate unit
def sizeFmt( num ):
   x = ''
   for x in [ 'bytes', 'KB', 'MB', 'GB', 'TB' ]:
      if num < 1024.0:
         break
      num /= 1024.0
   return f"{num:3.1f} {x}"

# Return a list of kernel interfaces names for all management ports on the dut
def getMgmtIntfList( mode ):
   mgmtIntf = {}
   # List all interfaces
   intfs = Intf.getAll( mode )

   # Go over all interfaces and find matching ones
   for intf in intfs:
      m = re.match( r"Management(\d+)(?:/(\d+))?$", intf.name )
      if m:
         # Obtain the vrfName of this management intf
         vrfName = 'default'
         ipIntfStatus = _asuData.ipStatus.ipIntfStatus.get( m.group( 0 ) )
         if ipIntfStatus:
            vrfName = ipIntfStatus.vrf
         # Convert it to kernel interface name
         kernelIntfName = "ma%s" % m.group( 1 )
         if m.group( 2 ) is not None:
            kernelIntfName += "_%s" % m.group( 2 )
         mgmtIntf[ kernelIntfName ] = vrfName
   return mgmtIntf

# Shut all mgmt interfaces
def shutMgmtIntfs( mode ):
   if not "FASTBOOT_SIM" in os.environ:
      print( "Shutting down management interface(s)" )
      _asuReloadCliHelper.logAsuPatch( 'Shutting down management interface(s)' )
      sys.stdout.flush()
      mgmtIntfs = getMgmtIntfList( mode )
      for mgmtIntf, vrfName in mgmtIntfs.items():
         if vrfName == 'default':
            intfShutCmd = os.environ.get( "INTF_SHUT_CMD",
                  "ip link set %s down" % mgmtIntf )
         else:
            intfShutCmd = os.environ.get( "INTF_SHUT_CMD",
                  "ip -n ns-%s link set %s down" % ( vrfName, mgmtIntf ) )
         _asuReloadCliHelper.logAsuPatch( 'Running command: %s' % intfShutCmd )
         Tac.run( intfShutCmd.split( " " ), asRoot=True,
               stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )

# Validate available storage space
def doSpaceCheck( url, fs, mode ):
   blockSize = 4096
   def _blocks( sz ):
      return sz // blockSize

   _, flashFree = url.fs.stat()
   freeBlocks = _blocks( flashFree )
   urlBlocks = _blocks( fs.size() ) + 1 # round up to be conservative
   blocksNeeded = urlBlocks
   blocksNeeded *= 1.05 # Add 5% slop factor to be conservative

   # This needs to be more elaborate to make sure we have space to copy
   # the .boot-image.swi to EOS.swi. Basically the space after removing
   # the current free space + current .boot-image.swi > fs.realFilename_
   if freeBlocks < blocksNeeded:
      mode.addError( "Insufficient free space on the internal flash disk to copy "
                     "the boot image." )
      mode.addError( "Free up at least %s and try again." % (
                                         sizeFmt( blocksNeeded * blockSize ) ) )
      return False
   return True

# Issue warning if the recommended amount of free space is not available to
# persist debug info in case of a fatal error during a 'reload hitless' or
# the equivalent 'reload fast-boot'
def doFatalErrorSpaceCheck( mode ):
   feConstants = Tac.Type( "Stage::FatalErrorConstants" )

   # determine the persistent storage device that will be used for persisting logs
   # and cores.
   def _getFatalErrorFilesystem( mode ):
      # check if ssd filesystem is present.
      fs = feConstants.primaryPersistentFs
      if not os.path.exists( fs ):
         # assume no SSDs present; debug info will be persisted at /mnt/flash
         fs = feConstants.secondaryPersistentFs
         if not os.path.exists( fs ):
            # Failing to do the space check should not prevent someone from
            # performing hitless reload. Just print a warning. 
            mode.addWarning( "No filesystem is available for storing debug "
                  "information in the event a problem occurs during system reload." )
            return None
      return fs 

   fs = _getFatalErrorFilesystem( mode )
   if not fs:
      return

   fsStat = os.statvfs( fs )
   freeBlocks = fsStat.f_bfree
   fsBlkSize = fsStat.f_bsize
   blocksNeeded = feConstants.persistedInfoBytesNeeded // fsBlkSize
   if freeBlocks < blocksNeeded:
      mode.addWarning( "It is recommended that at least "
            + sizeFmt( blocksNeeded * fsBlkSize ) + " of free space is available on "
            + fs + " to store debug information in the event that a problem arises"
            + " during system reload. Free up at least "
            + sizeFmt( ( blocksNeeded - freeBlocks ) * fsBlkSize )
            + " before continuing." )

def doExtractBoot0FromSwi( swiFilename ):
   unzipCmd = "unzip -o {} {} {} {} -d /tmp".format( swiFilename, 
                                                     "linux-i386",
                                                     "initrd-i386", "boot0" )
   cmd = os.environ.get( "UNZIP_SQUASHFS_CMD", unzipCmd )
   Tac.run( cmd.split( " " ), stdout=Tac.DISCARD, stderr=Tac.CAPTURE,
         asRoot=True )
   TacSigint.check()

# Generate the command line we will pass to the kexec
def generateProcOutput( fs, doDelayedSwiCopyBack=False ):
   regexMatch = [ re.compile( r"SWI=[\S]* " ),
                  re.compile( "kexec_jump_back_entry=0x[0-9A-Fa-f]*"),
                  re.compile( "arista.asu_reboot"),
                  re.compile( "arista.asu_hitless"),
                  re.compile( r"arista.doDelayedSwiCopyBack"),
                  re.compile( r"NETIP=[\S]*"),
                  re.compile( r"NETGW=[\S]*"),
                  re.compile( r"dmamem=[\w-]+" ),
                  re.compile( r"SKIP_CMP=[\S]*" ),
                  re.compile( r"systemd.default_standard_output=tty" ),
                  ]

   # pylint: disable-next=consider-using-with
   proc = subprocess.Popen( [ 'cat', '/proc/cmdline' ],
         stdout=subprocess.PIPE,
         universal_newlines=True )
   procOutput = proc.communicate()[ 0 ]
   procOutput = procOutput.strip( '\n' )
   for regex in regexMatch:
      procOutput = re.sub( regex, '', procOutput )

   # remove duplicates, later setting wins and preserve non-duplicates
   # order
   args = procOutput.split()
   procOutDict = {}
   procOutList = []
   for arg in args:
      n, v = arg, arg
      if '=' in arg:
         n = arg.split( '=' )[ 0 ]
      if n in procOutDict:
         # use later arg if v is different, else skip adding to list
         if v != procOutDict[ n ]:
            procOutList.remove( procOutDict[ n ] )
            procOutList.append( v )
      else:
         procOutList.append( v )
      procOutDict[ n ] = v

   procOutList.append( "SWI=%s" % fs.realFilename_ )
   procOutList.append( "arista.asu_hitless" )

   # Is the partition less than 2GB in size? We need to do some work in
   # boot1 script to accomodate that
   if doDelayedSwiCopyBack:
      procOutList.append( "arista.doDelayedSwiCopyBack" )

   return procOutList

# Save prefdl data to be used on the way back up
def savePrefdlData():

   def getPreFdl():
      path = '/etc/prefdl'
      if os.path.isfile( path ):
         with open( path, 'rb' ) as f:
            return f.read()
      else:
         return Tac.run( [ "/bin/sh", "-c", "/usr/bin/genprefdl" ],
                         stdout=Tac.CAPTURE,
                         stderr=Tac.DISCARD,
                         asRoot=True,
                         text=False )

   prefdlInEnvVariables = os.environb.get( b"PREFDL" )
   prefdl = prefdlInEnvVariables or getPreFdl()

   # Save identifyCell and prefdl output to use on boot up
   # and save time (only if not a test)
   if not prefdlInEnvVariables:

      asuConstants = Tac.Type( "Asu::AsuConstants" )
      Tac.run( [ "mkdir", "-p", asuConstants.asuReloadCacheDir ] )
      # Need to run as root because of ham file mapping permission issue
      Tac.run( [ "/usr/bin/identifyCell", "-d", 
         asuConstants.asuReloadCacheDir ],
         stdout=Tac.DISCARD, stderr=Tac.DISCARD, asRoot=True )

      prefdlPath = "%s/prefdl.txt" % asuConstants.asuReloadCacheDir
      with open( prefdlPath, 'wb+' ) as f:
         f.write( prefdl )
   return prefdl

def doAsuReloadPolicyChecks( mode, swiFile ):
   '''Run the ASU ReloadPolicy checks. If the checks produced any errors or warnings,
   then output those to the CLI. Returns a tuple of booleans indicating if the checks
   succeeded and if the feature-keys check was run.
   '''
   blockReload = False

   # Run the checks and determine if feature-keys check ran
   category = [ 'ASU' ]
   policyResult, caUsed = \
      ReloadPolicy.doCheck( swiFile, category=category, mode=mode,
                            abootSbStatus=_asuData.abootSbStatus )
   ranFeatureKeysCheck = 'AsuFeatureKeysCheck' in policyResult.policySuccess

   # Output errors/warnings to CLI
   for errStr in policyResult.errors:
      mode.addError( errStr )
   for warnStr in policyResult.warnings:
      mode.addWarning( warnStr )

   # Errors block reload
   if policyResult.errors:
      blockReload = True
      if ( 'AsuDowngradeCheck' in policyResult.policySuccess and
           not policyResult.policySuccess[ 'AsuDowngradeCheck' ] ):
         mode.addError( "ASU hitless upgrade failed downgrade check." )
      if ( ranFeatureKeysCheck and
           not policyResult.policySuccess[ 'AsuFeatureKeysCheck' ] ):
         mode.addError( "ASU hitless upgrade failed feature-keys check." )

   return blockReload, caUsed

# Create the .boot.swi file so that after reboot copy does not happen
def doMaybeCopyDotBootSWI( fs, doDelayedSwiCopyBack ):
   if not doDelayedSwiCopyBack:
      cpCmd = f"cp {fs.realFilename_} {_asuData.dotBootReloadSwi}"
      cmd = os.environ.get( "TEMP_IMAGE_CMD", cpCmd )
      Tac.run( cmd.split( " " ), asRoot=True )
      Tac.run( [ "sync" ], asRoot=True )

def doMaybeMoveDotBootSWI( fs, doDelayedSwiCopyBack ):
   # Call this function for small flash duts just before Kexec.
   # This will prevent the system from booting with a missing EOS.swi
   # in case a fatalError is triggerd before kexec
   if doDelayedSwiCopyBack:
      # If the partition is small, we just rename the EOS.swi instead of copying here
      # We will remember this, on the way back up, once the old .boot-image.swi has
      # been removed, we will make a copy of the .boot-reload.swi to EOS.swi
      mvCmd = f"mv {fs.realFilename_} {_asuData.dotBootReloadSwi}"
      cmd = os.environ.get( "TEMP_IMAGE_CMD", mvCmd )
      Tac.run( cmd.split( " " ), asRoot=True )
   
      _asuReloadCliHelper.logAsuPatch( 'doMaybeMoveDotBootSWI completed' )

      # Install a recovery SWI that can recover an EOS image to the
      # location that boot-config points to in case ASU fails. This
      # avoids a failure mode where we can get stuck in Aboot due to
      # the boot-config pointing to a non-existent SWI.
      if 'FASTBOOT_SIM' not in os.environ:
         Tac.run( ( "mv /usr/share/asuRecovery.swi %s" % fs.realFilename_ ).split(),
                  asRoot=True )

def requestFatalError( mode, reason ):
   mode.addError( "%s. Forcing cold reboot" % reason.capitalize() )
   fatalErrorResetMode = Tac.Type( "Stage::FatalError::ResetMode" )
   _asuData.fatalError.rebootReason = reason
   _asuData.fatalError.requestReboot = fatalErrorResetMode.resetLocal

def debugTimeoutValue():
   # increase watchdog timeout to 30 min for hold shutdown test
   timeout = None
   if _asuData.asuDebugConfig.holdHitlessShutdown:
      timeout = 1800
   return timeout

def watchdogTimeoutValue():
   return _asuData.asuHwStatus.watchdogTimeout

# Do all preprocessing work ( support for lacp, arp, pim etc )
def doASUPreProcessing( mode ):
   # Save a reload cause file under /mnt/flash/debug
   _asuReloadCliHelper.logAsuPatch( 'Writing reload cause' )
   if not "FASTBOOT_SIM" in os.environ:
      reloadCauseConstants = Tac.Value( "ReloadCause::ReloadCauseConstants" )
      reloadCauseConstants.writeLocalReloadCause(
                         reloadCauseConstants.hitlessReloadDesc, "", True )

   _asuData.asuShutDownEnabledConfig.enabled[ defaultStageInstance ] = True
   newTimeout = debugTimeoutValue() or 150
   _asuReloadCliHelper.logAsuPatch( 'Asu Shutdown stages starting' )
   try:
      instStatus = _asuData.asuShutDownStatus.instStatus
      Tac.waitFor( lambda: ( defaultStageInstance in instStatus ) and \
                      instStatus[ defaultStageInstance ].complete,
                   description="platform processing",
                   maxDelay=0.1, sleep=True, timeout=newTimeout )
   except Tac.Timeout:
      _asuReloadCliHelper.logAsuPatch( 'Platform processing timed out' )
      requestFatalError( mode, "platform processing timed out" )
      return False
   _asuReloadCliHelper.logAsuPatch( 'Asu Shutdown stages completed' )
   for agent, l in PersistentRestartLogUtils.getPersistentLogs().items():
      mode.addWarning( 'Agent ' + agent + ' reported a warning: ' + l )
   return True

def getPlatformKillRe( mode ):
   agentsToKill = []
   for agent, valid in _asuData.asuHwStatus.forwardingAgent.items():
      # Filter out any invalid or "null" agents.
      if valid and ( agent != "" ):
         try:
            if not os.environ.get( "PLATFORM_CMD" ):
               # Use signal 0 to check whether the agent is running
               cmd = "killall -q --signal 0 %s" % agent
               Tac.run( cmd.split(), asRoot=True )
            agentsToKill.append( agent )
         except Tac.SystemCommandError:
            # Agent not running
            mode.addMessage( "Skipping %s (not running)" % agent )

   return " ".join( agentsToKill )

def killForwardingAgents( mode ):
   _asuReloadCliHelper.logAsuPatch( 'Running doSaveAsuState' )
   AsuUtil.doSaveAsuState( mode.entityManager )

   agentsToKill = getPlatformKillRe( mode )
   _asuReloadCliHelper.logAsuPatch( 'Killing forwarding agents' )
   if agentsToKill:
      platformAgentCmd = os.environ.get( "PLATFORM_CMD",
                                         "killall -w -q %s" % agentsToKill )
      try:
         Tac.run( platformAgentCmd.split( " " ) , asRoot=True)
      except Tac.SystemCommandError:
         checkAgents = getPlatformKillRe( mode )
         # If there are no forwarding agents on recheck one or more shut themselves
         # down gracefully. Ignore the error
         for agent in checkAgents.split():
            mode.addWarning( "Unable to shutdown " + agent )
   _asuReloadCliHelper.logAsuPatch( 'killForwardingAgents completed' )

def getPlatformModel():
   platform = _asuData.asuHwStatus.platform
   assert platform
   return platform

def shutdownWatchdog( mode ):
   try:
      watchdogCmd = os.environ.get( "WATCHDOGKILL_CMD", "watchdog -k" )
      Tac.run( watchdogCmd.split( " " ), asRoot=True )

      newTimeout = debugTimeoutValue() or watchdogTimeoutValue()
      watchdogTimeoutCmd = os.environ.get( "WATCHDOGTIMEOUT_CMD", 
            "watchdog -o %d" % newTimeout )
      _asuReloadCliHelper.logAsuPatch( watchdogTimeoutCmd )
      Tac.run( watchdogTimeoutCmd.split( " " ), asRoot=True )

   except Tac.SystemCommandError as e:
      mode.addWarning(
         'unexpected error: ' + str( e )  + ' proceeding with normal reload' )
      watchdogCmd = os.environ.get( "WATCHDOGKILL_CMD", "watchdog -k" )
      #Kill watchdog again in case
      Tac.run( watchdogCmd.split( " " ), asRoot=True )
      shutdownCmd = os.environ.get( "SHUTDOWN_CMD", "shutdown -r now" )
      Tac.run( shutdownCmd.split( " " ), asRoot=True )

def cleanupSqshes():
   cmd = os.environ.get( "CLEANUP_FILE_SYSTEM" )
   if cmd:
      Tac.run( cmd.split( " " ), asRoot=True )
   else:
      sqshList = glob.glob( "/mnt/flash/*.sqsh" )
      if sqshList:
         Tac.run( [ "rm", "-rf" ] + sqshList, asRoot=True )

def doFailedCmdCleanup():
   if not isNamespaceDut():
      cleanupAsuPatchData()
   cmdDefault = "rm -rf %s" % _asuData.dotBootReloadSwi
   cmd = os.environ.get( "CLEANUP_TEMP_IMAGE", cmdDefault )
   Tac.run( cmd.split( " " ), asRoot=True )
   cmd = os.environ.get( "CLEANUP_KERNEL",
                         "rm -rf /tmp/initrd-i386 /tmp/linux-i386 /tmp/boot0" )
   Tac.run( cmd.split( " " ), asRoot=True )
   cmd = os.environ.get( "CLEANUP_HITLESS_MANIFEST", 
                         "rm -rf /tmp/asu-manifest /tmp/asu-manifestc" )
   Tac.run( cmd.split( " " ), asRoot=True )
   cmd = os.environ.get( "CLEANUP_HITLESS_POLICY", 
                         "rm -rf /tmp/asu-upgrade-policy /tmp/asu-upgrade-policyc" )
   Tac.run( cmd.split( " " ), asRoot=True )

   if os.path.exists( "/tmp/etc-bootconfig-created" ):
      bootconfigCleanup =  os.environ.get( "CLEANUP_ASUBOOTCONFIG",
                                           "rm -rf /etc/boot-config" )
      Tac.run( bootconfigCleanup.split( " " ), asRoot=True )
      Tac.run( "rm -rf /tmp/etc-bootconfig-created".split( " " ), asRoot=True )

   if os.path.exists( "/tmp/etc-cmdline-created" ):
      cmdlineCleanup = os.environ.get( "CLEANUP_ASUCMDLINE", "rm -rf /etc/cmdline" )
      Tac.run( cmdlineCleanup.split( " " ), asRoot=True )
      Tac.run( "rm -rf /tmp/etc-cmdline-created".split( " " ), asRoot=True )

   bootScriptCleanup = os.environ.get( "CLEANUP_BOOTSCRIPT",
                                       "rm -rf /tmp/boot-script" )
   Tac.run( bootScriptCleanup.split( " " ), asRoot=True )

   cleanupSqshes()

   Tac.run( cmd.split( " " ), asRoot=True )
   cmd = os.environ.get( "CLEANUP_FPGA_IMAGE", "rm -rf /tmp/squashfs-root" )
   Tac.run( cmd.split( " " ), asRoot=True )

def setupBootScript( fs, procOutList ):
   bootconfigFile = os.environ.get( "ASUBOOTCONFIG_FILE", "/etc/boot-config" )
   if not os.path.exists( bootconfigFile ):
      bootConfigCmd = "touch %s" % bootconfigFile
      Tac.run( bootConfigCmd.split( " " ), asRoot=True )
      Tac.run( "touch /tmp/etc-bootconfig-created".split( " " ), asRoot=True )

   cmdlineFile = os.environ.get( "ASUCMDLINE_FILE", "/etc/cmdline" )
   Tac.run( [ 'touch', cmdlineFile ], asRoot=True )
   Tac.run( [ 'chmod', '777', cmdlineFile ], asRoot=True )
   procOutStr = '\n'.join( procOutList ) + '\n'
   with open( cmdlineFile, "w" ) as f:
      f.write( procOutStr )
   Tac.run( [ 'chmod', '777', cmdlineFile ], asRoot=True )
   Tac.run( "touch /tmp/etc-cmdline-created".split( " " ), asRoot=True )

   bootscriptFile = os.environ.get( "BOOTSCRIPT_FILE", "/tmp/boot-script" )
   with open( bootscriptFile, "w" ) as f:
      f.write( bootScriptTemplate % _asuData.dotBootReloadSwi ) 
   cmd = "chmod 777 %s" % bootscriptFile
   Tac.run( cmd.split( " " ), asRoot=True )

def cancelDma():
   _asuData.asuCliConfig.dmaCancelInitiated = True
   newTimeout = debugTimeoutValue() or 60
   try:
      Tac.waitFor( lambda:_asuData.asuDmaCancelStatus.asuDmaCancelComplete,
                   description="dma quiesce",
                   maxDelay=0.1, sleep=True, timeout=newTimeout )
   except Tac.Timeout:
      return False

   return _asuData.asuDmaCancelStatus.asuDmaCancelSuccess

def doCheckSkipFpgaUpgrade( mode ):
   with open( '/proc/cmdline' ) as f:
      cmdlineStr = f.read()
   if "AristaDiagnosticsMode=NoFPGA" in cmdlineStr:
      mode.addMessage( 'Skipping FPGA upgrade due to kernel '
                       'AristaDiagnosticsMode parameter.' )
      return True
   if os.path.exists( "/mnt/flash/skipFpgaUpgrade" ):
      mode.addMessage( 'Skipping FPGA upgrade due to configuration in flash.' )
      return True
   return False

def doCheckFpgaVersionManifest( swiFilename, fpgaVersionManifestFile ):
   fpgaImageManifestFound = False
   # Try extracting the manifest from swi, ignore error on unzip (if any)
   unzipCmd = ( "unzip -q -o %s %s -d /tmp"
                % ( swiFilename, fpgaVersionManifestFile ) )
   cmd = os.environ.get( "UNZIP_FPGA_MANIFEST_CMD", unzipCmd )
   Tac.run( cmd.split( " " ), stdout=Tac.DISCARD, stderr=Tac.DISCARD,
            asRoot=True, ignoreReturnCode=True )
   if os.path.exists( "/tmp/" + fpgaVersionManifestFile ):
      fpgaImageManifestFound = True
   return fpgaImageManifestFound

def doPrepareFpgaUpgrade( mode, prefdl, fs ):
   flavor = FpgaUtil.translateSwiFlavor()
   fpgas = FpgaUtil.systemFpgas( prefdl, flavor ).fpgaList()
   qualifiedFpgas = [ fpga for fpga in fpgas
                      if fpga.isHitlessResetSupported() and fpga.spiEnabled() ]
   # Update the qualifedFpgas list based on the version found in the manifest file
   upgradeNeededFpgas = []
   if qualifiedFpgas:
      fpgaVersionManifestFile = "fpga-image-versions"
      fpgaImageManifestFound = doCheckFpgaVersionManifest( fs.realFilename_,
                                                           fpgaVersionManifestFile )
      # Perform a FPGA version fast check only when the manifest file is found
      # Otherwise, the list of qualifiedFpgas will not be updated and unsquashing
      # the file system will be needed
      if fpgaImageManifestFound:
         # Check against the manifest file rather than unsquashing the file system
         with open( "/tmp/" + fpgaVersionManifestFile, 'rb' ) as f:
            fpgaManifestVersions = pickle.load( f )
         for fpga in qualifiedFpgas:
            fpgaName = fpga.name()
            if fpgaName in fpgaManifestVersions:
               currentFpgaVersion = fpga.hardwareVersion()
               newFpgaVersion = fpgaManifestVersions[ fpgaName ]
               mode.addMessage( 'Checking Fpga %s image version: '
                                'Current version is %s, New version is %s' %
                                ( fpgaName, currentFpgaVersion.major,
                                  newFpgaVersion.major ) )
               _asuReloadCliHelper.logAsuPatch(
                     'Checking Fpga %s image version: '
                     'Current version is %s, New version is %s' %
                     ( fpgaName, currentFpgaVersion.major,
                        newFpgaVersion.major ) )
               # Upgrade is need if major versions differ or if minor version
               # of the new fpga is higher than the current one. However,
               # if forceUpgradeFile is present, extend the list of fpgas
               # to be updated.
               if ( ( currentFpgaVersion.major != newFpgaVersion.major or
                    currentFpgaVersion.minor < newFpgaVersion.minor ) or 
                    forceFpgaUpgrade() ):
                  upgradeNeededFpgas.append( fpga )
            else:
               upgradeNeededFpgas.append( fpga )
         # update the qualifiedFpgas list to be all Fpgas that need update
         qualifiedFpgas = upgradeNeededFpgas

   # Extract file system only when there are qualified FPGAs in switch
   # Important for cloverdale duts to reload hitless without upgrading FPGA
   # Peach FPGA in cloverdale should not be qualified
   if qualifiedFpgas:
      mode.addMessage( 'Extracting file system for upgrading FPGAs' )
      _asuReloadCliHelper.logAsuPatch(
            'Extracting file system for upgrading FPGAs' )
      mode.addMessage( 'This process may take a few minutes' )
      doExtractSqshWithFpgaFromSwi( fs.realFilename_ )
      skipFpgaUpgrade = doUnsquashFpgaImageFiles( mode, qualifiedFpgas )
      if skipFpgaUpgrade:
         qualifiedFpgas = []
   cleanupCmd = os.environ.get( "CLEANUP_FILE_SYSTEM",
                                "rm -rf /mnt/flash/rootfs-i386.sqsh" )
   Tac.run( cleanupCmd.split( " " ), asRoot=True )
   return qualifiedFpgas

def doExtractSqshWithFpgaFromSwi( swiFilename ):
   unzipListCmd = "unzip -l %s" % swiFilename
   swiFileTable = Tac.run( unzipListCmd.split(), stdout=Tac.CAPTURE,
                           stderr=Tac.CAPTURE, asRoot=True )
   # Check to see if the target image is SWIM, which installs
   # its FPGAs into sqshes named as *.fpga.sqsh
   if ".fpga.sqsh" in swiFileTable:
      sqsh = "*.fpga.sqsh"
   else:
      # Needed for ASU Upgrades to non-SWIM formatted
      # images until we switch to SWIM completely.
      sqsh = "rootfs-i386.sqsh"

   unzipCmd = f"unzip -o {swiFilename} {sqsh} -d /mnt/flash"
   cmd = os.environ.get( "UNZIP_SQUASHFS_CMD", unzipCmd )
   try:
      Tac.run( cmd.split( " " ), stdout=Tac.DISCARD, stderr=Tac.CAPTURE,
               asRoot=True )
   except ( Tac.SystemCommandError, Tac.Timeout ) :
      # pylint: disable-next=raise-missing-from
      raise FpgaUtil.UpdateFailed( "Can not extract file system from SWI image. "
                                   "Please check that the flash has enough free "
                                   "space." )
   TacSigint.check()

def forceFpgaUpgrade():
   return os.path.exists( "/mnt/flash/forceFpgaUpgrade" )

def doMove( fromPath, toPath ):
   mvCmd = f"mv {fromPath} {toPath}"
   cmd = os.environ.get( "TEMP_IMAGE_CMD", mvCmd )
   Tac.run( cmd.split( " " ), asRoot=True )

def programFpgaFlash( mode, fpgas ):
   # Assume all of the fpgas are programmed, and working currently.
   # Assume all FPGAs passed in are qualified and have images in /tmp
   for fpga in fpgas:
      # Need to copy because spiProgram use the path /usr/share/fpga/
      doCopyFpgaImageFiles( fpga )
      hwVersion = fpga.hardwareVersion()
      ( upgradeNeeded, _ ) = fpga.needsUpgrade( hwVersion )
      if upgradeNeeded or forceFpgaUpgrade():
         mode.addMessage( "Programming flash for %s FPGA" % fpga.name() )
         _asuReloadCliHelper.logAsuPatch( "Programming flash for %s FPGA"
               % fpga.name() )
         fpga.spiProgram()
      else:
         mode.addMessage( "No need to reprogram %s FPGA" % fpga.name() )
         _asuReloadCliHelper.logAsuPatch( "No need to reprogram %s FPGA"
               % fpga.name() )
   TacSigint.check()

# Assume the number of qualified FPGAs is greater than 0
def doUnsquashFpgaImageFiles( mode, qualifiedFpgas ):
   filesToUnsquash = ""
   for fpga in qualifiedFpgas:
      filesToUnsquash += fpga.imageFilePath() + " "
      filesToUnsquash += fpga.spiImageFilePath() + " "

   cmd = os.environ.get( "UNSQSH_CMD" )
   if cmd:
      Tac.run( cmd.split( " " ),asRoot=True )
   else:
      sqshList = glob.glob( "/mnt/flash/*.fpga.sqsh" )
      if not sqshList:
         # Needed for ASU Upgrades to non-SWIM formatted
         # images until we switch to SWIM completely.
         sqshList = [ "/mnt/flash/rootfs-i386.sqsh" ]
      for sqsh in sqshList:
         unsqshCmd = "unsquashfs -f -d {} {} {}".format( "/tmp/squashfs-root", sqsh,
                                                         filesToUnsquash )
         Tac.run( unsqshCmd.split( " " ), asRoot=True )

   skipFpgaUpgrade = False
   for fpga in qualifiedFpgas:
      if not ( os.path.isfile( "/tmp/squashfs-root" + fpga.imageFilePath() ) and
               os.path.isfile( "/tmp/squashfs-root" + fpga.spiImageFilePath() ) ):
         mode.addWarning( "Can not extract %s FPGA image from SWI" % fpga.name() )
         prompt = "Are you downgrading to an older version of EOS? "
         answer = BasicCliUtil.getChoice( mode, prompt, [ 'yes', 'no' ], 'no' )
         if answerMatches( answer, "yes" ):
            prompt = ( "WARNING: Downgrading with reload hitless is not supported. "
                       "You may experience unexpected behavior until you perform a "
                       "cold reboot. Proceed anyway? " )
            confirm = BasicCliUtil.getChoice( mode, prompt, [ 'yes', 'no' ], 'no' )
         if answerMatches( answer, "yes" ) and answerMatches( confirm, "yes" ):
            mode.addMessage( "Skipping FPGA upgrade" )
            skipFpgaUpgrade = True
         else:
            raise FpgaUtil.UpdateFailed( "Please contact technical support" )
   return skipFpgaUpgrade

def doCopyFpgaImageFiles( fpga ):
   newImagePath = "/tmp/squashfs-root" + fpga.imageFilePath()
   currentImagePath = fpga.imageFilePath()
   doCopy( currentImagePath, currentImagePath + '.old' )
   doCopy( newImagePath, currentImagePath )
   newSpiImagePath = "/tmp/squashfs-root" + fpga.spiImageFilePath()
   currentSpiImagePath = fpga.spiImageFilePath()
   doCopy( currentSpiImagePath, currentSpiImagePath + '.old' )
   doCopy( newSpiImagePath, currentSpiImagePath )

def rebootHitlessGuard( mode, token ):
   if productAttributes().bootAttributes.asuShutdownSupported() or \
      ( ( _asuData.AsuMode.hitless == _asuData.asuHwStatus.asuModeSupported ) and
        not isCeos() ):
      return None
   return CliParser.guardNotThisPlatform

def setHoldHitlessShutdown( hold ):
   _asuData.asuDebugConfig.holdHitlessShutdown = hold

def setSkipReloadPolicyBlock( value ):
   _asuData.asuDebugConfig.skipReloadPolicyBlock = value

def isReloadScheduled():
   return _asuData.reloadStatus and \
      _asuData.reloadStatus.reloadTime != Tac.endOfTime

def getScheduledReloadInfo():
   if isReloadScheduled():
      reloadStatus = _asuData.reloadStatus
      timeLeft = reloadStatus.reloadTime - Tac.now()
      assert timeLeft > 0, 'Scheduled reload in past found.'
      reloadTime = timeLeft + time.time()
      reloadReason = reloadStatus.reason
      infoStr = f'Reload scheduled for {time.ctime( reloadTime )}'
      minutes = int( timeLeft / 60 )
      hh, mm = divmod( minutes, 60 )
      if hh < 24:
         infoStr += f' (in {hh} hours {mm} minutes)'
      if reloadReason:
         infoStr += os.linesep + f'Reload reason: {reloadReason}'
      return infoStr
   return 'No reload is scheduled'

def scheduleReload( seconds, reason, mode ):
   _asuData.reloadConfig.reason = '' if not reason else reason
   _asuData.reloadConfig.all = False
   _asuData.reloadConfig.fastboot = True
   _asuData.reloadConfig.reloadTime = seconds + Tac.now()
   _asuData.mode = mode
   try:
      Tac.waitFor( isReloadScheduled,
                   description="reload to get scheduled",
                   sleep=True )
   except Tac.Timeout:
      print( 'Reload schedule could not be ascertained' )
      return False
   return True

def getFastbootFeatureDisruption( mode, args ):
   ( warningList_, blockingList_ ) = \
            AsuUtil.doCheckHitlessReloadSupported( mode.entityManager )
   inCmd_ = 'in' if 'in' in args else ''

   return ReloadFastbootDisruption( blockingList=blockingList_,
                                   warningList=warningList_,
                                   inCmd=inCmd_ )

class AsuFastbootReloadReactor( Tac.Notifiee ):
   notifierTypeName = 'System::Reload::Status'
   def __init__( self, reloadConfig, reloadStatus, schedConfig ):
      Tac.Notifiee.__init__( self, reloadStatus )
      self.reloadConfig = reloadConfig
      self.reloadStatus = reloadStatus
      self.schedConfig = schedConfig
      self.scheduleName = 'fast-boot-reload'
      self.fastbootReloadCmd = 'reload fast-boot now'
      self.SchedConfigType = Tac.Type( "System::CliScheduler::ScheduledCli" )
      self.clock_ = Tac.ClockNotifiee( handler=self.handleClock,
                                       timeMin=Tac.endOfTime )

   def handleClock( self ):
      t0( 'Start fast-boot reload process.' )
      if _unsavedChangesExist( _asuData.mode ) == UCE_NO_UNSAVED_CHANGES:
         t0( 'Execute "reload fast-boot now" command' )
         at, startAt, interval = 0, 0, 0
         maxLogFiles, maxTotalSize = 100, 0
         loggingVerbose = False
         timeout = 1200
         schedConfig = self.SchedConfigType( self.scheduleName,
                                    self.fastbootReloadCmd,
                                    at, startAt, interval, maxLogFiles,
                                    maxTotalSize, loggingVerbose, timeout,
                                    logDir='flash:schedule/' )
         self.schedConfig.scheduledCli.addMember( schedConfig )
      else:
         syslog.syslog( syslog.LOG_ERR,
            'Unsaved configuration changes abort the reload.' )
      t0( 'Cleanup scheduled reload in case blocking issue abort reload' )
      self._clearReload()


   def _clearReload(self):
      self.reloadConfig.reason = ''
      self.reloadConfig.all = False
      self.reloadConfig.reloadTime = Tac.endOfTime

   @Tac.handler( 'reloadTime' )
   def handleReloadTime( self ):
      t0( f'handleReloadTime: reloadTime = {self.notifier_.reloadTime}' )
      if not self.reloadConfig.fastboot:
         t0( 'Non-fast-boot reload operations. Do nothing.' )
         return
      if self.scheduleName in self.schedConfig.scheduledCli:
         # The workflow is schedule a reload, then cleanup "reloadTime", then this
         # handler is called and we do cleanup here so that this clean up is TACC
         # consistency model independent. 
         t0( 'Clean up failed/aborted schedule.' )
         del self.schedConfig.scheduledCli[ self.scheduleName ]
      if self.notifier_.reloadTime == Tac.endOfTime:
         # Cancel/cleanup reload, only reset fastboot AsuReloadCli only need to take
         # care of "fastboot" config.
         self.reloadConfig.fastboot = False
      else:
         seconds = self.notifier_.reloadTime - Tac.now()
         t0( f'Time before fast-boot reload: {seconds} seconds' )
      self.clock_.timeMin = self.notifier_.reloadTime
      t0( 'Finish handleReloadTime' )

def createFastbootReloadReactor( entityManager ):
   mg = entityManager.mountGroup()
   _asuData.reloadStatus = \
      mg.mount( Cell.path( 'sys/reload/status' ), 'System::Reload::Status', 'r' )
   _asuData.reloadConfig = \
      mg.mount( Cell.path( 'sys/reload/config' ), 'System::Reload::Config', 'fw' )

   def _finishMounts():
      t0( 'Creating AsuFastbootReloadReactor' )
      _asuData.asuFastbootReloadReactor = \
               AsuFastbootReloadReactor( _asuData.reloadConfig,
                     _asuData.reloadStatus, _asuData.schedConfig )

   mg.close( _finishMounts )

def Plugin( entityManager ):
   abootSbStatusPath = Cell.path( "aboot/sb/status" )
   asuShutdownEnabledConfigPath = Cell.path( "stageEnable/shutdown" )
   asuShutdownStatusPath = Cell.path( "stage/shutdown/status" )
   _asuData.abootSbStatus = LazyMount.mount( entityManager, abootSbStatusPath,
                                             "Aboot::Secureboot::Status", "r" )
   _asuData.asuHwStatus = LazyMount.mount( entityManager, "asu/hardware/status",
                                           "Asu::AsuStatus", "r" )
   _asuData.asuCliConfig = LazyMount.mount( entityManager, "asu/cli/config",
                                            "Asu::CliConfig", "w" )
   _asuData.asuShutDownEnabledConfig = LazyMount.mount( entityManager,
                                               asuShutdownEnabledConfigPath,
                                               "Stage::EnabledConfig", "w" )
   _asuData.asuShutDownStatus = LazyMount.mount( entityManager,
                                                 asuShutdownStatusPath,
                                                 "Stage::Status", "r" )
   _asuData.fatalError = LazyMount.mount( entityManager,
                               Cell.path( "fatalError/config/request/Asu-Cli" ),
                               "Stage::FatalError", "fcw" )
   _asuData.asuDmaCancelStatus = LazyMount.mount( entityManager,
                                                  "asu/status/dmaCancel",
                                                  "Asu::DmaCancelStatus", "r" )
   _asuData.asuDebugConfig = ConfigMount.mount( entityManager, "asu/debug/config",
                                              "Asu::DebugConfig", "w" )
   eventHistoryPersistentPath = Cell.path( "eventhistory/persistent" )
   _asuData.eventHistoryPersistentDir = LazyMount.mount( entityManager,
         eventHistoryPersistentPath, "Tac::Dir", "ri" )
   ehAsuPatchPath = Cell.path( "eventhistory/persistent/AsuPatch" )
   _asuData.ehAsuPatchDir = LazyMount.mount( entityManager, ehAsuPatchPath,
         "EventHistory::WriterDir", "wcf" )
   bootCompletionStatusDirPath = Cell.path( "stage/boot/completionstatus" )
   _asuData.bootCompletionStatusDir = LazyMount.mount( entityManager,
         bootCompletionStatusDirPath, "Stage::CompletionStatusDir", 'r' )
   redundancyStatusPath = Cell.path( "redundancy/status" )
   _asuData.redundancyStatus = LazyMount.mount( entityManager,
         redundancyStatusPath, 'Redundancy::RedundancyStatus', 'r' )
   _asuData.schedConfig = LazyMount.mount( entityManager,
         'sys/clischeduler/config', 'System::CliScheduler::Config', 'w' )
   _asuData.ipStatus = LazyMount.mount( entityManager,
                                        "ip/status", "Ip::Status", "r" )
   createFastbootReloadReactor( entityManager )
