# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliParser
import BasicCli
import CliCommand
import CliMatcher
import ShowCommand
import TacSigint
import sys
import AsuUtil

from CliPlugin.TechSupportCli import techSupportKwMatcher
from CliPlugin.AsuPStoreModel import ReloadHitlessDisruption
from CliPlugin.AsuPStoreModel import ReloadFastbootDisruption
from CliPlugin.AsuReloadCli import ( rebootHitlessGuard, asuInProgress, 
                                     asuInProgressMsg, getScheduledReloadInfo,
                                     _asuData )

def showFastbootFeatureDisruption( mode, args ):
   if asuInProgress():
      mode.addErrorAndStop( asuInProgressMsg )

   scheduledReloadInfo_ = getScheduledReloadInfo()
   ( warningList_, blockingList_ ) = \
            AsuUtil.doCheckHitlessReloadSupported( mode.entityManager )

   return ReloadFastbootDisruption( blockingList=blockingList_,
                                   warningList=warningList_,
                                   scheduledReloadInfo=scheduledReloadInfo_ )

def showHitlessFeatureDisruption( mode, args ):
   if asuInProgress():
      mode.addErrorAndStop( asuInProgressMsg )

   ( warningList_, blockingList_ ) = \
            AsuUtil.doCheckHitlessReloadSupported( mode.entityManager )

   return ReloadHitlessDisruption( blockingList=blockingList_,
                                   warningList=warningList_ )

def showTechSupportHitless( mode, args ):
   cmds = [ 'show rib ready vrf all',
            'show bgp convergence vrf all',
            'show boot stages log' ]
   for cmd in cmds:
      # pylint: disable-next=consider-using-f-string
      mode.addMessage( "\n------------ %s --------------\n" % cmd )
      sys.stdout.flush()
      try:
         mode.session_.runCmd( cmd, aaa=False )
      except CliParser.GuardError as e:
         # pylint: disable-next=consider-using-f-string
         mode.addError( "(unavailable: %s)" % e.guardCode )
      except CliParser.AlreadyHandledError:
         pass
      except KeyboardInterrupt:
         raise
      except: # pylint: disable=bare-except
         mode.addError( "(unavailable)" )
      TacSigint.check()

def showFastbootReloadPolicyCheck( mode, args ):
   policyCheckEnabled = not _asuData.asuDebugConfig.skipReloadPolicyBlock

   mode.addMessage( "Policy check is: " + ( "Enabled" if policyCheckEnabled
                                                      else "Disabled" ) )
   sys.stdout.flush()

# --------------------------------------------------------------------------------
# Common keyword matchers
# --------------------------------------------------------------------------------
reloadKwMatcherforShowCmd = CliMatcher.KeywordMatcher( "reload",
                                                       helpdesc="Display system "
                                                                "reload status" )

fastBootKwMatcherForShowCmd = CliCommand.guardedKeyword( "fast-boot",
                                            helpdesc="Check for feature disruption "
                                                     "from fast-boot reload",
                                            guard=rebootHitlessGuard )
#--------------------------------------------------------------------------------
# show reload hitless
#--------------------------------------------------------------------------------
class ShowHitlessFeatureDisruptionCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show reload hitless'
   data = {
      'reload' : reloadKwMatcherforShowCmd,
      'hitless' : CliCommand.guardedKeyword( "hitless",
                                          helpdesc="Check for feature disruption "
                                                   "from hitless reload",
                                          guard=rebootHitlessGuard ),
   }
   hidden = True
   handler = showHitlessFeatureDisruption
   cliModel = ReloadHitlessDisruption

BasicCli.addShowCommandClass( ShowHitlessFeatureDisruptionCmd )

#--------------------------------------------------------------------------------
# show reload fast-boot
#--------------------------------------------------------------------------------
class ShowFastbootFeatureDisruptionCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show reload fast-boot'
   data = {
      'reload' : reloadKwMatcherforShowCmd,
      'fast-boot': fastBootKwMatcherForShowCmd
   }
   handler = showFastbootFeatureDisruption
   cliModel = ReloadFastbootDisruption

BasicCli.addShowCommandClass( ShowFastbootFeatureDisruptionCmd )

#--------------------------------------------------------------------------------
# show tech-support reload-fast-boot
# show tech-support reload-hitless
# "reload-hitless" is an alternate of "reload-fast-boot"
#--------------------------------------------------------------------------------
class ShowTechSupportHitlessCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show tech-support reload-fast-boot'
   data = {
      'tech-support' : techSupportKwMatcher,
      'reload-fast-boot' : CliCommand.Node(
                              CliMatcher.KeywordMatcher( "reload-fast-boot",
                                              helpdesc="Show tech support about "
                                                       "reload fastboot",
                                              alternates=[ "reload-hitless" ] ),
                              guard=rebootHitlessGuard ),
   }
   handler = showTechSupportHitless

BasicCli.addShowCommandClass( ShowTechSupportHitlessCmd )

# --------------------------------------------------------------------------------
# show reload fast-boot debug policy-check
# --------------------------------------------------------------------------------
class ShowFastbootReloadPolicyCheckCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show reload fast-boot debug policy-check'
   data = {
      'reload': reloadKwMatcherforShowCmd,
      'fast-boot': fastBootKwMatcherForShowCmd,
      'debug': "Debug information",
      'policy-check': "Show if policy check is enabled",
   }

   hidden = True
   handler = showFastbootReloadPolicyCheck

BasicCli.addShowCommandClass( ShowFastbootReloadPolicyCheckCmd )
