#!/usr/bin/env python3
# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliMatcher
import CliCommand
from CliPlugin import IpAddrMatcher
import CliToken.Clear
from CliToken.AvtCliToken import (
   avtNodeForShow,
   pathMatcherForShow,
   detailMatcherForShow,
   appProfileMatcherForShow,
   AvtVrfExprFactory,
   AvtNameExprFactory,
   AvtDestinationExprFactory,
   AvtPathIdExprFactory,
   AvtAppExprFactory,
)
import Tac
import ShowCommand
from functools import partial
from Toggles.BgpCommonToggleLib import toggleEvpnL3DciWithDpsEnabled

IntfId = Tac.Type( "Arnet::IntfId" )

vrfFilterHelpStr = 'Filter by VRF name'
avtFilterHelpStr = 'Filter by adaptive virtual topology name'
destFilterHelpStr = 'Filter by destination'
pathIdFilterHelpStr = 'Filter by Path ID'
appFilterHelpStr = 'filter by appplication-profile name'

alphaNumericMatcher = partial( CliMatcher.PatternMatcher,
                               pattern='[a-zA-Z0-9_-]+',
                               helpname='WORD' )

# -----------------------------------------------------------------------------------
# show adaptive-virtual-topology path counters
#    [ vrf VRF ] [ avt AVT ] [ hop HOP ] [ path-id PATH-ID ]
# -----------------------------------------------------------------------------------
class ShowAvtPathCounters( ShowCommand.ShowCliCommandClass ):
   syntax = '''show adaptive-virtual-topology path counters
               [ VRF ] [ AVT ] [ hop HOP ] [ PATH-ID ]'''
   data = {
             'adaptive-virtual-topology': avtNodeForShow,
             'path': pathMatcherForShow,
             'counters': 'Show adaptive-virtual-topology counters',
             'VRF': AvtVrfExprFactory( vrfFilterHelpStr ),
             'AVT': AvtNameExprFactory( avtFilterHelpStr ),
             'hop': 'Show adaptive-virtual-topology counters filtered by '
                    'next hop VTEP',
             'HOP': IpAddrMatcher.IpAddrMatcher(
                       helpdesc="Ip address of the next hop VTEP" ),
             'PATH-ID': AvtPathIdExprFactory( pathIdFilterHelpStr ),
          }
   cliModel = "AvtCliModel.AvtCountersModel"
   handler = "AvtShowCliHandler.ShowAvtPathCountersHandler"

BasicCli.addShowCommandClass( ShowAvtPathCounters )

# -----------------------------------------------------------------------------------
# clear adaptive-virtual-topology path counters
# -----------------------------------------------------------------------------------
class ClearAvtPathCounters( CliCommand.CliCommandClass ):
   syntax = 'clear adaptive-virtual-topology path counters'
   data = {
             'clear': CliToken.Clear.clearKwNode,
             'adaptive-virtual-topology': avtNodeForShow,
             'path': pathMatcherForShow,
             'counters': 'Clear adaptive-virtual-topology counters',
          }

   handler = "AvtShowCliHandler.ClearAvtPathCountersHandler"

BasicCli.EnableMode.addCommandClass( ClearAvtPathCounters )

# -----------------------------------------------------------------------------------
# show adaptive-virtual-topology path [ vrf VRF [ avt AVT ] ] [ destination DEST ]
# -----------------------------------------------------------------------------------
class ShowAvtPaths( ShowCommand.ShowCliCommandClass ):
   syntax = '''show adaptive-virtual-topology path [ detail ]
               [ VRF [ AVT ] ] [ DEST ]'''
   data = {
      'adaptive-virtual-topology': avtNodeForShow,
      'path': pathMatcherForShow,
      'detail': detailMatcherForShow,
      'VRF': AvtVrfExprFactory( vrfFilterHelpStr ),
      'AVT': AvtNameExprFactory( avtFilterHelpStr ),
      'DEST': AvtDestinationExprFactory( destFilterHelpStr ),
   }

   cliModel = "AvtCliModel.AvtPathsModel"
   handler = "AvtShowCliHandler.ShowAvtPathsHandler"

BasicCli.addShowCommandClass( ShowAvtPaths )

# -----------------------------------------------------------------------------------
# show adaptive-virtual-topology application-profile
# -----------------------------------------------------------------------------------
class ShowAvtAppProfile( ShowCommand.ShowCliCommandClass ):
   syntax = ''' show adaptive-virtual-topology application-profile
                 [ VRF [ APPLICATION ] [ AVT ] ] '''

   data = {
      'adaptive-virtual-topology': avtNodeForShow,
      'application-profile': appProfileMatcherForShow,
      'VRF': AvtVrfExprFactory( vrfFilterHelpStr ),
      'APPLICATION': AvtAppExprFactory( appFilterHelpStr ),
      'AVT': AvtNameExprFactory( avtFilterHelpStr ),
   }

   cliModel = "AvtCliModel.AvtAppProfileModel"
   handler = "AvtShowCliHandler.ShowAvtAppProfileHandler"

BasicCli.addShowCommandClass( ShowAvtAppProfile )

# -----------------------------------------------------------------------------------
# show adaptive-virtual-topology next-hop tunnel encapsulation
#   [ path-selection | vxlan ]
# -----------------------------------------------------------------------------------
class ShowAvtTunnelEncap( ShowCommand.ShowCliCommandClass ):
   syntax = ''' show adaptive-virtual-topology next-hop tunnel encapsulation
                [ path-selection | vxlan ] '''

   data = {
      'adaptive-virtual-topology': avtNodeForShow,
      'next-hop': 'Next hop VTEP',
      'tunnel': 'Tunnel information',
      'encapsulation': 'Tunnel encapsulation type',
      'path-selection': 'Filter by dynamic path-selection (DPS) type',
      'vxlan': 'Filter by VXLAN type',
   }

   cliModel = "AvtCliModel.AvtNextHopModel"
   handler = "AvtShowCliHandler.ShowAvtTunnelEncapHandler"

if toggleEvpnL3DciWithDpsEnabled():
   BasicCli.addShowCommandClass( ShowAvtTunnelEncap )
