# Copyright (c) 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Ark
import BasicCli
import CliCommand
import CliGlobal
import CliMatcher
import CliParser
import CliPlugin.FruCli as FruCli # pylint: disable=consider-using-from-import
from CliPlugin import LedCommon
from CliPlugin.EthIntfCli import EthPhyAutoIntfType
from CliPlugin.FabricIntfCli import FabricIntfRangePlugin
import CommonGuards
import Intf.IntfRange
import ShowCommand
import Tac
from GenericReactor import GenericReactor

FabricAutoIntfType = FabricIntfRangePlugin.fabricAutoIntfType

verbose = False

beaconGlobal = CliGlobal.CliGlobal( dict( beaconMonitor=None ) )
ledConfig = None
hwLedConfig = None
# set to either "Fixed" or "Modular"
systemType = None
redundancyStatus = None
redundancyStatusReactor = None

plutoStatus = Tac.Type( "Led::PlutoStatus" )
beaconSetName = "beacon"

class BeaconMonitor:
   beaconCbs = []
   lock = None

   def __init__( self ):
      self.beacon = None
      self.clock = {}

   @Ark.synchronized( lock )
   def startMonitor( self, ledName ):
      if not self.beacon:
         self.beacon = LedCommon.getLedSet( beaconSetName, ledConfig )
      ledSetting = self.beacon.led.get( ledName )
      if not ( ledSetting and ledSetting.turnOffTime ):
         return
      self.clock[ ledName ] = Tac.ClockNotifiee( lambda: _setLed( ledName, True ),
                                                 ledSetting.turnOffTime )

   @Ark.synchronized( lock )
   def stopMonitor( self, ledName ):
      if ledName not in self.clock:
         return
      self.clock[ ledName ].timeMin = Tac.endOfTime
      del self.clock[ ledName ]

   def scanForTimedBeacons( self ):
      self.clock.clear()
      for f in self.beaconCbs:
         f.__func__()

# All Command classes that use beaconMonitor should register a static method
# by annotating the method with this decorator
def monitor( func ):
   assert isinstance( func, staticmethod )
   BeaconMonitor.beaconCbs.append( func )

# This method starts beaconMonitor once ConfigAgent finishes its mounts
def startBeaconMonitor( notifiee=None ):
   if not beaconGlobal.beaconMonitor:
      beaconGlobal.beaconMonitor = BeaconMonitor()
   if redundancyStatus and redundancyStatus.mode == 'active':
      beaconGlobal.beaconMonitor.scanForTimedBeacons()

def _registerSystemTypeFixed():
   global systemType
   systemType = "Fixed"
FruCli.registerFixedSystemCallback( _registerSystemTypeFixed )

def _registerSystemTypeModular():
   global systemType
   systemType = "Modular"
FruCli.registerModularSystemCallback( _registerSystemTypeModular )

# -----------------------------------------------------------------------------
# [no] locator-led { module { Supervisor <#> | Linecard <#> | Fabric <#> |
#                             Switchcard | <#> } |
#                    interface <interface range> |
#                    powersupply <powersupply number> |
#                    fan <fan number> |
#                    chassis |
#                    multifan |
#                    multipowersupply }
# -----------------------------------------------------------------------------

def _setLed( led, disable, duration=0 ):
   assert hwLedConfig != None # pylint: disable=singleton-comparison
   assert ledConfig != None # pylint: disable=singleton-comparison

   ledExists, led = LedCommon.checkLedExists( led, hwLedConfig )
   
   if not ledExists:
      print( "This LED cannot be used as locator-led target" )
      return

   beacon = LedCommon.getLedSet( beaconSetName, ledConfig )
   ledShowName = LedCommon.ledToShowName( led, systemType )

   if disable:
      if led in beacon.led:
         print( "Disabling locator led for", ledShowName )
         del beacon.led[ led ]
      else:
         print( "Locator led for", ledShowName, "already disabled" )
   else:
      ledSetting = beacon.led.get( led )
      # duration and ledSetting.turnOffTime will be same only if
      # duration is 0 now and before the command was run.
      if ledSetting and duration == ledSetting.turnOffTime:
         print( "Locator led for", ledShowName, "already enabled" )
         return

      ledSetting = Tac.Value( "Led::LightSetting" )
      ledSetting.rampRate = 0.0104
      ledSetting.maxBright = 100
      ledSetting.flashRate = 1
      ledSetting.turnOffTime = 0 if not duration else Tac.now() + duration
      ledSetting.ledFlashRate = 1
      ledSetting.plutoStatus = plutoStatus.plutoStatusBeacon
      # Check if Led supports blue beacon, if so, set to blue.  Otherwise,
      # beacons for interfaces are flashing yellow, while for modules
      # we use flashing red.
      LedColor = Tac.Type( "Led::Color" )
      hwLed = hwLedConfig.leds[ led ]
      if hwLed.ledAttribute and hwLed.ledAttribute.blue:
         ledSetting.blue = True
         ledSetting.color = LedColor.BLUE
      elif led.startswith( 'Ethernet' ):
         ledSetting.yellow = True
         ledSetting.color = LedColor.YELLOW
      else:
         ledSetting.red = True
         ledSetting.color = LedColor.RED
      if duration:
         # pylint: disable-next=consider-using-f-string
         print( "Enabling locator led for {}{}".format(
               # pylint: disable-next=consider-using-f-string
               ledShowName, " for %d seconds" % duration ) )
      else:
         print( "Enabling locator led for", ledShowName )
      beacon.led[ led ] = ledSetting

def _targetToLed( target ):
   if verbose:
      print( "Target tag: ", target.tag )
      print( "Target label: ", target.label )

   for prefix in ( 'Supervisor', 'Linecard', 'Fabric', 'Switchcard', 'PowerSupply',
                   'Fan' ):
      if target.tag.startswith( prefix ):
         if prefix == 'Supervisor':
            prefix = "Status"
         return f"{prefix}{target.label}"

   return None

def doLocatorLedTarget( mode, noOption=None, target=None ):
   assert target != None # pylint: disable=singleton-comparison
   if verbose:
      print( "NoOption: ", noOption )
      print( "Target: ", target )
      print( "Name :", target.name )

   _setLed( _targetToLed( target ), noOption )

def ledsStartingWith( prefix: str ):
   return ( led for led in hwLedConfig.leds if led.startswith( prefix ) )

def guardHasLedStartingWith( prefix: str ):
   if any( ledsStartingWith( prefix ) ):
      return None
   else:
      return CliParser.guardNotThisPlatform

def guardPowerSupplyBeacon( mode, token ):
   '''Only show `powersupply` command on systems with individual LEDs per PSU'''
   return guardHasLedStartingWith( "PowerSupply" )

def guardMultiPowerSupplyBeacon( mode, token ):
   '''Only show `multipowersupply` on systems with single MultiPowerSupply LED'''
   return guardHasLedStartingWith( "MultiPowerSupply" )

def guardMultiFanBeacon( mode, token ):
   '''Only show `multifan` on systems with MultiFan LED'''
   return guardHasLedStartingWith( "MultiFan" )

def guardFantrayBeacon( mode, token ):
   '''Only show `fantray` on systems with Fan LED'''
   return guardHasLedStartingWith( "Fan" )

nodeLocatorLed = CliCommand.guardedKeyword( 'locator-led',
      helpdesc='Set a specific LED to flash',
      guard=CommonGuards.ssoStandbyGuard )

#--------------------------------------------------------------------------------
# [ no ] locator-led fantray FAN-TRAY
#--------------------------------------------------------------------------------
class LocatorLedFantrayCmd( CliCommand.CliCommandClass ):
   syntax = 'locator-led fantray FAN_TRAY'
   noSyntax = syntax
   data = {
      'locator-led' : nodeLocatorLed,
      'fantray': CliCommand.guardedKeyword( 'fantray',
         helpdesc='Fan tray LED', guard=guardFantrayBeacon ),
      'FAN_TRAY' : FruCli.FanTrayMatcher(),

   }

   @staticmethod
   def handler( mode, args ):
      # The FanTrayMatcher will return the actual object that we want, making this
      # method quite concise. In fact, it's the exact same method as before.
      fanName = args.get( 'FAN_TRAY' )
      _setLed( fanName, CliCommand.isNoCmd( args ) )


   noHandler = handler

BasicCli.EnableMode.addCommandClass( LocatorLedFantrayCmd )

# --------------------------------------------------------------------------------
# [ no ] locator-led powersupply POWER_SUPPLY
# --------------------------------------------------------------------------------
class LocatorLedPowerSupplyCmd( CliCommand.CliCommandClass ):
   syntax = 'locator-led powersupply POWER_SUPPLY'
   noSyntax = syntax
   data = {
      'locator-led': nodeLocatorLed,
      'powersupply': CliCommand.guardedKeyword( 'powersupply',
         helpdesc='Power supply LED', guard=guardPowerSupplyBeacon ),
      'POWER_SUPPLY' : FruCli.PowerSupplyMatcher(),
   }

   @staticmethod
   def handler( mode, args ):
      target = args.get( 'POWER_SUPPLY' )
      doLocatorLedTarget( mode, CliCommand.isNoCmd( args ), target=target )

   noHandler = handler

BasicCli.EnableMode.addCommandClass( LocatorLedPowerSupplyCmd )

# --------------------------------------------------------------------------------
# [ no ] locator-led multipowersupply
# --------------------------------------------------------------------------------
class LocatorLedMultiPowerSupplyCmd( CliCommand.CliCommandClass ):
   syntax = 'locator-led multipowersupply'
   noSyntax = syntax
   data = {
      'locator-led': nodeLocatorLed,
      'multipowersupply': CliCommand.guardedKeyword( 'multipowersupply',
         helpdesc='Multi power supply LED', guard=guardMultiPowerSupplyBeacon ),
   }

   @staticmethod
   def handler( mode, args ):
      # This handles variation in PSU LED naming schemes (MultiPowerSupply1,
      # MultiPowerSupplyStatus1, etc.) and enables / disables them all. There
      # is typically one on a given system, but some like NewportPrime have
      # two (front and back) that must both be handled.
      ledNames = ledsStartingWith( "MultiPowerSupply" )
      for ledName in ledNames:
         _setLed( ledName, CliCommand.isNoCmd( args ) )

   noHandler = handler

BasicCli.EnableMode.addCommandClass( LocatorLedMultiPowerSupplyCmd )

#--------------------------------------------------------------------------------
# [ no ] locator-led multifan
#--------------------------------------------------------------------------------
class LocatorLedMultifanCmd( CliCommand.CliCommandClass ):
   syntax = 'locator-led multifan'
   noSyntax = syntax
   data = {
      'locator-led' : nodeLocatorLed,
      'multifan' : CliCommand.guardedKeyword( 'multifan', helpdesc='Multi fan LED',
         guard=guardMultiFanBeacon ),
   }

   @staticmethod
   def handler( mode, args ):
      # Enable / disable all multifan LEDs. While most systems have at most one,
      # some, like NewportPrime, have two (front and back).
      ledNames = ledsStartingWith( "MultiFan" )
      for ledName in ledNames:
         _setLed( ledName, CliCommand.isNoCmd( args ) )

   noHandler = handler

BasicCli.EnableMode.addCommandClass( LocatorLedMultifanCmd )

#--------------------------------------------------------------------------------
# [ no ] locator-led interface INTFS
#--------------------------------------------------------------------------------
class LocatorLedInterfaceIntf1Cmd( CliCommand.CliCommandClass ):
   syntax = 'locator-led interface INTFS'
   noSyntax = syntax
   data = {
      'locator-led' : nodeLocatorLed,
      'interface' : 'Interface LED',
      'INTFS' : Intf.IntfRange.IntfRangeMatcher(
         explicitIntfTypes=( EthPhyAutoIntfType, FabricAutoIntfType ) ),
   }

   @staticmethod
   def handler( mode, args ):
      no = CliCommand.isNoCmd( args )
      for led in args[ 'INTFS' ]:
         _setLed( led, no )

   noHandler = handler

BasicCli.EnableMode.addCommandClass( LocatorLedInterfaceIntf1Cmd )

#--------------------------------------------------------------------------------
# [ no ] locator-led module SLOT
#--------------------------------------------------------------------------------
class LocatorLedModuleCmd( CliCommand.CliCommandClass ):
   syntax = 'locator-led module SLOT'
   noSyntax = syntax
   data = {
      'locator-led' : nodeLocatorLed,
      'module' : CliCommand.guardedKeyword( 'module', helpdesc='Module LED',
         guard=FruCli.modularSystemGuard ),
      'SLOT' : FruCli.SlotExpressionFactory()
   }

   @staticmethod
   def handler( mode, args ):
      target = args[ 'SLOT' ].slot
      doLocatorLedTarget( mode, CliCommand.isNoCmd( args ), target=target )

   noHandler = handler

BasicCli.EnableMode.addCommandClass( LocatorLedModuleCmd )

# -----------------------------------------------------------------------------
# [ no ] locator-led chassis
# 'locator-led chassis' command is only available on FixedSystems that support
# a blue status Led.  The guard for this command just looks up the capability
# of the 'Status1' led and see if it supports blue.  Older platform (Napa)
# are not supported.
# -----------------------------------------------------------------------------
def guardChassisBeacon( mode, token ):
   if FruCli.fixedSystemGuard( mode, token ):
      return CliParser.guardNotThisPlatform

   for hwLed in hwLedConfig.leds.values():
      if hwLed.name.startswith( 'Status' ):
         if hwLed.ledAttribute.blue:
            return None
   return CliParser.guardNotThisPlatform

class LocatorLedChassisCmd( CliCommand.CliCommandClass ):
   syntax = 'locator-led chassis [ duration SECONDS ]'
   noSyntax = 'locator-led chassis'
   data = {
      'locator-led' : nodeLocatorLed,
      'chassis': CliCommand.guardedKeyword( 'chassis', 'Chassis beacon LED',
         guard=guardChassisBeacon ),
      'duration': CliCommand.singleKeyword( 'duration',
         helpdesc='Duration for which LED should be turned on' ),
      'SECONDS': CliMatcher.IntegerMatcher( 1, 4294967295, helpdesc='seconds' )
   }

   @staticmethod
   def handler( mode, args ):
      disable = CliCommand.isNoCmd( args )
      duration = args.get( 'SECONDS', 0 )
      for led in hwLedConfig.leds:
         if led.startswith( 'Status' ):
            if disable:
               _setLed( led, disable )
               beaconGlobal.beaconMonitor.stopMonitor( led )
            else:
               _setLed( led, disable, duration )
               if duration:
                  beaconGlobal.beaconMonitor.startMonitor( led )
               else:
                  beaconGlobal.beaconMonitor.stopMonitor( led )

   noHandler = handler

   @monitor
   @staticmethod
   def _scanForTimedLeds():
      # This method is a callback for beaconMonitor during ConfigAgent init to
      # restart monitoring of 'chassis' LEDs from previous life, if any.
      for led in hwLedConfig.leds:
         if led.startswith( 'Status' ):
            beaconGlobal.beaconMonitor.startMonitor( led )

BasicCli.EnableMode.addCommandClass( LocatorLedChassisCmd )

# -----------------------------------------------------------------------------
# show locator-led
#
# Shows the list of currently enabled locator-led (aka beacon led)
# -----------------------------------------------------------------------------
class ShowLocatorLedCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show locator-led'
   data = {
      'locator-led' : CliCommand.guardedKeyword( 'locator-led',
         helpdesc='Show specific LED to flash',
         guard=CommonGuards.ssoStandbyGuard ),
   }

   @staticmethod
   def handler( mode, args ):
      beacon = LedCommon.getLedSet( beaconSetName, ledConfig )

      if beacon.led:
         print( "Locator LED currently enabled" )
         print( "-----------------------------" )
         # Print chassis only once for FixedSystems
         chassisLeds = [ led for led in beacon.led if led.startswith( 'Status' ) ]
         if chassisLeds:
            if systemType == "Fixed":
               print( LedCommon.ledToShowName( chassisLeds[ 0 ], systemType ) )
            else:
               for led in sorted( chassisLeds ):
                  print( LedCommon.ledToShowName( led, systemType ) )
         otherLeds = [ led for led in beacon.led if led not in chassisLeds ]
         for led in sorted( otherLeds ):
            print( LedCommon.ledToShowName( led, systemType ) )
      else:
         print( "There are no locator LED enabled" )

BasicCli.addShowCommandClass( ShowLocatorLedCmd )

# -----------------------------------------------------------------------------
# Cli Plugin initialization
# -----------------------------------------------------------------------------

def Plugin( entityManager ):
   global hwLedConfig, ledConfig, redundancyStatus, redundancyStatusReactor

   redundancyStatus = entityManager.redundancyStatus()
   redundancyStatusReactor = GenericReactor( redundancyStatus, [ "mode" ],
                                             startBeaconMonitor )

   mg = entityManager.mountGroup()
   hwLedConfig = mg.mount( "hardware/led/configInit",
                           "Hardware::Led::LedSystemConfigDir", "r" )
   ledConfig = mg.mount( "led/config", "Tac::Dir", "iw" )
   mg.close( startBeaconMonitor )
