#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing
import ConfigMount
import LazyMount
from CliPlugin import RouterMulticastCliLib
from McastCommonCliLib import AddressFamily
import CliCommand
import CliMatcher
import CliParser
import Cell

# Bogus import to generate dependency on Sfe-cli
# Needed for "show platform sfe mroute" command in SfeMcastShowCli.py
# pkgdeps: import CliPlugin.SfeMcastShowCli (forced dependency, see AID 10)

traceHandle = Tracing.Handle( "BessMgrCli" )
t0 = traceHandle.trace0

bessAgentStatus = None
bessAgentConfig = None
phyConfigDir = None

FwdType = Tac.Type( 'McastCommon::BessAgentConfig::Forwarding' )

def hwPlatformGuard( mode, token ):
   pcdPath = f'hardware/cell/{Cell.cellId()}/phy/sfe/config'
   pcd = phyConfigDir
   if pcd is None:
      pcd = mode.sysdbRoot.entity[ pcdPath ]
   if not pcd.isSfeSecondaryForwardingAgent:
      return CliParser.guardNotThisPlatform
   else:
      return None

class ForwardingType( CliCommand.CliCommandClass ):
   syntax = 'software-forwarding FORWARDING_TYPE [ force ]'
   data = {
      'software-forwarding' : CliCommand.Node(
         CliMatcher.KeywordMatcher( 'software-forwarding',
            helpdesc='Multicast packet software forwarding configuration' ),
         guard=hwPlatformGuard ),
      'FORWARDING_TYPE' : CliMatcher.EnumMatcher( {
         'kernel' : 'Software forwarding by the Linux kernel',
         'sfe' : 'Software forwarding by SFE (Software Forwarding Engine)'
      } ),
      'force' : CliCommand.Node(
         CliMatcher.KeywordMatcher( 'force',
            helpdesc='Force software forwarding mode without switch reboot' ),
         hidden=True )
   }

   @staticmethod
   def handler( mode, args ):
      def warn( x ):
         mode.addWarning( f"Change {x} will take effect only after switch reboot" )
      forwardingType = args[ 'FORWARDING_TYPE' ]
      af = RouterMulticastCliLib.getAddressFamilyFromMode( mode )
      bessEnabled = forwardingType.title() == 'Sfe'
      fwdType = FwdType.sfe if bessEnabled else FwdType.kernel
      force = 'force' in args
      if af == AddressFamily.ipv6:
         bessAgentConfig.v6Forwarding = fwdType
         if mode.session.startupConfig() or force:
            bessAgentStatus.v6Enabled = bessEnabled
         elif bessAgentStatus.v6Enabled ^ bessEnabled:
            warn( fwdType )
      else:
         bessAgentConfig.v4Forwarding = fwdType
         if mode.session.startupConfig() or force:
            bessAgentStatus.v4Enabled = bessEnabled
         elif bessAgentStatus.v4Enabled ^ bessEnabled:
            warn( fwdType )

RouterMulticastCliLib.RouterMulticastNonVrfAfModelet.addCommandClass(
   ForwardingType )

def Plugin( entityManager ):
   global bessAgentStatus
   global bessAgentConfig
   global phyConfigDir
   bessAgentStatus = LazyMount.mount( entityManager,
      'bess/agentstatus/cli', 'McastCommon::BessAgentStatus', 'w' )
   bessAgentConfig = ConfigMount.mount( entityManager,
      'bess/config', 'McastCommon::BessAgentConfig', 'wi' )
   phyConfigDir = LazyMount.mount( entityManager,
         f'hardware/cell/{Cell.cellId()}/phy/sfe/config',
         'Hardware::Phy::SfePhyConfigDir', 'r' )
