# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliGlobal
import CliParser
import CliMatcher
import ConfigMount
import LazyMount
import Tac
from CliPlugin.VrfCli import DEFAULT_VRF, VrfNameExprFactory, vrfExists
from Toggles.BfdToggleLib import (
   toggleBfdConditionallyRunnableEnabled
)

perlinkMode = Tac.Type( "Bfd::PerlinkMode" )
sessionType = Tac.Type( "Bfd::SessionType" )
operSessionType = Tac.Type( "Bfd::OperSessionType" )
historyInfo = Tac.Value( "Bfd::HistoryInfo" )
operState = Tac.Type( "Bfd::OperState" )
authType = Tac.Type( 'Bfd::BfdAuthType' )
routedPortWarning = {}
existingSessionWarning = {}
perlinkOnWarning = {}
rfc7130SupportedWarning = {}
rfc7130BfdNeighborWarning = {}

# Global variable holder.
gv = CliGlobal.CliGlobal(
   dict(
      authSharedSecretConfig=None,
      routingConfig=None,
      routing6Config=None,
      testAppConfigDir=None,
      launcherControlDir=None,
   )
)

matcherAuthProfile = CliMatcher.DynamicNameMatcher( ( lambda mode:
   gv.authSharedSecretConfig.profile ), 'Shared-secret profile name' )
vrfNameExprFactory = VrfNameExprFactory( inclAllVrf=False, inclDefaultVrf=True )

def getBfdTestAppVrfAppConfig( testAppDir, vrf ):
   # pylint: disable-next=consider-using-f-string
   testAppVrfDir = testAppDir.newEntity( 'Tac::Dir', '%s' % vrf )
   testCfg = testAppVrfDir.newEntity( 'Bfd::AppConfig', 'test' )
   testCfg.appPid = 1
   return testCfg

def delPeer( peer, testCfg ):
   if peer in testCfg.peerConfigVer:
      del testCfg.peerConfigVer[ peer ]

def addPeer( peer, testCfg ):
   if peer not in testCfg.peerConfigVer:
      testCfg.newPeerConfigVer( peer, 1 )

def addOrRemoveBfdSession( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   peerAddr = args.get( 'PEERADDR' )
   intf = args.get( 'INTF' )
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   srcAddr = args.get( 'SRCADDR' )
   tunnelId = args.get( 'TUNNEL' )
   minTx = args.get( 'INTERVAL' )
   minRx = args.get( 'MIN_RX' )
   mult = args.get( 'MULTIPLIER' )
   sbfdEchoRttEnabled = 'rtt-echo-enabled' in args
   proxy = 'proxy' in args

   if not vrfExists( vrfName ):
      # pylint: disable-next=consider-using-f-string
      mode.addError( "VRF %s not configured." % vrfName )
      return

   testCfgDir = ConfigMount.force( gv.testAppConfigDir )
   testCfg = getBfdTestAppVrfAppConfig( testCfgDir, vrfName )
   add = not no
   if testCfg is None:
      print( 'Unable to mount test AppConfig!' )
      return
   # pylint: disable-next=consider-using-f-string
   addr = Tac.Value( 'Arnet::IpGenAddr', '%s' % peerAddr )
   peer = Tac.Value( 'Bfd::Peer', addr, vrfName )
   if intf:
      intf = Tac.Value( 'Arnet::IntfId', intf.name )
      peer.intf = intf

   if srcAddr:
      # pylint: disable-next=consider-using-f-string
      srcAddr = Tac.Value( 'Arnet::IpGenAddr', '%s' % srcAddr )
      peer.srcIp = srcAddr
      peer.type = sessionType.multihop
   elif tunnelId:
      peer.type = sessionType.sbfdInitiator
      peer.tunnelId = tunnelId

   if add:
      addPeer( peer, testCfg )
      if toggleBfdConditionallyRunnableEnabled():
         gv.launcherControlDir.newEntity( 'Tac::Dir', f'test-{peer.stringValue()}' )
      if peer.type == sessionType.sbfdInitiator:
         cfg = Tac.Value( 'Bfd::InitiatorConfig' )
         if minTx and mult:
            cfg.timer = Tac.Value( 'Bfd::InitiatorTimerConfig', minTx, mult )
         cfg.sbfdEchoRttEnabled = sbfdEchoRttEnabled
         cfg.sbfdProxyEnabled = proxy
         testCfg.peerToInitiatorConfigMap[ peer ] = cfg
      elif minRx:
         testCfg.peerConfigVer[ peer ].maybeIntervalConfig = Tac.Value(
               'Bfd::BfdIntervalConfig', minTx, minRx, mult )
      else:
         testCfg.peerConfigVer[ peer ].maybeIntervalConfig = None
   else:
      delPeer( peer, testCfg )
      del testCfg.peerToInitiatorConfigMap[ peer ]
      if toggleBfdConditionallyRunnableEnabled():
         gv.launcherControlDir.deleteEntity( f'test-{peer.stringValue()}' )

def getAuthModeFromArgs( args ):
   if 'simple' in args:
      return authType.authPassword
   elif 'md5' in args:
      if 'meticulous' in args:
         return authType.authMeticulousMd5
      else:
         return authType.authKeyedMd5
   elif 'sha1' in args:
      if 'meticulous' in args:
         return authType.authMeticulousSha1
      else:
         return authType.authKeyedSha1
   elif 'disabled' in args:
      return authType.authNone
   return None

class BfdIpIntfConfigModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.routingSupported()

def Plugin( entityManager ):
   gv.authSharedSecretConfig = LazyMount.mount( entityManager,
                                   'mgmt/security/sh-sec-prof/config',
                                   'Mgmt::Security::SharedSecretProfile::Config',
                                   'r' )
   gv.routingConfig = LazyMount.mount( entityManager, 'routing/config',
                                       'Routing::Config', 'r' )
   gv.routing6Config = LazyMount.mount( entityManager, 'routing6/config',
                                        'Routing6::Config', 'r' )
   gv.testAppConfigDir = ConfigMount.mount( entityManager,
                                            'bfd/config/app/test',
                                            'Tac::Dir', 'wic' )
   if toggleBfdConditionallyRunnableEnabled():
      gv.launcherControlDir = LazyMount.mount( entityManager, 'bfd/launcherControl',
                                          'Tac::Dir', 'wi' )
