# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import LazyMount
import CliCommand
import CliGlobal
import CliPlugin.IntfCli as IntfCli # pylint: disable=consider-using-from-import
from CliPlugin.BfdCli import (
      BfdIpIntfConfigModelet,
      getAuthModeFromArgs,
      matcherAuthProfile,
)
from CliPlugin.IpGenAddrMatcher import IpGenAddrMatcher
from CliToken.Bfd import (
      matcherBfd,
      matcherInterval,
      matcherMeticulous,
      matcherMinRx,
      matcherMinRxDeprecated,
      matcherMultihop,
      matcherMultiplier,
      matcherMultiplierVal,
      matcherTxRxIntervalMs,
)
import Tac
import ConfigMount
from TypeFuture import TacLazyType

from Toggles.BfdToggleLib import (
   toggleBfdStaticMultihopEnabled,
   toggleBfdConditionallyRunnableEnabled
 )

# Global variable holder.
gv = CliGlobal.CliGlobal(
   dict(
      bfdConfigIntf=None,
      bfdConfigGlobal=None,
      staticSessionRequests=None,
      authSharedSecretConfig=None,
      bfdLauncherControl=None,
   )
)

AuthType = TacLazyType( 'Bfd::BfdAuthType' )
SessionType = TacLazyType( "Bfd::SessionType" )

class BfdIpIntf ( IntfCli.IntfDependentBase ):
   def __init__( self, intf, sysdbRoot, createIfMissing=True ):
      IntfCli.IntfDependentBase.__init__( self, intf, sysdbRoot )

   def setDefault( self ):
      intf = IntfId( self.intf_.name )
      if intf in gv.bfdConfigIntf.intfConfig:
         del gv.bfdConfigIntf.intfConfig[ intf ]
      if intf in gv.bfdConfigIntf.echoOn:
         del gv.bfdConfigIntf.echoOn[ intf ]
      if intf in gv.bfdConfigIntf.authType:
         del gv.bfdConfigIntf.authType[ intf ]
      if intf in gv.staticSessionRequests.peerIpsByIntf:
         del gv.staticSessionRequests.peerIpsByIntf[ intf ]

routedPortWarning = {}
existingSessionWarning = {}
perlinkOnWarning = {}
rfc7130SupportedWarning = {}
rfc7130BfdNeighborWarning = {}

IntfCli.IntfConfigMode.addModelet( BfdIpIntfConfigModelet )
modelet = BfdIpIntfConfigModelet

# Interface mode config commands
# ------------------------------------------------------------------------------
# "[no|default] bfd interval <50-60000> min-rx <50-60000> multiplier <3-50>"

# "[no|default] bfd interval <50-60000> min_rx <50-60000> multiplier <3-50>"
# command in config-if mode (hidden command)
# "[no|default] bfd interval <50-60000> min_rx <50-60000> multiplier <3-50>
#  default" in config mode (hidden command)
# ------------------------------------------------------------------------------

def IntfId( intfName ):
   return Tac.Value( 'Arnet::IntfId', intfName )

def BfdIntervalConfig( minTx, minRx, mult ):
   return Tac.Value( 'Bfd::BfdIntervalConfig', minTx, minRx, mult )

def intfConfig( mode ):
   intf = IntfId( mode.intf.name )
   return gv.bfdConfigIntf.intfConfig.newMember( intf )

def setIntfBfdParams( mode, args ):
   doSetIntfBfdParams( mode,
                       args[ 'INTERVAL' ],
                       args[ 'MIN_RX' ],
                       args[ 'MULTIPLIER' ],
                       legacy=( 'min_rx' in args ) )

def doSetIntfBfdParams( mode, bfdTxInt, bfdRxInt, bfdMult, legacy=False ):
   # If a non-legacy command was used to configure this attribute, flip legacyConfig
   # in Bfd::ConfigGlobal to False, but only if it has not yet been transitioned.  We
   # _never_ want to set this back to True.
   gv.bfdConfigGlobal.legacyConfig = ( legacy if gv.bfdConfigGlobal.legacyConfig
                                       else False )
   intf = IntfId( mode.intf.name )
   gv.bfdConfigIntf.intfConfig[ intf ] = (
      BfdIntervalConfig( bfdTxInt, bfdRxInt, bfdMult ) )

def resetIntfBfdParams( mode, args ):
   intf = IntfId( mode.intf.name )
   del gv.bfdConfigIntf.intfConfig[ intf ]

class BfdIntfParamsCmd( CliCommand.CliCommandClass ):
   syntax = 'bfd interval INTERVAL ( min-rx | min_rx ) MIN_RX multiplier MULTIPLIER'
   noOrDefaultSyntax = 'bfd interval ...'
   data = {
      'bfd': matcherBfd,
      'interval': matcherInterval,
      'INTERVAL': matcherTxRxIntervalMs,
      'min-rx': matcherMinRx,
      'min_rx': CliCommand.Node( matcherMinRxDeprecated, hidden=True ),
      'MIN_RX': matcherTxRxIntervalMs,
      'multiplier': matcherMultiplier,
      'MULTIPLIER': matcherMultiplierVal,
   }
   handler = setIntfBfdParams
   noOrDefaultHandler = resetIntfBfdParams

modelet.addCommandClass( BfdIntfParamsCmd )

# ------------------------------------------------------------------------------
# "[no|default] bfd echo"
# command in config-if mode
# ------------------------------------------------------------------------------
def echoFunctionCmd( mode, no=False ):
   intf = IntfId( mode.intf.name )
   if no:
      if intf in gv.bfdConfigIntf.echoOn:
         del gv.bfdConfigIntf.echoOn[ intf ]
   else:
      gv.bfdConfigIntf.echoOn[ intf ] = True

class BfdIntfEchoCmd( CliCommand.CliCommandClass ):
   syntax = 'bfd echo'
   noOrDefaultSyntax = syntax
   data = {
      'bfd': matcherBfd,
      'echo': 'Echo function'
   }

   @staticmethod
   def handler( mode, args ):
      echoFunctionCmd( mode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      echoFunctionCmd( mode, no=True )

modelet.addCommandClass( BfdIntfEchoCmd )

# ------------------------------------------------------------------------------
# "[no|default] bfd static [multihop] neighbor" command in config-if mode
# ------------------------------------------------------------------------------
def staticNeighborFunctionCmd( mode, peerIp, multihop, disable=False ):
   intfName = mode.intf.name
   sessionType = SessionType.multihop if multihop else SessionType.normal

   if disable:
      if intfName in gv.staticSessionRequests.peerIpsByIntf:
         peerIpsByIntf = gv.staticSessionRequests.peerIpsByIntf[ intfName ]
         if peerIp:
            # Remove single peer of the passed type for this intf
            if peerIp in peerIpsByIntf.typesByPeerIp:
               del peerIpsByIntf.typesByPeerIp[ peerIp ].sessionTypes[ sessionType ]

               # Clean up the entry for this peerIp if we deleted the last
               # sessionType entry
               if not peerIpsByIntf.typesByPeerIp[ peerIp ].sessionTypes:
                  del peerIpsByIntf.typesByPeerIp[ peerIp ]
         else:
            # Implicit delete all peers of the passed type for this intf
            for ip in peerIpsByIntf.typesByPeerIp:
               del peerIpsByIntf.typesByPeerIp[ ip ].sessionTypes[ sessionType ]

               # Clean up the entry for this peerIp if we deleted the last
               # sessionType entry
               if not peerIpsByIntf.typesByPeerIp[ ip ].sessionTypes:
                  del peerIpsByIntf.typesByPeerIp[ ip ]

         # If we deleted the last peerIp entry for this intfName, then delete the
         # intfName entry as well.
         if not gv.staticSessionRequests.peerIpsByIntf[ intfName ].typesByPeerIp:
            del gv.staticSessionRequests.peerIpsByIntf[ intfName ]
            if toggleBfdConditionallyRunnableEnabled():
               gv.bfdLauncherControl.deleteEntity( "static-route-" + intfName )

   else: # Adding new config
      peerIpsByIntf = gv.staticSessionRequests.peerIpsByIntf.newMember( intfName )
      typesByPeerIp = peerIpsByIntf.typesByPeerIp.newMember( peerIp )
      typesByPeerIp.sessionTypes.add( sessionType )
      if toggleBfdConditionallyRunnableEnabled():
         gv.bfdLauncherControl.newEntity( "Tac::Dir", "static-route-" + intfName )

class BfdIntfStaticNeighborCmd( CliCommand.CliCommandClass ):
   syntax = 'bfd static neighbor PEERADDR'
   noOrDefaultSyntax = 'bfd static neighbor [ PEERADDR ]'
   if toggleBfdStaticMultihopEnabled():
      syntax = 'bfd static [ multihop ] neighbor PEERADDR'
      noOrDefaultSyntax = 'bfd static [ multihop ] neighbor [ PEERADDR ]'

   data = {
      'bfd': matcherBfd,
      'static': 'Configure static BFD',
      'multihop': matcherMultihop,
      'neighbor': 'Add a single-hop static BFD neighbor',
      'PEERADDR': IpGenAddrMatcher( helpdesc='IPv4/IPv6 neighbor address' ),
   }

   @staticmethod
   def handler( mode, args ):
      staticNeighborFunctionCmd( mode,
                                 args[ 'PEERADDR' ],
                                 ( 'multihop' in args ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      staticNeighborFunctionCmd( mode,
                                 args.get( 'PEERADDR' ),
                                 ( 'multihop' in args ),
                                 disable=True )

modelet.addCommandClass( BfdIntfStaticNeighborCmd )

# ------------------------------------------------------------------------------
# Authentication command in config-if mode:
# [ no | default ] authentication mode ( disabled |
#                                        simple shared-secret profile PROFILE |
#                                        md5 shared-secret profile PROFILE
#                                            [ meticulous ] |
#                                        sha1 shared-secret profile PROFILE
#                                            [ meticulous ] )
# ------------------------------------------------------------------------------
class BfdIntfAuthModeCmd( CliCommand.CliCommandClass ):
   # Having a kw arg passed to the CliParser named "mode" is problematic, since that
   # name is used all over the code as an argument.  To get around this, we have to
   # name the arg for this command "<mode>" in the data instead of simply "mode".
   syntax = ( 'bfd authentication mode ( disabled |'
                                     '( simple shared-secret profile PROFILE ) |'
                                     '( ( sha1 | md5 ) shared-secret profile PROFILE'
                                       '[ meticulous ] ) )' )
   noOrDefaultSyntax = 'bfd authentication ...'
   data = {
            'bfd': matcherBfd,
            'authentication': 'Configure BFD authentication',
            'mode': 'Specify BFD authentication mode',
            'disabled': 'Authentication disabled',
            'simple': 'Simple password authentication',
            'md5': 'Keyed MD5 authentication',
            'sha1': 'Keyed Sha1 authentication',
            'shared-secret': 'Specify a shared-secret',
            'profile': 'Shared-secret profile',
            'PROFILE': matcherAuthProfile,
            'meticulous': CliCommand.Node( matcherMeticulous, hidden=True ),
          }

   @staticmethod
   def handler( mode, args ):
      intf = IntfId( mode.intf.name )
      gv.bfdConfigIntf.authType[ intf ] = getAuthModeFromArgs( args )

      # If the mode configured is disabled, then there's no need to configure a
      # secret
      gv.bfdConfigIntf.secretProfileName[ intf ] = args.get( 'PROFILE', '' )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intf = IntfId( mode.intf.name )
      gv.bfdConfigIntf.authType[ intf ] = AuthType.authInherit
      gv.bfdConfigIntf.secretProfileName[ intf ] = ''

modelet.addCommandClass( BfdIntfAuthModeCmd )

def Plugin( entityManager ):
   gv.bfdConfigIntf = ConfigMount.mount( entityManager, 'bfd/config/intf',
                                         'Bfd::ConfigIntf', 'w' )
   gv.bfdConfigGlobal = ConfigMount.mount( entityManager, 'bfd/config/global',
                                           'Bfd::ConfigGlobal', 'w' )
   gv.staticSessionRequests = ConfigMount.mount( entityManager,
                                                 'bfd/config/static',
                                                 'Bfd::StaticConfigRequests',
                                                 'w' )
   gv.authSharedSecretConfig = LazyMount.mount( entityManager,
                                   'mgmt/security/sh-sec-prof/config',
                                   'Mgmt::Security::SharedSecretProfile::Config',
                                   'r' )
   if toggleBfdConditionallyRunnableEnabled():
      gv.bfdLauncherControl = LazyMount.mount( entityManager, 'bfd/launcherControl',
                                             'Tac::Dir', 'wi' )

   IntfCli.Intf.registerDependentClass( BfdIpIntf )
