#!/usr/bin/env python3
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# CliPlugin module for Bgp Mac Vrf commands
import Cell
import CliCommand
import CliParser
import BasicCli
import MultiRangeRule
import ConfigMount
import LazyMount
from CliMatcher import (
   EnumMatcher,
   IntegerMatcher,
   KeywordMatcher,
   PatternMatcher,
   StringMatcher,
)
import CliPlugin.RcfCliLib as RcfCliLib # pylint: disable=consider-using-from-import
# pylint: disable-next=consider-using-from-import
import CliPlugin.RouteDistinguisher as RouteDistinguisher
from CliPlugin.RoutingBgpCli import RouterBgpBaseMode, deleteRouterBgpMacVrfHook
from CliPlugin.RouteMapCli import RtSooExtCommCliMatcher, AsNumCliExpr
from CliMode.BgpMacVrfConfigMode import BgpMacVrfMode, BgpMacVrfVpwsPwMode
from BgpLib import (
   pwNamePattern,
   routeTargetToExtCommU64Value,
)
from PseudowireLib import (
   COLOR_MIN,
   COLOR_MAX,
)
from Toggles.ArBgpToggleLib import (
   toggleEvpnVpwsRemoteDomainEnabled,
   toggleEvpnMacVrfBundleAutoRdRtEnabled,
   toggleVpwsRouteTargetExportRcfEnabled,
   toggleEvpnUmrEnabled,
   toggleMacVrfRedistGatewayIpPhase1Enabled,
   toggleMacVrfRedistGatewayIpPhase2Enabled,
)
from Toggles.McastVpnLibToggleLib import (
   toggleMcastVpnL2StubEnabled )
import Tac
from BridgeIdHelper import vlanIdToBrId
from TypeFuture import TacLazyType

#-----------------------------------------------------------------------------
# Globals
#-----------------------------------------------------------------------------
bgpMacVrfConfigDir = None 
bgpConfig = None
bgpMacVrfRdMap = None
bridgingHwCapabilities = None
entityManager = None

EvpnEtid = TacLazyType( "Evpn::Etid" )
MacVrfTypeEnum = TacLazyType( "Routing::Bgp::MacVrfType" )
MacVrfSubTypeEnum = TacLazyType( "Routing::Bgp::MacVrfSubType" )
MacVrfPseudowireConfig = TacLazyType( "Routing::Bgp::MacVrfPseudowireConfig" )
TristatePrefMode = TacLazyType( "Routing::Bgp::TristatePreferenceMode" )

dfKw = 'Designated PE for broadcast, unknown unicast and multicast packets'
electionKw = 'Election'
MPLS = 'MPLS transport configuration'

macVrfPrefixDict = {
   MacVrfTypeEnum.macVrfTypeVni : "vnibundle.",
   MacVrfTypeEnum.macVrfTypeVlan : "vlanbundle.",
}

def deleteBgpMacVrfConfigHook():
   bgpMacVrfConfigDir.config.clear()

#-----------------------------------------------------------------------------
# MacVrfConfig Mode
#------------------------------------------------------------------------------

def RtToU64( rt ):
   return routeTargetToExtCommU64Value( rt )

class BgpMacVrfConfigModeBase( BgpMacVrfMode, BasicCli.ConfigModeBase ):

   def __init__( self, parent, session, macVrfName, isBundle, isVniMacVrf, isVpws ):

      # create new macVrfConfig
      self.name = macVrfName 
      self.session_ = session
      self.macVrfType = None
      if isVpws:
         assert not isBundle
         assert not isVniMacVrf
         self.macVrfType = MacVrfTypeEnum.macVrfTypeVpws
      elif isVniMacVrf:
         self.macVrfType = MacVrfTypeEnum.macVrfTypeVni
      else:
         self.macVrfType = MacVrfTypeEnum.macVrfTypeVlan
      self.macVrfSubType = MacVrfSubTypeEnum.macVrfSubTypeNone
      bgpMacVrfConfigDir.config.newMember( self.name, isBundle, self.macVrfType,
                                           self.macVrfSubType )
      BgpMacVrfMode.__init__( self, self.name, isBundle, isVniMacVrf, isVpws )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def config( self ):
      assert self.name in bgpMacVrfConfigDir.config, "MacVrf not found in Config"
      return bgpMacVrfConfigDir.config[ self.name ]

   def setRouteDistinguisher( self, args ):
      rd = args.get( 'RD' )
      self.config().rdAll = False
      self.config().autoRd = False
      rdToVrfList = bgpMacVrfRdMap.rdToVrfList.get( rd )
      if rdToVrfList and self.name not in rdToVrfList.vrfName:
         # pylint: disable-next=consider-using-f-string
         self.addWarning( "RD %s already used by other MAC-VRF(s)" % rd )
      if bgpConfig.rdAutoUnifiedConfig.rdAutoAfi.l2evpn is True:
         arnetRd = Tac.Value( "Arnet::RouteDistinguisher" )
         arnetRd.stringValue = rd
         if ( arnetRd.rdType() == "type1" and
               bgpConfig.rdAutoUnifiedConfig.assignedNumberStart <=
               arnetRd.type1LocallyAssignedNumber() <=
               bgpConfig.rdAutoUnifiedConfig.assignedNumberEnd ):
            self.addWarning(
                  f"Rd {rd} is in auto assigned rd range and can cause conflicts" )

      self.config().rd = rd

   def noRouteDistinguisher( self, args ):
      self.config().rd = 'INVALID'
      self.config().rdAll = False
      self.config().autoRd = False

   def setRemoteRouteDistinguisher( self, args ):
      rd = args.get( 'RD' )
      remoteRdToVrfList = bgpMacVrfRdMap.remoteRdToVrfList.get( rd )
      if 'remote' in args:
         self.config().rdAll = False
         if ( remoteRdToVrfList and self.name not in
               remoteRdToVrfList.vrfName ):
            # pylint: disable-next=consider-using-f-string
            self.addWarning( "RD %s already used by other MAC-VRF(s)" % rd )
         self.config().remoteRd = rd
      elif 'all' in args:
         rdToVrfList = bgpMacVrfRdMap.rdToVrfList.get( rd )
         self.config().autoRd = False
         self.config().rdAll = True
         if ( ( remoteRdToVrfList and self.name not in
                remoteRdToVrfList.vrfName ) or
              ( rdToVrfList and self.name not in rdToVrfList.vrfName ) ):
            # pylint: disable-next=consider-using-f-string
            self.addWarning( "RD %s already used by other MAC-VRF(s)" % rd )
         self.config().rd = rd
         self.config().remoteRd = rd

   def noRemoteRouteDistinguisher( self, args ):
      self.config().rdAll = False
      self.config().remoteRd = 'INVALID'
      if 'all' in args:
         self.config().rd = 'INVALID'
         self.config().autoRd = False

   def setRouteTargetCliConf( self, rt, importExportAll=None, importAll=None,
                              exportAll=None, importExport=None, both=None,
                              importExportRemote=None ):
      "Interface to set flags for the route-target CLI configuration map"
      u64Rt = RtToU64( rt )
      rtMap = self.config().routeTargetCli
      val = rtMap[ u64Rt ].value if u64Rt in rtMap else 0x0
      rtCliFlags = Tac.Value( "Routing::Bgp::MacVrfConfig::routeTargetCliFlags",
                              val )
      if importExportAll is not None:
         rtCliFlags.importExportAll = importExportAll
      if importAll is not None:
         rtCliFlags.importAll = importAll
      if exportAll is not None:
         rtCliFlags.exportAll = exportAll
      if importExport is not None:
         rtCliFlags.importExport = importExport
      if both is not None:
         rtCliFlags.both = both
      if importExportRemote is not None:
         rtCliFlags.importExportRemote = importExportRemote
      # If all the flags are set to False, we can remove the entry from
      # the collection
      if rtCliFlags.value == 0x0:
         del rtMap[ u64Rt ]
      else:
         rtMap[ u64Rt ] = rtCliFlags

   def setRouteTargetImport( self, rt ):
      self.config().importRtList[ RtToU64( rt ) ] = True
      self.setRouteTargetCliConf( rt, importExportAll=False, importAll=False,
                                  importExport=False, both=False )

   def noRouteTargetImport( self, rt ):
      del self.config().importRtList[ RtToU64( rt ) ]
      self.setRouteTargetCliConf( rt, importExportAll=False, importAll=False,
                                  importExport=False, both=False )

   def setRouteTargetExport( self, rt ):
      self.config().exportRtList[ RtToU64( rt ) ] = True
      self.setRouteTargetCliConf( rt, importExportAll=False, exportAll=False,
                                  importExport=False, both=False )

   def noRouteTargetExport( self, rt ):
      del self.config().exportRtList[ RtToU64( rt ) ]
      self.setRouteTargetCliConf( rt, importExportAll=False, exportAll=False,
                                  importExport=False, both=False )

   def setRouteTargetConcatImportExport( self, rt, **kwargs ):
      u64Rt = RtToU64( rt )
      self.config().importRtList[ u64Rt ] = True
      self.config().exportRtList[ u64Rt ] = True
      self.setRouteTargetCliConf( rt, importExportAll=False, importAll=False,
                                  exportAll=False, importExport=True, both=False )

   def noRouteTargetConcatImportExport( self, rt, **kwargs ):
      u64Rt = RtToU64( rt )
      del self.config().importRtList[ u64Rt ]
      del self.config().exportRtList[ u64Rt ]
      self.setRouteTargetCliConf( rt, importExportAll=False, importAll=False,
                                  exportAll=False, importExport=False, both=False )

   @staticmethod
   def _computeImportExport( importExport ):
      """
      Handle the importExport argument from "ruleImportExport"

      It can match one of 'import', 'export' or 'both'.
      """
      if importExport == 'both':
         setImport = True
         setExport = True
      else:
         setImport = importExport == 'import'
         setExport = importExport == 'export'
      return setImport, setExport

   def setRouteTargetImportExport( self, importExport, rt ):
      setImport, setExport = self._computeImportExport( importExport )
      u64Rt = RtToU64( rt )
      if importExport == 'both':
         self.config().importRtList[ u64Rt ] = True
         self.config().exportRtList[ u64Rt ] = True
         self.setRouteTargetCliConf( rt, importExportAll=False, importAll=False,
                                     exportAll=False, importExport=False, both=True )
      elif setImport:
         self.config().importRtList[ u64Rt ] = True
         self.setRouteTargetCliConf( rt, importExportAll=False, importAll=False,
                                     importExport=False, both=False )
      elif setExport:
         self.config().exportRtList[ u64Rt ] = True
         self.setRouteTargetCliConf( rt, importExportAll=False, exportAll=False,
                                     importExport=False, both=False )

   def noRouteTargetImportExport( self, importExport, rt ):
      clearImport, clearExport = self._computeImportExport( importExport )
      u64Rt = RtToU64( rt )
      if clearImport:
         del self.config().importRtList[ u64Rt ]
         self.setRouteTargetCliConf( rt, importExportAll=False, importAll=False,
                                     importExport=False, both=False )
      if clearExport:
         del self.config().exportRtList[ u64Rt ]
         self.setRouteTargetCliConf( rt, importExportAll=False, exportAll=False,
                                     importExport=False, both=False )

   def noRouteTargetExportRcf( self ):
      self.config().exportRtRcf = ''

   def noRouteTargetAll( self ):
      self.config().importRtList.clear()
      self.config().exportRtList.clear()
      self.config().importRemoteDomainRtList.clear()
      self.config().exportRemoteDomainRtList.clear()
      self.config().routeTargetCli.clear()
      self.noRouteTargetExportRcf()

   def handleRemoteDomainRt( self, args ):
      config = self.config()
      rt = args.get( 'RT' )
      u64Rt = RtToU64( rt )
      isImport = 'import' in args
      isExport = 'export' in args
      if 'remote' in args:
         if isImport:
            config.importRemoteDomainRtList[ u64Rt ] = True
         if isExport:
            config.exportRemoteDomainRtList[ u64Rt ] = True
         importExportRemote = isImport and isExport
         importAll = False if importExportRemote else None
         exportAll = False if importExportRemote else None
         self.setRouteTargetCliConf( rt, importExportAll=False, importAll=importAll,
                                     exportAll=exportAll,
                                     importExportRemote=importExportRemote )
      elif 'all' in args:
         if isImport:
            config.importRtList[ u64Rt ] = True
            config.importRemoteDomainRtList[ u64Rt ] = True
         if isExport:
            config.exportRtList[ u64Rt ] = True
            config.exportRemoteDomainRtList[ u64Rt ] = True
         importAll = None if not isImport else isImport and not isExport
         exportAll = None if not isExport else isExport and not isImport
         importExportAll = isExport and isImport
         self.setRouteTargetCliConf( rt, importExportAll=importExportAll,
                                     importAll=importAll,
                                     exportAll=exportAll, importExport=False,
                                     both=False, importExportRemote=False )

   def noHandleRemoteDomainRt( self, args ):
      config = self.config()
      rt = args.get( 'RT' )
      u64Rt = RtToU64( rt )
      isImport = 'import' in args
      isExport = 'export' in args
      if 'remote' in args:
         if isImport:
            del config.importRemoteDomainRtList[ u64Rt ]
         if isExport:
            del config.exportRemoteDomainRtList[ u64Rt ]
         importExportRemote = False
         importAll = False if isImport else None
         exportAll = False if isExport else None
         self.setRouteTargetCliConf( rt, importExportAll=False, importAll=importAll,
                                     exportAll=exportAll,
                                     importExportRemote=importExportRemote )
      elif 'all' in args:
         if isImport:
            del config.importRtList[ u64Rt ]
            del config.importRemoteDomainRtList[ u64Rt ]
         if isExport:
            del config.exportRtList[ u64Rt ]
            del config.exportRemoteDomainRtList[ u64Rt ]
         importAll = False if isImport else None
         exportAll = False if isExport else None
         self.setRouteTargetCliConf( rt, importExportAll=False,
                                     importAll=importAll,
                                     exportAll=exportAll, importExport=False,
                                     both=False, importExportRemote=False )

   def setRouteTargetImportExportAuto( self, args ):
      config = self.config()
      # pylint: disable-msg=simplifiable-if-statement
      if not toggleEvpnMacVrfBundleAutoRdRtEnabled():
         asn = args.get( 'AS_NUM' )
         if asn is None:
            config.autoExportRt = True
         else:
            config.autoImportRtList[ int( asn ) ] = True
         return
      # toggleEvpnMacVrfBundleAutoRdRtEnabled
      if 'export' in args:
         asn = args.get( '2BAS_NUM' )
         # route-target export auto [ asn ]
         config.autoExportRt = True
         config.autoExportRtAsn = int( asn ) if asn else 0

      else:
         asn = args.get( 'AS_NUM' )
         # import route-target auto asn
         config.autoImportRtList[ int( asn ) ] = True

   def noRouteTargetImportExportAuto( self, args ):
      config = self.config()
      asn = args.get( 'AS_NUM' )
      if not toggleEvpnMacVrfBundleAutoRdRtEnabled():
         if asn is None:
            config.autoExportRt = False
         else:
            del config.autoImportRtList[ int( asn ) ]
         return
      # toggleEvpnMacVrfBundleAutoRdRtEnabled
      if 'export' in args:
         # no route-target export auto
         config.autoExportRt = False
         config.autoExportRtAsn = 0
      else:
         # no import route-target auto asn
         del config.autoImportRtList[ int( asn ) ]

   def setUmrOriginator( self ):
      self.config().originateUmr = True

   def noUmrOriginator( self ):
      self.config().originateUmr = False

   def setReAdvertiseMacIpDisable( self, args ):
      self.config().reAdvertiseMacIpIntoLocalDomain = not args.get( 'DISABLED' )

   def noReAdvertiseMacIpDisable( self ):
      self.config().reAdvertiseMacIpIntoLocalDomain = True

   def setRedistribute( self, source ):
      def shouldWarn( cfg1, cfg2 ):
         """ We should warn if `cfg1` is being configured while `cfg2` already
         exists, and vice versa. """
         return ( source == cfg1 and self.config().isInRedistribute( cfg2 ) ) or \
                ( source == cfg2 and self.config().isInRedistribute( cfg1 ) )

      if shouldWarn( 'redistributeSysMacGateway',
                     'redistributeRouterMacVirtualIp' ) or \
         shouldWarn( 'redistributeSysMacGateway', 'redistributeSysMacPrimaryIp' ):
         self.addWarning( 'Redistribution of virtual and primary gateway IPs are '
                          'overridden in favor of default gateway extended '
                          'community advertisements' )

      self.config().addSource( source )

   def noRedistribute( self, source ):
      self.config().removeSource( source )

   def defaultRedistribute( self, source ):
      self.config().defaultSource( source )

   def setMacAliasDefaultGateway( self ):
      self.config().macAliasDefaultGateway = True

   def noMacAliasDefaultGateway( self ):
      self.config().macAliasDefaultGateway = False

   def setRouteDistinguisherAuto( self, args ):
      self.config().rd = 'INVALID'
      self.config().autoRd = True

   def setPreferenceRule( self, args ):
      if 'low' in args:
         self.config().preferenceMode = TristatePrefMode.valueSet( 'preferenceLow' )
      else:
         self.config().preferenceMode = TristatePrefMode.valueSet( 'preferenceHigh' )

   def noPreferenceRule( self, args ):
      self.config().preferenceMode = TristatePrefMode.valueInvalid()

   def setMaxRoutes( self, args ):
      numRoutes = args.get( 'NUM' )
      self.config().maxRoutes = numRoutes

   def noSetMaxRoutes( self, args ):
      self.config().maxRoutes = 0

   def setControlWord( self, args ):
      """Use default when value=None"""
      self.config().mplsControlWord = True

   def noControlWord( self, args ):
      self.config().mplsControlWord = False

   def setFlowLabel( self, args ):
      self.config().flowLabel = True

   def noFlowLabel( self, args ):
      self.config().flowLabel = False

   def setMtu( self, args ):
      self.config().pseudowireMtu = args.get( 'MTU', 0 )

   def setFxcSignaling( self, args ):
      self.config().fxcMode = 'defaultFxc'

   def noOrDefaultFxcSignaling( self, args ):
      self.config().fxcMode = 'notFxc'

#------------------------------------------------------------------------------
# (config-router-bgp)# [no|default] vni-aware-bundle <bundle name>
#------------------------------------------------------------------------------

class BgpMacVrfVniBundleMode( BgpMacVrfConfigModeBase ):
   # Attributes required for every Mode Class
   name = 'vnimacvrf-bundle'

   def __init__( self, parent, session, macVrfName ):
      BgpMacVrfConfigModeBase.__init__( self, parent, session, macVrfName, 
            isBundle=True, isVniMacVrf=True, isVpws=False )

   def setRedistVxlan( self, args ):
      enable = not CliCommand.isNoOrDefaultCmd( args )
      self.config().redistributeVcs = enable

def getBundleName( macVrfType, mode, bundleName, validate=True ):
   prefix = macVrfPrefixDict[ macVrfType ]
   fullBundleName = prefix + bundleName
   if validate:
      maxLength = Tac.Type( "L3::VrfName" ).maxLength
      if len( fullBundleName ) > maxLength:
         mode.addErrorAndStop(
            # pylint: disable-next=consider-using-f-string
            "'%s' too long: must be no more than %d characters"
            % ( bundleName, maxLength - len( prefix ) )
         )
   return fullBundleName

class BgpVniBundleCommand( CliCommand.CliCommandClass ):
   syntax = 'vni-aware-bundle NAME'
   noOrDefaultSyntax = syntax
   data = {
       'vni-aware-bundle' : 'Configure MAC VRF BGP for multiple VNI support',
       'NAME' : StringMatcher( helpdesc='Unique name to identify VNI Aware Bundle',
                               helpname='VNI Bundle Name' )
   }
   handler = "BgpMacVrfConfigCliHandler.BgpVniBundleCommand_handler"
   noOrDefaultHandler = \
      "BgpMacVrfConfigCliHandler.BgpVniBundleCommand_noOrDefaultHandler"

RouterBgpBaseMode.addCommandClass( BgpVniBundleCommand )

#------------------------------------------------------------------------------
# (config-router-bgp)# [no|default] vlan <vlan id>
#------------------------------------------------------------------------------

class BgpMacVrfStandAloneMode( BgpMacVrfConfigModeBase ):
   # Attributes required for every Mode Class
   name = 'macvrf-standalone'

   def __init__( self, parent, session, macVrfName ):
      BgpMacVrfConfigModeBase.__init__( self, parent, session, macVrfName, 
            isBundle=False, isVniMacVrf=False, isVpws=False )
      cfg = bgpMacVrfConfigDir.config[ self.name ]
      vlanId = int( self.macVrfId )
      brId = vlanIdToBrId( vlanId ) 
      cfg.brIdToEtId[ brId  ] = 0
      
def getStandAloneName( vlanId ):
   return "vlan.%d" % vlanId # pylint: disable=consider-using-f-string

class BgpMacVrfStandAloneCommand( CliCommand.CliCommandClass ):
   syntax = 'vlan VLAN_ID'
   noOrDefaultSyntax = syntax
   data = {
       'vlan' : 'Configure MAC VRF BGP for single VLAN support',
       'VLAN_ID' : IntegerMatcher( 1, 4094, helpdesc='Identifier for a Virtual LAN' )
   }
   handler = "BgpMacVrfConfigCliHandler.BgpMacVrfStandAloneCommand_handler"
   noOrDefaultHandler = \
      "BgpMacVrfConfigCliHandler.BgpMacVrfStandAloneCommand_noOrDefaultHandler"

RouterBgpBaseMode.addCommandClass( BgpMacVrfStandAloneCommand )

#------------------------------------------------------------------------------
# (config-router-bgp)# [no|default] vlan-aware-bundle <bundle name>
#------------------------------------------------------------------------------

class BgpMacVrfVlanBundleMode( BgpMacVrfConfigModeBase ):
   # Attributes required for every Mode Class
   name = 'macvrf-bundle'

   def __init__( self, parent, session, macVrfName ):
      BgpMacVrfConfigModeBase.__init__( self, parent, session, macVrfName, 
            isBundle=True, isVniMacVrf=False, isVpws=False )
      
   def setVlanRange( self, vlanSet ):
      macVrfConfig = self.config()
      current = { brId.vlanId for brId, etid in
                  macVrfConfig.brIdToEtId.items() if etid == 0 }
      newSet = set( vlanSet )
      removeSet = current.difference( newSet )
      addSet = newSet.difference( current )
      for vlanId in removeSet:
         brId = vlanIdToBrId( vlanId ) 
         del macVrfConfig.brIdToEtId[ brId ]
      for vlanId in addSet:
         brId = vlanIdToBrId( vlanId ) 
         macVrfConfig.brIdToEtId[ brId ] = 0

   def noVlanRange( self, vlanSet ):
      macVrfConfig = self.config()
      for vlanId in vlanSet:
         brId = vlanIdToBrId( vlanId ) 
         del macVrfConfig.brIdToEtId[ brId ]

   def setVlanToEtidMap( self, args ):
      vlanId = args.get( 'VLAN_ID' )
      etid = args.get( 'ET_ID' )
      brId = vlanIdToBrId( vlanId ) 
      macVrfConfig = self.config()
      # check for duplicates
      for v, e in macVrfConfig.brIdToEtId.items():
         if e == etid and v != brId:
            self.session_.addWarning( "Duplicated ETID assignment detected" )
      macVrfConfig.brIdToEtId[ brId ] = etid

   def addVlanRange( self, vlanSet ):
      macVrfConfig = self.config()
      currSet = { brId.vlanId for brId in macVrfConfig.brIdToEtId }
      diffSet = set( vlanSet ).difference( currSet )
      for vlanId in diffSet:
         brId = vlanIdToBrId( vlanId )
         macVrfConfig.brIdToEtId[ brId ] = 0

   def removeVlanRange( self, vlanSet ):
      macVrfConfig = self.config()
      for vlanId in vlanSet:
         brId = vlanIdToBrId( vlanId )
         del macVrfConfig.brIdToEtId[ brId ]

class BgpMacVrfBundleCommand( CliCommand.CliCommandClass ):
   syntax = 'vlan-aware-bundle NAME'
   noOrDefaultSyntax = syntax
   data = {
       'vlan-aware-bundle' : 'Configure MAC VRF BGP for multiple VLAN support',
       'NAME' : StringMatcher( helpdesc='Unique name to identify VLAN Aware Bundle',
                               helpname='Bundle Name' )
   }
   handler = "BgpMacVrfConfigCliHandler.BgpMacVrfBundleCommand_handler"
   noOrDefaultHandler = \
      "BgpMacVrfConfigCliHandler.BgpMacVrfBundleCommand_noOrDefaultHandler"

RouterBgpBaseMode.addCommandClass( BgpMacVrfBundleCommand )

#------------------------------------------------------------------------------
# (config-router-bgp)# [no|default] vpws <name>
#------------------------------------------------------------------------------

class BgpMacVrfVpwsMode( BgpMacVrfConfigModeBase ):
   name = 'macvrf-vpws'

   def __init__( self, parent, session, macVrfName ):
      BgpMacVrfConfigModeBase.__init__( self, parent, session, macVrfName,
            isBundle=False, isVniMacVrf=False, isVpws=True )

class BgpMacVrfVpwsCommand( CliCommand.CliCommandClass ):
   syntax = 'vpws VPWS'
   noOrDefaultSyntax = syntax
   data = {
      'vpws' : 'Configure EVPN instance for VPWS',
      'VPWS' : PatternMatcher(
         pattern='.+',
         helpdesc='Unique name to identify the EVPN instance',
         helpname='WORD' ),
   }
   handler = "BgpMacVrfConfigCliHandler.BgpMacVrfVpwsCommand_handler"
   noOrDefaultHandler = \
      "BgpMacVrfConfigCliHandler.BgpMacVrfVpwsCommand_noOrDefaultHandler"

RouterBgpBaseMode.addCommandClass( BgpMacVrfVpwsCommand )

#------------------------------------------------------------------------------
# (config-macvrf-<id>)# [no|default] vlan [add|remove] <range>
#------------------------------------------------------------------------------
def vlanIdListFunc( mode, grList ):
   return list( grList.values() )

bundleVlanSet = MultiRangeRule.MultiRangeMatcher(
   lambda: ( 1, 4094 ),
   False,
   'VLAN IDs of the MAC VRF',
   value=vlanIdListFunc )

vlanKwMatcher = KeywordMatcher( 'vlan',
      helpdesc='Set of VLANs that forms a MAC VRF' )

class VlanRangeCmd( CliCommand.CliCommandClass ):
   syntax = ' vlan [add|remove] <range> '
   noOrDefaultSyntax = 'vlan <range> ...'
   data = {
      'vlan' : vlanKwMatcher,
      'add' : 'Add VLANs to the current list',
      'remove' : 'Remove VLANs from the current list',
      '<range>' : bundleVlanSet
   }
   handler = "BgpMacVrfConfigCliHandler.VlanRangeCmd_handler"
   noOrDefaultHandler = "BgpMacVrfConfigCliHandler.VlanRangeCmd_noOrDefaultHandler"

BgpMacVrfVlanBundleMode.addCommandClass( VlanRangeCmd )

#------------------------------------------------------------------------------
# (config-macvrf-<id>)# [no|default] vlan <vlanId> etid <etid> ]
#------------------------------------------------------------------------------
# Restrict ETID range to 1-4094 in vlan-aware bundle mode. ETID is used as
# dot1q tag in Lfib and anything outside this range will cause ArBgp to crash.
# Per RFC, ETID can either be 24-bit or 12-bit. In our EVPN MPLS
# implementation, it is not useful to have ETID range greater than vlan range.

class VlanToEtidMapCommand( CliCommand.CliCommandClass ):
   syntax = 'vlan VLAN_ID etid ET_ID'
   data = {
       'vlan' : vlanKwMatcher,
       'VLAN_ID' : IntegerMatcher( 1, 4094,
          helpdesc='Identifier for a Virtual LAN' ),
       'etid' : 'ETID mapping for VLAN',
       'ET_ID' : IntegerMatcher( 1, 4094, helpdesc='ETID' )
   }
   handler = "BgpMacVrfConfigCliHandler.VlanToEtidMapCommand_handler"

BgpMacVrfVlanBundleMode.addCommandClass( VlanToEtidMapCommand )
# Note: The noOrDefault command is taken care of by [ no|default ] vlan <range>

#------------------------------------------------------------------------------
# (config-macvrf-<id>)# [no|default] rd <auto | route distinguisher>
#------------------------------------------------------------------------------

rdKwMatcher = KeywordMatcher( 'rd', helpdesc='BGP route distinguisher' )
rdMatcher = RouteDistinguisher.RdDistinguisherMatcher(
      helpdesc='BGP route distinguisher' )

class RdCommand( CliCommand.CliCommandClass ):
   syntax = 'rd RD'
   noOrDefaultSyntax = 'rd [ RD ]'
   data = {
         'rd' : rdKwMatcher,
         'RD' : rdMatcher,
   }
   handler = "BgpMacVrfConfigCliHandler.RdCommand_handler"
   noOrDefaultHandler = "BgpMacVrfConfigCliHandler.RdCommand_noOrDefaultHandler"

BgpMacVrfStandAloneMode.addCommandClass( RdCommand )
BgpMacVrfVlanBundleMode.addCommandClass( RdCommand )
BgpMacVrfVniBundleMode.addCommandClass( RdCommand )
BgpMacVrfVpwsMode.addCommandClass( RdCommand )

def remoteDomainVtepBridgingSupportedGuard( mode, token ):
   if bridgingHwCapabilities.remoteDomainVtepBridgingSupported:
      return None
   return CliParser.guardNotThisPlatform

#------------------------------------------------------------------------------
# (config-macvrf-<id>)# [no|default] rd evpn domain remote RD
#------------------------------------------------------------------------------
domainKwMatcher = KeywordMatcher( 'domain',
      helpdesc='Select scope for multi-domain gateway config' )
remoteKwMatcher = KeywordMatcher( 'remote',
      helpdesc='Apply config to only remote domain on multi-domain gateway' )
allKwMatcher = KeywordMatcher( 'all',
      helpdesc='Apply config to both local and remote domains on multi-domain ' \
               'gateway' )
evpnRdDomainKwMatcher = KeywordMatcher( 'evpn',
      helpdesc='Configure route distinguisher for EVPN address family' )
evpnRdDomainNode = CliCommand.Node( matcher=evpnRdDomainKwMatcher,
      guard=remoteDomainVtepBridgingSupportedGuard )

class RemoteRdCommand( CliCommand.CliCommandClass ):
   syntax = 'rd evpn domain ( remote | all ) RD'
   noOrDefaultSyntax = 'rd evpn domain ( remote | all ) [ RD ]'
   data = {
         'rd' : rdKwMatcher,
         'RD' : rdMatcher,
         'evpn' : evpnRdDomainNode,
         'domain' : domainKwMatcher,
         'remote' : remoteKwMatcher,
         'all' : allKwMatcher,
   }
   handler = "BgpMacVrfConfigCliHandler.RemoteRdCommand_handler"
   noOrDefaultHandler = \
      "BgpMacVrfConfigCliHandler.RemoteRdCommand_noOrDefaultHandler"

BgpMacVrfStandAloneMode.addCommandClass( RemoteRdCommand )
BgpMacVrfVlanBundleMode.addCommandClass( RemoteRdCommand )
if toggleEvpnVpwsRemoteDomainEnabled():
   BgpMacVrfVpwsMode.addCommandClass( RemoteRdCommand )

class RdAutoCommand( CliCommand.CliCommandClass ):
   syntax = 'rd auto'
   noOrDefaultSyntax = 'rd auto'
   data = {
         'rd' : rdKwMatcher,
         'auto' : 'Auto Generate Route Distinguisher',
   }
   handler = "BgpMacVrfConfigCliHandler.RdAutoCommand_handler"
   noOrDefaultHandler = "BgpMacVrfConfigCliHandler.RdAutoCommand_noOrDefaultHandler"

# rd auto is applicable for VLAN-based and vlan-aware bundle service
BgpMacVrfStandAloneMode.addCommandClass( RdAutoCommand )
if toggleEvpnMacVrfBundleAutoRdRtEnabled():
   BgpMacVrfVlanBundleMode.addCommandClass( RdAutoCommand )

#---------------------------------------------------------------------------------
# (config-macvrf-<id>)# [no|default] designated-forwarder election preference rule
#           ( low | high )
#
#---------------------------------------------------------------------------------

class ElectionPreferenceRuleCommand( CliCommand.CliCommandClass ):
   syntax = 'designated-forwarder election preference rule ( low | high )'
   noOrDefaultSyntax = 'designated-forwarder election preference rule ...'
   data = {
      'designated-forwarder' : dfKw,
      'election' : electionKw,
      'preference' : 'Options for preference-based election',
      'rule' : 'Rule for selecting the DF given a preference from each candidate',
      'low' : 'Select the candidate with the lowest preference',
      'high' : 'Select the candidate with the highest preference (default)',
   }
   handler = "BgpMacVrfConfigCliHandler.ElectionPreferenceRuleCommand_handler"
   noOrDefaultHandler = \
      "BgpMacVrfConfigCliHandler.ElectionPreferenceRuleCommand_noOrDefaultHandler"

BgpMacVrfStandAloneMode.addCommandClass( ElectionPreferenceRuleCommand )
BgpMacVrfVlanBundleMode.addCommandClass( ElectionPreferenceRuleCommand )

#---------------------------------------------------------------------------------
# (config-macvrf-<id>)# [no|default] route-target 
#           < import | export | import export > evpn domain
#           < remote | all > rt
# e.g.
#   route-target import evpn domain remote <rt>
#   route-target export evpn domain remote <rt>
#   route-target import export evpn domain all <rt>
#---------------------------------------------------------------------------------

importKwMatcher = KeywordMatcher( 'import', helpdesc='Route Import' )
exportKwMatcher = KeywordMatcher( 'export', helpdesc='Route Export' )
rtKwMatcher = KeywordMatcher( 'route-target', helpdesc='Route target' )
evpnRemoteRtKwMatcher = KeywordMatcher( 'evpn',
      helpdesc='Configure route target for EVPN address family' )
evpnRemoteRtNode = CliCommand.Node( matcher=evpnRemoteRtKwMatcher,
      guard=remoteDomainVtepBridgingSupportedGuard )

class RemoteRtCommand( CliCommand.CliCommandClass ):
   syntax = 'route-target ( ( import [ export ] ) | export ) evpn domain \
             ( remote | all ) RT'
   noOrDefaultSyntax = syntax
   data = {
      'route-target' : rtKwMatcher,
      'import' : importKwMatcher,
      'export' : exportKwMatcher,
      'RT' : RtSooExtCommCliMatcher( 'Route target' ),
      'evpn' : evpnRemoteRtNode,
      'domain' : domainKwMatcher,
      'remote' : remoteKwMatcher,
      'all' : allKwMatcher,
   }
   handler = "BgpMacVrfConfigCliHandler.RemoteRtCommand_handler"
   noOrDefaultHandler = \
      "BgpMacVrfConfigCliHandler.RemoteRtCommand_noOrDefaultHandler"

BgpMacVrfStandAloneMode.addCommandClass( RemoteRtCommand )
BgpMacVrfVlanBundleMode.addCommandClass( RemoteRtCommand )
if toggleEvpnVpwsRemoteDomainEnabled():
   BgpMacVrfVpwsMode.addCommandClass( RemoteRtCommand )

#---------------------------------------------------------------------------------
# (config-macvrf-<id>)# [no|default] route-target
#           < import | export | import export | both | all > < auto > < rt | asn >
# e.g.
#   route-target import <rt>
#   route-target export <rt>
#   route-target import export <rt>
#   route-target both <rt>
#   route-target import auto <rt>
#   route-target export auto
#---------------------------------------------------------------------------------

class RtCommand( CliCommand.CliCommandClass ):
   syntax = 'route-target ( ( ( import [ export ] ) | export | both ) RT )'
   noOrDefaultSyntax = syntax + ' | all'
   data = {
      'route-target' : rtKwMatcher,
      'import' : importKwMatcher,
      'export' : exportKwMatcher,
      'both' : 'Both Route Import and Export',
      'RT' : RtSooExtCommCliMatcher( "Route target" ),
      'all' : 'removes all import and export route targets'
   }
   handler = "BgpMacVrfConfigCliHandler.RtCommand_handler"
   noOrDefaultHandler = "BgpMacVrfConfigCliHandler.RtCommand_noOrDefaultHandler"

BgpMacVrfStandAloneMode.addCommandClass( RtCommand )
BgpMacVrfVlanBundleMode.addCommandClass( RtCommand )
BgpMacVrfVniBundleMode.addCommandClass( RtCommand )

class RtAutoCommand( CliCommand.CliCommandClass ):
   if toggleEvpnMacVrfBundleAutoRdRtEnabled():
      syntax = 'route-target ( ( export auto [ 2BAS_NUM ] ) |'\
         ' ( import auto AS_NUM ) )'
   else:
      syntax = 'route-target ( ( export auto ) | ( import auto AS_NUM ) )'
   noOrDefaultSyntax = syntax
   data = {
     'route-target' : rtKwMatcher,
     'import' : importKwMatcher,
     'export' : exportKwMatcher,
     'auto' : KeywordMatcher( 'auto', 'Auto Generate Route Target' ),
     'AS_NUM' : AsNumCliExpr,
     '2BAS_NUM' : IntegerMatcher( 1, 65535,
         helpdesc='2 octets AS number' )
   }
   handler = "BgpMacVrfConfigCliHandler.RtAutoCommand_handler"
   noOrDefaultHandler = \
      "BgpMacVrfConfigCliHandler.RtAutoCommand_noOrDefaultHandler"

# route-target export auto is applicable for VLAN-based and vlan-aware bundle
BgpMacVrfStandAloneMode.addCommandClass( RtAutoCommand )
if toggleEvpnMacVrfBundleAutoRdRtEnabled():
   BgpMacVrfVlanBundleMode.addCommandClass( RtAutoCommand )

class UmrOriginateCommand( CliCommand.CliCommandClass ):
   syntax = 'unknown-mac-route originate'
   noOrDefaultSyntax = syntax
   data = {
      'unknown-mac-route' : KeywordMatcher(
         'unknown-mac-route', helpdesc='EVPN Type-2 Unknown MAC route (UMR)' ),
      'originate' : KeywordMatcher(
         'originate', helpdesc='Originate Unknown MAC route (UMR) in this MAC-VRF' ),
   }
   handler = "BgpMacVrfConfigCliHandler.UmrOriginateCommand_handler"
   noOrDefaultHandler = \
      "BgpMacVrfConfigCliHandler.UmrOriginateCommand_noOrDefaultHandler"

if toggleEvpnUmrEnabled():
   BgpMacVrfStandAloneMode.addCommandClass( UmrOriginateCommand )
   BgpMacVrfVlanBundleMode.addCommandClass( UmrOriginateCommand )

vpwsRouteTargetEvpnKwMatcher = KeywordMatcher( 'evpn',
   helpdesc='Import or export only EVPN VPWS routes' )

class VpwsRouteTargetCommand( CliCommand.CliCommandClass ):
   syntax = 'route-target ( ( import [ export ] ) | export ) evpn RT'
   noOrDefaultSyntax = (
      'route-target ( ( ( import [ export ] ) | export ) evpn RT ) | all' )
   data = {
      'route-target' : rtKwMatcher,
      'import' : CliCommand.Node( KeywordMatcher( 'import', 'Route import' ),
                                  maxMatches=1 ),
      'export' : CliCommand.Node( KeywordMatcher( 'export', 'Route export' ),
                                  maxMatches=1 ),
      'evpn' : vpwsRouteTargetEvpnKwMatcher,
      'RT' : RtSooExtCommCliMatcher( "Route target" ),
      'all' : 'Remove all import and export route targets',
   }
   handler = "BgpMacVrfConfigCliHandler.RtCommand_handler"
   noOrDefaultHandler = "BgpMacVrfConfigCliHandler.RtCommand_noOrDefaultHandler"

BgpMacVrfVpwsMode.addCommandClass( VpwsRouteTargetCommand )

class VpwsRouteTargetRcfCommand( CliCommand.CliCommandClass ):
   # Only export support for now, but import support may follow
   syntax = 'route-target export evpn rcf FUNC_NAME'
   noOrDefaultSyntax = syntax.replace( 'FUNC_NAME', '...' )
   data = {
      'route-target' : rtKwMatcher,
      'export' : exportKwMatcher,
      'evpn' : vpwsRouteTargetEvpnKwMatcher,
      'rcf' : RcfCliLib.rcfKw,
      'FUNC_NAME' : RcfCliLib.rcfFunctionMatcher,
   }
   handler = "BgpMacVrfConfigCliHandler.VpwsRouteTargetRcfCommand_handler"
   noOrDefaultHandler = \
      "BgpMacVrfConfigCliHandler.VpwsRouteTargetRcfCommand_noOrDefaultHandler"

if toggleVpwsRouteTargetExportRcfEnabled():
   BgpMacVrfVpwsMode.addCommandClass( VpwsRouteTargetRcfCommand )

#------------------------------------------------------------------------------
# (config-macvrf-<id>)# [no|default] redistribute
#      < ( learned [ remote ] ) |
#        static |
#        dot1x |
#        ( link-local ipv6 ) |
#        ( router-mac [ system [ default-gateway | ip ] |
#                       ( next-hop vtep primary ) |
#                       ( virtual-ip [ next-hop vtep primary ] ) ] ) |
#        host-route |
#        igmp |
#        mld |
#        multicast-router >
#------------------------------------------------------------------------------

def vtepClassificationBridgingSupportGuard( mode, token ):
   if ( bridgingHwCapabilities.vxlanVtepToVtepBridgingSupported and
        bridgingHwCapabilities.vtepClassificationBridgingSupported ):
      return None
   return CliParser.guardNotThisPlatform

def macAliasDefaultGatewaySupportGuard( mode, token ):
   if bridgingHwCapabilities.evpnMacAliasDefaultGatewaySupported:
      return None
   return CliParser.guardNotThisPlatform

routerMacKwMatcher = KeywordMatcher( 'router-mac',
      helpdesc='router Ethernet Address' )
redistKwMatcher = KeywordMatcher( 'redistribute',
      helpdesc='Redistribute routes in to MAC-VRF' )
systemKwMatcher = KeywordMatcher( 'system',
      helpdesc='System Ethernet address' )
defaultGateway = KeywordMatcher( 'default-gateway',
      helpdesc='Default gateway Etherenet address' )
remoteWithGuard = CliCommand.guardedKeyword( 'remote',
      helpdesc='datapath learned remote Ethernet addresses',
      guard=vtepClassificationBridgingSupportGuard )
defaultGatewayWithGuard = CliCommand.Node( matcher=defaultGateway,
      guard=macAliasDefaultGatewaySupportGuard )

class MacVrfRedistributeCommand( CliCommand.CliCommandClass ):
   syntax = ( 'redistribute '
              '( ( learned [ remote ] ) '
              '| static '
              '| dot1x '
              '| ( link-local ipv6 ) '
              '| ( router-mac '
                  '[ next-hop vtep primary ] ) '
              '| host-route '
              '| igmp '
              '| mld '
              '| multicast-router )' )
   data = {
      'redistribute' : redistKwMatcher,
      'learned' : 'locally learned Ethernet addresses',
      'static' : 'statically defined Ethernet addresses',
      'router-mac' : routerMacKwMatcher,
      'host-route' : 'host route using symmetric IRB',
      'dot1x' : 'dot1x Ethernet addresses',
      'igmp' : 'IGMP proxy for EVPN',
      'mld' : 'MLD proxy for EVPN',
      'link-local' : 'link local adresses',
      'ipv6' : 'Ipv6 link local adresses',
      'next-hop' : 'configure the advertised next-hop',
      'vtep' : 'associate next-hop with a VTEP',
      'primary' : 'use the primary VTEP IP',
      'remote' : remoteWithGuard,
      'multicast-router' : 'multicast router for EVPN',
   }

   if not toggleMcastVpnL2StubEnabled():
      syntax = syntax.replace( ' | multicast-router', '' )
      del data[ 'multicast-router' ]

   noOrDefaultSyntax = syntax
   handler = "BgpMacVrfConfigCliHandler.MacVrfRedistributeCommand_handler"
   noOrDefaultHandler = \
      "BgpMacVrfConfigCliHandler.MacVrfRedistributeCommand_noOrDefaultHandler"

BgpMacVrfStandAloneMode.addCommandClass( MacVrfRedistributeCommand )
BgpMacVrfVlanBundleMode.addCommandClass( MacVrfRedistributeCommand )

class MacVrfRedistSysMacCommand( CliCommand.CliCommandClass ):
   syntax = 'redistribute router-mac system [ default-gateway ]'
   noOrDefaultSyntax = 'redistribute router-mac system ...'
   data = {
      'redistribute' : redistKwMatcher,
      'router-mac' : routerMacKwMatcher,
      'system' : systemKwMatcher,
      'default-gateway' : defaultGateway,
   }
   handler = "BgpMacVrfConfigCliHandler.MacVrfRedistSysMacCommand_handler"
   noHandler = "BgpMacVrfConfigCliHandler.MacVrfRedistSysMacCommand_noHandler"
   defaultHandler = \
      "BgpMacVrfConfigCliHandler.MacVrfRedistSysMacCommand_defaultHandler"

BgpMacVrfStandAloneMode.addCommandClass( MacVrfRedistSysMacCommand )
BgpMacVrfVlanBundleMode.addCommandClass( MacVrfRedistSysMacCommand )

class MacVrfRedistSysMacPrimaryIpCommand( CliCommand.CliCommandClass ):
   syntax = 'redistribute router-mac system ip'
   noOrDefaultSyntax = 'redistribute router-mac system ip'
   data = {
      'redistribute' : redistKwMatcher,
      'router-mac' : routerMacKwMatcher,
      'system' : systemKwMatcher,
      'ip' : 'Use the primary IP address'
   }
   handler = "BgpMacVrfConfigCliHandler.MacVrfRedistSysMacPrimaryIpCommand_handler"
   noOrDefaultHandler = \
         "BgpMacVrfConfigCliHandler." \
         "MacVrfRedistSysMacPrimaryIpCommand_noOrDefaultHandler"

if toggleMacVrfRedistGatewayIpPhase2Enabled():
   BgpMacVrfStandAloneMode.addCommandClass( MacVrfRedistSysMacPrimaryIpCommand )
   BgpMacVrfVlanBundleMode.addCommandClass( MacVrfRedistSysMacPrimaryIpCommand )

class MacVrfRedistRouterMacVirtualIpCommand( CliCommand.CliCommandClass ):
   syntax = 'redistribute router-mac virtual-ip'
   noOrDefaultSyntax = 'redistribute router-mac virtual-ip'
   data = {
      'redistribute' : redistKwMatcher,
      'router-mac' : routerMacKwMatcher,
      'virtual-ip' : 'SVI virtual IP address',
   }
   if toggleMacVrfRedistGatewayIpPhase2Enabled():
      noOrDefaultSyntax += ' ...'
      syntax += ' [ next-hop vtep primary ]'
      data.update( {
         'next-hop' : 'Configure the advertised next-hop',
         'vtep' : 'Associate next-hop with a VTEP',
         'primary' : 'Use the primary VTEP IP',
      } )
   handler = \
      "BgpMacVrfConfigCliHandler.MacVrfRedistRouterMacVirtualIpCommand_handler"
   noHandler = \
      "BgpMacVrfConfigCliHandler.MacVrfRedistRouterMacVirtualIpCommand_noHandler"
   defaultHandler = \
      "BgpMacVrfConfigCliHandler." \
      "MacVrfRedistRouterMacVirtualIpCommand_defaultHandler"

if toggleMacVrfRedistGatewayIpPhase1Enabled():
   BgpMacVrfStandAloneMode.addCommandClass( MacVrfRedistRouterMacVirtualIpCommand )
   BgpMacVrfVlanBundleMode.addCommandClass( MacVrfRedistRouterMacVirtualIpCommand )

#------------------------------------------------------------------------------
# (config-macvrf-<id>)# [no|default] router mac-address alias default-gateway
#------------------------------------------------------------------------------

class MacVrfMacAliasDefaultGateway( CliCommand.CliCommandClass ):
   syntax = ( 'router mac-address alias default-gateway' )
   noOrDefaultSyntax = syntax
   data = {
      'router' : 'routing command',
      'mac-address' : 'Ethernet address',
      'alias' : 'Ethernet address aliasing',
      'default-gateway' : defaultGatewayWithGuard,
   }
   handler = "BgpMacVrfConfigCliHandler.MacVrfMacAliasDefaultGateway_handler"
   noOrDefaultHandler = \
      "BgpMacVrfConfigCliHandler.MacVrfMacAliasDefaultGateway_noOrDefaultHandler"

BgpMacVrfStandAloneMode.addCommandClass( MacVrfMacAliasDefaultGateway )
BgpMacVrfVlanBundleMode.addCommandClass( MacVrfMacAliasDefaultGateway )

#------------------------------------------------------------------------------
# (config-macvrf-name)# [no|default] redistribute service vxlan
#------------------------------------------------------------------------------
# This redist command only applies to the vni-macvrfs

class RedistVxlanCommand( CliCommand.CliCommandClass ):
   syntax = 'redistribute service vxlan'
   noOrDefaultSyntax = 'redistribute service vxlan'
   data = {
      'redistribute' : redistKwMatcher,
      'service' : 'Redistribute routes from a service into a MAC-VRF',
      'vxlan' : 'Redistribute routes from VXLAN service into a MAC-VRF'
   }
   handler = "BgpMacVrfConfigCliHandler.RedistVxlanCommand_handler"
   noOrDefaultHandler = handler

BgpMacVrfVniBundleMode.addCommandClass( RedistVxlanCommand )

#------------------------------------------------------------------------------
# (config-macvrf-name)# [no|default] maximum-routes <num>
#------------------------------------------------------------------------------

class MaxRoutesCommand( CliCommand.CliCommandClass ):
   syntax = 'maximum-routes NUM'
   noOrDefaultSyntax = 'maximum-routes ...'
   data = {
      'maximum-routes' : 'Maximum number of routes accepted on this MAC-VRF',
      'NUM' : IntegerMatcher( 0, 4294967294,
         helpdesc='Maximum number of accepted routes (0 means unlimited)' )
   }
   handler = "BgpMacVrfConfigCliHandler.MaxRoutesCommand_handler"
   noOrDefaultHandler = \
      "BgpMacVrfConfigCliHandler.MaxRoutesCommand_noOrDefaultHandler"

BgpMacVrfStandAloneMode.addCommandClass( MaxRoutesCommand )
BgpMacVrfVlanBundleMode.addCommandClass( MaxRoutesCommand )
BgpMacVrfVniBundleMode.addCommandClass( MaxRoutesCommand )

#------------------------------------------------------------------------------
# (config-macvrf-<name>)# [no|default] mpls control-word
#------------------------------------------------------------------------------

class VpwsMplsControlWordCommand( CliCommand.CliCommandClass ):
   syntax = 'mpls control-word'
   noOrDefaultSyntax = syntax
   data = {
      'mpls': MPLS,
      'control-word': 'Enable control word',
   }
   handler = "BgpMacVrfConfigCliHandler.VpwsMplsControlWordCommand_handler"
   noOrDefaultHandler = \
      "BgpMacVrfConfigCliHandler.VpwsMplsControlWordCommand_noOrDefaultHandler"

BgpMacVrfVpwsMode.addCommandClass( VpwsMplsControlWordCommand )

#------------------------------------------------------------------------------
# (config-macvrf-<name>)# [no|default] label flow
#------------------------------------------------------------------------------

class VpwsFlowLabelCommand( CliCommand.CliCommandClass ):
   syntax = 'label flow'
   noOrDefaultSyntax = syntax
   data = {
      'label' : 'Label operation',
      'flow' : 'Enable flow label',
   }
   handler = "BgpMacVrfConfigCliHandler.VpwsFlowLabelCommand_handler"
   noOrDefaultHandler = \
      "BgpMacVrfConfigCliHandler.VpwsFlowLabelCommand_noOrDefaultHandler"

BgpMacVrfVpwsMode.addCommandClass( VpwsFlowLabelCommand )

#------------------------------------------------------------------------------
# (config-macvrf-<name>)# [no|default] mtu (ignore|<num>)
#------------------------------------------------------------------------------

class VpwsMtuCommand( CliCommand.CliCommandClass ):
   syntax = 'mtu ( ignore | MTU )'
   noOrDefaultSyntax = 'mtu ...'
   data = {
      'mtu' : 'MTU value to use for VPWS signaling and validation',
      'ignore' : 'Do not send MTU and ignore received MTU',
      'MTU' : IntegerMatcher( 1, 65535,
         helpdesc='Maximum transmission unit in bytes' ),
   }
   handler = "BgpMacVrfConfigCliHandler.VpwsMtuCommand_handler"
   noOrDefaultHandler = handler

BgpMacVrfVpwsMode.addCommandClass( VpwsMtuCommand )

#------------------------------------------------------------------------------
# (config-macvrf) [no|default] flexible-cross-connect [vlan]
# Note: [vlan] will be added later
#------------------------------------------------------------------------------

class FxcSignalingCommand( CliCommand.CliCommandClass ):
   syntax = 'flexible-cross-connect'
   noOrDefaultSyntax = 'flexible-cross-connect ...'
   data = {
      'flexible-cross-connect': 'Flexible cross-connect (FXC) configuration',
      #'vlan': 'Use VLAN-signaled FXC cross-connect',
   }
   handler = "BgpMacVrfConfigCliHandler.FxcSignalingCommand_handler"
   noOrDefaultHandler = \
      "BgpMacVrfConfigCliHandler.FxcSignalingCommand_noOrDefaultHandler"

BgpMacVrfVpwsMode.addCommandClass( FxcSignalingCommand )

#----------------------------------------------------------------------------------
# (vpws-evi)# [no|default] designated-forwarder election preference rule (low|high)
#----------------------------------------------------------------------------------
BgpMacVrfVpwsMode.addCommandClass( ElectionPreferenceRuleCommand )

#------------------------------------------------------------------------------
# (config-macvrf-<name>)# pseudowire <name>
#------------------------------------------------------------------------------

class BgpMacVrfVpwsPwConfigMode( BgpMacVrfVpwsPwMode, BasicCli.ConfigModeBase ):
   name = 'BGP VPWS pseudowire configuration'

   def __init__( self, parent, session, pwName ):
      BgpMacVrfVpwsPwMode.__init__( self, pwName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.parent = parent
      configColl = parent.config().pseudowireConfig
      if pwName not in configColl:
         configColl.addMember( MacVrfPseudowireConfig( pwName ) )

   def config( self ):
      configColl = self.parent.config().pseudowireConfig
      if self.pwName in configColl:
         return Tac.nonConst( configColl[ self.pwName ] )
      else:
         # Config was deleted elsewhere, just return the default value
         return MacVrfPseudowireConfig()

   def setConfig( self, pwConfig ):
      configColl = self.parent.config().pseudowireConfig
      if self.pwName in configColl:
         # Only update if the pseudowire config was not deleted in another session
         configColl.addMember( pwConfig )

   def setVpwsId( self, args ):
      local = args[ 'LOCAL' ]
      remote = args[ 'REMOTE' ]
      pwConfig = self.config()
      pwConfig.vpwsIdLocal = local
      pwConfig.vpwsIdRemote = remote
      self.setConfig( pwConfig )

   def noOrDefaultVpwsId( self, args ):
      pwConfig = self.config()
      pwConfig.vpwsIdLocal = EvpnEtid.max
      pwConfig.vpwsIdRemote = EvpnEtid.max
      self.setConfig( pwConfig )

   cwDirMap = { 'receive': 'rxOverride', 'transmit': 'txOverride' }
   cwStatusMap = { 'always': 'cwAlways', 'disabled': 'cwDisabled' }

   def setDataPlaneCw( self, direction, status ):
      pwConfig = self.config()
      override = Tac.nonConst( pwConfig.controlWordOverride )
      setattr( override, self.cwDirMap[ direction ], self.cwStatusMap[ status ] )
      pwConfig.controlWordOverride = override
      self.setConfig( pwConfig )

   def noOrDefaultDataPlaneCw( self, direction=None ):
      pwConfig = self.config()
      override = Tac.nonConst( pwConfig.controlWordOverride )
      direction = [ direction ] if direction else self.cwDirMap
      for clearDirection in direction:
         setattr( override, self.cwDirMap[ clearDirection ], 'noOverride' )
      pwConfig.controlWordOverride = override
      self.setConfig( pwConfig )

class BgpMacVrfVpwsPwCommand( CliCommand.CliCommandClass ):
   syntax = 'pseudowire NAME'
   noOrDefaultSyntax = syntax
   data = {
      'pseudowire' : 'Configure VPWS pseudowire for this EVPN instance',
      'NAME' : PatternMatcher(
         pattern=pwNamePattern,
         helpdesc='Unique name to identify the pseudowire',
         helpname='WORD' ),
   }
   handler = "BgpMacVrfConfigCliHandler.BgpMacVrfVpwsPwCommand_handler"
   noOrDefaultHandler = \
      "BgpMacVrfConfigCliHandler.BgpMacVrfVpwsPwCommand_noOrDefaultHandler"

BgpMacVrfVpwsMode.addCommandClass( BgpMacVrfVpwsPwCommand )

#------------------------------------------------------------------------------
# (config-vpws-pw-<name>)# evpn vpws id local <LOCAL> remote <REMOTE>
#------------------------------------------------------------------------------
vpwsIdMatcher = IntegerMatcher( 1, EvpnEtid.max - 1,
                                helpdesc='VPWS service instance identifier' )

class VpwsPwEvpnVpwsIdCommand( CliCommand.CliCommandClass ):
   syntax = 'evpn vpws id local LOCAL remote REMOTE'
   noOrDefaultSyntax = 'evpn vpws id ...'
   data = {
      'evpn' : 'EVPN pseudowire parameters',
      'vpws' : 'EVPN VPWS pseudowire parameters',
      'id' : 'EVPN VPWS service instance identifiers',
      'local' : 'VPWS service instance identifier advertised to peer',
      'LOCAL' : vpwsIdMatcher,
      'remote' : 'VPWS service instance identifier advertised by peer',
      'REMOTE' : vpwsIdMatcher,
   }
   handler = "BgpMacVrfConfigCliHandler.VpwsPwEvpnVpwsIdCommand_handler"
   noOrDefaultHandler = \
      "BgpMacVrfConfigCliHandler.VpwsPwEvpnVpwsIdCommand_noOrDefaultHandler"

BgpMacVrfVpwsPwConfigMode.addCommandClass( VpwsPwEvpnVpwsIdCommand )

#------------------------------------------------------------------------------
# (config-vpws-pw-<name>)#
# mpls control-word data-plane (receive|transmit) (always|disabled)
#------------------------------------------------------------------------------
directionMatcher = EnumMatcher( {
   'receive': 'Receive (decapsulation) direction',
   'transmit': 'Transmit (encapsulation) direction',
} )
statusMatcher = EnumMatcher( {
   'always': 'Control word is always used in this direction',
   'disabled': 'Control word is never used in this direction',
} )

class VpwsPwControlWordDataPlaneCmd( CliCommand.CliCommandClass ):
   syntax = 'mpls control-word data-plane DIRECTION STATUS'
   noOrDefaultSyntax = 'mpls control-word data-plane [ DIRECTION ] ...'
   data = {
      'mpls': MPLS,
      'control-word': 'Override control word behavior',
      'data-plane': 'Data-plane configuration',
      'DIRECTION': directionMatcher,
      'STATUS': statusMatcher,
   }
   handler = "BgpMacVrfConfigCliHandler.VpwsPwControlWordDataPlaneCmd_handler"
   noOrDefaultHandler = \
      "BgpMacVrfConfigCliHandler.VpwsPwControlWordDataPlaneCmd_noOrDefaultHandler"

BgpMacVrfVpwsPwConfigMode.addCommandClass( VpwsPwControlWordDataPlaneCmd )

# -------------------------------------------------------------------------------
# (config-vpws-pw-<name>)#
# color [value]
# -------------------------------------------------------------------------------
class VpwsPwColorCmd( CliCommand.CliCommandClass ):
   syntax = 'color COLOR'
   noOrDefaultSyntax = 'color ...'
   data = {
      'color' : 'Color',
      'COLOR' : IntegerMatcher( COLOR_MIN, COLOR_MAX, helpdesc='Color value' ),
   }
   handler = "BgpMacVrfConfigCliHandler.VpwsPwColorCmd_handler"
   noOrDefaultHandler = "BgpMacVrfConfigCliHandler.VpwsPwColorCmd_noOrDefaultHandler"

BgpMacVrfVpwsPwConfigMode.addCommandClass( VpwsPwColorCmd )

#------------------------------------------------------------------------------
# CliPlugin init hook
#------------------------------------------------------------------------------
def Plugin( em ):
   global entityManager
   global bgpMacVrfConfigDir
   global bgpConfig
   global bgpMacVrfRdMap
   global bridgingHwCapabilities
   entityManager = em
   bgpMacVrfConfigDir = ConfigMount.mount( 
      entityManager, 'routing/bgp/macvrf', 
      'Routing::Bgp::MacVrfConfigDir', 'w' )
   bgpConfig = ConfigMount.mount(
      entityManager, 'routing/bgp/config',
      'Routing::Bgp::Config', 'w' )
   bgpMacVrfRdMap = LazyMount.mount(
      entityManager, Cell.path( 'routing/bgp/macvrfrdmap' ),
      'Routing::Bgp::RdToVrfListDir', 'r' )
   bridgingHwCapabilities = LazyMount.mount(
      entityManager, 'bridging/hwcapabilities',
      'Bridging::HwCapabilities', 'r' )

   deleteRouterBgpMacVrfHook.addExtension( deleteBgpMacVrfConfigHook )
