# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import Tac
from Toggles import RcfLibToggleLib
import BasicCli
import CliMatcher
import ConfigMount
import CliCommand
from CliMode.BgpMaintenanceProfile import BgpMaintenanceProfileMode
from CliMode.BgpMaintenanceMode import BgpMaintenanceMode
from CliPlugin.BgpGroupCli import (
   BgpGroupConfigMode,
   BgpBuiltinGroupConfigMode,
)
from CliPlugin.VrfCli import getAllVrfNames
from CliPlugin.MaintenanceMode import MaintenanceConfigMode
from CliPlugin.MaintenanceModels import profilesCleanupHook
from CliPlugin.RouteMapCli import mapNameMatcher
from CliPlugin.RcfCliLib import rcfFunctionMatcher
from CliPlugin.RoutingBgpCli import PeerCliExpression
from CliPlugin.MaintenanceCliLib import (
   defaultProfile,
   dynamicUnitName,
   maintenanceKwMatcher,
   Profile,
   profileMatcher,
   profileNameMatcher,
   quiesceMatcher,
)
from CliPlugin.IntfGroupLib import (
   IntfBuiltinGroupConfigMode,
   IntfGroupConfigMode,
)
from ConfigConsistencyChecker import UndefinedReferenceChecker

globalBgpProfileConfigDir = None
globalMaintenanceUnitInputDir = None
globalDefaultBgpProfile = None

GroupOrigin = Tac.Type( "Group::GroupOrigin" )

matcherBgp = CliMatcher.KeywordMatcher( 'bgp',
                                        helpdesc='Configure BGP' )
matcherVrf = CliMatcher.KeywordMatcher( 'vrf', helpdesc='VRF name' )
matcherBgpExec = CliMatcher.KeywordMatcher( 'bgp', helpdesc='Bgp' )

#-------------------------------------------------------------------------------
# [no] maintenance profile bgp <profileName>
# in "config-group-bgp-<groupName>" mode
#-------------------------------------------------------------------------------
bgpProfileNameMatcher = profileNameMatcher(
   lambda mode: globalBgpProfileConfigDir.config )

class MaintenanceProfileBgpCmd( CliCommand.CliCommandClass ):
   syntax = 'maintenance profile bgp PROFILE_NAME'
   noOrDefaultSyntax = 'maintenance profile bgp [ PROFILE_NAME ]'
   data = {
      'maintenance': maintenanceKwMatcher,
      'profile': profileMatcher,
      'bgp': matcherBgp,
      'PROFILE_NAME': bgpProfileNameMatcher,
   }

   handler = "BgpMaintenanceCliHandler.MaintenanceProfileBgpCmd_handler"
   noOrDefaultHandler = "BgpMaintenanceCliHandler." \
      "MaintenanceProfileBgpCmd_noOrDefaultHandler"

IntfGroupConfigMode.addCommandClass( MaintenanceProfileBgpCmd )
IntfBuiltinGroupConfigMode.addCommandClass( MaintenanceProfileBgpCmd )
BgpGroupConfigMode.addCommandClass( MaintenanceProfileBgpCmd )
BgpBuiltinGroupConfigMode.addCommandClass( MaintenanceProfileBgpCmd )

#-------------------------------------------------------------------------------
# [no] profile bgp <profileName>
# in "config-maintenance" mode
#-------------------------------------------------------------------------------

class BgpMaintenanceProfileConfigMode( BgpMaintenanceProfileMode,
                                       BasicCli.ConfigModeBase ):
   name = "Bgp Maintenance profile configuration"

   def __init__( self, parent, session, bgpProfile ):
      self.bgpProfile_ = bgpProfile
      self.session_ = session
      BgpMaintenanceProfileMode.__init__( self, self.bgpProfile_.name() )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#
# BgpProfile class: Holds cli state for each configured BGP profile
#
class BgpProfile( Profile ):
   def __init__( self, profileName ):
      self.name_ = profileName
      Profile.__init__( self )

   def addProfile( self, profileName ): # pylint: disable=arguments-renamed
      if profileName not in globalBgpProfileConfigDir.config:
         profileConfig = globalBgpProfileConfigDir.newConfig( profileName )
         Profile.addProfile( self, profileConfig.key )

   def remProfile( self, profileName ): # pylint: disable=arguments-renamed
      if profileName in globalBgpProfileConfigDir.config:
         profileConfig = globalBgpProfileConfigDir.config.get( profileName )
         Profile.remProfile( self, profileConfig.key )
         del globalBgpProfileConfigDir.config[ profileName ]

   def name( self ):
      return self.name_

class ProfileBgpCmd( CliCommand.CliCommandClass ):
   syntax = 'profile bgp PROFILE_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'profile': profileMatcher,
      'bgp': matcherBgp,
      'PROFILE_NAME': bgpProfileNameMatcher,
   }

   @staticmethod
   def delBgpProfileConfig_( mode, profileName ):
      bgpProfile = BgpProfile( profileName )
      bgpProfile.remProfile( profileName )

   handler = "BgpMaintenanceCliHandler.ProfileBgpCmd_handler"
   noOrDefaultHandler = "BgpMaintenanceCliHandler.ProfileBgpCmd_noOrDefaultHandler"

MaintenanceConfigMode.addCommandClass( ProfileBgpCmd )

#---------------------------------------------------------------------------------
# [ no | default ] initiator route-map MAPNAME ( inout | in | out )
# [ no | default ] initiator rcf ( inout | in | out ) FUNCTION
# and
# [ no | default ] initiator route-map -- removes all the route-map names
# [ no | default ] initiator rcf -- removes all the RCF config
# in "maint-profile-bgp" mode
#---------------------------------------------------------------------------------
class InitiatorPolicyCmd( CliCommand.CliCommandClass ):
   syntax = 'initiator route-map MAPNAME DIRECTION'
   noOrDefaultSyntax = 'initiator route-map [ MAPNAME DIRECTION ]'
   if RcfLibToggleLib.toggleRcfMaintenancePoaEnabled():
      syntax = '''initiator ( ( route-map MAPNAME DIRECTION ) | 
      ( rcf DIRECTION FUNCTION ) )'''
      noOrDefaultSyntax = '''initiator ( ( route-map [ MAPNAME DIRECTION ] ) | 
      ( rcf [ DIRECTION FUNCTION ] ) )'''
   data = {
      'initiator': 'Maintenance Mode Initiator',
      'route-map':
      'Configure route-map to override default inbound and outbound policy',
      'rcf':
      'Configure RCF to override default inbound and outbound policy',
      'MAPNAME': mapNameMatcher,
      'FUNCTION': rcfFunctionMatcher,
      'DIRECTION': CliMatcher.EnumMatcher( {
         'in': 'Apply inbound only',
         'out': 'Apply outbound only',
         'inout': 'Apply inbound and outbound',
      } ),
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      directionToAttr = {
         'in': 'policyIn',
         'out': 'policyOut',
         'inout': 'policyInOut',
      }
      if 'DIRECTION' in args:
         args[ 'directionAttr' ] = directionToAttr[ args[ 'DIRECTION' ] ]

   handler = "BgpMaintenanceCliHandler.InitiatorPolicyCmd_handler"
   noOrDefaultHandler = "BgpMaintenanceCliHandler." \
      "InitiatorPolicyCmd_noOrDefaultHandler"

BgpMaintenanceProfileConfigMode.addCommandClass( InitiatorPolicyCmd )

#-------------------------------------------------------------------------------
# [no] bgp <peerName> [ vrf <vrfName> ]
# in "config-maintenance" mode
#-------------------------------------------------------------------------------

class BgpMaintenanceConfigMode( BgpMaintenanceMode,
                                BasicCli.ConfigModeBase ):
   name = "Bgp Maintenance Mode configuration"

   def __init__( self, parent, session, bgpPeer, vrfName=None ):
      if vrfName:
         self.bgpPeer_ = dynamicUnitName( bgpPeer, vrfName )
      else:
         self.bgpPeer_ = dynamicUnitName( bgpPeer, 'default' )

      self.session_ = session
      BgpMaintenanceMode.__init__( self, self.bgpPeer_ )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class BgpMaintenanceConfigModeCmd( CliCommand.CliCommandClass ):
   syntax = 'bgp PEER [ vrf VRFNAME ]'
   noOrDefaultSyntax = syntax
   data = {
      'bgp': matcherBgpExec,
      'PEER': PeerCliExpression,
      'vrf': matcherVrf,
      'VRFNAME': CliMatcher.DynamicNameMatcher(
         getAllVrfNames,
         helpdesc='VRF name'
      ),
   }

   handler = "BgpMaintenanceCliHandler.BgpMaintenanceConfigModeCmd_handler"
   noOrDefaultHandler = "BgpMaintenanceCliHandler." \
      "BgpMaintenanceConfigModeCmd_noOrDefaultHandler"

MaintenanceConfigMode.addCommandClass( BgpMaintenanceConfigModeCmd )

#-------------------------------------------------------------------------------
# [ no | default ] quiesce
# in "config-maint-bgp-<peerName>" mode
#-------------------------------------------------------------------------------
class QuiesceCmd( CliCommand.CliCommandClass ):
   syntax = 'quiesce'
   noOrDefaultSyntax = syntax
   data = {
      'quiesce': quiesceMatcher,
   }

   handler = "BgpMaintenanceCliHandler.QuiesceCmd_handler"
   noOrDefaultHandler = "BgpMaintenanceCliHandler.QuiesceCmd_noOrDefaultHandler"

BgpMaintenanceConfigMode.addCommandClass( QuiesceCmd )

#-------------------------------------------------------------------------------
# "[no] profile bgp <profileName> default"
# in "config-maintenance" mode
#-------------------------------------------------------------------------------
class ProfileBgpDefaultCmd( CliCommand.CliCommandClass ):
   syntax = 'profile bgp PROFILE_NAME default'
   noOrDefaultSyntax = syntax

   data = {
      'maintenance': maintenanceKwMatcher,
      'profile': profileMatcher,
      'bgp': matcherBgp,
      'PROFILE_NAME': bgpProfileNameMatcher,
      'default': 'default profile',
   }

   handler = "BgpMaintenanceCliHandler.ProfileBgpDefaultCmd_handler"
   noOrDefaultHandler = "BgpMaintenanceCliHandler." \
      "ProfileBgpDefaultCmd_noOrDefaultHandler"

MaintenanceConfigMode.addCommandClass( ProfileBgpDefaultCmd )

def bgpProfileCleanup( mode ):
   for profileName in globalBgpProfileConfigDir.config:
      if profileName != defaultProfile:
         ProfileBgpCmd.delBgpProfileConfig_( mode, profileName )
   globalDefaultBgpProfile.profileName = defaultProfile

# pylint: disable-next=useless-object-inheritance
class BgpMaintRouteMapReferenceGatherer( object ):
   """
   This class gathers references to route maps in 'initiator route-map' commands.
   """
   @staticmethod
   def gather( feature ):
      references = set()
      configColl = globalBgpProfileConfigDir.config
      for profileName in configColl:
         config = configColl[ profileName ]
         for direction in [ "In", "Out", "InOut" ]:
            policyAttr = "policy" + direction
            rmName = getattr( config, policyAttr ).policyName
            if rmName:
               references.add( rmName )
      return references

UndefinedReferenceChecker.addReferenceGatherer( [ "Route map" ],
                                                BgpMaintRouteMapReferenceGatherer )

def Plugin( entityManager ):
   global globalBgpProfileConfigDir
   global globalDefaultBgpProfile
   global globalMaintenanceUnitInputDir
   profilesCleanupHook.addExtension( bgpProfileCleanup )
   globalBgpProfileConfigDir = ConfigMount.mount(
      entityManager, 'maintenance/profile/config/bgp',
      'BgpMaintenanceProfile::ConfigDir', 'w' )
   globalDefaultBgpProfile = ConfigMount.mount(
      entityManager, 'maintenance/profile/config/default/bgp',
      'Maintenance::Profile::DefaultProfile', 'w' )
   globalMaintenanceUnitInputDir = ConfigMount.mount(
      entityManager, 'maintenance/unit/input/cli',
      'Maintenance::Unit::Input', 'w' )
