#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import BasicCli
import CliMatcher
from CliDynamicSymbol import (
   LazyCallback, 
   CliDynamicPlugin,
)
import ShowCommand
import LazyMount
from CliPlugin.BgpGroupCli import bgpGroupNameMatcher
from CliPlugin.MaintenanceCliLib import profileNameMatcher
from CliPlugin.MaintenanceCliLib import showMaintenanceMatcher, defaultMatcher
from CliPlugin.MaintenanceModels import (
   BgpGroups, 
   BgpProfiles,
   MaintenanceDebugInfo,
)
from CliPlugin.MaintenanceModels import bgpGroupsHook, bgpProfilesHook
from CliPlugin.MaintenanceModels import showMaintenanceInterfaceHook
from CliPlugin.RouteMapCliModels import RouteMaps
from CliPlugin.MaintenanceModels import bgpMaintenanceDebugInfoHook
from CliPlugin.RoutingBgpCli import V4V6PeerKeyCliExpression, PeerCliExpression
from CliPlugin.RoutingBgpShowCli import allVrfExprFactory
import CliToken

bgpDefaultProfileDir = None
bgpGroupConfigDir = None
bgpProfileStatusDir = None
mapConfig = None
defaultInitiatorRmDir = None

#-------------------------------------------------------------------------------
# show maintenance groups bgp [ groupName ]
#
# 1. show maintenance groups bgp
# 2. show maintenance groups bgp <groupName>
#-------------------------------------------------------------------------------

bgpMatcher = CliMatcher.KeywordMatcher( 'bgp',
                                        helpdesc='BGP groups information' )

groupsMatcher = CliMatcher.KeywordMatcher( 'groups',
                                           helpdesc='Groups information' )

class ShowMaintenanceGroupsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show maintenance groups bgp [ GROUP_NAME ]"
   data = { "maintenance": showMaintenanceMatcher,
            "groups": groupsMatcher,
            "bgp": bgpMatcher,
            "GROUP_NAME": bgpGroupNameMatcher }

   handler = "BgpMaintenanceShowCliHandler.ShowMaintenanceGroupsCmd_handler"
   cliModel = BgpGroups

BasicCli.addShowCommandClass( ShowMaintenanceGroupsCmd )

#-------------------------------------------------------------------------------
# show maintenance profiles bgp [ profileName ]
#
# 1. show maintenance profiles bgp
# 2. show maintenance profiles bgp <profileName>
#-------------------------------------------------------------------------------

bgpProfileNameMatcher = profileNameMatcher( lambda mode:
                                            bgpProfileStatusDir.status )

profilesMatcher = CliMatcher.KeywordMatcher( 'profiles',
                                             helpdesc='Profiles information' )

class ShowMaintenanceProfilesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show maintenance profiles bgp [ PROFILE_NAME ]"
   data = { "maintenance": showMaintenanceMatcher,
            "profiles": profilesMatcher,
            "bgp": bgpMatcher,
            "PROFILE_NAME": bgpProfileNameMatcher }
   handler = "BgpMaintenanceShowCliHandler.handlerShowMaintenanceProfilesCmd"
   cliModel = BgpProfiles

BasicCli.addShowCommandClass( ShowMaintenanceProfilesCmd )

#-------------------------------------------------------------------------------
# "show maintenance bgp ip all [vrf <vrfName>]"
#-------------------------------------------------------------------------------
allMatcher = CliMatcher.KeywordMatcher( 'all',
                                        helpdesc="All the bgp neighbors" )

class ShowMaintenanceBgpIpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show maintenance bgp ip all [ VRF ]"
   data = { "maintenance": showMaintenanceMatcher,
            "bgp": bgpMatcher,
            "ip": CliToken.Ip.ipMatcherForShow,
            "all": allMatcher,
            "VRF": allVrfExprFactory, }
   handler = "BgpMaintenanceShowCliHandler.handlerShowMaintenanceBgpIpCmd"
   cliModel = "BgpMaintenanceShowCliHandler.BgpMaintenanceStatusVrf"

BasicCli.addShowCommandClass( ShowMaintenanceBgpIpCmd )

#-------------------------------------------------------------------------------
# "show maintenance bgp ipv6 all [vrf <vrfName>]"
#-------------------------------------------------------------------------------
class ShowMaintenanceBgpIPv6Cmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show maintenance bgp ipv6 all [ VRF ]"
   data = { "maintenance": showMaintenanceMatcher,
            "bgp": bgpMatcher,
            "ipv6": CliToken.Ipv6.ipv6MatcherForShow,
            "all": allMatcher,
            "VRF": allVrfExprFactory, }
   handler = "BgpMaintenanceShowCliHandler.handlerShowMaintenanceBgpIPv6Cmd"
   cliModel = "BgpMaintenanceShowCliHandler.BgpMaintenanceStatusVrf"

BasicCli.addShowCommandClass( ShowMaintenanceBgpIPv6Cmd )

#-------------------------------------------------------------------------------
# "show maintenance profiles bgp default"
#-------------------------------------------------------------------------------

class ShowDefaultProfilesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show maintenance profiles bgp default"
   data = { "maintenance": showMaintenanceMatcher,
            "profiles": profilesMatcher,
            "bgp": bgpMatcher,
            "default": defaultMatcher }
   handler = "BgpMaintenanceShowCliHandler." \
      "doShowMaintenanceBgpDefaultProfile_handler"
   cliModel = "BgpMaintenanceModels.BgpMaintenanceDefaultProfile"

BasicCli.addShowCommandClass( ShowDefaultProfilesCmd )

#-------------------------------------------------------------------------------
# "show maintenance bgp <addr> [vrf <vrfName>]"
#-------------------------------------------------------------------------------

class ShowAddrVrfCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show maintenance bgp PEER [ VRF ]"
   data = { "maintenance": showMaintenanceMatcher,
            "bgp": bgpMatcher,
            "PEER": V4V6PeerKeyCliExpression,
            "VRF": allVrfExprFactory, }

   handler = "BgpMaintenanceShowCliHandler.ShowAddrVrfCmd_handler"
   cliModel = "BgpMaintenanceShowCliHandler.BgpPeerMaintenanceInfoVrf"

BasicCli.addShowCommandClass( ShowAddrVrfCmd )

#-------------------------------------------------------------------------------
# "show maintenance debug bgp"
#-------------------------------------------------------------------------------

debugMatcher = CliMatcher.KeywordMatcher( 'debug',
                                          helpdesc='Debug information' )

class ShowMaintenanceDebugBgpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show maintenance debug bgp"
   data = { "maintenance": showMaintenanceMatcher,
            "debug": debugMatcher,
            "bgp": bgpMatcher }

   handler = "BgpMaintenanceShowCliHandler.ShowMaintenanceDebugBgpCmd_handler"
   cliModel = MaintenanceDebugInfo

BasicCli.addShowCommandClass( ShowMaintenanceDebugBgpCmd )

#-------------------------------------------------------------------------------
# show maintenance debug bgp <peer> [ vrf <vrf_name> ]
#-------------------------------------------------------------------------------

class ShowPeerDebugCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show maintenance debug bgp PEER [ VRF ]"
   data = { "maintenance": showMaintenanceMatcher,
            "debug": debugMatcher,
            "bgp": bgpMatcher,
            "PEER": PeerCliExpression,
            "VRF": allVrfExprFactory, }

   handler = "BgpMaintenanceShowCliHandler.ShowPeerDebugCmd_handler"
   cliModel = MaintenanceDebugInfo

BasicCli.addShowCommandClass( ShowPeerDebugCmd )

#-------------------------------------------------------------------------------
# show maintenance bgp receiver route-map
#-------------------------------------------------------------------------------
receiverMatcher = CliMatcher.KeywordMatcher( 'receiver',
                                             helpdesc='Show receiver route map' )
routemapMatcher = CliMatcher.KeywordMatcher( 'route-map',
                                             helpdesc='Show receiver route map' )

class ShowReceiverRouteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show maintenance bgp receiver route-map"
   data = { "maintenance": showMaintenanceMatcher,
            "bgp": bgpMatcher,
            "receiver": receiverMatcher,
            "route-map": routemapMatcher }
   handler = "BgpMaintenanceShowCliHandler.doShowMaintenanceBgpReceiverRm"
   cliModel = RouteMaps

BasicCli.addShowCommandClass( ShowReceiverRouteCmd )

def Plugin( entityManager ):
   global bgpGroupConfigDir, bgpProfileStatusDir
   global bgpDefaultProfileDir
   global mapConfig, defaultInitiatorRmDir

   bgpGroupsHook.addExtension(
      LazyCallback( "BgpMaintenanceShowCliHandler.showBgpGroupsHook" ) )
   bgpProfilesHook.addExtension(
      LazyCallback( "BgpMaintenanceShowCliHandler.showBgpProfilesHook" ) )
   bgpMaintenanceDebugInfoHook.addExtension(
      lambda mode=None, ignoreDisplay=False:
      CliDynamicPlugin( 'BgpMaintenanceShowCliHandler' )
      .showMaintenanceAllBgpDebugInfo( mode, ignoreDisplay=ignoreDisplay ) )
   bgpGroupConfigDir = LazyMount.mount( entityManager,
                                        'group/config/bgp',
                                        'BgpGroup::ConfigDir', 'r' )
   bgpProfileStatusDir = LazyMount.mount( entityManager,
                                          'maintenance/profile/status/bgp',
                                          'BgpMaintenanceProfile::StatusDir', 'r' )
   bgpDefaultProfileDir = LazyMount.mount(
      entityManager,
      'maintenance/profile/config/default/bgp',
      'Maintenance::Profile::DefaultProfile', 'r' )
   defaultInitiatorRmDir = LazyMount.mount(
      entityManager, 'maintenance/profile/config/default/initiatorRM/',
      'Routing::RouteMap::Config', 'r' )
   mapConfig = LazyMount.mount(
      entityManager, 'routing/routemap/config', 'Routing::RouteMap::Config', 'r' )
   showMaintenanceInterfaceHook.addExtension(
      LazyCallback( "BgpMaintenanceShowCliHandler.bgpShowMaintenanceIntf" ) )
