#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
from BgpLib import updateConfigAttrsAfMap
import CliCommand
from CliMatcher import (
    DynamicNameMatcher,
    IntegerMatcher,
)
from CliMode.BgpCommon import RoutingBgpBaseAfMode
from CliMode.BgpVpls import (
   BgpVplsGroupMode,
   BgpVplsGroupSignalingMode,
   BgpVplsInstanceMode,
)
import CliParser
from CliPlugin.RouteDistinguisher import RdDistinguisherMatcher
from CliPlugin.RouteMapCli import RtSooExtCommCliMatcher
from CliPlugin.RoutingBgpCli import (
   RouterBgpBaseMode,
   afModeExtensionHook,
   deleteRouterBgpMacVrfHook,
   RouterBgpAfUnsupportedPrefixListSharedModelet,
)
from CliPlugin.RoutingBgpInstanceCli import (
   SetGracefulRestartCmd,
)
from CliPlugin.RoutingBgpNeighborCli import (
   RouterBgpAfActivateCommand,
   SetNeighborGracefulRestartCmd,
   SetNeighborRcfCmd,
)
from CliPlugin import RcfCliLib
import CliToken.RoutingBgp as bgpTokens
import ConfigMount
from IpLibConsts import DEFAULT_VRF
import LazyMount
from PseudowireLib import vplsNameRegex
from Toggles.BgpCommonToggleLib import (
   toggleBgpVplsRcfImportExportEnabled,
)
from Toggles.PseudowireToggleLib import toggleVplsBgpSignalingEnabled
from TypeFuture import TacLazyType

PseudowireProfileDir = TacLazyType( 'Pseudowire::PseudowireProfileDir' )
BgpVplsGroupConfig = TacLazyType( 'Routing::Bgp::BgpVplsPlugin::BgpVplsGroupConfig' )

VPLS_AF_STR = "vpls"

# The mounted PseudowireProfileDir collection containing the common pseudowire
# profiles.
pwProfileDir = None

# Matcher that matches (and auto-completes) a common profile name.
pwProfileNameMatcher = DynamicNameMatcher(
      lambda mode: ( pwProfileDir.profile ),
      'MPLS pseudowire profile name' )

# -----------------------------------------------------------------------------
# Globals
# -----------------------------------------------------------------------------
bgpVplsConfigDir = None

class RouterBgpVplsAfMode( RoutingBgpBaseAfMode, BasicCli.ConfigModeBase ):
   name = "BGP address family VPLS configuration"

   def __init__( self, parent, session ):
      self.vrfName = DEFAULT_VRF
      RoutingBgpBaseAfMode.__init__(
         self, VPLS_AF_STR, modeKey="router-bgp-af-vpls" )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpVplsAfModelet( CliParser.Modelet ):
   pass

RouterBgpVplsAfMode.addModelet( RouterBgpVplsAfModelet )
RouterBgpVplsAfMode.addModelet( RouterBgpAfUnsupportedPrefixListSharedModelet )

# -------------------------------------------------------------------------------
# "[no|default] neighbor <ipaddr|peerGroup> activate"
# -------------------------------------------------------------------------------
RouterBgpVplsAfModelet.addCommandClass( RouterBgpAfActivateCommand )

# -------------------------------------------------------------------------------
# [no|default] neighbor PEER rcf [in|out] FUNCTION
# -------------------------------------------------------------------------------
RouterBgpVplsAfModelet.addCommandClass( SetNeighborRcfCmd )

# -------------------------------------------------------------------------------
# [no|default] graceful-restart
# -------------------------------------------------------------------------------
RouterBgpVplsAfModelet.addCommandClass( SetGracefulRestartCmd )

# -------------------------------------------------------------------------------
# [no|default] neighbor PEER graceful-restart [disabled]
# -------------------------------------------------------------------------------
RouterBgpVplsAfModelet.addCommandClass( SetNeighborGracefulRestartCmd )

# -------------------------------------------------------------------------------
# "[no|default] address-family vpls"
# -------------------------------------------------------------------------------
class VplsAfModeCmd( CliCommand.CliCommandClass ):
   syntax = "address-family vpls"
   noOrDefaultSyntax = syntax

   data = {
      "address-family": bgpTokens.addrFamily,
      "vpls": "Virtual private LAN service",
   }
   handler = "BgpVplsHandler.VplsAfModeCmd_handler"
   noOrDefaultHandler = "BgpVplsHandler.VplsAfModeCmd_noOrDefaultHandler"

# ------------------------------------------------------------------------------
# (config-router-bgp)# [no|default] vpls <inst>
# ------------------------------------------------------------------------------
class BgpVplsInstanceConfigMode( BgpVplsInstanceMode, BasicCli.ConfigModeBase ):
   name = "BGP VPLS instance configuration"

   def __init__( self, parent, session, vplsName ):
      BgpVplsInstanceMode.__init__( self, vplsName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      if vplsName not in bgpVplsConfigDir.instance:
         bgpVplsConfigDir.newInstance( vplsName )

   def config( self ):
      return bgpVplsConfigDir.instance[ self.vplsName ]

class BgpVplsCommand( CliCommand.CliCommandClass ):
   syntax = "vpls VPLS_INSTANCE_NAME"
   noOrDefaultSyntax = syntax
   data = {
      "vpls": "Virtual Private LAN Service",
      "VPLS_INSTANCE_NAME": DynamicNameMatcher(
                                lambda mode: bgpVplsConfigDir.instance,
                                helpdesc="VPLS instance name",
                                pattern=vplsNameRegex ),
   }
   handler = "BgpVplsHandler.BgpVplsCommand_handler"
   noOrDefaultHandler = "BgpVplsHandler.BgpVplsCommand_noOrDefaultHandler"

afModeExtensionHook.addAfModeExtension( VPLS_AF_STR, RouterBgpVplsAfMode )
RouterBgpBaseMode.addCommandClass( VplsAfModeCmd )
updateConfigAttrsAfMap( { "vpls": [ "Vpls", "AfVpls" ] }, vrfSupported=False )

# Command for BGP VPLS config
RouterBgpBaseMode.addCommandClass( BgpVplsCommand )

# ------------------------------------------------------------------------------
# (config-router-bgp-vpls-<>)# [no|default] pseudowire <group-name>
# ------------------------------------------------------------------------------
class BgpVplsGroupConfigMode( BgpVplsGroupMode, BasicCli.ConfigModeBase ):
   name = "BGP VPLS group configuration"

   def __init__( self, parent, session, vplsName, groupName ):
      BgpVplsGroupMode.__init__( self, vplsName, groupName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

      instanceConfig = parent.config()
      if groupName not in instanceConfig.group:
         instanceConfig.newGroup( groupName )

   def config( self ):
      return self.parent_.config().group[ self.groupName ]

class BgpVplsGroupCommand( CliCommand.CliCommandClass ):
   syntax = "pseudowire GROUP_NAME"
   noOrDefaultSyntax = syntax
   data = {
      "pseudowire": "Pseudowire configuration",
      "GROUP_NAME": DynamicNameMatcher(
                        lambda mode: mode.config().group,
                        helpdesc="VPLS group name",
                        pattern=vplsNameRegex ),
   }
   handler = "BgpVplsHandler.BgpVplsGroupCommand_handler"
   noOrDefaultHandler = "BgpVplsHandler.BgpVplsGroupCommand_noOrDefaultHandler"

BgpVplsInstanceConfigMode.addCommandClass( BgpVplsGroupCommand )

# ------------------------------------------------------------------------------
# (config-router-bgp-vpls-<>-group-<>)# [no|default] rd <route-distinguisher>
# ------------------------------------------------------------------------------
class RdCommand( CliCommand.CliCommandClass ):
   syntax = "rd RD"
   noOrDefaultSyntax = "rd ..."
   data = {
      "rd": "BGP route distinguisher",
      "RD": RdDistinguisherMatcher( "BGP route distinguisher" ),
   }
   handler = "BgpVplsHandler.RdCommand_handler"
   noOrDefaultHandler = "BgpVplsHandler.RdCommand_noOrDefaultHandler"

BgpVplsGroupConfigMode.addCommandClass( RdCommand )

# ------------------------------------------------------------------------------
# (config-router-bgp-vpls-<>-group-<>)# [no|default] route-target import/export
# ------------------------------------------------------------------------------
class RtCommand( CliCommand.CliCommandClass ):
   syntax = "route-target ( ( import [ export ] ) | export ) vpls RT"
   noOrDefaultSyntax = (
      "route-target ( ( ( import [ export ] ) | export ) vpls RT ) | all" )
   data = {
      "route-target": "Route target",
      "import": "Route import",
      "export": "Route export",
      "vpls": "Import or export only VPLS routes",
      "RT": RtSooExtCommCliMatcher( "Route target" ),
      "all": "Remove all import and export route targets",
   }
   handler = "BgpVplsHandler.RtCommand_handler"
   noOrDefaultHandler = "BgpVplsHandler.RtCommand_noOrDefaultHandler"

BgpVplsGroupConfigMode.addCommandClass( RtCommand )

# ------------------------------------------------------------------------------
# [no|default] route-target import/export vpls rcf <rcf>
# ------------------------------------------------------------------------------
class RtRcfCommand( CliCommand.CliCommandClass ):
   syntax = "route-target ( ( import [ export ] ) | export ) vpls rcf RCF_FUNCTION"
   noOrDefaultSyntax = syntax.replace( 'RCF_FUNCTION', '...' )
   data = {
      "route-target": "Route target",
      "import": "Route import",
      "export": "Route export",
      "vpls": "Import or export only VPLS routes",
      "rcf": bgpTokens.rcf,
      "RCF_FUNCTION": RcfCliLib.rcfFunctionMatcher,
   }
   handler = "BgpVplsHandler.RtRcfCommand_handler"
   noOrDefaultHandler = "BgpVplsHandler.RtRcfCommand_noOrDefaultHandler"

if toggleBgpVplsRcfImportExportEnabled():
   BgpVplsGroupConfigMode.addCommandClass( RtRcfCommand )

# ------------------------------------------------------------------------------
# (config-router-bgp-vpls-<>-group-<>)# [no|default] signaling
# ------------------------------------------------------------------------------
class BgpVplsGroupSignalingConfigMode( BgpVplsGroupSignalingMode,
                                       BasicCli.ConfigModeBase ):
   name = "BGP VPLS group signaling configuration"

   def __init__( self, parent, session, vplsName, groupName ):
      BgpVplsGroupSignalingMode.__init__( self, vplsName, groupName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def config( self ):
      return self.parent_.config()

class BgpSignalingCommand( CliCommand.CliCommandClass ):
   syntax = "signaling"
   noOrDefaultSyntax = syntax
   data = {
      "signaling": "BGP signaling configuration for VPLS"
   }
   handler = "BgpVplsHandler.BgpSignalingCommand_handler"
   noOrDefaultHandler = "BgpVplsHandler.BgpSignalingCommand_noOrDefaultHandler"

if toggleVplsBgpSignalingEnabled():
   BgpVplsGroupConfigMode.addCommandClass( BgpSignalingCommand )

# ------------------------------------------------------------------------------
# (config-router-bgp-vpls-<>-group-<>-signaling)# [no|default] ve-id VE_ID
# ------------------------------------------------------------------------------
class BgpSignalingVeIdCommand( CliCommand.CliCommandClass ):
   syntax = "ve-id VE_ID"
   noOrDefaultSyntax = "ve-id ..."
   veIdMin_ = BgpVplsGroupConfig.bgpSignalingVeIdMin
   veIdMax_ = BgpVplsGroupConfig.bgpSignalingVeIdMax
   data = {
      "ve-id": "Configure the local VPLS Edge identifier",
      'VE_ID': IntegerMatcher( veIdMin_, veIdMax_,
                               helpdesc="Local VPLS Edge identifier for the "
                                        "pseudowire group" ),
   }
   handler = "BgpVplsHandler.BgpSignalingVeIdCommand_handler"
   noOrDefaultHandler = "BgpVplsHandler.BgpSignalingVeIdCommand_noOrDefaultHandler"

BgpVplsGroupSignalingConfigMode.addCommandClass( BgpSignalingVeIdCommand )

# ------------------------------------------------------------------------------
# (config-router-bgp-vpls-<>-group-<>-signaling)# [no|default] ve-block-size SIZE
# ------------------------------------------------------------------------------
class BgpSignalingVeBlockSizeCommand( CliCommand.CliCommandClass ):
   syntax = "ve-block-size SIZE"
   noOrDefaultSyntax = "ve-block-size ..."
   sizeMin_ = BgpVplsGroupConfig.bgpSignalingVbsMin
   sizeMax_ = BgpVplsGroupConfig.bgpSignalingVbsMax
   sizeDefault_ = BgpVplsGroupConfig.bgpSignalingVbsDefault
   data = {
      "ve-block-size": f"Configure the size of each generated label block "
                       f"(default {sizeDefault_})",
      'SIZE': IntegerMatcher( sizeMin_, sizeMax_,
                              helpdesc=f"The number of labels in each generated "
                                       f"label block (default {sizeDefault_})" ),
   }
   handler = "BgpVplsHandler.BgpSignalingVeBlockSizeCommand_handler"
   noOrDefaultHandler = \
      "BgpVplsHandler.BgpSignalingVeBlockSizeCommand_noOrDefaultHandler"

BgpVplsGroupSignalingConfigMode.addCommandClass( BgpSignalingVeBlockSizeCommand )

# ------------------------------------------------------------------------------
# (config-router-bgp-vpls-<>-group-<>-signaling)# [no|default] profile PROFILE_NAME
# ------------------------------------------------------------------------------
class BgpSignalingProfileCommand( CliCommand.CliCommandClass ):
   syntax = "profile PROFILE_NAME"
   noOrDefaultSyntax = "profile ..."
   data = {
      "profile": "Configure the MPLS pseudowire profile to use",
      'PROFILE_NAME': pwProfileNameMatcher
   }
   handler = "BgpVplsHandler.BgpSignalingProfileCommand_handler"
   noOrDefaultHandler = \
      "BgpVplsHandler.BgpSignalingProfileCommand_noOrDefaultHandler"

BgpVplsGroupSignalingConfigMode.addCommandClass( BgpSignalingProfileCommand )

def Plugin( entityManager ):
   global bgpVplsConfigDir
   bgpVplsConfigDir = ConfigMount.mount(
      entityManager,
      "vpls/bgp/config",
      "Routing::Bgp::BgpVplsPlugin::BgpVplsConfigDir",
      "w" )

   # Also mount the common pseudowire profiles to get the list of profile names (for
   # auto-completion when configuring BGP signaling).
   if toggleVplsBgpSignalingEnabled():
      global pwProfileDir
      pwProfileDir = LazyMount.mount( entityManager,
                                      PseudowireProfileDir.mountPath,
                                      "Pseudowire::PseudowireProfileDir",
                                      "r" )

   # Clear all config with "no router bgp" -- not worth adding a new hook so using
   # deleteRouterBgpMacVrfHook here. Note that "bgpVplsConfigDir.instance.clear"
   # can't be passed directly to deleteRouterBgpMacVrfHook.addExtension because
   # bgpVplsConfigDir is a LazyMount and not accessible during Cli initialization.
   def clearBgpVplsConfig():
      bgpVplsConfigDir.instance.clear()

   deleteRouterBgpMacVrfHook.addExtension( clearBgpVplsConfig )
