#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-in

import ArnetLib
import CliCommand
import CliMatcher
import CliParserCommon
import Tac
import Tracing
from CliPlugin.RouteMapCli import asdotNumMatcher, asplainNumMatcher

t0 = Tracing.trace0

class DomainIdMatcher( CliMatcher.PatternMatcher ):
   __slots__ = ( 'globalAdminMatcher_', 'localAdminMatcher_' )

   def __init__( self, globalAdminMatcher ):
      # Apply the pattern in layers where a generic integer pattern is
      # applied to take in domain-id which marks two numeric
      # sub-tokens separated by ':'. Apply the global and local admin
      # matchers on either side of separator respectively.
      pattern = r'([0-9\.]+):([0-9]+)'
      super().__init__(
         pattern,
         helpdesc="Domain Identifier",
         helpname="ASN(asplain):local admin or ASN(asdot):local admin" )
      self.globalAdminMatcher_ = globalAdminMatcher
      self.localAdminMatcher_ = CliMatcher.IntegerMatcher(
         0, ArnetLib.U16_MAX_VALUE, helpdesc="2 byte local admin field" )

   def match( self, mode, context, token ):
      # Override the base match method to combine global admin and
      # local admin.
      result = self.re_.match( token )
      if result is None:
         t0( "No match for", token )
         return CliParserCommon.noMatch
      groups = result.groups()
      globalAdminToken = groups[ 0 ]
      localAdminToken = groups[ 1 ]
      globalAdmin = self.matchAdminToken( self.globalAdminMatcher_,
                                          globalAdminToken )
      localAdmin = self.matchAdminToken( self.localAdminMatcher_,
                                         localAdminToken )
      if globalAdmin == CliParserCommon.noMatch or \
         localAdmin == CliParserCommon.noMatch:
         return CliParserCommon.noMatch
      domainToken = str( globalAdmin ) + ':' + str( localAdmin )
      return CliParserCommon.MatchResult( domainToken, domainToken )

   def matchAdminToken( self, adminMatcher, token ):
      t0( "admin matcher", adminMatcher, "token", token )
      adminValue = adminMatcher.match( None, None, token )
      if adminValue == CliParserCommon.noMatch:
         t0( "Invalid token", token )
      return adminValue

   def completions( self, mode, context, token ):
      # Override completions to match the token with global and local
      # admin matchers. Our parent matcher is too generic and will
      # allow completions for invalid tokens in global/local
      # admin. This results in ambiguous command error instead of
      # invalid input error when attempting match domain-id tokens
      # like '10:', '1:65536', etc.
      if token == '' or \
         ( self.match( mode, context, token ) is not CliParserCommon.noMatch ):
         return [ CliParserCommon.Completion( self.helpname_, self.helpdesc_,
                                              False, common=self.common_ ) ]
      return []

class DomainIdExpression( CliCommand.CliExpression ):
   """DOMAIN-ID is a 6-octet field that represents a domain. It is
   composed of a 4-octet Global Administrator sub-field and a 2-octet
   Local Administrator sub-field. The Global Administrator sub-field
   MAY be filled with an Autonomous System Number (ASN), an IPv4
   address, or any value that guarantees the uniqueness of the DOMAIN-
   ID when the tenant network is connected to multiple Operators.
   Current model is to match ASN formatted token for global admin
   where ASN can be plain number or dotted format. See ArnetLib.py for
   examples. This can be extended to match on IP address if/when
   needed.
   """
   expression = 'ASPLAIN:LOCAL_ADMIN | ASDOT:LOCAL_ADMIN'
   data = { 'ASPLAIN:LOCAL_ADMIN' : DomainIdMatcher( asplainNumMatcher ),
            'ASDOT:LOCAL_ADMIN' : DomainIdMatcher( asdotNumMatcher ),
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      domainIdStr = None
      globalAdminType = 'unknown'
      if 'ASPLAIN:LOCAL_ADMIN' in args:
         domainIdStr = args[ 'ASPLAIN:LOCAL_ADMIN' ]
         globalAdminType = 'asnAsPlain'
      elif 'ASDOT:LOCAL_ADMIN' in args:
         domainIdStr = args[ 'ASDOT:LOCAL_ADMIN' ]
         globalAdminType = 'asnAsDot'
      if not domainIdStr:
         return
      t0( "domainIdStr", domainIdStr )
      adminFields = domainIdStr.split( ':' )
      if not adminFields or len( adminFields ) != 2:
         return
      globalAdmin = ArnetLib.asnStrToNum( adminFields[ 0 ] )
      localAdmin = ArnetLib.asnStrToNum( adminFields[ 1 ], minValue=0,
                                        maxValue=ArnetLib.U16_MAX_VALUE )
      if globalAdmin is None or localAdmin is None:
         return
      domainId = Tac.Type( 'Routing::Bgp::VpnDomainId' ).getVpnDomainId(
         globalAdmin, localAdmin )
      args[ 'DOMAIN_ID' ] = Tac.Value( 'Routing::Bgp::VpnDomainIdConfig',
                                       globalAdminType, domainId )

tokenDomain = CliMatcher.KeywordMatcher(
   'domain', helpdesc="domain identifier configuration" )
tokenIdentifier = CliMatcher.KeywordMatcher(
   'identifier',
   helpdesc="domain identifier configuration" )
tokenRemote = CliMatcher.KeywordMatcher(
   'remote', helpdesc="remote domain identifier configuration" )
tokenAttach = CliMatcher.KeywordMatcher(
   'attach', helpdesc="attach domain identifier configuration" )
