#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import tempfile

import BasicCli
import BiosLib
import CliCommand
import CliMatcher
import CliParser
import FileReplicationCmds
import Fru
import Logging
import ShowCommand
import Url

from AbootLogMsgs import HARDWARE_SYSTEM_ABOOT_UPDATE_FAILED, \
   HARDWARE_SYSTEM_ABOOT_UPDATE_SUCCEEDED
from BiosInstallLib import ( ABOOT_UPDATE_DIR, ABOOT_UPDATE_HISTORY,
                             InstallerErrorCode, Supervisor )
from CliPlugin import BiosCliModel, BiosCliLib
from CliPlugin.BiosCliLib import isSpiLocked, isSPIUpdateEnabled
from CliPlugin.BiosInstall import BiosInstaller, readHistory
from CliToken.Install import installMatcher, sourceMatcher, reloadMatcher, nowMatcher
from CliToken.Platform import platformMatcherForShow
from Tac import DISCARD, run, SystemCommandError
from Toggles.AbootEosToggleLib import toggleShowQueuedUpdatesEnabled

def guardFieldUpdateableAboot( mode, token ):
   # If the spi flash is locked we must check that aboot spi updating is supported
   sbStatus = BiosCliLib.getSecurebootStatus( mode.entityManager )

   if not sbStatus.loadFailed and sbStatus.supported:
      if sbStatus.hardwiredSpiFlashWP:
         return CliParser.guardNotThisPlatform

      if isSpiLocked( sbStatus ) and not isSPIUpdateEnabled( sbStatus ):
         return CliParser.guardNotThisAbootVersion

   return None

def guardSupervisor( mode, token ):
   '''
   Only display supervisor tokens in the case that they can be used. Must be a
   modular system, and the standby token is only available if the peer is in standby
   mode.
   '''
   entityMibRoot = BiosCliLib.getEntityMibRoot( mode.entityManager )

   if BiosCliLib.isModular( entityMibRoot ):
      if token == 'active':
         return None

      redundancyStatus = BiosCliLib.getRedundancyStatus( mode.entityManager )
      if BiosCliLib.isPeerStandby( redundancyStatus ) and token == 'standby':
         return None

   return CliParser.guardNotThisPlatform

biosMatcherForShow = CliMatcher.KeywordMatcher( 'bios', helpdesc='BIOS info' )
historyMatcherForShow = CliCommand.guardedKeyword( 'history',
                                                   helpdesc='BIOS install history',
                                                   guard=guardFieldUpdateableAboot )

biosMatcherForInstall = CliCommand.guardedKeyword( 'bios', helpdesc='BIOS install',
                                                   guard=guardFieldUpdateableAboot )

supervisorEnum = CliMatcher.EnumMatcher( {
   'active': 'Install only on the active supervisor',
   'standby': 'Install only on the standby supervisor',
} )
supervisorMatcher = CliCommand.Node( matcher=supervisorEnum, guard=guardSupervisor )

def _readAufDir( model, aufDir ):
   import Auf
   if os.path.exists( aufDir ):
      for filename in os.listdir( aufDir ):
         if filename.endswith( '.auf' ):
            aufObj = Auf.Auf( os.path.join( aufDir, filename ) )
            model.queuedVersions.append( aufObj.name )

def populateBiosVersions( mode, model, standby=False ):
   ( running, sysName ) = BiosCliModel.getRunningVersionAndSysName( mode,
                                                                    standby=standby )
   if not sysName:
      return
   model.sysName = sysName
   model.runningVersion = BiosCliModel.BiosVersion().toModel( running )

   norcal = running.norcal
   abootReader = BiosLib.getAbootReader( norcal )

   try:
      ( programmed, fallback ) = \
                     abootReader.getProgrammedAndFallbackVersions( standby=standby )
   except SystemCommandError:
      programmed, fallback = None, None

   if programmed:
      programmed.reconcileMissingFields( running )
      model.programmedVersion = BiosCliModel.BiosVersion().toModel( programmed )

   if fallback:
      fallback.reconcileMissingFields( running )
      model.fallbackVersion = BiosCliModel.BiosVersion().toModel( fallback )

   if ( running.norcalId is None or running.norcalId in [ 3, 5 ] ):
      mode.addError( 'Reading flash not supported on this BIOS version' )
   elif not programmed:
      mode.addError( 'Unable to read flash%s' %
                     ( ' on standby' if standby else '' ) )

   if toggleShowQueuedUpdatesEnabled():
      aufDir = Url.parseUrl( ABOOT_UPDATE_DIR,
                             Url.Context( *Url.urlArgsFromMode( mode ) ) )
      aufDir = aufDir.localFilename()
      if not standby:
         _readAufDir( model, aufDir )
      else:
         simulation = 'SIMULATION_VMID' in os.environ
         with tempfile.TemporaryDirectory() as tmpDir:
            cmd = FileReplicationCmds.copyFile( Fru.slotId(), tmpDir,
                                                loopback=simulation,
                                                source=aufDir, useKey=True,
                                                peerSource=( not simulation ),
                                                preservePerm=True )
            try:
               run( cmd, asRoot=True, stdout=DISCARD, stderr=DISCARD )
            except SystemCommandError as e:
               FILE_ERROR_CODE = 23
               if e.error != FILE_ERROR_CODE:
                  mode.addError( f'Failed to fetch info from standby supervisor:'
                                 f' error code { e.error }' )
            _readAufDir( model, tmpDir )

def showBios( mode, args ):
   result = BiosCliModel.DualSupBiosVersions()
   populateBiosVersions( mode, result )
   BiosCliLib.populateBiosDeprecated( mode, result )

   if not result.sysName:
      mode.addError( 'Unable to determinate system information' )
      return None

   entityMibRoot = BiosCliLib.getEntityMibRoot( mode.entityManager )
   redundancyStatus = BiosCliLib.getRedundancyStatus( mode.entityManager )
   if BiosCliLib.hasStandby( entityMibRoot, redundancyStatus ):
      result.standbyVersions = BiosCliModel.BiosVersions()

      populateBiosVersions( mode, result.standbyVersions, standby=True )
      BiosCliLib.populateBiosDeprecated( mode, result.standbyVersions, standby=True )

      if not result.standbyVersions.sysName:
         mode.addError( 'Unable to determinate standby system information' )
         return None

   return result

def populateBiosHistory( mode, model, historyFile, standby=False ):
   ( _, sysName ) = BiosCliModel.getRunningVersionAndSysName( mode, standby=standby )
   model.sysName = sysName

   try:
      for history in readHistory( historyFile, standby=standby ):
         model.insert( history.epoch, history.name, history.sha, history.errorCode )
   except OSError:
      # No history - go ahead and print an empty table
      pass

def showBiosHistory( mode, args ):
   detail = 'detail' in args

   result = BiosCliModel.DualSupBiosHistory( _detail=detail )

   updateDir = Url.parseUrl( ABOOT_UPDATE_DIR,
                             Url.Context( *Url.urlArgsFromMode( mode ) ) )
   historyFile = os.path.join( updateDir.localFilename(), ABOOT_UPDATE_HISTORY )

   populateBiosHistory( mode, result, historyFile )

   if not result.sysName:
      mode.addError( 'Unable to determinate system information' )
      return None

   entityMibRoot = BiosCliLib.getEntityMibRoot( mode.entityManager )
   redundancyStatus = BiosCliLib.getRedundancyStatus( mode.entityManager )
   if BiosCliLib.hasStandby( entityMibRoot, redundancyStatus ):
      result.standbyHistory = BiosCliModel.BiosHistory( _detail=detail )

      populateBiosHistory( mode, result.standbyHistory, historyFile, standby=True )

      if not result.standbyHistory.sysName:
         mode.addError( 'Unable to determinate standby system information' )
         return None

   return result

def logInstallStatus( mode, status, sup, isModular ):
   supName = ""
   if isModular and sup in [ Supervisor.ACTIVE, Supervisor.STANDBY ]:
      supName = " on the %s" % str( sup )

   if status == InstallerErrorCode.SUCCESS:
      mode.addMessage( "BIOS installation succeeded%s." % supName )
   elif status == InstallerErrorCode.SETUP_SUCCESS:
      mode.addMessage( "BIOS installation setup succeeded%s." % supName )
   elif status == InstallerErrorCode.ABORT:
      mode.addMessage( "BIOS installation aborted%s." % supName )
   elif status == InstallerErrorCode.FAIL:
      mode.addError( "BIOS installation failed%s." % supName )
   elif status == InstallerErrorCode.UPGRADE_FAIL:
      mode.addError( "BIOS installation failed in a partially flashed state%s."
                     % supName )
      Logging.log( HARDWARE_SYSTEM_ABOOT_UPDATE_FAILED, "Flashrom failed%s."
                   % supName )
   else:
      mode.addError( "BIOS installation returned an unknown state%s." % supName )

def finishInstall( status, installer ):
   if status == InstallerErrorCode.SUCCESS:
      Logging.log( HARDWARE_SYSTEM_ABOOT_UPDATE_SUCCEEDED )
   elif status == InstallerErrorCode.FAIL:
      Logging.log( HARDWARE_SYSTEM_ABOOT_UPDATE_FAILED, "Invalid AUF." )

   # Let the installer only reboot on successful results
   installer.finish()

def checkInstallStatus( mode, installStatus, installer, sup, isModular, hasStandby ):
   successResults = [ InstallerErrorCode.SUCCESS, InstallerErrorCode.SETUP_SUCCESS ]

   if sup in [ Supervisor.ALL, Supervisor.ACTIVE ]:
      logInstallStatus( mode, installStatus.active, Supervisor.ACTIVE, isModular )
   if hasStandby and sup in [ Supervisor.ALL, Supervisor.STANDBY ]:
      logInstallStatus( mode, installStatus.standby, Supervisor.STANDBY, isModular )

   if not hasStandby or sup == Supervisor.ACTIVE:
      finishInstall( installStatus.active, installer )
   elif sup == Supervisor.STANDBY:
      finishInstall( installStatus.standby, installer )
   else: # ALL case on modular with standby
      if installStatus.active == installStatus.standby:
         finishInstall( installStatus.active, installer )
      else:
         # Different results on each supervisor
         if installStatus.active in successResults and \
            installStatus.standby in successResults:
            # Both succeeded, but one needs Aboot to process the update
            finishInstall( InstallerErrorCode.SETUP_SUCCESS, installer )
         elif installStatus.active in successResults or \
              installStatus.standby in successResults:
            # One of the updates failed, the other succeeded
            # Let the installer finish installation for the successful one
            finishInstall( InstallerErrorCode.SETUP_SUCCESS, installer )
         else:
            # None succeeded, one failed, the other was cancelled
            if installStatus.active == InstallerErrorCode.ABORT:
               finishInstall( installStatus.standby, installer )
            else:
               finishInstall( installStatus.active, installer )

def doInstall( mode, args ):
   src = args[ 'SOURCE' ]
   reboot = 'reload' in args
   now = 'now' in args

   sup = Supervisor.ALL
   if 'SUPERVISOR' in args:
      if 'active' in args[ 'SUPERVISOR' ]:
         sup = Supervisor.ACTIVE
      elif 'standby' in args[ 'SUPERVISOR' ]:
         sup = Supervisor.STANDBY
      else:
         mode.addError( "Supervisor token not accepted" )
         return

   entityMibRoot = BiosCliLib.getEntityMibRoot( mode.entityManager )
   redundancyStatus = BiosCliLib.getRedundancyStatus( mode.entityManager )
   isModular = BiosCliLib.isModular( entityMibRoot )
   hasStandby = BiosCliLib.hasStandby( entityMibRoot, redundancyStatus )
   installer = BiosInstaller( mode, sup, reboot, now )
   installStatus = installer.run( src )
   checkInstallStatus( mode, installStatus, installer, sup, isModular, hasStandby )

#--------------------------------------------------------------------------------
# show platform bios
#--------------------------------------------------------------------------------

class ShowBiosCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform bios'
   data = {
      'platform': platformMatcherForShow,
      'bios': biosMatcherForShow,
   }
   cliModel = BiosCliModel.DualSupBiosVersions
   handler = showBios

BasicCli.addShowCommandClass( ShowBiosCmd )

#--------------------------------------------------------------------------------
# show platform bios history [ detail ]
#--------------------------------------------------------------------------------

class ShowBiosHistoryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform bios history [ detail ]'
   data = {
      'platform': platformMatcherForShow,
      'bios': biosMatcherForShow,
      'history': historyMatcherForShow,
      'detail': 'Detailed history',
   }
   cliModel = BiosCliModel.DualSupBiosHistory
   handler = showBiosHistory

BasicCli.addShowCommandClass( ShowBiosHistoryCmd )

#--------------------------------------------------------------------------------
# install bios source SOURCE [ SUPERVISOR ] [ reload ] [ now ]
#--------------------------------------------------------------------------------
class InstallBiosCmd( CliCommand.CliCommandClass ):
   syntax = 'install bios source SOURCE [ SUPERVISOR ] [ reload ] [ now ]'
   data = {
      'install': installMatcher,
      'bios': biosMatcherForInstall,
      'source': sourceMatcher,
      'SOURCE': Url.UrlMatcher( fsFunc=lambda fs: ( fs.fsType == 'network' or
                                                    fs.fsType == 'flash' ),
                                helpdesc='Source path', allowAllPaths=True ),
      'SUPERVISOR': supervisorMatcher,
      'reload': reloadMatcher,
      'now': nowMatcher,
   }
   handler = doInstall

BasicCli.EnableMode.addCommandClass( InstallBiosCmd )
