#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import tempfile

import BasicCliUtil
import BiosLib
import FileReplicationCmds
import Fru
import Tac
import Url

from AbootVersion import AbootVersion
import BiosInstallLib
from BiosInstallLib import ( HistoryCode, InstallStatus, InstallerErrorCode,
                             Supervisor )
from CliPlugin import ReloadCli, BiosCliLib

class HistoryReader:
   def __init__( self, line ):
      line = line.strip()
      segments = line.split( ',' )

      self.epoch = int( segments[ 0 ] )
      self.sha = segments[ 1 ]
      self.name = segments[ 2 ]
      self.errorCode = int( segments[ 3 ] )

def readHistory( fp, standby=False ):
   for line in BiosLib.readFileOnSupe( fp, standby=standby ):
      yield HistoryReader( line )

class CliInstaller( BiosInstallLib.LocalInstaller ):
   def __init__( self, updateDir, mode, isModular=False ):
      super().__init__( updateDir )

      self.mode = mode
      self.isModular = isModular

      # Sysdb entities
      self.entityMibRoot = BiosCliLib.getEntityMibRoot( self.mode.entityManager )
      self.sbStatus = BiosCliLib.getSecurebootStatus( self.mode.entityManager,
                                                      standby=self.isStandby )

   @property
   def isStandby( self ):
      return False

   def printSupeMsg( self, msg ):
      self.logger.printMsg( msg, standby=self.isStandby if self.isModular else None )

   def isSpiLocked( self ):
      return BiosCliLib.isSpiLocked( self.sbStatus )

   def isSpiUpdateEnabled( self ):
      return BiosCliLib.isSPIUpdateEnabled( self.sbStatus )

   def isSpiFlashWpHardwired( self ):
      return self.sbStatus.hardwiredSpiFlashWP

   def getAbootVersion( self ):
      # Use sysdb in simulation to get the running version. On the product just
      # read directly from /proc/cmdline
      if self.simulation:
         return AbootVersion( BiosCliLib.getSysdbAbootVersion( self.entityMibRoot ) )
      return BiosLib.getAbootVersion( standby=self.isStandby )

class CliStandbyInstaller( CliInstaller ):
   def __init__( self, updateDir, mode ):
      # Running sup's slot ID for replication commands.
      # Needs to be set before parent's __init__ as some overriden methods
      # require it.
      self.simulation = 'SIMULATION_VMID' in os.environ
      self.runningSlotId = None if self.simulation else Fru.slotId()

      super().__init__( updateDir, mode, isModular=True )

      self.sbStatus = BiosCliLib.getSecurebootStatus( self.mode.entityManager,
                                                      standby=True )

   @property
   def isStandby( self ):
      return True

   def makeDir( self, directory ):
      cmd = FileReplicationCmds.createDir( self.runningSlotId, directory,
                                           useKey=True, loopback=self.simulation )
      Tac.run( cmd, asRoot=True, stdout=Tac.DISCARD, stderr=Tac.DISCARD )

   def createTmpFile( self ):
      return Tac.run( FileReplicationCmds.runCmd( self.runningSlotId, 'mktemp',
                                                  useKey=True,
                                                  loopback=self.simulation ),
                             stdout=Tac.CAPTURE, stderr=Tac.DISCARD,
                             asRoot=True )

   def getLocalFilepath( self, filePath ):
      # Copy file to standby
      standbyTmpFile = self.createTmpFile()
      self.copyFile( filePath, standbyTmpFile )
      return standbyTmpFile

   def copyFile( self, src, dst ):
      Tac.run( FileReplicationCmds.copyFile( self.runningSlotId, dst, src,
                                             useKey=True,
                                             loopback=self.simulation ),
               stdout=Tac.DISCARD, stderr=Tac.DISCARD, asRoot=True )

   def moveAuf( self, src, dest ):
      self.logger.printMsg( 'Transferring AUF to the standby supervisor...' )
      self.copyFile( src.localFilename(), dest.localFilename() )
      cmd = FileReplicationCmds.syncFile( self.runningSlotId,
                                          dest.localFilename(),
                                          useKey=True, loopback=self.simulation )
      Tac.run( cmd, asRoot=True, stdout=Tac.DISCARD, stderr=Tac.DISCARD )

   def runScript( self, script ):
      cmd = 'bash %s'

      scriptDst = self.createTmpFile()
      self.copyFile( script, scriptDst )
      cmd = FileReplicationCmds.runCmd( self.runningSlotId, cmd % ( scriptDst ),
                                        useKey=True, loopback=self.simulation )
      return Tac.run( cmd, asRoot=True, stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )

   def runCmd( self, cmd ):
      cmd = FileReplicationCmds.runCmd( self.runningSlotId, cmd,
                                         useKey=True, loopback=self.simulation )
      return Tac.run( cmd, asRoot=True, stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )

   def writeHistory( self, aufSha, errorType, aufName='' ):
      historyLog = BiosInstallLib.historyFormat( aufSha, aufName, errorType )

      cmd = FileReplicationCmds.writeFile( self.runningSlotId, self.historyPath,
                                           historyLog, useKey=True,
                                           loopback=self.simulation )
      # FileReplicationCmds.writeFile() returns a command with shell
      # redirection (">>").
      # Through ssh this works fine, without ssh we need the command to be
      # executed in a shell for it to actually write in the file.
      if self.simulation:
         # FileReplicationCmds functions return arrays.
         # With `shell=True`, if cmd is an array only the first word will be
         # executed, so join it back into a string.
         cmd = ' '.join( cmd )
         Tac.run( cmd, shell=True, stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )
      else:
         Tac.run( cmd, asRoot=True, stdout=Tac.DISCARD, stderr=Tac.DISCARD )

      # Sync history
      cmd = FileReplicationCmds.syncFile( self.runningSlotId, self.historyPath,
                                          useKey=True, loopback=self.simulation )
      Tac.run( cmd, asRoot=True, stdout=Tac.DISCARD, stderr=Tac.DISCARD )

class BiosInstaller:
   def __init__( self, mode, sup, reboot, now ):
      self.mode = mode
      self.sup = sup
      self.reboot = reboot
      self.now = now

      # Sysdb entities
      self.entityMibRoot = BiosCliLib.getEntityMibRoot( self.mode.entityManager )
      self.redStatus = BiosCliLib.getRedundancyStatus( self.mode.entityManager )

      # This is accessed a few times, so only get it once
      self.isModular = BiosCliLib.isModular( self.entityMibRoot )

      # Create aboot update directories if they don't exist
      updateDir = Url.parseUrl( BiosInstallLib.ABOOT_UPDATE_DIR,
                                Url.Context( *Url.urlArgsFromMode( self.mode ) ) )
      updateDir = updateDir.localFilename()
      logDir = Url.parseUrl( BiosInstallLib.ABOOT_UPDATE_LOG_DIR,
                             Url.Context( *Url.urlArgsFromMode( self.mode ) ) )
      logDir = logDir.localFilename()

      self.runningSupe = CliInstaller( updateDir, mode, isModular=self.isModular )
      self.runningSupe.makeDir( logDir )
      self.logFile = tempfile.NamedTemporaryFile( suffix='.log',
                                                  dir=logDir,
                                                  delete=False,
                                                  mode='w' )
      self.logger = BiosInstallLib.Logger( self.mode, self.logFile )
      self.runningSupe.setLogger( self.logger )
      self.standbySupe = None

      self.supesToUpdate = []
      if sup in [ Supervisor.ALL, Supervisor.ACTIVE ]:
         self.supesToUpdate.append( self.runningSupe )
      if self.isModular and BiosCliLib.hasStandby( self.entityMibRoot,
                                                   self.redStatus ):
         if sup in [ Supervisor.ALL, Supervisor.STANDBY ]:
            self.standbySupe = CliStandbyInstaller( updateDir, mode )
            self.standbySupe.setLogger( self.logger )
            self.supesToUpdate.append( self.standbySupe )

   def promptUser( self, question ):
      if self.now:
         return True

      return BasicCliUtil.confirm( self.mode, question )

   def getInstallStatus( self ):
      activeStatus = None
      standbyStatus = None
      if self.runningSupe in self.supesToUpdate:
         activeStatus = self.runningSupe.installStatus
      if self.standbySupe and self.standbySupe in self.supesToUpdate:
         standbyStatus = self.standbySupe.installStatus
      return InstallStatus( activeStatus, standbyStatus )

   def setGlobalInstallStatus( self, errorCode ):
      for supe in self.supesToUpdate:
         supe.installStatus = errorCode
      return self.getInstallStatus()

   def run( self, src ):
      # Lazy import AUF as this import zipfile which is a memory hog. This avoids
      # it acting as a memory hog for each CLI process.  Only the CLI using this
      # command will be affected, rather than globally.
      from Auf import Auf, calcSha

      # Make sure we're running on the right sup
      if self.isModular and not BiosCliLib.isActive( self.redStatus ):
         self.logger.printMsg( 'This command is only available%s.', error=True,
                               standby=False )
         return self.getInstallStatus()

      # Check if src is a valid filetype
      filename = src.basename()
      if not filename.endswith( '.auf' ):
         self.logger.printMsg( 'File is not a \'.auf\'.', error=True )
         return self.setGlobalInstallStatus( InstallerErrorCode.FAIL )

      # Create our destination in /mnt/flash/aboot/update/
      dest = Url.parseUrl( os.path.join( BiosInstallLib.ABOOT_UPDATE_DIR, filename ),
                           Url.Context( *Url.urlArgsFromMode( self.mode ) ) )

      # Download if file isn't on system
      if src.fs.fsType == 'network':
         question = 'Download AUF? [confirm]'
         if not self.promptUser( question ):
            return self.setGlobalInstallStatus( InstallerErrorCode.ABORT )

         downloadUrl = src

         # Create temporary file for validation
         tmp = tempfile.NamedTemporaryFile( suffix='.auf' )
         src = Url.parseUrl( 'file:' + tmp.name,
                             Url.Context( *Url.urlArgsFromMode( self.mode ) ) )

         self.logger.printMsg( 'Downloading AUF...' )
         try:
            downloadUrl.writeLocalFile( downloadUrl, src.localFilename() )
         except OSError as e:
            self.logger.printMsg(
                  "AUF download failed with error: %s" % ( format( e ) ),
                           error=True )
            return self.setGlobalInstallStatus( InstallerErrorCode.FAIL )
         self.logger.printMsg( 'Done.' )

      if not os.path.exists( src.localFilename() ):
         self.logger.printMsg( 'No such source AUF.', error=True )
         return self.setGlobalInstallStatus( InstallerErrorCode.FAIL )

      # Open AUF
      aufSha = calcSha( src.localFilename() )
      try:
         auf = Auf( src.localFilename() )
         self.logger.writeLog( str( auf ) )
      except ( OSError, SyntaxError ) as e:
         for supe in self.supesToUpdate:
            supe.writeHistory( aufSha, HistoryCode.ZIP_ERROR )
         self.logger.printMsg( 'AUF parsing failed with error: %s' % ( format( e ) ),
                        error=True )
         return self.setGlobalInstallStatus( InstallerErrorCode.FAIL )
      except ( TypeError, ValueError ) as e:
         for supe in self.supesToUpdate:
            supe.writeHistory( aufSha, HistoryCode.SHA_ERROR )
         self.logger.printMsg( 'AUF parsing failed with error: %s' % ( format( e ) ),
                        error=True )
         return self.setGlobalInstallStatus( InstallerErrorCode.FAIL )
      aufName = auf.getName()

      stepFailed = False
      for supe in self.supesToUpdate:
         ret = supe.validateAuf( auf )
         if ret:
            supe.writeHistory( aufSha, ret, aufName )
            supe.installStatus = InstallerErrorCode.FAIL
            stepFailed = True
      if stepFailed:
         return self.getInstallStatus()

      forceAbootInstall = auf.abootOnly
      for supe in self.supesToUpdate:
         supeFeatures = BiosLib.getAbootFeatures( standby=supe.isStandby )
         supeSpiUpgradeRevision = \
               supeFeatures.get( BiosLib.ABOOT_FEATURE_SPIUPGRADE, -1 )

         if forceAbootInstall:
            if supeSpiUpgradeRevision == -1:
               self.logger.printMsg(
                     'Aboot SPI update not supported but AUF requires it%s',
                     standby=supe.isStandby if supe.isModular else None,
                     error=True )
               supe.installStatus = InstallerErrorCode.ABORT
               stepFailed = True
            elif not supe.isSpiUpdateEnabled():
               self.logger.printMsg(
                     'Aboot SPI update disabled but AUF requires it%s',
                     standby=supe.isStandby if supe.isModular else None,
                     error=True )
               supe.installStatus = InstallerErrorCode.ABORT
               stepFailed = True
         elif supe.isSpiLocked() and not supe.isSpiUpdateEnabled():
            self.logger.printMsg( 'Aboot SPI update disabled and SPI flash locked%s',
                            standby=supe.isStandby if supe.isModular else None,
                            error=True )
            supe.installStatus = InstallerErrorCode.ABORT
            stepFailed = True
         elif supe.isSpiFlashWpHardwired():
            # This case should be guarded by the Cli, but it doesn't hurt to prevent
            # it here too.
            self.logger.printMsg( 'SPI flash permanently locked%s',
                            standby=supe.isStandby if supe.isModular else None,
                            error=True )
            stepFailed = True

         # Install will happen in Aboot, check feature revision
         if not stepFailed and ( forceAbootInstall or supe.isSpiLocked() ):
            if auf.abootInstallerMinRev != 0 and \
               supeSpiUpgradeRevision < auf.abootInstallerMinRev:
               self.logger.printMsg( (
                        'Aboot installer revision required %d, has %d' % (
                        auf.abootInstallerMinRev, supeSpiUpgradeRevision )
                        ) + '%s',
                     standby=supe.isStandby if supe.isModular else None,
                     error=True )
               supe.installStatus = InstallerErrorCode.ABORT
               stepFailed = True

      if stepFailed:
         return self.getInstallStatus()

      question = 'Are you sure you want to upgrade the BIOS now? [confirm]'
      userConfirmed = False
      for supe in self.supesToUpdate:
         if forceAbootInstall or supe.isSpiLocked():
            supe.moveAuf( src, dest )
            supe.installStatus = InstallerErrorCode.SETUP_SUCCESS
         else:
            if not userConfirmed:
               if not self.promptUser( question ):
                  supe.installStatus = InstallerErrorCode.ABORT
                  return self.getInstallStatus()
               userConfirmed = True

            ret = supe.flashAufSection( auf )
            supe.writeHistory( aufSha, ret, aufName )
            if ret:
               supe.installStatus = InstallerErrorCode.UPGRADE_FAIL
               return self.getInstallStatus()
            supe.installStatus = InstallerErrorCode.SUCCESS

      return self.getInstallStatus()

   def finish( self ):
      if not self.reboot:
         return

      if self.standbySupe and self.standbySupe in self.supesToUpdate:
         if self.standbySupe.installStatus in [ InstallerErrorCode.SUCCESS,
                                                InstallerErrorCode.SETUP_SUCCESS ]:
            self.standbySupe.printSupeMsg( 'Reloading%s...' )
            ReloadCli.reloadPeerSupervisor( quiet=True )
            # This is set so that we don't warn the user that the
            # peer supervisor is disabled, since we just reloaded it
            setattr( self.mode, 'disableElectionCheckBeforeReload', True )

      if self.runningSupe in self.supesToUpdate:
         if self.runningSupe.installStatus in [ InstallerErrorCode.SUCCESS,
                                                InstallerErrorCode.SETUP_SUCCESS ]:
            self.runningSupe.printSupeMsg( 'Reloading%s...' )
            ReloadCli.doReload( self.mode, power=True, now=self.now )

      setattr( self.mode, 'disableElectionCheckBeforeReload', False )
