#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliMatcher
from CliMatcher import IntegerMatcher
import CliPlugin.BmpCli as BmpCli # pylint: disable=consider-using-from-import
from CliPlugin.RoutingBgpCli import (
    BgpCmdBaseClass,
    RouterBgpBaseMode,
)
import ConfigMount
import Toggles.BmpToggleLib
import Tac

# Tac Entities
bmpConfig = None

# Enum Type
matcherMonitoring = CliMatcher.KeywordMatcher(
   'monitoring', helpdesc='BGP monitoring protocol configuration' )
matcherReceived = CliMatcher.KeywordMatcher(
   'received', helpdesc='BGP monitoring protocol received route selection' )
matcherAdvertised = CliMatcher.KeywordMatcher(
   'advertised', helpdesc='BGP monitoring protocol advertised route selection' )
matcherRoutes = CliMatcher.KeywordMatcher(
   'routes', helpdesc='BGP monitoring protocol route selection' )
matcherTimestamp = CliMatcher.KeywordMatcher(
   'timestamp',
   helpdesc='BGP monitoring protocol Per-Peer Header timestamp behavior' )
matcherQos = CliMatcher.KeywordMatcher(
   'qos',
   helpdesc='Set QoS parameters for BGP monitoring protocol' )
matcherDscp = CliMatcher.KeywordMatcher(
   'DSCP',
   helpdesc='Set DSCP value for BGP monitoring protocol' )
matcherDscpValue = IntegerMatcher(
   0,
   63,
   helpdesc='DSCP value' )

#--------------------------------------------------------------------------------
# [ no | default ] monitoring port PORT
# XXX: BUG157364 add support for VRF
#--------------------------------------------------------------------------------
class MonitoringPortCmd( CliCommand.CliCommandClass ):
   syntax = 'monitoring port PORT'
   noOrDefaultSyntax = 'monitoring port ...'
   data = {
      'monitoring': matcherMonitoring,
      'port': CliMatcher.KeywordMatcher(
         'port',
         helpdesc=( 'BGP monitoring protocol port number '
                    'for stations in passive mode' ) ),
      'PORT': CliMatcher.IntegerMatcher(
         1024, 65535, helpdesc='BGP monitoring protocol port number' ),
   }
   handler = "BmpRouterBgpBaseModeHandler.handlerMonitoringPortCmd"
   noOrDefaultHandler = \
      "BmpRouterBgpBaseModeHandler.noOrDefaultHandlerMonitoringPortCmd"

RouterBgpBaseMode.addCommandClass( MonitoringPortCmd )

#--------------------------------------------------------------------------------
# [ no | default ] monitoring qos dscp DSCP_VALUE
#--------------------------------------------------------------------------------
class MonitoringQosDscpCmd( CliCommand.CliCommandClass ):
   syntax = 'monitoring qos dscp DSCP_VALUE'
   noOrDefaultSyntax = 'monitoring qos dscp ...'
   data = {
      'monitoring': matcherMonitoring,
      'qos': matcherQos,
      'dscp': matcherDscp,
      'DSCP_VALUE': matcherDscpValue,
   }
   handler = "BmpRouterBgpBaseModeHandler.handlerMonitoringQosDscpCmd"
   noOrDefaultHandler = \
      "BmpRouterBgpBaseModeHandler.noOrDefaultHandlerMonitoringQosDscpCmd"

RouterBgpBaseMode.addCommandClass( MonitoringQosDscpCmd )

#--------------------------------------------------------------------------------
# [ no | default ] monitoring received routes post-policy
#--------------------------------------------------------------------------------
class MonitoringReceivedRoutesPostPolicyCmd( CliCommand.CliCommandClass ):
   syntax = 'monitoring received routes post-policy'
   noSyntax = 'monitoring received routes post-policy ...'
   defaultSyntax = 'monitoring received routes post-policy ...'
   data = {
      'monitoring': matcherMonitoring,
      'received': matcherReceived,
      'routes': matcherRoutes,
      'post-policy': CliMatcher.KeywordMatcher(
         'post-policy',
         helpdesc='Export BGP routes after input policies are applied' ),
   }
   handler = \
      "BmpRouterBgpBaseModeHandler.handlerMonitoringReceivedRoutesPostPolicyCmd"
   noHandler = \
      "BmpRouterBgpBaseModeHandler.noHandlerMonitoringReceivedRoutesPostPolicyCmd"
   defaultHandler = \
      "BmpRouterBgpBaseModeHandler." + \
      "defaultHandlerMonitoringReceivedRoutesPostPolicyCmd"

RouterBgpBaseMode.addCommandClass( MonitoringReceivedRoutesPostPolicyCmd )

#--------------------------------------------------------------------------------
# [ no | default ] monitoring received routes pre-policy
#--------------------------------------------------------------------------------
class MonitoringReceivedRoutesPrePolicyCmd( CliCommand.CliCommandClass ):
   syntax = 'monitoring received routes pre-policy'
   noSyntax = 'monitoring received routes pre-policy ...'
   defaultSyntax = 'monitoring received routes pre-policy ...'
   data = {
      'monitoring': matcherMonitoring,
      'received': matcherReceived,
      'routes': matcherRoutes,
      'pre-policy': CliMatcher.KeywordMatcher(
         'pre-policy',
         helpdesc='Export BGP routes before input policies are applied' ),
   }
   handler = \
      "BmpRouterBgpBaseModeHandler.handlerMonitoringReceivedRoutesPrePolicyCmd"
   noHandler = \
      "BmpRouterBgpBaseModeHandler.noHandlerMonitoringReceivedRoutesPrePolicyCmd"
   defaultHandler = \
      "BmpRouterBgpBaseModeHandler." + \
      "defaultHandlerMonitoringReceivedRoutesPrePolicyCmd"

RouterBgpBaseMode.addCommandClass( MonitoringReceivedRoutesPrePolicyCmd )

# --------------------------------------------------------------------------------
# [ no | default ] monitoring advertised routes post-policy
# --------------------------------------------------------------------------------
class MonitoringAdvertisedRoutesPostPolicyCmd( CliCommand.CliCommandClass ):
   syntax = 'monitoring advertised routes post-policy'
   noSyntax = 'monitoring advertised routes post-policy ...'
   defaultSyntax = 'monitoring advertised routes post-policy ...'
   data = {
      'monitoring': matcherMonitoring,
      'advertised': matcherAdvertised,
      'routes': matcherRoutes,
      'post-policy': 'Export BGP routes after output policies are applied',
   }
   handler = \
      "BmpRouterBgpBaseModeHandler.handlerMonitoringAdvertisedRoutesPostPolicyCmd"
   noHandler = \
      "BmpRouterBgpBaseModeHandler.noHandlerMonitoringAdvertisedRoutesPostPolicyCmd"
   defaultHandler = \
      "BmpRouterBgpBaseModeHandler." + \
      "defaultHandlerMonitoringAdvertisedRoutesPostPolicyCmd"

if Toggles.BmpToggleLib.toggleBmpAdjRibOutExportEnabled():
   RouterBgpBaseMode.addCommandClass( MonitoringAdvertisedRoutesPostPolicyCmd )

# --------------------------------------------------------------------------------
# [ no | default ] monitoring bgp rib bestpaths [ disabled ]
# --------------------------------------------------------------------------------
class MonitoringBribRouteCmd( BgpCmdBaseClass ):
   syntax = 'monitoring bgp rib bestpaths [ disabled ]'
   noOrDefaultSyntax = syntax
   data = BgpCmdBaseClass._createSyntaxData( {
      'monitoring': matcherMonitoring,
      'bgp': BmpCli.matcherBgp,
      'rib': BmpCli.matcherRib,
      'bestpaths': BmpCli.matcherBestPaths,
   } )
   handler = "BmpRouterBgpBaseModeHandler.handlerMonitoringBribRouteCmd"
   noOrDefaultHandler = \
      "BmpRouterBgpBaseModeHandler.noOrDefaultHandlerMonitoringBribRouteCmd"

if Toggles.BmpToggleLib.toggleBmpLocRibEnabled():
   RouterBgpBaseMode.addCommandClass( MonitoringBribRouteCmd )

#--------------------------------------------------------------------------------
# [ no | default ] monitoring received routes address-family
#                  ( ipv4 unicast [ disable ] ) |
#                  ( ipv6 ( unicast [ disable ] ) |
#                         ( labeled-unicast { 6pe | tunnel } ) ) |
#                  ( vpn-ipv4 ) | ( vpn-ipv6 ) |
#                  flow-spec ( ipv4 | ipv6 )
#--------------------------------------------------------------------------------
class MonitoringReceivedRoutesAfiSafiCmd( CliCommand.CliCommandClass ):
   syntax = (
      'monitoring received routes address-family'
      ' ( ipv4 unicast [ disable ] ) |'
      ' ( ipv6 ( unicast [ disable ] ) |'
      '        ( labeled-unicast { 6pe | tunnel } ) ) |'
      ' ( vpn-ipv4 ) | '
      ' ( vpn-ipv6 ) ' )
   noOrDefaultSyntax = (
      'monitoring received routes address-family'
      ' ( ipv4 unicast ) |'
      ' ( ipv6 ( unicast | labeled-unicast ) ) |' )
   data = {
      'monitoring': matcherMonitoring,
      'received': matcherReceived,
      'routes': matcherRoutes,
      'address-family': 'Export specified address family',
      'ipv4': 'IPv4 related',
      'ipv6': 'IPv6 related',
      'unicast': 'Unicast sub-address family',
      'labeled-unicast': 'Labeled-unicast sub-address family',
      '6pe': CliCommand.Node(
         CliMatcher.KeywordMatcher(
            '6pe',
            helpdesc='Export 6PE paths' ),
         maxMatches=1 ),
      'tunnel': CliCommand.Node(
         CliMatcher.KeywordMatcher(
            'tunnel',
            helpdesc='Export IPv6 Labeled-unicast tunnel paths' ),
         maxMatches=1,
         hidden=True ),
      'disable': 'Disable export of this address family',
      'vpn-ipv4': 'MPLS L3 VPN IPv4 unicast address family',
      'vpn-ipv6': 'MPLS L3 VPN IPv6 unicast address family',
   }
   if Toggles.BmpToggleLib.toggleBmpFlowspecSafiEnabled():
      syntax += ' | ( flow-spec ( ipv4 | ipv6 ) ) '
      noOrDefaultSyntax += (
         ' ( vpn-ipv4 ) | ( vpn-ipv6 ) |'
         ' ( flow-spec ( ipv4 | ipv6 ) ) ... ' )
      data[ 'flow-spec' ] = 'Flowspec address family'
   else:
      noOrDefaultSyntax += (
         ' ( vpn-ipv4 ) | ( vpn-ipv6 ) ... ' )
   handler = "BmpRouterBgpBaseModeHandler.handlerMonitoringReceivedRoutesAfiSafiCmd"
   noOrDefaultHandler = \
      "BmpRouterBgpBaseModeHandler." + \
      "noOrDefaultHandlerMonitoringReceivedRoutesAfiSafiCmd"

   @staticmethod
   def adapter( mode, args, argsList ):
      afi = 'afiIpv4' if 'ipv4' in args or 'vpn-ipv4' in args else 'afiIpv6'
      if 'labeled-unicast' in args:
         safi = 'safiMplsLabels'
      elif 'vpn-ipv4' in args or 'vpn-ipv6' in args:
         safi = 'safiMplsVpn'
      elif 'flow-spec' in args:
         safi = 'safiFlowspec'
      else:
         safi = 'safiUnicast'
      args[ 'AFISAFI' ] = Tac.Value( 'Routing::Bgp::AfiSafi', afi, safi )

RouterBgpBaseMode.addCommandClass( MonitoringReceivedRoutesAfiSafiCmd )

#--------------------------------------------------------------------------------
# [ no | default ] monitoring station BMPSTATIONNAME
#--------------------------------------------------------------------------------
class MonitoringStationBmpstationnameCmd( CliCommand.CliCommandClass ):
   syntax = 'monitoring station BMPSTATIONNAME'
   noOrDefaultSyntax = 'monitoring station BMPSTATIONNAME ...'
   data = {
      'monitoring': matcherMonitoring,
      'station': 'BGP monitoring station configuration',
      'BMPSTATIONNAME': CliMatcher.DynamicNameMatcher(
         lambda mode: ( sorted( bmpConfig.bmpStation.members() ) ),
         helpdesc='BGP monitoring station name' ),
   }
   handler = "BmpRouterBgpBaseModeHandler.handlerMonitoringStationBmpstationnameCmd"
   noOrDefaultHandler = \
      "BmpRouterBgpBaseModeHandler." + \
      "noOrDefaultHandlerMonitoringStationBmpstationnameCmd"

RouterBgpBaseMode.addCommandClass( MonitoringStationBmpstationnameCmd )

#--------------------------------------------------------------------------------
# ( no | default ) monitoring timestamp ( none | send-time )
#--------------------------------------------------------------------------------
class MonitoringTimestampCmd( CliCommand.CliCommandClass ):
   syntax = 'monitoring timestamp ( none | send-time )'
   noOrDefaultSyntax = 'monitoring timestamp ...'
   data = {
      'monitoring': matcherMonitoring,
      'timestamp': matcherTimestamp,
      'none': 'Zero',
      'send-time': 'Time when BGP monitoring protocol message is sent',
   }
   handler = "BmpRouterBgpBaseModeHandler.handlerMonitoringTimestampCmd"
   noOrDefaultHandler = \
      "BmpRouterBgpBaseModeHandler.noOrDefaultHandlerMonitoringTimestampCmd"

RouterBgpBaseMode.addCommandClass( MonitoringTimestampCmd )

# --------------------------------------------------------------------------------
# [ no | default ] monitoring statistics interval STATSINTERVAL seconds
# --------------------------------------------------------------------------------
class MonitoringStatsIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'monitoring statistics interval STATSINTERVAL seconds'
   noOrDefaultSyntax = 'monitoring statistics interval ...'
   data = {
      'monitoring': matcherMonitoring,
      'statistics': 'Statistics report related configuration for the '
      'BGP monitoring station',
      'interval': 'Configure statistics interval',
      'STATSINTERVAL': CliMatcher.IntegerMatcher(
         15, 3600,
         helpdesc='Interval for sending statistics report in seconds' ),
      'seconds': 'Interval to be specified in seconds',
   }
   handler = "BmpRouterBgpBaseModeHandler.handlerMonitoringStatsIntervalCmd"
   noOrDefaultHandler = \
      "BmpRouterBgpBaseModeHandler.noOrDefaultHandlerMonitoringStatsIntervalCmd"

RouterBgpBaseMode.addCommandClass( MonitoringStatsIntervalCmd )

def Plugin( entityManager ):
   global bmpConfig
   bmpConfig = ConfigMount.mount( entityManager, 'routing/bmp/config',
                                  'Routing::Bmp::BmpConfig', 'w' )
