#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliMatcher
from CliMatcher import StringMatcher
from CliPlugin.BmpCli import (
   matcherBgp,
   matcherRib,
   matcherBestPaths,
   RouterBmpStationMode,
)

from CliPlugin.IntfCli import Intf
from CliPlugin.RoutingBgpCli import (
    BgpCmdBaseClass,
)
import CliPlugin.VrfCli as VrfCli # pylint: disable=consider-using-from-import
from HostnameCli import IpAddrOrHostnameMatcher
import Toggles.BmpToggleLib
import ReversibleSecretCli

matcherConnection = CliMatcher.KeywordMatcher(
   'connection',
   helpdesc='Connection configuration for the BGP monitoring session' )

#--------------------------------------------------------------------------------
# [no | default ] authentication-key ( [ 0 ] AUTHUNENCRYPTEDPASSWD ) |
#                                    ( 7 AUTHENCRYPTEDPASSWD )
#--------------------------------------------------------------------------------
def generateBmpEncrytionKey( mode, args ):
   return mode.bmpStationName + '_passwd'

class AuthenticationKeyCmd( CliCommand.CliCommandClass ):
   syntax = 'authentication-key AUTH_PASSWD'
   noOrDefaultSyntax = 'authentication-key ...'
   data = {
      'authentication-key': CliMatcher.KeywordMatcher(
         'authentication-key',
         helpdesc='Configure authentication key for the BGP monitoring station' ),
      'AUTH_PASSWD': ReversibleSecretCli.ReversiblePasswordCliExpression(
         cleartextMatcher=ReversibleSecretCli.cleartextAuthMatcher,
         obfuscatedTextMatcher=ReversibleSecretCli.type7AuthMatcher,
         uniqueKeyGenerator=generateBmpEncrytionKey,
         algorithm='DES', errorMsg='Invalid encrypted password' )
   }
   handler = "BmpRouterBmpStationModeHandler.handlerAuthenticationKeyCmd"
   noOrDefaultHandler = \
      "BmpRouterBmpStationModeHandler.noOrDefaultHandlerAuthenticationKeyCmd"

RouterBmpStationMode.addCommandClass( AuthenticationKeyCmd )

#--------------------------------------------------------------------------------
# [ no | default ] connection address [ vrf VRF ]
#--------------------------------------------------------------------------------
class ConnectionAddressCmd( CliCommand.CliCommandClass ):
   syntax = 'connection address HOST [ VRF ]'
   noOrDefaultSyntax = 'connection address ...'
   data = {
      'connection': 'Connection configuration for the BGP monitoring session',
      'address': "Specify the BGP monitoring station's remote address",
      'HOST': IpAddrOrHostnameMatcher( ipv6=True ),
      'VRF': VrfCli.VrfExprFactory( helpdesc='Configure BMP in a VRF' ),
   }
   handler = "BmpRouterBmpStationModeHandler.handlerConnectionAddressCmd"
   noOrDefaultHandler = \
      "BmpRouterBmpStationModeHandler.noOrDefaultHandlerConnectionAddressCmd"

RouterBmpStationMode.addCommandClass( ConnectionAddressCmd )

#--------------------------------------------------------------------------------
# [ no | default ] connection keepalive IDLETIME PROBEINTERVAL PROBECOUNT
#--------------------------------------------------------------------------------
class ConnectionKeepaliveCmd(
      CliCommand.CliCommandClass ):
   syntax = 'connection keepalive IDLETIME PROBEINTERVAL PROBECOUNT'
   noOrDefaultSyntax = 'connection keepalive ...'
   data = {
      'connection': matcherConnection,
      'keepalive': CliMatcher.KeywordMatcher(
         'keepalive',
         helpdesc="Specify the BGP monitoring station's TCP Keepalive parameters" ),
      'IDLETIME': CliMatcher.IntegerMatcher(
         1, 86400, helpdesc='Idle time (seconds) before TCP Keepalive' ),
      'PROBEINTERVAL': CliMatcher.IntegerMatcher(
         1, 3600, helpdesc='Interval (seconds) between TCP Keepalive Probes' ),
      'PROBECOUNT': CliMatcher.IntegerMatcher(
         1, 1000,
         helpdesc='Number of Keepalive probes before closing connection' ),
   }
   handler = "BmpRouterBmpStationModeHandler.handlerConnectionKeepaliveCmd"
   noOrDefaultHandler = \
      "BmpRouterBmpStationModeHandler.noOrDefaultHandlerConnectionKeepaliveCmd"

RouterBmpStationMode.addCommandClass( ConnectionKeepaliveCmd )

#--------------------------------------------------------------------------------
# [ no | default ] connection mode
#    ( active port PORT [ reconnect - timer RECONNECTTIME ] ) | ( passive )
#--------------------------------------------------------------------------------
class ConnectionModeCmd( CliCommand.CliCommandClass ):
   syntax = ( 'connection mode '
              '( active port PORT [ reconnect-timer RECONNECTTIME ] ) | '
              '( passive )' )
   noOrDefaultSyntax = 'connection mode ...'
   data = {
      'connection': matcherConnection,
      'mode': CliMatcher.KeywordMatcher(
         'mode',
         helpdesc="Specify the BGP monitoring station's connection mode" ),
      'active': 'Active mode',
      'port': 'BGP monitoring protocol port number for stations in passive mode',
      'PORT': CliMatcher.IntegerMatcher(
         1, 65535, helpdesc='Configure the BMP port to this value' ),
      'reconnect-timer': "Configure the BGP monitoring session's reconnect timer",
      'RECONNECTTIME': CliMatcher.IntegerMatcher(
         1, 3600, helpdesc='Configure the reconnect timer to this value' ),
      'passive': 'Passive mode',
   }
   handler = "BmpRouterBmpStationModeHandler.handlerConnectionModeCmd"
   noOrDefaultHandler = \
      "BmpRouterBmpStationModeHandler.noOrDefaultHandlerConnectionModeCmd"

RouterBmpStationMode.addCommandClass( ConnectionModeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] description DESCRIPTION
#--------------------------------------------------------------------------------
class DescriptionCmd( CliCommand.CliCommandClass ):
   syntax = 'description DESCRIPTION'
   noOrDefaultSyntax = 'description ...'
   data = {
      'description': 'Assign a description to the BGP monitoring station',
      'DESCRIPTION': StringMatcher(
         helpdesc='Text describing this BGP monitoring station' ),
   }
   handler = "BmpRouterBmpStationModeHandler.handlerDescriptionCmd"
   noOrDefaultHandler = \
      "BmpRouterBmpStationModeHandler.noOrDefaultHandlerDescriptionCmd"

RouterBmpStationMode.addCommandClass( DescriptionCmd )

#--------------------------------------------------------------------------------
# [ no | default ] export-policy received routes { pre-policy | post-policy }
#--------------------------------------------------------------------------------
class ExportPolicyReceivedRoutesCmd( CliCommand.CliCommandClass ):
   syntax = 'export-policy received routes { POLICIES }'
   noOrDefaultSyntax = 'export-policy received routes [ POLICIES ]'
   data = {
      'export-policy': 'Per-station BGP monitoring protocol export policy',
      'received': CliMatcher.KeywordMatcher(
         'received',
         helpdesc='BGP monitoring protocol received route selection' ),
      'routes': CliMatcher.KeywordMatcher(
         'routes',
         helpdesc='BGP monitoring protocol received route selection' ),
      'POLICIES': CliCommand.SetEnumMatcher( {
         'pre-policy': 'Export BGP routes before input policies are applied',
         'post-policy': 'Export BGP routes after input policies are applied',
      } ),
   }
   handler = "BmpRouterBmpStationModeHandler.handlerExportPolicyReceivedRoutesCmd"
   noHandler = \
      "BmpRouterBmpStationModeHandler.noHandlerExportPolicyReceivedRoutesCmd"
   defaultHandler = \
      "BmpRouterBmpStationModeHandler.defaultHandlerExportPolicyReceivedRoutesCmd"

RouterBmpStationMode.addCommandClass( ExportPolicyReceivedRoutesCmd )

# --------------------------------------------------------------------------------
# [ no | default ] export-policy received routes additional-paths
# --------------------------------------------------------------------------------
class ExportPolicyAdditionalPathsCmd( CliCommand.CliCommandClass ):
   syntax = 'export-policy received routes additional-paths'
   noOrDefaultSyntax = 'export-policy received routes additional-paths'
   data = {
      'export-policy': 'Per-station BGP monitoring protocol export policy',
      'received': 'BGP monitoring protocol received route selection',
      'routes': 'BGP monitoring protocol received route selection',
      'additional-paths': 'Per-station status of parsing additional paths',
   }
   handler = "BmpRouterBmpStationModeHandler.handlerExportPolicyAdditionalPathsCmd"
   noOrDefaultHandler = \
      "BmpRouterBmpStationModeHandler." + \
      "noOrDefaultHandlerExportPolicyAdditionalPathsCmd"

RouterBmpStationMode.addCommandClass( ExportPolicyAdditionalPathsCmd )

# --------------------------------------------------------------------------------
# [ no | default ] export-policy advertised routes post-policy
# --------------------------------------------------------------------------------
class ExportPolicyAdvertisedRoutesCmd( CliCommand.CliCommandClass ):
   syntax = 'export-policy advertised routes post-policy'
   noOrDefaultSyntax = 'export-policy advertised routes ...'
   data = {
      'export-policy': 'Per-station BGP monitoring protocol export policy',
      'advertised': 'BGP monitoring protocol advertised route selection',
      'routes': 'BGP monitoring protocol advertised route selection',
      'post-policy': 'Export BGP routes after output policies are applied',
   }
   handler = "BmpRouterBmpStationModeHandler.handlerExportPolicyAdvertisedRoutesCmd"
   noHandler = \
      "BmpRouterBmpStationModeHandler.noHandlerExportPolicyAdvertisedRoutesCmd"
   defaultHandler = \
      "BmpRouterBmpStationModeHandler.defaultHandlerExportPolicyAdvertisedRoutesCmd"

if Toggles.BmpToggleLib.toggleBmpAdjRibOutExportEnabled():
   RouterBmpStationMode.addCommandClass( ExportPolicyAdvertisedRoutesCmd )

# --------------------------------------------------------------------------------
# [ no | default ] export-policy bgp rib bestpaths
# --------------------------------------------------------------------------------
class ExportPolicyBribBestPathsCmd( BgpCmdBaseClass ):
   syntax = 'export-policy bgp rib bestpaths [ disabled ]'
   noOrDefaultSyntax = syntax

   data = BgpCmdBaseClass._createSyntaxData( {
      'export-policy': 'Per-station BGP monitoring protocol export policy',
      'bgp': matcherBgp,
      'rib': matcherRib,
      'bestpaths': matcherBestPaths,
   } )
   handler = "BmpRouterBmpStationModeHandler.handlerExportPolicyBribBestPathsCmd"
   noOrDefaultHandler = \
      "BmpRouterBmpStationModeHandler." + \
      "noOrDefaultHandlerExportPolicyBribBestPathsCmd"

if Toggles.BmpToggleLib.toggleBmpLocRibEnabled():
   RouterBmpStationMode.addCommandClass( ExportPolicyBribBestPathsCmd )


#--------------------------------------------------------------------------------
# [ no | default ] shutdown [ reason MESSAGE ]
#--------------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown [ reason MESSAGE ]'
   noOrDefaultSyntax = 'shutdown ...'
   data = {
      'shutdown': 'Administratively shut down the BGP monitoring station',
      'reason': 'BGP monitoring station shutdown reason to be saved',
      'MESSAGE': CliMatcher.StringMatcher( helpname='MESSAGE',
         helpdesc='Shut down message' )
   }
   handler = "BmpRouterBmpStationModeHandler.handlerShutdownCmd"
   noOrDefaultHandler = \
      "BmpRouterBmpStationModeHandler.noOrDefaultHandlerShutdownCmd"

RouterBmpStationMode.addCommandClass( ShutdownCmd )

#--------------------------------------------------------------------------------
# [ no | default ] statistics
#--------------------------------------------------------------------------------
class StatisticsEnableCmd( CliCommand.CliCommandClass ):
   syntax = 'statistics'
   noOrDefaultSyntax = syntax
   data = {
      'statistics': 'Enable statistics report for the BGP monitoring session',
   }
   handler = "BmpRouterBmpStationModeHandler.handlerStatisticsEnableCmd"
   noOrDefaultHandler = \
      "BmpRouterBmpStationModeHandler.noOrDefaultHandlerStatisticsEnableCmd"

RouterBmpStationMode.addCommandClass( StatisticsEnableCmd )

#--------------------------------------------------------------------------------
# update-source INTF
#--------------------------------------------------------------------------------
class UpdateSourceCmd( CliCommand.CliCommandClass ):
   syntax = 'update-source INTF'
   noOrDefaultSyntax = 'update-source ...'
   data = {
      'update-source': CliMatcher.KeywordMatcher(
         'update-source',
         helpdesc=(
            'Specify a local source interface for the BGP monitoring session' )
      ),
      'INTF': Intf.matcherWithIpSupport,
   }
   handler = "BmpRouterBmpStationModeHandler.handlerUpdateSourceCmd"
   noOrDefaultHandler = \
      "BmpRouterBmpStationModeHandler.noOrDefaultHandlerUpdateSourceCmd"

RouterBmpStationMode.addCommandClass( UpdateSourceCmd )
