#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
from CliPlugin import ShowTaskSchedulerModel
import CliPlugin.TechSupportCli
import CliPlugin.VrfCli as VrfCli # pylint: disable=consider-using-from-import
import CommonGuards
import LazyMount
import ShowCommand
import Tracing

traceHandle = Tracing.Handle( 'Bmp' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1
t2 = traceHandle.trace2

bmpConfig = None

matcherActive = CliMatcher.KeywordMatcher( 'active',
      helpdesc='Active station information' )
matcherBgp = CliMatcher.KeywordMatcher( 'bgp', helpdesc='BGP information' )
matcherMonitoring = CliMatcher.KeywordMatcher( 'monitoring',
      helpdesc='BGP monitoring' )
matcherPassive = CliMatcher.KeywordMatcher( 'passive',
      helpdesc='Listener information' )
matcherStation = CliMatcher.KeywordMatcher( 'station',
      helpdesc='Monitoring station information' )
matcherSummary = CliMatcher.KeywordMatcher( 'summary', helpdesc='Summary view' )
nodeInternal = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'internal',
         helpdesc='BMP Agent internal status' ),
      maxMatches=1, hidden=True, guard=CommonGuards.standbyGuard )

stationNameMatcher = CliMatcher.DynamicNameMatcher( lambda mode: (
                         bmpConfig.bmpStation.members() ),
                        'BGP monitoring station name' )

#--------------------------------------------------------------------------------
# show bgp monitoring active [ station STATION ]
#--------------------------------------------------------------------------------
class BgpMonitoringActiveCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp monitoring active [ station STATION ]'
   data = {
      'bgp': matcherBgp,
      'monitoring': matcherMonitoring,
      'active': matcherActive,
      'station': matcherStation,
      'STATION': stationNameMatcher
   }
   cliModel = "BmpCliModels.AllBmpActive"
   handler = "BmpShowCliHandler.handlerBgpMonitoringActiveCmd"

BasicCli.addShowCommandClass( BgpMonitoringActiveCmd )

#--------------------------------------------------------------------------------
# show bgp monitoring active summary
#--------------------------------------------------------------------------------
class BgpMonitoringActiveSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp monitoring active summary'
   data = {
      'bgp': matcherBgp,
      'monitoring': matcherMonitoring,
      'active': matcherActive,
      'summary': matcherSummary,
   }
   cliModel = "BmpCliModels.AllBmpActiveSummary"
   handler = "BmpShowCliHandler.handlerBgpMonitoringActiveSummaryCmd"

BasicCli.addShowCommandClass( BgpMonitoringActiveSummaryCmd )

#--------------------------------------------------------------------------------
# show bgp monitoring [ station STATION ] [ detail ]
#--------------------------------------------------------------------------------
class BgpMonitoringCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp monitoring [ station STATION ] [ detail ]'
   data = {
      'bgp': matcherBgp,
      'monitoring': matcherMonitoring,
      'station': matcherStation,
      'STATION': stationNameMatcher,
      'detail': 'Detailed view'
   }
   cliModel = "BmpCliModels.AllBmpStation"
   handler = "BmpShowCliHandler.handlerBgpMonitoringCmd"

BasicCli.addShowCommandClass( BgpMonitoringCmd )

#--------------------------------------------------------------------------------
# show bgp monitoring internal
#--------------------------------------------------------------------------------
class BgpMonitoringInternalCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp monitoring internal'
   data = {
      'bgp': matcherBgp,
      'monitoring': matcherMonitoring,
      'internal': nodeInternal,
   }
   privileged = True
   handler = "BmpShowCliHandler.handlerBgpMonitoringInternalCmd"

BasicCli.addShowCommandClass( BgpMonitoringInternalCmd )

#--------------------------------------------------------------------------------
# show bgp monitoring internal convergence
#--------------------------------------------------------------------------------
class BgpMonitoringInternalConvergenceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp monitoring internal convergence'
   data = {
      'bgp': matcherBgp,
      'monitoring': matcherMonitoring,
      'internal': nodeInternal,
      'convergence': CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
         'convergence', helpdesc='BMP convergence status' ),
         guard=CommonGuards.standbyGuard ),
   }
   privileged = True
   cliModel = "BmpCliModels.AllBmpConvergence"
   handler = "BmpShowCliHandler.handlerBgpMonitoringInternalConvergenceCmd"

BasicCli.addShowCommandClass( BgpMonitoringInternalConvergenceCmd )

#--------------------------------------------------------------------------------
# Deprecate 'show bgp monitoring internal scheduler reset'.  It was a
# lazy implementation and doesn't fit the new one.
# --------------------------------------------------------------------------------
class BgpMonitoringInternalSchedulerResetCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp monitoring internal scheduler reset'
   data = {
      'bgp': matcherBgp,
      'monitoring': matcherMonitoring,
      'internal': nodeInternal,
      'scheduler': CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
         'scheduler', helpdesc='BMP Agent scheduler' ),
         guard=CommonGuards.standbyGuard ),
      'reset': CliCommand.singleNode( CliMatcher.KeywordMatcher( 'reset',
         helpdesc='Reset scheduler statistics' ),
         deprecatedByCmd='clear agent Bmp task scheduler' ),
   }
   handler = "BmpShowCliHandler.handlerBgpMonitoringInternalSchedulerResetCmd"

BasicCli.addShowCommandClass( BgpMonitoringInternalSchedulerResetCmd )

#--------------------------------------------------------------------------------
# show bgp monitoring internal scheduler
#       [ { ( verbose | history ) } ]
#--------------------------------------------------------------------------------
class BgpMonitoringInternalSchedulerCmd( ShowCommand.ShowCliCommandClass ):
   syntax = \
      'show bgp monitoring internal scheduler' \
      ' [ { ( verbose | history ) } ]'
   data = {
      'bgp': matcherBgp,
      'monitoring': matcherMonitoring,
      'internal': nodeInternal,
      'scheduler': CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
         'scheduler', helpdesc='BMP Agent scheduler' ),
         guard=CommonGuards.standbyGuard ),
      'verbose': 'Include more information',
      'history': 'Display scheduler event history',
   }
   privileged = True
   cliModel = ShowTaskSchedulerModel.Overall
   handler = "BmpShowCliHandler.handlerBgpMonitoringInternalSchedulerCmd"

BasicCli.addShowCommandClass( BgpMonitoringInternalSchedulerCmd )

#--------------------------------------------------------------------------------
# show bgp monitoring passive [ VRF ]
#--------------------------------------------------------------------------------
class BgpMonitoringPassiveCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp monitoring passive [ VRF ]'
   data = {
      'bgp': matcherBgp,
      'monitoring': matcherMonitoring,
      'passive': matcherPassive,
      'VRF': VrfCli.VrfExprFactory( helpdesc='VRF name', inclDefaultVrf=True ),
   }
   cliModel = "BmpCliModels.AllBmpPassive"
   handler = "BmpShowCliHandler.handlerBgpMonitoringPassiveCmd"

BasicCli.addShowCommandClass( BgpMonitoringPassiveCmd )

#--------------------------------------------------------------------------------
# show bgp monitoring passive summary
#--------------------------------------------------------------------------------
class BgpMonitoringPassiveSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp monitoring passive summary'
   data = {
      'bgp': matcherBgp,
      'monitoring': matcherMonitoring,
      'passive': matcherPassive,
      'summary': matcherSummary,
   }
   cliModel = "BmpCliModels.AllBmpPassiveSummary"
   handler = "BmpShowCliHandler.handlerBgpMonitoringPassiveSummaryCmd"

BasicCli.addShowCommandClass( BgpMonitoringPassiveSummaryCmd )

#--------------------------------------------------------------------------------
# show bgp monitoring summary
#--------------------------------------------------------------------------------
class BgpMonitoringSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp monitoring summary'
   data = {
      'bgp': matcherBgp,
      'monitoring': matcherMonitoring,
      'summary': matcherSummary,
   }
   cliModel = "BmpCliModels.AllBmpStationSummary"
   handler = "BmpShowCliHandler.handlerBgpMonitoringSummaryCmd"

BasicCli.addShowCommandClass( BgpMonitoringSummaryCmd )

#-------------------------------------------------------------------------------
# BMP commands in "show tech-support"
#-------------------------------------------------------------------------------
def bmpIsEnabled():
   return bmpConfig.bmpActivate

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2017-08-25 11:30:00',
   cmds=[ 'show bgp monitoring',
          'show bgp monitoring summary',
          'show bgp monitoring active',
          'show bgp monitoring passive',
          'show bgp monitoring internal',
          'show agent Bmp task scheduler' ],
   cmdsGuard=bmpIsEnabled,
   summaryCmds=[ 'show bgp monitoring summary' ] )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global bmpConfig

   bmpConfig = LazyMount.mount( entityManager, 'routing/bmp/config',
                                'Routing::Bmp::BmpConfig', 'r' )
