# Copyright (c) 2006-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=not-callable

#-------------------------------------------------------------------------------
# This module implements bridging configuration.  In particular, it provides:
# -  the "[no] mac address-table aging-time" command
# -  the "show bridge mac-address-table aging time" command
# -  the "[no] mac address-table static" command
# -  the "show mac address-table" command
# -  the "show mac address-table count" command
# -  the "show interfaces [<intf>] switchport" command.
# -  the "show dot1q-tunnel [<intf>]" command, in "enable" mode.
# -  the "[no|default] switch forwarding-mode 
#                              cut-through | store-and-forward" command.
# -  the "[no|default] mac address-table reserved forward" command
# -  the "[no|default] mac pause-frame pass-through" command
# -  the "show mac pause-frame" command
#
# This module does not contain VLAN-specific commands.
#-------------------------------------------------------------------------------
import sys
import Arnet
import BasicCli
import BasicCliModes
import CliCommand
from CliDynamicSymbol import CliDynamicPlugin
import CliExtensions
import CliGlobal
import CliMatcher
import CliParser
from CliPlugin import BridgingCliModel
from CliPlugin.BridgingCliModel import MacAddressTable
from CliPlugin.BridgingCliModel import MacAddressTableAgingTime
from CliPlugin.BridgingCliModel import SwitchPorts
# pylint: disable-next=consider-using-from-import
import CliPlugin.EbraCliLib as EbraCliLib
# pylint: disable-next=consider-using-from-import
import CliPlugin.EbraEthIntfCli as EbraEthIntfCli
import CliPlugin.IntfCli as IntfCli # pylint: disable=consider-using-from-import
import CliPlugin.MacAddr as MacAddr # pylint: disable=consider-using-from-import
import CliPlugin.VlanCli as VlanCli # pylint: disable=consider-using-from-import
from CliPlugin.VlanCli import switchportMatcher
from CliPlugin.VlanCli import vlanIdMatcher
import CliToken.Mac
import CliToken.Switch
import ConfigMount
from EbraLib import defaultTpid
from Intf.IntfRange import IntfRangeMatcher, intfRangeMatcher
import LazyMount
import ShowCommand
import SmashLazyMount
import Tac
from TypeFuture import TacLazyType
from Vlan import VlanIdRangeList

BridgingDynCliModel = CliDynamicPlugin( "BridgingCliModel" )

PortChannelIntfId = TacLazyType( 'Arnet::PortChannelIntfId' )
SwitchForwardingMode = Tac.Type( "Bridging::SwitchForwardingMode" )

gv = CliGlobal.CliGlobal(
        bridgingConfig=None,
        bridgingFdbConfig=None,
        bridgingHwCapabilities=None,
        bridgingInCliConfig=None,
        bridgingStatus=None,
        bridgingSwitchIntfConfig=None,
        dynAllowedVlanDir=None,
        dynBlockedVlanDir=None,
        intfTpidStatus=None,
        vlanTagFormatDropStatusDir=None )

# -------------------------------------------------------------------------------
# Basic tokens for MAC address tables.
# -------------------------------------------------------------------------------
matcherAddressTable = CliMatcher.KeywordMatcher( 'address-table',
      helpdesc='MAC forwarding table' )
matcherAgingTimeConfig = CliMatcher.KeywordMatcher( 'aging-time',
      helpdesc='Set MAC address table entry maximum age')
nodeMacAddressTable = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'mac-address-table', helpdesc='_' ),
      hidden=True )

# -------------------------------------------------------------------------------
# Basic networking tokens.
# -------------------------------------------------------------------------------
matcherAgingTimeShow = CliMatcher.KeywordMatcher( 'aging-time',
      helpdesc='aging-time keyword')
matcherInterface = CliMatcher.KeywordMatcher( 'interface',
      helpdesc='Interface keyword' )
matcherVlan = CliMatcher.KeywordMatcher( 'vlan',
      helpdesc='VLAN Keyword' )
matcherCutThrough = CliMatcher.KeywordMatcher( 'cut-through',
      helpdesc='Set the switch to cut-through packets' )
matcherCutThroughDelay = CliMatcher.KeywordMatcher( 'delay',
      helpdesc='Cut through delay' )

class MacAddrTableExprForConfig( CliCommand.CliExpression ):
   expression = '( mac address-table ) | mac-address-table'
   data = {
      'mac': CliToken.Mac.macMatcherForConfig,
      'address-table': matcherAddressTable,
      'mac-address-table': nodeMacAddressTable,
   }

class MacAddrTableExprForClear( CliCommand.CliExpression ):
   expression = '( mac address-table ) | mac-address-table'
   data = {
      'mac': CliToken.Mac.macMatcherForClear,
      'address-table': matcherAddressTable,
      'mac-address-table': nodeMacAddressTable,
   }

class MacAddrTableExprForShow( CliCommand.CliExpression ):
   expression = '( mac address-table ) | mac-address-table'
   data = {
      'mac': CliToken.Mac.macMatcherForShow,
      'address-table': matcherAddressTable,
      'mac-address-table': nodeMacAddressTable,
   }

#--------------------------------------------------------------------------------
# [ no | default ] ( ( mac address-table ) | mac-address-table ) 
# aging-time ( DISABLE_AGING | AGING_TIME )
#--------------------------------------------------------------------------------
class SetAgingTimeCmd( CliCommand.CliCommandClass ):
   syntax = 'MAC_ADDR_TABLE aging-time ( DISABLE_AGING | AGING_TIME )'
   noOrDefaultSyntax = 'MAC_ADDR_TABLE aging-time ...'
   data = {
      'MAC_ADDR_TABLE': MacAddrTableExprForConfig,
      'aging-time': matcherAgingTimeConfig,
      'DISABLE_AGING': CliMatcher.IntegerMatcher( 0, 0,
         helpdesc='Enter 0 to disable aging' ),
      'AGING_TIME': CliMatcher.IntegerMatcher( 10, 1000000,
         helpdesc='Aging time in seconds' ),
   }
   handler = "BridgingCliHandler.setAgingTime"
   noOrDefaultHandler = "BridgingCliHandler.noAgingTime"

BasicCliModes.GlobalConfigMode.addCommandClass( SetAgingTimeCmd )

#-------------------------------------------------------------------------------
# The "show bridge mac-address-table aging timeout" command, in "enable" mode.
#
# The full syntax of this command is:
#   show bridge mac-address-table aging timeout
# legacy:
#   show mac address-table aging-time
#-------------------------------------------------------------------------------

def doShowAgingTime( mode, args ):
   return MacAddressTableAgingTime(
                 hostAgingTime=gv.bridgingInCliConfig.hostAgingTime )

matcherAgingTimeLegacy = CliCommand.Node( matcher=matcherAgingTimeShow, 
                           deprecatedByCmd=
                           'show bridge mac address-table aging timeout' )

class BridgeAgingTimeoutCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bridge MAC_ADDR_TABLE aging timeout'
   data = {
      'bridge': 'Bridge information',
      'MAC_ADDR_TABLE': MacAddrTableExprForShow,
      'aging': 'Aging keyword',
      'timeout': 'Timeout keyword',
   }
   handler = doShowAgingTime
   cliModel = MacAddressTableAgingTime

BasicCli.addShowCommandClass( BridgeAgingTimeoutCmd )

#--------------------------------------------------------------------------------
# show mac address-table aging-time
#--------------------------------------------------------------------------------
class AgingTimeCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show MAC_ADDR_TABLE aging-time'
   data = {
      'MAC_ADDR_TABLE': MacAddrTableExprForShow,
      'aging-time': matcherAgingTimeLegacy,
   }
   handler = doShowAgingTime
   cliModel = MacAddressTableAgingTime

BasicCli.addShowCommandClass( AgingTimeCmd )

# Providing Cli Hooks to individual features for static (configured)
# mac entries.
#
# 1. bridgingCheckStaticMacHook: Check if configuredHost can be added to
#    fdbConfig. Registered extension functions (hooks) return True/False.
# 2. bridgingAddStaticMacHook: Any extra work to be done when a static
#    mac address is added. Hooks return True/False.
# 3. bridgingDelStaticMacHook: Any extra work to be done when a static mac
#    address is deleted. Hooks return True/False.
# 4. bridgingExtraStaticMacHook: To get any additional static mac entries,
#    e.g., when the 'show mac address-table configured' is issued. Hooks
#    return a list of Bridging::ConfiguredHost.
#
# These hooks can be utilized by external individual features that manipulate
# static mac entries. For instance, Vxlan, which keeps vxlan mac table for
# remote hosts adds the hook extensions to these Cli Hooks.

bridgingCheckStaticMacHook = CliExtensions.CliHook()
bridgingAddStaticMacHook = CliExtensions.CliHook()
bridgingDelStaticMacHook = CliExtensions.CliHook()
bridgingExtraStaticMacHook = CliExtensions.CliHook()

# Callback for Trident4 platforms to warn of L2 configuration in unsupported
# UFT modes
warnMacTableUnsupportedUFTModeHook = CliExtensions.CliHook()

#-------------------------------------------------------------------------------
# The "[no] mac address-table static" command, in "config" mode.
#
# The full syntax of this command is:
#
#   mac address-table static <mac_addr> vlan <vlan_id>
#                             {interface <eth_intf>+|drop <address destination>}
#   {no|default} mac address-table static <mac_addr> vlan <vlan_id>
#                             {interface <eth_intf>+|drop <address destination>}
#-------------------------------------------------------------------------------
staticKwMatcher = CliMatcher.KeywordMatcher( 'static', helpdesc='static keyword' )

def destinationDropMacGuard( mode, token ):
   if gv.bridgingHwCapabilities.destinationDropMacSupported:
      return None
   return CliParser.guardNotThisPlatform

def addressDropMacGuard( mode, token ):
   return destinationDropMacGuard( mode, token )

macAddressStaticIntfGuards_ = {}

def registerMacAddressStaticIntfGuard( intfType, guard ):
   macAddressStaticIntfGuards_[ intfType.tagLong ] = guard

class MacAddressStaticCmd( CliCommand.CliCommandClass ):
   syntax = ''' MAC_ADDR_TABLE static MACADDR vlan VLAN_ID
               ( ( drop [ address destination ] ) |
                 ( interface { INTFS } ) ) '''
   noOrDefaultSyntax = ''' MAC_ADDR_TABLE static MACADDR vlan VLAN_ID
                           [ ( drop [ address destination ] ) |
                             ( interface { INTFS } ) ] '''
   data = {
      'MAC_ADDR_TABLE': MacAddrTableExprForConfig,
      'static': staticKwMatcher,
      'MACADDR': MacAddr.MacAddrMatcher(),
      'vlan': 'On VLAN',
      'VLAN_ID': vlanIdMatcher,
      'drop': 'Drop frames',
      'address': CliCommand.guardedKeyword( 'address',
                     helpdesc='Address for drop frames',
                     guard=addressDropMacGuard ),
      'destination': CliCommand.guardedKeyword( 'destination',
                     helpdesc='Match destination address in drop frames',
                     guard=destinationDropMacGuard ),
      'interface': 'Interface',
      'INTFS': IntfRangeMatcher(
                     explicitIntfTypes=EbraEthIntfCli.switchPortIntfTypes,
                     intfTypeGuards=macAddressStaticIntfGuards_ )
   }

   handler = "BridgingCliHandler.setStatic"
   noOrDefaultHandler = handler

class MacAddressStaticWithForwardingEligibleCmd( CliCommand.CliCommandClass ):
   syntax = ''' MAC_ADDR_TABLE static MACADDR vlan VLAN_ID
                ( ( drop [ address destination ] )
                | ( interface { INTFS } [ eligibility forwarding ] ) ) '''

   noOrDefaultSyntax = ''' MAC_ADDR_TABLE static MACADDR vlan VLAN_ID
                           [ ( drop [ address destination ] )
                           | ( [ interface { INTFS } ] [ eligibility forwarding ] )
                           ] '''

   data = {
      'MAC_ADDR_TABLE': MacAddrTableExprForConfig,
      'static': staticKwMatcher,
      'MACADDR': MacAddr.MacAddrMatcher(),
      'vlan': 'On VLAN',
      'VLAN_ID': vlanIdMatcher,
      'drop': 'Drop frames',
      'address': CliCommand.guardedKeyword( 'address',
                     helpdesc='Address for drop frames',
                     guard=addressDropMacGuard ),
      'destination': CliCommand.guardedKeyword( 'destination',
                     helpdesc='Match destination address in drop frames',
                     guard=destinationDropMacGuard ),
      'interface': 'Interface',
      'INTFS': IntfRangeMatcher(
                     explicitIntfTypes=EbraEthIntfCli.switchPortIntfTypes,
                     intfTypeGuards=macAddressStaticIntfGuards_ ),
      'eligibility': 'Specify eligiblity criteria',
      'forwarding': 'Ability to forward traffic on the '
                    'interface with the specified vlan',
   }

   handler = "BridgingCliHandler.setStatic"
   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass(
   MacAddressStaticWithForwardingEligibleCmd )

#-------------------------------------------------------------------------------
# The "show mac address-table" command, in "enable" mode.
#
# The full syntax of this command is:
#
#   show mac address-table [static|dynamic|unicast|configured] [address <mac_addr>]
#                                            [interface <intf>] [vlan <vlan_id>]
#-------------------------------------------------------------------------------
def filterUnicastMacAddr( macEntry, intfSet, addrTypeFilter='' ):
   # Filter by intf
   if len( intfSet ) > 0 and \
          len( { macEntry.intf } & intfSet ) == 0:
      return False
   # Filter by type
   if addrTypeFilter is not None and addrTypeFilter != 'unicast':
      if addrTypeFilter == 'static':
         return macEntry.entryType in [ 'configuredStaticMac', 'peerStaticMac',
                                        'configuredRemoteMac',
                                        'peerConfiguredRemoteMac',
                                        'evpnConfiguredRemoteMac',
                                        'dfwConfiguredStaticMac' ]
      elif addrTypeFilter == 'configured':
         return macEntry.entryType in [ 'configuredStaticMac',
                                        'configuredRemoteMac' ]
      else:
         return macEntry.entryType in [ 'learnedDynamicMac', 'peerDynamicMac',
                                        'learnedRemoteMac', 'receivedRemoteMac',
                                        'peerLearnedRemoteMac',
                                        'peerReceivedRemoteMac',
                                        'evpnDyamicRemoteMac',
                                        'peerEvpnRemoteMac',
                                        'configuredDynamicMac',
                                        'vplsDynamicRemoteMac' ]

   # Don't filter by type
   else:
      return True

def filterMulticastMacAddr( macEntry, intfs, addrTypeFilter='' ):
   # Do not return multicast entries if dynamic
   assert addrTypeFilter != 'unicast'
   if addrTypeFilter is not None and addrTypeFilter == 'dynamic':
      return False

   if not intfs:
      return True

   # Filter by intf
   return set( macEntry.intf ).intersection( intfs )

def getStaticMulticastTable( mode, vlans, macAddr, intfs, vlanConfigs,
                             addrTypeFilter='' ):

   if mode.session_.guardsEnabled() \
          and not gv.bridgingHwCapabilities.staticMcastSupported:
      return None

   multicastTable = BridgingCliModel.MulticastMacAddressTable()

   for v in vlans:
      if isinstance( v, VlanCli.Vlan ):
         vlanId = v.id
      else:
         vlanId = int( v )
      vlanConfig = vlanConfigs.get( vlanId )
      if not vlanConfig and addrTypeFilter != "configured":
         continue
      fdbConfig = gv.bridgingFdbConfig.fdbConfig.get( vlanId )

      if fdbConfig:
         groupTable = fdbConfig.ethGroup
      else:
         continue

      addresses = []

      if macAddr is not None:
         macEntry = groupTable.get( macAddr )
         if macEntry and filterMulticastMacAddr( macEntry, intfs,
               addrTypeFilter=addrTypeFilter ):
            addresses = [ macEntry ]
      else:
         addresses = ( addr for addr in groupTable.values()
                       if filterMulticastMacAddr( addr, intfs,
                          addrTypeFilter=addrTypeFilter ) )

      for addr in sorted( addresses, key=lambda x: x.addr ):
         printIntfs = Arnet.sortIntf( addr.intf )

         mac = addr.getRawAttribute( "addr" )  # Avoid string conversion.
         multicastTable.tableEntries.append(
            BridgingCliModel.MulticastMacAddressTable.MulticastTableEntry(
                                 vlanId=vlanId,
                                 macAddress=mac,
                                 interfaces=printIntfs,
                                 entryType="static" ) )
   return multicastTable

def doShowTable( mode, args ):
   showMacTable = Tac.newInstance( "ShowMacTableNs::ShowMacTable",
                                   gv.bridgingInCliConfig.force(),
                                   gv.bridgingStatus,
                                   gv.bridgingFdbConfig.force(),
                                   gv.bridgingConfig.force(),
                                   IntfCli.intfIdDisplayContextHelper )

   # Pass the filters to C++
   vlan = args.get( 'VLAN_ID' )
   vlanId = vlan.id if vlan else 0
   showMacTable.vlanIdFilter = Tac.Value( 'Bridging::VlanIdOrNone', vlanId )
   if mac := args.get( 'MAC_ADDRESS' ):
      newMacFilter = Tac.Value( 'Arnet::EthAddr' )
      newMacFilter.stringValue = mac
      showMacTable.macFilter = newMacFilter
      showMacTable.macFilterPresent = True

   addrTypeFilter = args.get( 'ADDRTYPE_FILTER', '' )
   showMacTable.addrTypeFilter = addrTypeFilter
   for intf in args.get( 'INTFS', () ):
      showMacTable.intfFilter.add( Tac.Value( 'Arnet::IntfId', intf ) )

   # Should the multicast table be printed ?
   if addrTypeFilter != 'unicast':
      if not mode.session_.guardsEnabled() \
         or gv.bridgingHwCapabilities.staticMcastSupported:
         showMacTable.populateMulticast = True

   # Get Mac addresses from other modules
   # for example, Vxlan has a list of MACs to be shown.
   if addrTypeFilter == 'configured':
      if vlan:
         vlans = []
         if vlanId in gv.bridgingFdbConfig.fdbConfig:
            vlans = [ vlanId ]
      else:
         vlans = gv.bridgingFdbConfig.fdbConfig

      for v in vlans:
         notifiedAddresses = []
         for hook in bridgingExtraStaticMacHook.extensions():
            notifiedAddresses.extend( hook( None, v ) )

         vlanObj = Tac.Value( 'Bridging::VlanId', v )
         notifiedMac = showMacTable.notifiedMac.newMember( vlanObj )
         for count, cHost in enumerate( notifiedAddresses ):
            notifiedMac.host[ count ] = cHost

   # Populate the entries
   # implemented in C++ for efficiency
   showMacTable.populateMacTblDisplayBuffer()

   # Print the entries
   # Implemented in C++ for efficiency
   fd = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()
   revision = mode.session_.requestedModelRevision()
   showMacTable.render( fd, fmt, revision )
   sys.stdout.flush()

   # Deferred Model.
   # Return the empty model without data
   return BridgingCliModel.MacAddressTable

class MacAddressTableCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show MAC_ADDR_TABLE ' 
              '[ ADDRTYPE_FILTER ] [ address MAC_ADDRESS ] [ interface INTFS ] ' 
              ' [ vlan VLAN_ID ]' )
   data = {
      'MAC_ADDR_TABLE': MacAddrTableExprForShow,
      'ADDRTYPE_FILTER': CliMatcher.EnumMatcher( {
         'static': 'Static entry type',
         'dynamic': 'Dynamic entry type',
         'unicast': 'Unicast entry type',
         'configured': 'Configured MAC entries',
      } ),
      'address': 'Address keyword',
      'MAC_ADDRESS': MacAddr.MacAddrMatcher(),
      'interface': 'Interface',
      'INTFS': intfRangeMatcher,
      'vlan': 'On VLAN',
      'VLAN_ID': vlanIdMatcher,
   }

   cliModel = MacAddressTable
   handler = doShowTable

BasicCli.addShowCommandClass( MacAddressTableCmd )

#-------------------------------------------------------------------------------
# The "show mac address-table count" command, in "enable" mode.
#
# The full syntax of this command is:
#
#   show mac address-table count [vlan <vlan_id>]
#-------------------------------------------------------------------------------
def getCountersViaSmash( vlans ):
   counters = {}
   vlanConfigs = gv.bridgingConfig.vlanConfig

   # initialize the counters dictionary with the valid vlans
   for v in vlans:
      vlanConfig = vlanConfigs.get( v.id )
      if not vlanConfig:
         continue

      counters[ v.id ] = { 'dynamic': 0, 'static': 0, 'multicast': 0 }
      fdbConfig = gv.bridgingFdbConfig.fdbConfig.get( v.id )
      if fdbConfig:
         counters[ v.id ][ 'multicast' ] = len( fdbConfig.ethGroup )

   # iterate through the entire Smash table and populate the counters dictionary
   for entry in gv.bridgingStatus.smashFdbStatus.values():
      if entry.fid in counters:
         if entry.entryType in [ 'configuredStaticMac', 'peerStaticMac',
                                 'configuredRemoteMac', 'peerConfiguredRemoteMac',
                                 'evpnConfiguredRemoteMac', 'evpnIntfStaticMac',
                                 'authenticatedMac', 'peerAuthenticatedMac',
                                 'dfwConfiguredStaticMac' ]:
            counters[ entry.fid ][ 'static' ] += 1
         else:
            counters[ entry.fid ][ 'dynamic' ] += 1

   return counters

def doShowCount( mode, args ):
   ret = BridgingCliModel.MacAddressTableCount()
   if args.get( 'VLAN_ID' ):
      if args[ 'VLAN_ID' ].lookup( mode ):
         vlans = [ args[ 'VLAN_ID' ] ]
      else:
         # Just fail silently if the VLAN doesn't exist.  This is compatible
         # with the industry-standard.
         return ret
   else:
      vlans = VlanCli.Vlan.getAll( mode, secondaryVlans=False )

   # get counters through smash
   smashCounters = getCountersViaSmash( vlans )

   for v in vlans:
      counters = smashCounters.get( v.id )
      if counters is None:
         continue

      vlanCount = BridgingCliModel.MacAddressTableCount.VlanCount()
      vlanCount.dynamic = counters[ 'dynamic' ]
      vlanCount.unicast = counters[ 'static' ]
      vlanCount.multicast = counters[ 'multicast' ]
      ret.vlanCounts[ v.id ] = vlanCount

   return ret

class MacAddressTableCountCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show MAC_ADDR_TABLE count [ vlan VLAN_ID ]'
   data = { 
      'MAC_ADDR_TABLE': MacAddrTableExprForShow,
      'count': 'Total number of MAC entries',
      'vlan': 'On VLAN',
      'VLAN_ID': vlanIdMatcher,
   }

   cliModel = BridgingCliModel.MacAddressTableCount
   handler = doShowCount

BasicCli.addShowCommandClass( MacAddressTableCountCmd )     

#-------------------------------------------------------------------------------
# The "show interfaces [<intf>] switchport" command, in "enable" mode.
#
# The full syntax of this command is:
#
#   show interfaces [<intf>] switchport
#   show interfaces switchport module <mod>
#   show interfaces module <mod> switchport
#-------------------------------------------------------------------------------
def doShowSwitchport( mode, args ): # pylint: disable=inconsistent-return-statements
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )

   def convertToPrintForm( allowedVlans ):
      if allowedVlans == '':
         return 'NONE'
      elif allowedVlans == '1-4094':
         return 'ALL'
      else:
         return allowedVlans

   switchPorts = BridgingCliModel.SwitchPorts()
   intfs = IntfCli.Intf.getAll( mode, intf, mod )
   if intfs is None:
      return

   # Note that if no interface name is specified, we only show the interfaces that
   # are configured as switchports.  However, if an interface name is specified, we
   # show that interface regardless of whether it is configured as a switchport.
   if intf is None:
      print( 'Default switchport mode: '
             f'{ gv.bridgingInCliConfig.defaultSwitchportMode }' )
      if gv.bridgingInCliConfig.defaultPhoneVlan:
         print( 'Default switchport phone VLAN: '
                f'{ gv.bridgingInCliConfig.defaultPhoneVlan }' )
      print( 'Default switchport phone COS: '
             f'{ gv.bridgingInCliConfig.defaultPhoneCos }' )
      tagString = 'untagged' if gv.bridgingInCliConfig.defaultPhoneTrunkTagMode == \
                                'phoneTrunkUntagged' else 'tagged'
      print( f'Default switchport phone trunk tagging: { tagString }' )
      print()
      intfs = [ i for i in intfs if switchIntfConfigIfEnabled( mode, i ) ]

   vlanTagFormatDrop = {}
   vlanTagRequiredDropFormats = \
      Tac.newInstance( "Bridging::VlanTagRequired" ).vlanTagFormatDrop
   vlanTagDisallowedDropFormats = \
      Tac.newInstance( "Bridging::VlanTagDisallowed" ).vlanTagFormatDrop
   for vlanTagFormatDropStatus in gv.vlanTagFormatDropStatusDir.entityPtr.values():
      for intf, status in \
            vlanTagFormatDropStatus.intfVlanTagFormatDropStatus.items():
         vlanTagFormatDrop[ intf ] = dict( list( status.vlanTagFormatDrop.items() ) )

   for intf in intfs:
      if PortChannelIntfId.isPortChannelIntfId( intf.name ):
         # get active lagMembers for port channel
         # we cannot do Tac.Type( "Lag::LagStatusFilter" ).filterOnActive here
         # as we cannot depend on Lag package
         lagMembers =\
           list( EbraCliLib.getLagMembersCallBack( # pylint: disable-msg=not-callable
              mode, intf.name, lacpOnly=False, status='filterOnActive' ) )
         vlanTagFormatDrop[ intf.name ] = {}
         if lagMembers:
            for tagFormat in vlanTagRequiredDropFormats:
               if all( vlanTagFormatDrop.get( m, {} ).get( tagFormat, False )
                       for m in lagMembers ):
                  vlanTagFormatDrop[ intf.name ][ tagFormat ] = True
            for tagFormat in vlanTagDisallowedDropFormats:
               if all( vlanTagFormatDrop.get( m, {} ).get( tagFormat, False )
                       for m in lagMembers ):
                  vlanTagFormatDrop[ intf.name ][ tagFormat ] = True
            for p in lagMembers:
               vlanTagFormatDrop.pop( p, None )

   for i in intfs: # pylint: disable=too-many-nested-blocks
      switchport = BridgingCliModel.SwitchPort()
      switchIntfConfig = switchIntfConfigIfEnabled( mode, i )
      switchIntfBridgeConfig = switchIntfBridgeConfigIfEnabled( mode, i )
      switchport.enabled = bool( switchIntfConfig )

      if switchIntfConfig:
         switchportInfo = BridgingCliModel.SwitchPortInfo()
         accessVlan = VlanCli.Vlan( switchIntfConfig.accessVlan )
         if accessVlan.lookup( mode ):
            accessVlanName = accessVlan.name( mode )
         else:
            accessVlanName = 'inactive'

         trunkNativeVlan = VlanCli.Vlan( switchIntfConfig.trunkNativeVlan )
         if trunkNativeVlan.lookup( mode ):
            trunkNativeVlanName = trunkNativeVlan.name( mode )
         else:
            trunkNativeVlanName = 'inactive'

         switchportInfo.mode = switchIntfConfig.switchportMode
         switchportInfo.phoneTrunk = switchIntfConfig.phoneTrunk
         switchportInfo.macLearning = switchIntfConfig.macLearningEnabled

         if hasattr( i, 'intfId' ):
            intfName = i.intfId
         else:
            intfName = i.name

         if gv.bridgingHwCapabilities.vlanTpidSupported:
            tpid = switchIntfConfig.vlanTpid
            if not intfName.startswith( 'Et' ):
               status = None
            else:
               e = gv.intfTpidStatus.entry.get( intfName )
               if e is None:
                  if tpid == defaultTpid:
                     status = True
                  else:
                     status = None
               else:
                  status = ( e.tpid == tpid and e.active )
            switchportInfo.tpid = hex( tpid )
            if status is not None:
               switchportInfo.tpidStatus = status
         else:
            # Set the default TPID ( 0x8100 )
            switchportInfo.tpid = hex( defaultTpid )
            switchportInfo.tpidStatus = True

         # Set dot1qVlanTagRequired and dot1qVlanTagRequiredStatus
         switchportInfo.dot1qVlanTagRequired = all(
            switchIntfConfig.vlanTagFormatDrop.get( tagFormat, False )
            for tagFormat in vlanTagRequiredDropFormats )
         dot1qVlanTagRequiredStatus = False
         dot1qVlanTagDisallowed = False
         vlanTagFormatDropStatus = vlanTagFormatDrop.get( intfName, None )
         if vlanTagFormatDropStatus:
            # Checks if vlanTagFormatDropStatus has all vlanTagRequiredDropFormats
            # vlanTagRequiredDropFormats collection has priority and untagged
            # set to true
            # If they are present then dot1qVlanTagRequiredStatus is set to true
            dot1qVlanTagRequiredStatus = all(
               vlanTagFormatDropStatus.get( tagFormat,
                  False ) for tagFormat in vlanTagRequiredDropFormats )
            # Checks if vlanTagFormatDropStatus has all vlanTagDisallowedDropFormats
            # vlanTagDisallowedDropFormats collection has singleTagged set to true
            # If they are present then dot1qVlanTagDisallowed is set to true
            dot1qVlanTagDisallowed = all(
               vlanTagFormatDropStatus.get( tagFormat,
                  False ) for tagFormat in vlanTagDisallowedDropFormats )
         switchportInfo.dot1qVlanTagRequiredStatus = dot1qVlanTagRequiredStatus
         switchportInfo.dot1qVlanTagDisallowed = dot1qVlanTagDisallowed

         # BUG210 The operational mode should be 'down' if the port is down.
         switchportInfo.accessVlanId = accessVlan.id
         switchportInfo.accessVlanName = accessVlanName
         switchportInfo.trunkingNativeVlanId = trunkNativeVlan.id
         if trunkNativeVlan.id != 0:
            switchportInfo.trunkingNativeVlanName = trunkNativeVlanName

         if gv.bridgingHwCapabilities.privateVlansSupported:
            if not switchIntfConfig.defaultPrivateVlanMapping:
               pvSet = Tac.newInstance( 'EbraUtils::VlanIdSet' )
               for v in switchIntfConfig.privateVlanMapping:
                  pvSet.doAdd( v )
               pvStr = pvSet.multiRangeString()
               switchportInfo.privateVlanMapping = convertToPrintForm( pvStr )
         dynamicTrunkGroupNames = []

         # Make a list of dynamic allowed vlans for this interface from different
         # agents. Also concatenate these vlans strings into a single allowed vlan
         # string.
         dynAllowedVlans = { }
         allVlans = VlanIdRangeList(
                       initialString=switchIntfConfig.trunkAllowedVlans )
         intfId = i.name
         isRecirc = Tac.Type( "Arnet::PortChannelIntfId" ).isRecirc( intfId )
         isUnconnectedEth = Tac.Type( "Arnet::EthIntfId" ).isUnconnected( intfId )
         noDynamicInfo = i.isSubIntf() or isRecirc or isUnconnectedEth
         if not noDynamicInfo:
            for key in gv.dynAllowedVlanDir.entityPtr:
               entity = gv.dynAllowedVlanDir.entityPtr.get( key, None )
               if entity:
                  for groupName, tg in entity.trunkGroup.items():
                     if intfId in tg.intfList:
                        dynamicTrunkGroupNames.append( groupName )
                  allowedVlans = entity.vlans.get( intfId, None )
                  if allowedVlans is not None and allowedVlans != "":
                     dynAllowedSet = VlanIdRangeList( initialString=allowedVlans )
                     allVlans.addVlanIdRangeList( dynAllowedSet )
                     canonicalStr = str( dynAllowedSet )
                     dynAllowedVlans[ key ] = convertToPrintForm( canonicalStr )

         # If there were any dynamic allowed vlans
         # then print union of dynamic and static allowed vlans
         # as the VLANs enabled on this interface and
         # also print each set separately.
         if len( dynAllowedVlans ):
            switchportInfo.staticAllowedVlans = convertToPrintForm(
               switchIntfConfig.trunkAllowedVlans )
            dynVlans = {}
            for key in dynAllowedVlans: # pylint: disable=consider-using-dict-items
               dynVlans[ key ] = dynAllowedVlans[ key ]
            switchportInfo.dynamicAllowedVlans = dynVlans

         # If any VLANs are blocked, include that information
         dynamicBlockedVlans = {}
         for agent in gv.dynBlockedVlanDir.entityPtr:
            entity = gv.dynBlockedVlanDir.entityPtr.get( agent )
            if entity:
               blockVlans = entity.vlans.get( intfId )
               if blockVlans:
                  blockVlanSet = VlanIdRangeList( initialString=blockVlans )
                  allVlans.removeVlanRangeList( blockVlanSet )
                  dynamicBlockedVlans[ agent ] = convertToPrintForm(
                     str( blockVlanSet ) )

         switchportInfo.dynamicBlockedVlans = dynamicBlockedVlans
         switchportInfo.trunkAllowedVlans = convertToPrintForm( str( allVlans ) )

         # BUG3226 Need to re-enable this output when the 'switchport block' commands
         # are re-enabled.
         # print 'Unknown unicast blocked: %s' % blockUnicastStr
         # print 'Unknown multicast blocked: %s' % blockMulticastStr

         staticTrunkGroupNames = list( switchIntfConfig.trunkGroup )
         staticTrunkGroupNames.sort()
         switchportInfo.staticTrunkGroups = staticTrunkGroupNames
         dynamicTrunkGroupNames.sort()
         switchportInfo.dynamicTrunkGroups = dynamicTrunkGroupNames
         switchportInfo.phoneVlan = switchIntfBridgeConfig.phoneVlan
         switchportInfo.phoneTrunkUntagged = (
              switchIntfBridgeConfig.phoneTrunkTagMode == 'phoneTrunkUntagged' )

         switchportInfo.mbvaEnabled = False
         if switchIntfBridgeConfig.phoneTrunk:
            if switchportInfo.phoneTrunkUntagged or (
                  switchIntfBridgeConfig.phoneTrunkTagMode == 'notSet' and
                  gv.bridgingConfig.defaultPhoneTrunkTagMode ==
                  'phoneTrunkUntagged' ):
               switchportInfo.mbvaEnabled = True

         if not gv.bridgingHwCapabilities.sourceportFilterSupported:
            switchportInfo.sourceportFilterMode = 'enabled'
         else:
            if switchIntfConfig.sourceportFilterMode == 'sourceportFilterEnabled':
               switchportInfo.sourceportFilterMode = 'enabled'
            elif switchIntfConfig.sourceportFilterMode == \
                  'sourceportFilterMulticastDisabled':
               switchportInfo.sourceportFilterMode = 'multicastDisabled'
            else:
               switchportInfo.sourceportFilterMode = 'disabled'
         if switchIntfConfig.vlanForwardingMode == \
               'vlanForwardingModeAllowedVlansOnPort':
            switchportInfo.vlanForwardingMode = BridgingCliModel.allowedVlansOnlyStr
         else:
            switchportInfo.vlanForwardingMode = \
               BridgingCliModel.allConfiguredVlansStr
         switchport.switchportInfo = switchportInfo
      switchPorts.switchports[ i.name ] = switchport
   return switchPorts

def switchIntfBridgeConfigIfEnabled( mode, intf ):
   switchIntfConfigs = gv.bridgingConfig.switchIntfConfig

   if hasattr( intf, 'intfid' ):
      switchIntfConfig = switchIntfConfigs.switchIntfConfig.get( intf.intfid )
   else:
      switchIntfConfig = switchIntfConfigs.get( intf.name )
   return switchIntfConfig

# If the SwitchInterfaceConfig for the given interface is present and not
# disabled, return it.  Otherwise return None.
def switchIntfConfigIfEnabled( mode, intf ):
   switchIntfConfigs = gv.bridgingSwitchIntfConfig
   # this function is called with different types of intfs, some have the
   # intfId attribute, use this attr it if has it instead of name
   # BUG27653
   if hasattr( intf, 'intfId' ):
      switchIntfConfig = switchIntfConfigs.switchIntfConfig.get( intf.intfId )
   else:
      switchIntfConfig = switchIntfConfigs.switchIntfConfig.get( intf.name )
   enabled = switchIntfConfig and switchIntfConfig.enabled
   if enabled:
      return switchIntfConfig
   else:
      return None

class ShowIntfSwitchport( IntfCli.ShowIntfCommand ):
   syntax = "show interfaces switchport"
   data = dict( switchport='Details on switchports' )
   cliModel = SwitchPorts
   handler = doShowSwitchport
   moduleAtEnd = True

BasicCli.addShowCommandClass( ShowIntfSwitchport )

#-------------------------------------------------------------------------------
# The "show dot1q-tunnel [interface <intf>]" command, in "enable" mode.
#
# The full syntax of this command is:
#
#   show dot1q-tunnel [interface <intf>]
#-------------------------------------------------------------------------------

class Dot1QTunnelCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show dot1q-tunnel [ interface INTERFACE ]'
   data = {
         'dot1q-tunnel': 'Details for 802.1q tunnels',
         'interface': 'Interface keyword',
         'INTERFACE': intfRangeMatcher,
         }
   handler = "BridgingCliHandler.doShowDot1QTunnel"
   cliModel = "BridgingCliModel.Dot1QTunnel"

BasicCli.addShowCommandClass( Dot1QTunnelCmd )

#-------------------------------------------------------------------------------
# The "switch forwarding-mode cut-through | store-and-forward" command, 
# in "config" mode.
#
# The full syntax of this command is:
#
#   switch forwarding-mode cut-through
#   switch forwarding-mode store-and-forward
#
#-------------------------------------------------------------------------------
# we only enable the command if the platform supports it. 

def supportedForwardingModeGuard( mode, token ):
   if len( gv.bridgingHwCapabilities.supportedForwardingMode ) >= 1:
      return None
   return CliParser.guardNotThisPlatform

def cutThroughGuard( mode, token ):
   if len( gv.bridgingHwCapabilities.supportedForwardingMode ) and \
          ( SwitchForwardingMode.cutThrough in 
            list( gv.bridgingHwCapabilities.supportedForwardingMode.values() ) ):
      return None
   return CliParser.guardNotThisPlatform

matcherForwardingMode = CliCommand.guardedKeyword( 'forwarding-mode',
                            helpdesc='Set the switch forwarding mode',
                            guard=supportedForwardingModeGuard )

def storeAndForwardGuard( mode, token ):
   if len( gv.bridgingHwCapabilities.supportedForwardingMode ) and \
          ( SwitchForwardingMode.storeAndForward in
            list( gv.bridgingHwCapabilities.supportedForwardingMode.values() ) ):
      return None
   return CliParser.guardNotThisPlatform

class CutThroughCmd( CliCommand.CliCommandClass ):
   syntax = 'switch forwarding-mode ( cut-through | store-and-forward )'
   noOrDefaultSyntax = 'switch forwarding-mode [ cut-through | store-and-forward ]'
   data = {
      'switch': CliToken.Switch.matcherSwitchForConfig,
      'forwarding-mode': matcherForwardingMode,
      'cut-through': CliCommand.Node( matcher=matcherCutThrough,
                                      guard=cutThroughGuard ),
      'store-and-forward' : CliCommand.guardedKeyword( 'store-and-forward',
                               helpdesc='Set the switch to store-and-forward '
                               'packets', guard=storeAndForwardGuard ),
      }

   handler = "BridgingCliHandler.setForwardingMode"
   noOrDefaultHandler = "BridgingCliHandler.noForwardingMode"

BasicCli.GlobalConfigMode.addCommandClass( CutThroughCmd )

#-------------------------------------------------------------------------------
# The "switch forwarding-mode store-and-forward multicast" command, 
# in "config" mode.
#
# The full syntax of this command is:
#
#   [ no | default ] switch forwarding-mode store-and-forward multicast
#
#-------------------------------------------------------------------------------
# we only enable the command if the platform supports it. 

def mcastForwardingModeGuard( mode, token ):
   if len( gv.bridgingHwCapabilities.supportedL3McastForwardingMode ):
      return None
   return CliParser.guardNotThisPlatform

def mcastStoreAndForwardGuard( mode, token ):
   if len( gv.bridgingHwCapabilities.supportedL3McastForwardingMode ) and \
      ( SwitchForwardingMode.storeAndForward in
         list( gv.bridgingHwCapabilities.supportedL3McastForwardingMode.values() ) ):
      return None
   return CliParser.guardNotThisPlatform

class McastStoreAndForwardCmd( CliCommand.CliCommandClass ):
   syntax = 'switch forwarding-mode store-and-forward multicast'
   noOrDefaultSyntax = syntax
   data = {
      'switch' : CliToken.Switch.matcherSwitchForConfig,
      'forwarding-mode' : matcherForwardingMode,
      'store-and-forward' : CliCommand.guardedKeyword( 'store-and-forward',
                               helpdesc='Set the switch to store-and-forward '
                               'packets', guard=mcastStoreAndForwardGuard,
                                hidden=True ),
      'multicast' : CliCommand.guardedKeyword( 'multicast',
                       helpdesc='Set the forwarding mode for multicast packets only',
                       guard=mcastForwardingModeGuard, hidden=True ),
      }
   handler = "BridgingCliHandler.setMcastStoreAndForward"
   noOrDefaultHandler = "BridgingCliHandler.noForwardingMode"

BasicCli.GlobalConfigMode.addCommandClass( McastStoreAndForwardCmd )

#-------------------------------------------------------------------------------
# The "show switch forwarding-mode" command
#
# The full syntax of this command is:
#
#   show switch forwarding-mode
#-------------------------------------------------------------------------------
def doShowForwardingMode( mode, args ):
   ret = BridgingCliModel.SwitchForwardingMode()
   ret.currentMode = gv.bridgingConfig.forwardingMode
   for supportedMode in gv.bridgingHwCapabilities.supportedForwardingMode.values():
      ret.supportedModes.append( ret.SupportedMode( supportedMode=supportedMode ) )
   ret.currentL3McastMode = gv.bridgingConfig.l3McastForwardingMode
   for supportedMode in \
         gv.bridgingHwCapabilities.supportedL3McastForwardingMode.values():
      SupportedL3McastMode = ret.SupportedMode( supportedMode=supportedMode )
      ret.supportedL3McastModes.append( SupportedL3McastMode )
   return ret

class ShowForwardingModeCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show switch forwarding-mode'
   data = {
      'switch' : CliToken.Switch.matcherSwitchForShow,
      'forwarding-mode' : 'Show switching mode configuration',
      }

   handler = doShowForwardingMode
   cliModel = BridgingCliModel.SwitchForwardingMode

BasicCli.addShowCommandClass( ShowForwardingModeCmd )

#-------------------------------------------------------------------------------
# The "switchport default mode access | routed" command, 
# in "config" mode.
#
# The full syntax of this command is:
#
#   switchport default mode access
#   switchport default mode routed
#-------------------------------------------------------------------------------
# we only enable the command if the platform supports it. 

class SwitchportDefaultModeCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport default mode MODE_TYPE'
   noOrDefaultSyntax = 'switchport default mode ...'
   data = {
      'switchport': switchportMatcher,
      'default': 'Configure switchport default behavior',
      'mode': 'Set the switchport default mode',
      'MODE_TYPE': CliMatcher.EnumMatcher( {
         'access': 'Set the switchport default mode to access',
         'routed': 'Set the switchport default mode to routed',
      } ),
   }

   handler = "BridgingCliHandler.setSwitchportDefaultMode"
   noOrDefaultHandler = handler

BasicCliModes.GlobalConfigMode.addCommandClass( SwitchportDefaultModeCmd )

#-------------------------------------------------------------------------------
# The "[no|default] mac address-table reserved forward" command, in "config mode".
#
# The full syntax of this command is:
#
#    mac address-table reserved forward <mac_addr>
#    no mac address-table reserved forward <mac_addr>
#    default mac address-table reserved forward <mac_addr>
#-------------------------------------------------------------------------------
def ieeeReservedMacForwardSupportedGuard( mode, token ):
   if ( gv.bridgingHwCapabilities.ieeeReservedMacForwardAddrSupported or
        gv.bridgingHwCapabilities.ieeeReservedMacForwardGroup ):
      return None
   else:
      return CliParser.guardNotThisPlatform

def ieeeReservedMacForwardAddrSupportedGuard( mode, token ):
   if gv.bridgingHwCapabilities.ieeeReservedMacForwardAddrSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def ieeeReservedMacForwardMicroBfdSupportedGuard( mode, token ):
   if gv.bridgingHwCapabilities.microBfdReservedForwardingSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def ieeeReservedMacForwardGroupSupportedGuard( mode, token ):
   if gv.bridgingHwCapabilities.ieeeReservedMacForwardGroup:
      return None
   else:
      return CliParser.guardNotThisPlatform

class macAddressExpression( CliCommand.CliExpression ):
   expression = '''MAC_FORWARD_NODE | MAC_FORWARD_ADDR_NODE |
                MAC_FORWARD_MICROBFD_NODE | MAC_FORWARD_GROUP_NODE'''
   data = {
           'MAC_FORWARD_NODE': CliCommand.Node(
                matcher=CliMatcher.EnumMatcher( {
                           '0180.c200.0001-0180.c200.000f': 'Forward all frames '
                                'destined to 0180.c200.0001 to 0180.c200.000f',
                           '0180.c200.0001': 'Forward all frames destined to '
                                '0180.c200.0001 except Ethernet Pause frames',
                           '0180.c200.0002': 'Forward all frames destined to '
                                '0180.c200.0002',
                } ),
                alias='MAC_ADDR',
                guard=ieeeReservedMacForwardSupportedGuard ),
           'MAC_FORWARD_ADDR_NODE': CliCommand.Node(
                matcher=CliMatcher.EnumMatcher( {
                           '0180.c200.0003': 'Forward all frames destined to '
                                '0180.c200.0003',
                           '0180.c200.0004': 'Forward all frames destined to '
                                '0180.c200.0004',
                           '0180.c200.0005': 'Forward all frames destined to '
                                '0180.c200.0005',
                           '0180.c200.0006': 'Forward all frames destined to '
                                '0180.c200.0006',
                           '0180.c200.0007': 'Forward all frames destined to '
                                '0180.c200.0007',
                           '0180.c200.0008': 'Forward all frames destined to '
                                '0180.c200.0008',
                           '0180.c200.0009': 'Forward all frames destined to '
                                '0180.c200.0009',
                           '0180.c200.000a': 'Forward all frames destined to '
                                '0180.c200.000a',
                           '0180.c200.000b': 'Forward all frames destined to '
                                '0180.c200.000b',
                           '0180.c200.000c': 'Forward all frames destined to '
                                '0180.c200.000c',
                           '0180.c200.000d': 'Forward all frames destined to '
                                '0180.c200.000d',
                           '0180.c200.000e': 'Forward all frames destined to '
                                '0180.c200.000e',
                           '0180.c200.000f': 'Forward all frames destined to '
                                '0180.c200.000f',
                } ),
                alias='MAC_ADDR',
                guard=ieeeReservedMacForwardAddrSupportedGuard ),
           'MAC_FORWARD_MICROBFD_NODE': CliCommand.Node(
                matcher=CliMatcher.KeywordMatcher(
                   '0100.5e90.0001',
                   helpdesc='Forward all frames destined to 0100.5e90.0001' ),
              alias='MAC_ADDR',
              guard=ieeeReservedMacForwardMicroBfdSupportedGuard ),
           'MAC_FORWARD_GROUP_NODE': CliCommand.Node(
                matcher=CliMatcher.EnumMatcher( {
                           '0180.c200.0003,0180.c200.000e': 'Forward all frames '
                                'destined to 0180.c200.0003, and 0180.c200.000e',
                           '0180.c200.0004-0180.c200.000d,0180.c200.000f':
                                'Forward all frames destined to 0180.c200.0004 '
                                'to 0180.c200.000d, and 0180.c200.000f',
                } ),
                alias='MAC_ADDR',
                guard=ieeeReservedMacForwardGroupSupportedGuard ),
         }

class ReservedForwardCmd( CliCommand.CliCommandClass ):
   syntax = 'MAC_ADDR_TABLE reserved forward MAC_ADDR'
   noOrDefaultSyntax = syntax
   data = {
      'MAC_ADDR_TABLE': MacAddrTableExprForConfig,
      'reserved': CliCommand.guardedKeyword( 'reserved',
                     helpdesc='Reserved',
                     guard=ieeeReservedMacForwardSupportedGuard ),
      'forward': 'Forward',
      'MAC_ADDR': macAddressExpression,
   }

   handler = "BridgingCliHandler.setIeeeReservedMacCommon"
   noOrDefaultHandler = "BridgingCliHandler.noIeeeReservedMacCommon"

BasicCliModes.GlobalConfigMode.addCommandClass( ReservedForwardCmd )

#-------------------------------------------------------------------------------
# The "[no|default] mac pause-frame pass-through" command, 
# in "config" mode.
#
# The full syntax of this command is:
#
#   mac pause-frame pass-through
#   no mac pause-frame pass-through
#   default mac pause-frame pass-through
#-------------------------------------------------------------------------------


# we only enable the command if the platform supports it. 

matcherPauseFrame = CliMatcher.KeywordMatcher( 'pause-frame',
                    helpdesc='Pause frame visibility' )

def supportedPausePassThroughGuard( mode, token ):
   if gv.bridgingHwCapabilities.pausePassThroughSupported:
      return None
   return CliParser.guardNotThisPlatform

class MacPauseFramePassThroughCmd( CliCommand.CliCommandClass ):
   syntax = 'mac pause-frame pass-through'
   noOrDefaultSyntax = syntax
   data = {
      'mac': CliToken.Mac.macMatcherForConfig,
      'pause-frame': CliCommand.Node( matcher=matcherPauseFrame,
                     guard=supportedPausePassThroughGuard ),
      'pass-through': ( 'Set pause frame to be visible '
                        'across ethernet MAC chip pipeline' )
   }
   handler = "BridgingCliHandler.setPauseFramePassThrough"
   noOrDefaultHandler = "BridgingCliHandler.noPauseFramePassThrough"

BasicCliModes.GlobalConfigMode.addCommandClass( MacPauseFramePassThroughCmd )

#-------------------------------------------------------------------------------
# The "show mac pause-frame" command
#
# The full syntax of this command is:
#
#   show mac pause-frame
#-------------------------------------------------------------------------------

class MacPauseFrameCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show mac pause-frame'
   data = {
      'mac': CliToken.Mac.macMatcherForShow,
      'pause-frame': matcherPauseFrame
   }
   handler = "BridgingCliHandler.doShowPauseFramePassThrough"
   cliModel = "BridgingCliModel.PauseFramePassThrough"

BasicCli.addShowCommandClass( MacPauseFrameCmd )

#-------------------------------------------------------------------------------
# [no | default] mac address-table flushing interface vlan aggregation disabled
#-------------------------------------------------------------------------------

class DisableVlanMacFlushAgg( CliCommand.CliCommandClass ):
   syntax = 'mac address-table flushing interface vlan aggregation disabled'
   noOrDefaultSyntax = syntax
   data = {
         'mac': CliToken.Mac.macMatcherForConfig,
         'address-table': matcherAddressTable,
         'flushing': 'Flush MAC entries',
         'interface': 'Interface where the address is learned',
         'vlan': matcherVlan,
         'aggregation': 'Flushing all interface VLANs to reduce overall '
                        'flushing overhead',
         'disabled': 'Disabled',
   }
   handler = "BridgingCliHandler.disableVlanMacFlushAgg"
   noOrDefaultHandler = "BridgingCliHandler.noDisableVlanMacFlushAgg"

BasicCliModes.GlobalConfigMode.addCommandClass( DisableVlanMacFlushAgg )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   mount = LazyMount.mount
   gv.bridgingConfig = mount( entityManager, "bridging/config",
                              "Bridging::Config", "r" )
   gv.bridgingFdbConfig = mount( entityManager, "bridging/fdbConfigDir",
                                 "Bridging::FdbConfigDir", "r" )
   gv.bridgingHwCapabilities = mount( entityManager, "bridging/hwcapabilities",
                                      "Bridging::HwCapabilities", "r" )
   gv.bridgingInCliConfig = ConfigMount.mount( entityManager,
                                     "bridging/input/config/cli",
                                     "Bridging::Input::CliConfig", "w" )
   gv.bridgingStatus = SmashLazyMount.mount( entityManager, "bridging/status",
                                             "Smash::Bridging::Status",
                                             SmashLazyMount.mountInfo( 'reader' ) )
   gv.bridgingSwitchIntfConfig = mount( entityManager, "bridging/switchIntfConfig",
                                        "Bridging::SwitchIntfConfigDir", "r" )
   gv.dynAllowedVlanDir = mount( entityManager, "bridging/input/dynvlan/allowedvlan",
                                 "Tac::Dir", "ri" )
   gv.dynBlockedVlanDir = mount( entityManager, "bridging/input/dynBlockedVlan",
                                 "Tac::Dir", "ri" )
   gv.intfTpidStatus = mount( entityManager, "bridging/tpid/status",
                              "Bridging::IntfTpidStatus", "r" )
   gv.vlanTagFormatDropStatusDir = mount( entityManager,
                                          "bridging/vlanTagFormatDrop/status",
                                          "Tac::Dir", "ri" )

