# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os

import Arnet.Bpf
import BasicCli
import CliCommand
import CliExtensions
import CliMatcher
import CliModel
import CliParser
import CliToken.Clear
import CliToken.Platform
import ConfigMount
import LazyMount
import ShowCommand
import Tac
import AgentCommandRequest
import AgentTerminate
import json
from io import StringIO

from BasicCliModes import ConfigModeBase, GlobalConfigMode
from CliMode.EtbaMode import PlatformTfaBaseMode
from CliModel import cliPrinted
from CliPlugin import BridgingCli
from CliPlugin import IraNexthopGroupCli
from CliPlugin import TechSupportCli
from CliPlugin import IntfCli
from CliPlugin.EtbaModel import HandlerCounter, HandlerCounters, ArfaCounterGroups
from CliPlugin.EtbaModel import PamCollection
from CliPlugin.EtbaModel import AllGenericPipelines
from CliPlugin.EtbaModel import DemuxEndpoints
from CliPlugin.EtbaModel import LookupsTable
from CliPlugin.EtbaModel import AgingTables
from CliPlugin.EtbaModel import PktCounterTables
from CliPlugin.EtbaModel import AllArfaPlugins
from CliPlugin.EtbaModel import ArfaRateCounterCollection
from CliPlugin.EthIntfCli import EthPhyAutoIntfType
from TypeFuture import TacLazyType
from Intf import IntfRange

arfaTracingConfig = None
etbaCliConfig = None
ebraCounterConfig = None
ebraCounterStatus = None
fwdIntfCounterConfig = None
fwdIntfCounterStatus = None
etbaStatus = None
routingHwStatus = None

ArfaModeType = TacLazyType( "Arfa::ArfaMode" )
EthIntfId = TacLazyType( 'Arnet::EthIntfId' )

def nexthopGroupCounter( mode ):
   return routingHwStatus.nexthopGroupCounterSupported and \
      ( etbaStatus.arfaMode == ArfaModeType.arfaOnlyMode )

# Add a CliHook extension for supporting NHG counters in Arfa mode on Etba platform
IraNexthopGroupCli.showNexthopGroupCounterHook.addExtension( nexthopGroupCounter )

# Token for etba configuration commands
def etbaGuard( mode, token ):
   if 'NSPATH' in os.environ:
      return None
   return CliParser.guardNotThisPlatform

def tfaGuard( mode, token ):
   if etbaStatus.arfaMode == ArfaModeType.arfaOnlyMode:
      return None
   return CliParser.guardNotThisPlatform

def notTfaGuard( mode, token ):
   if etbaStatus.arfaMode != ArfaModeType.arfaOnlyMode:
      return None
   return CliParser.guardNotThisPlatform

clearMatcher = CliToken.Clear.clearKwNode
countersMatcher = CliCommand.guardedKeyword(
   'counters', helpdesc='Routing handler counters',
   guard=notTfaGuard )
etbaMatcher = CliMatcher.KeywordMatcher(
   'etba', helpdesc='Ebra Test Bridge configuration commands' )
etbaNode = CliCommand.Node( matcher=etbaMatcher, guard=etbaGuard )
phyMatcher = CliMatcher.KeywordMatcher(
   'phy', helpdesc='Commands for counterparts of physical things' )
platformMatcherForClear = CliToken.Platform.platformMatcherForClear
platformMatcherForShow = CliToken.Platform.platformMatcherForShow
platformMatcherForConfig = CliToken.Platform.platformMatcherForConfig
tfaNode = CliCommand.guardedKeyword( 'tfa', helpdesc='Test Forwarding Agent',
                                     guard=tfaGuard )
tfaNodeUnguarded = CliMatcher.KeywordMatcher(
   "tfa", helpdesc='Test Forwarding Agent' )
tfaCountersNode = CliMatcher.KeywordMatcher(
   'counters', helpdesc='Counters for TFA' )
tfaKernelNode = CliMatcher.KeywordMatcher(
   'kernel', helpdesc='Kernel counters for TFA' )
controlFrameMatcher = CliMatcher.KeywordMatcher(
   'control-frame', helpdesc='PHY Control Frame' )

tfaConfigModeCleanupHook = CliExtensions.CliHook()

class PlatformTfaConfigMode( PlatformTfaBaseMode, ConfigModeBase ):
   name = 'platform tfa configuration'

   def __init__( self, parent, session ):
      PlatformTfaBaseMode.__init__( self )
      ConfigModeBase.__init__( self, parent, session )

class PlatformTfaConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'platform tfa'
   noOrDefaultSyntax = syntax
   data = {
      'platform': platformMatcherForConfig,
      'tfa': tfaNodeUnguarded,
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( PlatformTfaConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      childMode = mode.childMode( PlatformTfaConfigMode )
      childMode.removeComment()
      tfaConfigModeCleanupHook.notifyExtensions( mode )
      # any new commands migrated under 'platform tfa' needs cleanup here.
      # poMemberIntfHwOffload is one such cmd added from SftArfaCli.
      # if we move that CLI here, then it results in EtbaDut importing tokens from
      # FlowTracker etc
      etbaCliConfig.poMemberIntfHwOffload = False
      noQtraceFilter( mode, None )
      noVerboseTracingBitmap( mode, None )
      etbaCliConfig.arfaMode = ArfaModeType.unknownMode

GlobalConfigMode.addCommandClass( PlatformTfaConfigCmd )

#-------------------------------------------------------------
# show platform etba counters [detail]
#-------------------------------------------------------------
detailMatcher = CliMatcher.KeywordMatcher(
   'detail', helpdesc="Show counters for each routing handler" )

def updateCounters( mode, config, status ):
   config.counterUpdateRequestTime = Tac.now()
   def countersUpdated():
      return ( status.counterUpdateTime >=
               config.counterUpdateRequestTime )
   try:
      Tac.waitFor( countersUpdated, description='Counters update',
                   maxDelay=1, sleep=True, timeout=5 )
   except Tac.Timeout:
      if config == ebraCounterConfig:
         mode.addWarning( 'Displaying stale counters for '
                          'EbraTestBridge:routingHandlers' )
      else:
         mode.addWarning( 'Displaying stale counters for '
                          'FwdIntfDevice:routingHandlers' )

def showEtbaCounter( mode, args ):
   updateCounters( mode, ebraCounterConfig, ebraCounterStatus )
   updateCounters( mode, fwdIntfCounterConfig, fwdIntfCounterStatus )

   ebraCounters = None
   fwdIntfCounters = None
   if args.get( 'detail' ):
      ebraCounters = {}
      fwdIntfCounters = {}
      for handler, counter in ebraCounterStatus.routingHandlerCounter.items():
         if handler != "overall":
            ebraCounters[ handler ] = (
               HandlerCounter( ignored=counter.ignored, routed=counter.routed ) )

      for handler, counter in fwdIntfCounterStatus.routingHandlerCounter.items():
         fwdIntfCounters[ handler ] = (
            HandlerCounter( ignored=counter.ignored, routed=counter.routed ) )

   # Can't do this with ebraIgnored/ebraRouted because even when a packet
   # is successully processed by an EbraTestBridge routing handler,
   # every other EbraTestBridge routing handler will still try to process the packet.
   # When a FwdIntfDevice routing handler successfully processes a packet,
   # other FwdIntfDevice routing handlers won't be called.
   # Also, Arfa mode does not support these counters currently. Arfa has been hooked
   # into the normal interface counters.
   fwdIntfIgnored = 0
   fwdIntfRouted = 0
   if fwdIntfCounterStatus.routingHandlerCounter:
      fwdIntfIgnored = min( fwdIntfCounterStatus.routingHandlerCounter.values(),
                            key=( lambda counter: counter.ignored ) )
      fwdIntfIgnored = fwdIntfIgnored.ignored
      fwdIntfRouted = sum( counter.routed for counter in
                           fwdIntfCounterStatus.routingHandlerCounter.values() )

   overallCounters = ebraCounterStatus.routingHandlerCounter.get( "overall" )
   overallIgnored = 0
   overallRouted = 0
   if overallCounters:
      overallIgnored = overallCounters.ignored
      overallRouted = overallCounters.routed

   return HandlerCounters(
      ebraCounters=ebraCounters,
      ebraIgnored=overallIgnored,
      ebraRouted=overallRouted,
      fwdIntfCounters=fwdIntfCounters,
      fwdIntfIgnored=fwdIntfIgnored,
      fwdIntfRouted=fwdIntfRouted )

class ShowPlatformEtbaCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform etba counters [ detail ]'
   data = {
            'platform': platformMatcherForShow,
            'etba': etbaNode,
            'counters': countersMatcher,
            'detail': detailMatcher,
          }
   handler = showEtbaCounter
   cliModel = HandlerCounters

BasicCli.addShowCommandClass( ShowPlatformEtbaCountersCmd )

#-------------------------------------------------------------
# clear platform etba counters
#-------------------------------------------------------------
def clearCounters( mode, args ):
   ebraCounterConfig.resetCountersTrigger = (
      ( ebraCounterConfig.resetCountersTrigger + 1 ) % 256 )
   fwdIntfCounterConfig.resetCountersTrigger = (
      ( fwdIntfCounterConfig.resetCountersTrigger + 1 ) % 256 )

class clearCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear platform etba counters'
   data = {
            'clear': clearMatcher,
            'platform': platformMatcherForClear,
            'etba': etbaNode,
            'counters': countersMatcher,
          }
   handler = clearCounters

BasicCli.EnableMode.addCommandClass( clearCountersCmd )

# -------------------------------------------------------------
# clear platform tfa phy fault
# -------------------------------------------------------------
def clearPhyRemoteFault( mode, args ):
   # Get interfaces given in cli
   intfsRcvd = IntfCli.Intf.getAll( mode, args.get( 'INTF' ) )
   for intf in intfsRcvd:
      prev = etbaCliConfig.clearLinkStatusTrigger.get( intf.name, 0 )
      etbaCliConfig.clearLinkStatusTrigger[ intf.name ] = ( prev + 1 ) % 256

class clearPhyRemoteFaultCmd( CliCommand.CliCommandClass ):
   syntax = 'clear platform tfa phy fault [ interface INTF ]'
   data = {
            'clear': clearMatcher,
            'platform': platformMatcherForClear,
            'tfa': tfaNode,
            'phy': phyMatcher,
            'fault': 'Clear mac rx remote fault',
            'interface': 'Clear mac rx remote fault for specific interface(s)',
            'INTF': IntfRange.IntfRangeMatcher(
                              explicitIntfTypes=[ EthPhyAutoIntfType ] ),
          }
   handler = clearPhyRemoteFault

BasicCli.EnableMode.addCommandClass( clearPhyRemoteFaultCmd )

#-----------------------------------------------------------------------------------
# "platform tfa qtrace packet filter FILTER_NAME ( ( EXPRESSION [ interface INTF ] )
#                                                            | ( interface INTF ) )'"
# command, in config mode
#-----------------------------------------------------------------------------------
def setQtraceFilter( mode, args ):
   filterName = args[ 'FILTER_NAME' ]
   filterExpression = args.get( 'EXPRESSION' )
   intf = args.get( 'INTF' )

   validExpression = ( filterExpression is not None and
                       Arnet.Bpf.isValidPcapFilterExpression( filterExpression ) )

   if intf is not None or validExpression:
      qtraceFilter = etbaCliConfig.qtraceFilter.newMember( filterName )
      qtraceFilter.bpfFilter = filterExpression if validExpression else ''
      qtraceFilter.intf = \
         Tac.newInstance( "Arnet::IntfId", intf.name if intf else '' )
   else:
      mode.addError( f"Syntax error in expression '{filterExpression}'" )

def noQtraceFilter( mode, args ):
   if args:
      filterName = args[ 'FILTER_NAME' ]
      del etbaCliConfig.qtraceFilter[ filterName ]
   else:
      etbaCliConfig.qtraceFilter.clear()

class EtbaQtraceFilterBase( CliCommand.CliCommandClass ):
   syntax = 'qtrace packet filter FILTER_NAME ( ( EXPRESSION [ interface INTF ] ) '\
            '| ( interface INTF ) )'
   noOrDefaultSyntax = 'qtrace packet filter FILTER_NAME ...'

   data = {
            'qtrace': 'Quicktrace logging',
            'packet': 'Packet information',
            'filter': 'PCAP filter',
            'FILTER_NAME': CliMatcher.PatternMatcher(
                              pattern=r'[a-zA-Z0-9_-]+',
                              helpname='WORD', helpdesc='Filter name' ),
            'EXPRESSION': CliMatcher.StringMatcher(
                              pattern='^((?!interface).)*$',
                              helpdesc='PCAP filter expression',
                              helpname='WORD' ),
            'interface': 'Interface',
            'INTF': IntfCli.Intf.matcher,
   }

   handler = setQtraceFilter
   noOrDefaultHandler = noQtraceFilter

class EtbaQtraceFilterCmd( EtbaQtraceFilterBase ):
   data = EtbaQtraceFilterBase.data.copy()

PlatformTfaConfigMode.addCommandClass( EtbaQtraceFilterCmd )

class EtbaQtraceFilterDeprecatedCmd( EtbaQtraceFilterBase ):
   syntax = 'platform tfa ' + EtbaQtraceFilterBase.syntax
   noOrDefaultSyntax = 'platform tfa ' + EtbaQtraceFilterBase.noOrDefaultSyntax

   data = EtbaQtraceFilterBase.data.copy()
   data.update( {
            'platform': platformMatcherForConfig,
            'tfa': tfaNode,
   } )

   hidden = True

BasicCli.GlobalConfigMode.addCommandClass( EtbaQtraceFilterDeprecatedCmd )

def clearArfaCounters( mode, args ):
   cmd = "CLEAR_COUNTERS"
   return actionArfaAcrJson( mode, cmd )

class clearArfaCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'clear platform tfa counters debug'
   data = {
      'clear': clearMatcher,
      'platform': platformMatcherForClear,
      'tfa': tfaNodeUnguarded,
      'counters': tfaCountersNode,
      'debug': 'Debug counters for TFA',
   }

   handler = clearArfaCounters

BasicCli.addShowCommandClass( clearArfaCountersCmd )

def doClearCounters( mode, intfs, sessionOnly, allIntfs ):
   request = etbaCliConfig.clearCountersRequest

   # We don't support per-session clear
   if sessionOnly:
      return

   # Clear all
   if allIntfs:
      request[ 'all' ] = Tac.now()
      return

   # Clear interface list
   for intf in intfs:
      request[ intf.name ] = Tac.now()

IntfCli.registerClearCountersHook( doClearCounters )

def showArfaShowCounters( mode, args ):
   cmd = "SHOW_COUNTERS"
   return showArfaAcrJson( mode, args, ArfaCounterGroups, cmd )

class ShowArfaShowCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform tfa counters debug'
   data = {
      'platform': platformMatcherForShow,
      'tfa': tfaNodeUnguarded,
      'counters': tfaCountersNode,
      'debug': 'Debug counters for TFA',
   }

   handler = showArfaShowCounters
   cliModel = ArfaCounterGroups

def showPersonality( mode, args ):
   displayMap = {
      ArfaModeType.arfaOnlyMode: "arfa",
      ArfaModeType.hybridMode: "python",
      ArfaModeType.unknownMode: "unknown",
   }

   print( displayMap[ etbaStatus.arfaMode ] )

class ShowArfaPersonalityCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform tfa personality'
   data = {
      'platform': platformMatcherForShow,
      'tfa': tfaNodeUnguarded,
      'personality': 'arfa or python',
   }

   handler = showPersonality
   hidden = True

def setPersonality( mode, args ):
   personality = args[ "VALUE" ]

   arfaMode = ( ArfaModeType.arfaOnlyMode if personality == "arfa" else
                ArfaModeType.hybridMode )

   etbaCliConfig.arfaMode = arfaMode

   # Only terminate the Etba agent if needed
   if arfaMode == etbaStatus.arfaMode:
      # pylint: disable-next=consider-using-f-string
      print( "Already in %s mode" % personality )
      return

   if os.getenv( "SIMULATION_ARFA_CLISAVE" ):
      # Shortcut for breadth tests, so that the etba agent is not restarted
      etbaStatus.arfaMode = arfaMode

def noPersonality( mode, args ):
   if etbaCliConfig.arfaMode == ArfaModeType.unknownMode:
      # already cleared
      return

   etbaCliConfig.arfaMode = ArfaModeType.unknownMode
   sysname = mode.entityManager.sysname()
   AgentTerminate.terminate( sysname, [ "Etba" ] )

class SetArfaPersonalityBase( CliCommand.CliCommandClass ):
   syntax = 'personality VALUE'
   noOrDefaultSyntax = 'personality ...'
   data = {
      'personality': 'arfa or python',
      'VALUE': CliMatcher.EnumMatcher( { "arfa": "arfa",
                                         "python": "python" } ),
   }

   handler = setPersonality
   noOrDefaultHandler = noPersonality
   hidden = True

class SetArfaPersonalityCmd( SetArfaPersonalityBase ):
   data = SetArfaPersonalityBase.data.copy()

class SetArfaPersonalityCmdDeprecated( SetArfaPersonalityBase ):
   syntax = "platform tfa " + SetArfaPersonalityBase.syntax
   noOrDefaultSyntax = "platform tfa " + SetArfaPersonalityBase.noOrDefaultSyntax

   data = SetArfaPersonalityBase.data.copy()
   data.update( {
            'platform': platformMatcherForConfig,
            'tfa': tfaNodeUnguarded,
   } )

PlatformTfaConfigMode.addCommandClass( SetArfaPersonalityCmd )
BasicCli.GlobalConfigMode.addCommandClass( SetArfaPersonalityCmdDeprecated )

BasicCli.addShowCommandClass( ShowArfaShowCountersCmd )
BasicCli.addShowCommandClass( ShowArfaPersonalityCmd )

def setVerboseTracingBitmap( mode, args ):
   arfaTracingConfig.verboseTracingBitmap = args[ 'LEVEL' ]

def noVerboseTracingBitmap( mode, args ):
   arfaTracingConfig.verboseTracingBitmap = 0

class SetArfaVerboseTracingBitmapBase( CliCommand.CliCommandClass ):
   syntax = 'tracing verbose LEVEL'
   noOrDefaultSyntax = 'tracing verbose ...'
   data = {
      'tracing': 'Tracing related configuration',
      'verbose': 'Verbose tracing',
      'LEVEL': CliMatcher.IntegerMatcher( 1, 0xFFFFFFFF,
                                          helpdesc="verbose levels" )
   }
   handler = setVerboseTracingBitmap
   noOrDefaultHandler = noVerboseTracingBitmap
   hidden = True

class SetArfaVerboseTracingBitmapCmd( SetArfaVerboseTracingBitmapBase ):
   data = SetArfaVerboseTracingBitmapBase.data.copy()

class SetArfaVerboseTracingBitmapDeprecated( SetArfaVerboseTracingBitmapBase ):
   syntax = 'platform tfa ' + SetArfaVerboseTracingBitmapBase.syntax
   noOrDefaultSyntax = ( 'platform tfa ' +
                         SetArfaVerboseTracingBitmapBase.noOrDefaultSyntax )

   data = SetArfaVerboseTracingBitmapBase.data.copy()
   data.update( {
            'platform': platformMatcherForConfig,
            'tfa': tfaNodeUnguarded,
   } )

PlatformTfaConfigMode.addCommandClass( SetArfaVerboseTracingBitmapCmd )
BasicCli.GlobalConfigMode.addCommandClass( SetArfaVerboseTracingBitmapDeprecated )

def actionArfaAcrJson( mode, cmd ):
   AgentCommandRequest.runCliPrintSocketCommand(
      mode.entityManager, "Etba", "debug", cmd, mode )

def showArfaAcr( mode, args, model, cmd ):
   AgentCommandRequest.runCliPrintSocketCommand(
      mode.entityManager, "Etba", "debug", cmd, mode )
   # Just return the output from the agent directly to the
   # user. Use cliPrinted() to let the infrastructure know
   # that we've done this.
   return cliPrinted( model )

def showArfaAcrJson( mode, args, model, cmd ):
   output = StringIO()
   AgentCommandRequest.runCliPrintSocketCommand(
      mode.entityManager, "Etba", "debug", cmd, mode, stringBuff=output,
      forceOutputFormat="json" )
   jsonTxt = output.getvalue()
   output.close()
   j = json.loads( jsonTxt )

   errors = j.pop( "errors", None )
   if errors:
      for error in errors:
         mode.addError( error )
      return model()

   ret = CliModel.unmarshalModel( model, j )

   return ret

def showArfaShowKernelIntfCounters( mode, args ):
   cmd = "SHOW_KERNEL_INTF_COUNTERS"
   return showArfaAcr( mode, args, PamCollection, cmd )

class ShowArfaShowKernelIntfCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform tfa counters kernel interfaces'
   data = {
      'platform': platformMatcherForShow,
      'tfa': tfaNode,
      'counters': tfaCountersNode,
      'kernel': tfaKernelNode,
      'interfaces': 'Kernel interfaces\' counters for TFA',
   }

   handler = showArfaShowKernelIntfCounters
   cliModel = PamCollection

BasicCli.addShowCommandClass( ShowArfaShowKernelIntfCountersCmd )

def showArfaShowPipelines( mode, args ):
   cmd = "SHOW_PIPELINES"
   return showArfaAcrJson( mode, args, AllGenericPipelines, cmd )

class ShowArfaShowPipelinesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform tfa pipelines'
   data = {
      'platform': platformMatcherForShow,
      'tfa': tfaNode,
      'pipelines': 'Pipelines for TFA',
   }

   hidden = True
   handler = showArfaShowPipelines
   cliModel = AllGenericPipelines

BasicCli.addShowCommandClass( ShowArfaShowPipelinesCmd )

def showArfaDemuxCounters( mode, args ):
   cmd = "SHOW_DEMUX_COUNTERS"
   return showArfaAcrJson( mode, args, DemuxEndpoints, cmd )

class ShowArfaShowDemuxCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform tfa counters demux'
   data = {
      'platform': platformMatcherForShow,
      'tfa': tfaNodeUnguarded,
      'counters': tfaCountersNode,
      'demux': 'Demux features for TFA',
   }

   hidden = True
   handler = showArfaDemuxCounters
   cliModel = DemuxEndpoints

BasicCli.addShowCommandClass( ShowArfaShowDemuxCountersCmd )

def showArfaLookups( mode, args ):
   cmd = "SHOW_LOOKUPS"
   return showArfaAcr( mode, args, LookupsTable, cmd )

class ShowArfaShowLookupsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform tfa lookups'
   data = {
      'platform': platformMatcherForShow,
      'tfa': tfaNodeUnguarded,
      'lookups': 'Lookups for TFA',
   }

   handler = showArfaLookups
   cliModel = LookupsTable

BasicCli.addShowCommandClass( ShowArfaShowLookupsCmd )

def showArfaMacAging( mode, args ):
   cmd = "SHOW_AGING"
   return showArfaAcrJson( mode, args, AgingTables, cmd )

class ShowArfaMacAgingCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform tfa mac address-table aging'
   data = {
      'platform': platformMatcherForShow,
      'tfa': tfaNode,
      'mac': CliToken.Mac.macMatcherForConfig,
      'address-table': BridgingCli.matcherAddressTable,
      'aging': 'Aging'
   }

   hidden = True
   handler = showArfaMacAging
   cliModel = AgingTables

BasicCli.addShowCommandClass( ShowArfaMacAgingCountersCmd )

def showArfaPktCounters( mode, args ):
   cmd = "SHOW_PKT_COUNTERS"
   return showArfaAcrJson( mode, args, PktCounterTables, cmd )

class ShowArfaPktCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform tfa counters shadow'
   data = {
      'platform': platformMatcherForShow,
      'tfa': tfaNodeUnguarded,
      'counters': tfaCountersNode,
      'shadow': 'agent-internal shadow of packet counters'
   }

   hidden = True
   handler = showArfaPktCounters
   cliModel = PktCounterTables

BasicCli.addShowCommandClass( ShowArfaPktCountersCmd )

def showArfaRateCounters( mode, args ):
   cmd = "SHOW_PKT_RATES"
   return showArfaAcr( mode, args, ArfaRateCounterCollection, cmd )

class ShowArfaRateCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform tfa counters rate'
   data = {
      'platform': platformMatcherForShow,
      'tfa': tfaNodeUnguarded,
      'counters': tfaCountersNode,
      'rate': 'Rates for counters'
   }

   hidden = True
   handler = showArfaRateCounters
   cliModel = ArfaRateCounterCollection

BasicCli.addShowCommandClass( ShowArfaRateCountersCmd )

def showArfaShowPlugins( mode, args ):
   cmd = 'SHOW_PLUGINS'
   return showArfaAcrJson( mode, args, AllArfaPlugins, cmd )

class ShowArfaShowPluginsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform tfa plugins'
   data = {
      'platform': platformMatcherForShow,
      'tfa': tfaNode,
      'plugins': 'Plugins for TFA',
   }

   hidden = True
   handler = showArfaShowPlugins
   cliModel = AllArfaPlugins

BasicCli.addShowCommandClass( ShowArfaShowPluginsCmd )

def noConfigurePhyControlActivity( mode, args ):
   del etbaCliConfig.phyControlDisabled[ mode.intf.name ]

def configurePhyControlActivity( mode, args ):
   etbaCliConfig.phyControlDisabled.add( mode.intf.name )

class ConfigurePhyControlActivityCmd( CliCommand.CliCommandClass ):
   syntax = 'platform tfa phy control-frame disabled'
   noOrDefaultSyntax = syntax
   data = {
      'platform': platformMatcherForConfig,
      'tfa': tfaNode,
      'phy': phyMatcher,
      'control-frame': controlFrameMatcher,
      'disabled': 'Disable phy-control frame',
   }
   handler = configurePhyControlActivity
   noOrDefaultHandler = noConfigurePhyControlActivity

class PhyControlIgnoreModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      # Phy-control is only relevant for EthIntf.
      return EthIntfId.isEthIntfId( mode.intf.name )

IntfCli.IntfConfigMode.addModelet( PhyControlIgnoreModelet )
PhyControlIgnoreModelet.addCommandClass( ConfigurePhyControlActivityCmd )

class PhyControlIgnoreIntfCleaner( IntfCli.IntfDependentBase ):
   # If interface is cleared with "default interface",
   # clear phy control-frame configuration.
   def setDefault( self ):
      del etbaCliConfig.phyControlDisabled[ self.intf_.name ]

# Add all of these commands to "show tech-support extended tfa"
TechSupportCli.registerShowTechSupportCmd(
      '2021-09-30 14:14:14', cmds=[
         "show platform tfa counters debug",
         "show platform tfa personality",
         "show platform tfa pipelines",
         "show platform tfa counters demux",
         "show platform tfa mac address-table aging",
         "show platform tfa counters shadow",
         "show platform tfa counters kernel interfaces",
         ],
      extended="tfa" )

def Plugin( entityManager ):
   global arfaTracingConfig
   global etbaCliConfig
   global ebraCounterConfig
   global ebraCounterStatus
   global fwdIntfCounterConfig
   global fwdIntfCounterStatus
   global etbaStatus
   global routingHwStatus

   arfaTracingConfig = ConfigMount.mount(
      entityManager, "bridging/etba/cli/tracingConfig",
      "Arfa::TracingConfig", "w" )
   etbaCliConfig = ConfigMount.mount(
      entityManager, "bridging/etba/cli/config",
      "Bridging::Etba::CliConfig", "w" )
   ebraCounterConfig = ConfigMount.mount(
      entityManager, "bridging/etba/counter/ebra/config",
      "Bridging::Etba::RoutingHandlerCounterConfig", "w" )
   ebraCounterStatus = LazyMount.mount(
      entityManager, "bridging/etba/counter/ebra/status",
      "Bridging::Etba::RoutingHandlerCounterStatus", "r" )
   fwdIntfCounterConfig = ConfigMount.mount(
      entityManager, "bridging/etba/counter/fwdIntf/config",
      "Bridging::Etba::RoutingHandlerCounterConfig", "w" )
   fwdIntfCounterStatus = LazyMount.mount(
      entityManager, "bridging/etba/counter/fwdIntf/status",
      "Bridging::Etba::RoutingHandlerCounterStatus", "r" )
   etbaStatus = LazyMount.mount(
      entityManager, "bridging/etba/status", "Bridging::Etba::Status", "r" )
   routingHwStatus = LazyMount.mount(
      entityManager, "routing/hardware/status",
      "Routing::Hardware::Status", "r" )

   IntfCli.Intf.registerDependentClass( PhyControlIgnoreIntfCleaner )
