#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
from Ark import switchTimeToUtc, utcTimeRelativeToNowStr
import CliModel
import IntfModels
import TableOutput
import Tac
import time

# Below dicts map tac types defined in PhyEee/BaseTCable.tac
# to str values in CAPI model.

# Maps "Hardware::BaseT::CableDiagnosticsRunState"
cableDiagnosticsRunStateToModel = {
   "cableDiagnosticsRunStateUntested" : "untested",
   "cableDiagnosticsRunStateRunning" : "running",
   "cableDiagnosticsRunStateCompleted" : "completed",
   "cableDiagnosticsRunStateFailed" : "failed",
   }

# Maps "Hardware::BaseT::CableState" to CAPI values
cableStateToModel = {
   "cableStateInconclusive" : "inconclusive",
   "cableStateOk" : "ok",
   "cableStateFaulty" : "faulty",
   "cableStateNotconnect" : "notconnect",
   }

# Maps "Hardware::BaseT::CablePairState" to CAPI values
cablePairStateToModelState = {
   "pairStateUndetermined" : "inconclusive",
   "pairStateOk" : "ok",
   "pairStateOpen" : "open",
   "pairStateIntraPairShort" : "intraPairShort",
   "pairStateInterPairShort" : "interPairShort",
   "pairStateBusy" : "busy",
   }

# This dict maps CAPI values for pairState to strings used for rendering.
cablePairStateStr = {
   "inconclusive" : "inconclusive",
   "ok" : "ok",
   "open" : "open",
   "intraPairShort" : "intrapair short",
   "interPairShort" : "interpair short",
   "busy" : "busy",
   }

class TestingStatus( CliModel.Model ):
   numTestRuns = CliModel.Int( help="Number of cable test runs" )
   testState = CliModel.Enum( values=list(
      cableDiagnosticsRunStateToModel.values() ),
                              help="Current testing state" )
   stateChanges = CliModel.Int( help="Number of test state changes" )
   lastStateChange = CliModel.Float( help="Last time test state changed" )

   def toModel( self, diagStatus ):
      self.numTestRuns = diagStatus.cableTestRuns
      self.testState = cableDiagnosticsRunStateToModel[ diagStatus.diagnosticsState ]
      self.stateChanges = diagStatus.changes
      self.lastStateChange = switchTimeToUtc( diagStatus.lastChange )
      return self

   def renderOnlyTestStatus( self ):
      return self.testState in [ 'untested', 'running' ]

   def renderLengthAccuracy( self ):
      return self.testState in [ 'completed', 'failed' ]

class CableStatus( CliModel.Model ):
   cableState = CliModel.Enum( values=list( cableStateToModel.values() ),
                               help="Overall cable state information" )
   stateChanges = CliModel.Int( help="Number of cable state changes" )
   lastStateChange = CliModel.Float( help="Last time cable state changed" )


   def toModel( self, cableStatus ):
      self.cableState = cableStateToModel[ cableStatus.cableState ]
      self.stateChanges = cableStatus.stateChanges
      self.lastStateChange = switchTimeToUtc( cableStatus.stateLastChange )
      return self

   def mayNeedAnotherTestRun( self ):
      return self.cableState == 'inconclusive'

class CablePairStatus( CliModel.Model ):
   pairState = CliModel.Enum( values=list( cablePairStateToModelState.values() ),
                              help="State of cable pair" )
   stateChanges = CliModel.Int( help="Number of cable pair state changes" )
   lastStateChange = CliModel.Float( help="Last time cable pair state changed" )
   pairLength = CliModel.Int( help="Estimated length of the cable pair in meters" )
   lengthChanges = CliModel.Int( help="Number of cable pair length changes" )
   lastLengthChange = CliModel.Float(
      help="Last time cable pair length changed" )

   def toModel( self, pairStatus ):
      self.pairState = cablePairStateToModelState[ pairStatus.pairState ]
      self.stateChanges = pairStatus.pairStateChanges
      self.lastStateChange = switchTimeToUtc( pairStatus.pairStateLastChange )
      self.pairLength = pairStatus.pairLength
      self.lengthChanges = pairStatus.pairLengthChanges
      self.lastLengthChange = switchTimeToUtc( pairStatus.pairLengthLastChange )
      return self

class BaseTCableTestStatus( CliModel.Model ):
   testStatus = CliModel.Submodel( valueType=TestingStatus,
                                      help="Status of cable test run" )
   cableStatus = CliModel.Submodel( valueType=CableStatus,
                                    help="Overall cable health status" )
   pairAStatus = CliModel.Submodel( valueType=CablePairStatus,
                                    help="Status of cable pair A" )
   pairBStatus = CliModel.Submodel( valueType=CablePairStatus,
                                    help="Status of cable pair B" )
   pairCStatus = CliModel.Submodel( valueType=CablePairStatus,
                                    help="Status of cable pair C" )
   pairDStatus = CliModel.Submodel( valueType=CablePairStatus,
                                    help="Status of cable pair D" )
   lengthUncertainty = CliModel.Int(
      help="Actual length is within this value of the reported length in meters" )

   def toModel( self, cableTestStatus ):
      self.testStatus = TestingStatus().toModel(
         cableTestStatus.diagnosticsStatus )
      self.cableStatus = CableStatus().toModel( cableTestStatus.cableStatus )
      self.pairAStatus = CablePairStatus().toModel(
         cableTestStatus.pairAStatus )
      self.pairBStatus = CablePairStatus().toModel(
         cableTestStatus.pairBStatus )
      self.pairCStatus = CablePairStatus().toModel(
         cableTestStatus.pairCStatus )
      self.pairDStatus = CablePairStatus().toModel(
         cableTestStatus.pairDStatus )
      self.lengthUncertainty = cableTestStatus.lengthAccuracy
      return self

   def render( self ):
      table = TableOutput.TableFormatter()
      alignL = TableOutput.Format( justify='left' )
      alignL.noPadLeftIs( True )
      alignL.padLimitIs( True )
      alignR = TableOutput.Format( justify='right' )
      alignR.padLimitIs( True )

      testStatus = self.testStatus
      # pylint: disable-next=consider-using-f-string
      print( 'Cable test runs: %d' % testStatus.numTestRuns )
      if testStatus.renderLengthAccuracy():
         # pylint: disable-next=consider-using-f-string
         print( 'Cable length accuracy: +/-%dm' % self.lengthUncertainty )

      headings = ( "", "Current State", "Changes", "Last Change" )
      table = TableOutput.createTable( headings )
      table.formatColumns( alignL, alignR, alignR, alignR )
      table.newRow( 'Diagnostics status',
                    testStatus.testState,
                    testStatus.stateChanges,
                    utcTimeRelativeToNowStr( testStatus.lastStateChange ) )
      if testStatus.renderOnlyTestStatus():
         print( table.output() )
         # Skip rest of the output if cable test is still running/untested
         return

      cableStatus = self.cableStatus
      table.newRow( 'Cable status',
                    cableStatus.cableState,
                    cableStatus.stateChanges,
                    utcTimeRelativeToNowStr( cableStatus.lastStateChange ) )

      table.newRow( 'Status of pair A',
                    cablePairStateStr[ self.pairAStatus.pairState ],
                    self.pairAStatus.stateChanges,
                    utcTimeRelativeToNowStr( self.pairAStatus.lastStateChange ) )
      table.newRow( 'Status of pair B',
                    cablePairStateStr[ self.pairBStatus.pairState ],
                    self.pairBStatus.stateChanges,
                    utcTimeRelativeToNowStr( self.pairBStatus.lastStateChange ) )
      table.newRow( 'Status of pair C',
                    cablePairStateStr[ self.pairCStatus.pairState ],
                    self.pairCStatus.stateChanges,
                    utcTimeRelativeToNowStr( self.pairCStatus.lastStateChange ) )
      table.newRow( 'Status of pair D',
                    cablePairStateStr[ self.pairDStatus.pairState ],
                    self.pairDStatus.stateChanges,
                    utcTimeRelativeToNowStr( self.pairDStatus.lastStateChange ) )

      table.newRow( 'Length of pair A',
                    str( self.pairAStatus.pairLength ) + "m",
                    self.pairAStatus.lengthChanges,
                    utcTimeRelativeToNowStr( self.pairAStatus.lastLengthChange ) )

      table.newRow( 'Length of pair B',
                    str( self.pairBStatus.pairLength ) + "m",
                    self.pairBStatus.lengthChanges,
                    utcTimeRelativeToNowStr( self.pairBStatus.lastLengthChange ) )

      table.newRow( 'Length of pair C',
                    str( self.pairCStatus.pairLength ) + "m",
                    self.pairCStatus.lengthChanges,
                    utcTimeRelativeToNowStr( self.pairCStatus.lastLengthChange ) )

      table.newRow( 'Length of pair D',
                    str( self.pairDStatus.pairLength ) + "m",
                    self.pairDStatus.lengthChanges,
                    utcTimeRelativeToNowStr( self.pairDStatus.lastLengthChange ) )
      print( table.output() )
      if cableStatus.mayNeedAnotherTestRun() :
         print( '! Cable test result is inconclusive, another test run may help' )
         print()

class InterfaceBaseTCableTestStatuses( CliModel.Model ):
   interfaces = CliModel.Dict(
      keyType=IntfModels.Interface,
      valueType=BaseTCableTestStatus,
      help="Mapping of base-t interfaces to cable test status " )

   def render( self ):
      print( "Current system time: " + str( time.ctime( Tac.utcNow() ) ) )
      for intf in Arnet.sortIntf( self.interfaces ):
         print( intf )
         self.interfaces[ intf ].render()
