#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
from CliPlugin import TechSupportCli
import ShowCommand
import CbfModel
import SharedMem
import Smash
import LazyMount
from TypeFuture import TacLazyType
from IpLibConsts import DEFAULT_VRF
from CliToken.SegmentRoutingToken import matcherSegmentRoutingForShow
from Toggles.IpLibToggleLib import toggleMplsCbfLdpOverRsvpTeEnabled

class FecOverrideTableMounter:
   def __init__( self ):
      self.entityManager = None
      self.shmemEm = None
      self.fecOverrideTable = None

   def setEntityManager( self, em ):
      self.entityManager = em
      self.shmemEm = SharedMem.entityManager( sysdbEm=self.entityManager )

   def getFecOverrideTable( self ):
      if self.fecOverrideTable is None:
         readerInfo = Smash.mountInfo( 'reader' )
         mountPath = "te/segmentrouting/fecoverride/config"
         self.fecOverrideTable = self.shmemEm.doMount( mountPath,
                                    "Qos::Smash::FecOverrideConfig",
                                    readerInfo )
      return self.fecOverrideTable

class SrTeColorDscpMapMounter:
   def __init__( self ):
      self.entityManager = None
      self._srTeCbfColorDscpMap = None

   def setEntityManager( self, em ):
      self.entityManager = em

   @property
   def srTeCbfColorDscpMap( self ):
      if self._srTeCbfColorDscpMap is None:
         self._srTeCbfColorDscpMap = LazyMount.mount( self.entityManager,
               'te/segmentrouting/cbf/color-dscp/config',
               'TrafficEngineering::CbfConfigMap', 'r' )
      return self._srTeCbfColorDscpMap

class TeColorDscpMapMounter:
   def __init__( self ):
      self.entityManager = None
      self._teCbfColorDscpMap = None

   def setEntityManager( self, em ):
      self.entityManager = em

   @property
   def teCbfColorDscpMap( self ):
      if self._teCbfColorDscpMap is None:
         self._teCbfColorDscpMap = LazyMount.mount( self.entityManager,
               'te/cbf/color-dscp/config',
               'TrafficEngineering::CbfConfigMap', 'r' )
      return self._teCbfColorDscpMap

class TeColorTcMapMounter:
   def __init__( self ):
      self.entityManager = None
      self._teCbfColorTcMap = None

   def setEntityManager( self, em ):
      self.entityManager = em

   @property
   def teCbfColorTcMap( self ):
      if self._teCbfColorTcMap is None:
         self._teCbfColorTcMap = LazyMount.mount( self.entityManager,
               'te/cbf/color-tc/config',
               'TrafficEngineering::CbfConfigMap', 'r' )
      return self._teCbfColorTcMap

class LfibVskOverrideTableMounter:
   def __init__( self ):
      self.entityManager = None
      self.shmemEm = None
      self.lfibVskOverrideConfigTable = None
      self.lfibVskOverrideStatusTable = None

   def setEntityManager( self, em ):
      self.entityManager = em
      self.shmemEm = SharedMem.entityManager( sysdbEm=self.entityManager )

   def getLfibVskOverrideConfigTable( self ):
      if self.lfibVskOverrideConfigTable is None:
         readerInfo = Smash.mountInfo( 'reader' )
         mountPath = "te/cbf/vskoverride/mpls/config"
         self.lfibVskOverrideConfigTable = self.shmemEm.doMount( mountPath,
                                             "Mpls::Override::LfibVskOverrideConfig",
                                             readerInfo )
      return self.lfibVskOverrideConfigTable

   def getLfibVskOverrideStatusTable( self ):
      if self.lfibVskOverrideStatusTable is None:
         readerInfo = Smash.mountInfo( 'reader' )
         mountPath = "te/cbf/vskoverride/mpls/status"
         self.lfibVskOverrideStatusTable = self.shmemEm.doMount( mountPath,
                                             "Mpls::Override::LfibVskOverrideStatus",
                                             readerInfo )
      return self.lfibVskOverrideStatusTable

class LfibAndTunnelNameStatusMounter:
   def __init__( self ):
      self.entityManager = None
      self.shmemEm = None
      self.transitLfib = None
      self.overrideLfib = None
      self.protocolTunnelNameStatus = None
      self.forwardingGenStatus = None

   def setEntityManager( self, em ):
      self.entityManager = em
      self.shmemEm = SharedMem.entityManager( sysdbEm=self.entityManager )

   def getTransitLfib( self ):
      if self.transitLfib is None:
         readerInfo = Smash.mountInfo( 'reader' )
         mountPath = 'mpls/transitLfib'
         self.transitLfib = self.shmemEm.doMount( mountPath, "Mpls::LfibStatus",
               readerInfo )
      return self.transitLfib

   def getOverrideLfib( self ):
      if self.overrideLfib is None:
         readerInfo = Smash.mountInfo( 'reader' )
         mountPath = 'mpls/cbf/overrideLfib'
         self.overrideLfib = self.shmemEm.doMount( mountPath,
               "Mpls::LfibViaSetStatus",
               readerInfo )
      return self.overrideLfib

   def getProtocolTunnelNameStatus( self ):
      if self.protocolTunnelNameStatus is None:
         mountPath = 'tunnel/tunnelNameStatus'
         self.protocolTunnelNameStatus = LazyMount.mount( self.entityManager,
               mountPath, 'Tunnel::TunnelTable::ProtocolTunnelNameStatus',
               'r' )
      return self.protocolTunnelNameStatus

   def getForwardingGenStatus( self ):
      if self.forwardingGenStatus is None:
         readerInfo = Smash.mountInfo( 'reader' )
         mountPath = 'forwardingGen/unifiedStatus'
         self.forwardingGenStatus = self.shmemEm.doMount(
            mountPath, 'Smash::FibGen::ForwardingStatus', readerInfo )
      return self.forwardingGenStatus

entMan = None
fecOverrideTableMounter = FecOverrideTableMounter()
srTeColorDscpMapMounter = SrTeColorDscpMapMounter()
teColorDscpMapMounter = TeColorDscpMapMounter()
teColorTcMapMounter = TeColorTcMapMounter()
lfibVskOverrideTableMounter = LfibVskOverrideTableMounter()
lfibAndTunnelNameStatusMounter = LfibAndTunnelNameStatusMounter()

# -x-x-x-x-x-x-x-x-x-x- Implement show commands below -x-x-x-x-x-x-x-x-x-x-x-x-x-

TunnelIdType = TacLazyType( "Tunnel::TunnelTable::TunnelId" )
TunnelType = TacLazyType( "Tunnel::TunnelTable::TunnelType" )
LfibSource = TacLazyType( "Mpls::LFib::Source" )
DyTunIntfId = TacLazyType( "Arnet::DynamicTunnelIntfId" )
LfibViaType = TacLazyType( "Mpls::LfibViaType" )

# -------------------------------------------------------------------------------
# Implements show traffic-engineering cbf fec [ FECID | SOURCE ]
# -------------------------------------------------------------------------------

def matchOverrideTunnelType( sourceType, overrideTunnelType ):
   if sourceType == 'rsvp-ler':
      return TunnelType.rsvpLerTunnel == overrideTunnelType
   elif sourceType == 'sr-te':
      return TunnelType.srTePolicyTunnel == overrideTunnelType
   else:
      return False

def showTeCbfFec( mode, args ):
   if srTeColorDscpMapMounter.srTeCbfColorDscpMap.colorToValue:
      mode.addWarning( "Color/DSCP and Color/TC mappings"
                       " in the traffic-engineering"
                       " mode will not take effect until the mappings in the"
                       " te-segment-routing mode are removed" )
   elif teColorDscpMapMounter.teCbfColorDscpMap.colorToValue and \
            teColorTcMapMounter.teCbfColorTcMap.colorToValue:
      mode.addWarning( "Color/TC mappings in the traffic-engineering"
                       " mode will not take effect until Color/DSCP mappings in"
                       " traffic-engineering mode and te-segment-routing mode are"
                       " removed" )

   vrfModel = CbfModel.CbfFecVrfModel()
   cbfFecModel = CbfModel.CbfFecModel()
   fecId = args.get( 'FEC_ID' )
   sourceType = args.get( 'TUNNEL_TYPE' )
   fecOverrideTable = fecOverrideTableMounter.getFecOverrideTable()
   for overrideKey in fecOverrideTable.override:
      if fecId and fecId != overrideKey.fecId:
         continue
      overrideEntry = fecOverrideTable.override.get( overrideKey )
      if not overrideEntry:
         continue
      if sourceType and not matchOverrideTunnelType( sourceType,
                                                 overrideEntry.tunnelType ):
         continue
      fecOverrideEntry = CbfModel.CbfFecOverride()
      fecOverrideEntry.defaultFecId = overrideEntry.key.fecId
      fecOverrideEntry.overrideFecId = overrideEntry.fecId
      fecOverrideEntry.ipVersion = overrideEntry.key.af()
      if overrideEntry.key.tc != CbfModel.invalidTc:
         fecOverrideEntry.tc = overrideEntry.key.tc
      elif overrideEntry.key.dscp != CbfModel.invalidDscp:
         fecOverrideEntry.dscp = overrideEntry.key.dscp
      fecOverrideEntry.tunnelType = TunnelIdType.tunnelTypeToCliStrep(
                                          overrideEntry.tunnelType, True )
      cbfFecModel.overrides.append( fecOverrideEntry )

   vrfModel.vrfs[ DEFAULT_VRF ] = cbfFecModel
   return vrfModel

# -------------------------------------------------------------------------------
# Implements show traffic-engineering cbf mpls [ index INDEX_ID ]
# -------------------------------------------------------------------------------
def getTunnelNames( viaSet, lfib, protocolTunnelNameStatus, forwardingGenStatus ):
   tunnelNames = []
   if not viaSet:
      return tunnelNames
   for viaKey in viaSet.viaKey.values():
      intfs = []
      if viaKey.viaType == LfibViaType.viaTypeExternalFec:
         fecVia = lfib.extFecVia.get( viaKey )
         if not fecVia:
            continue
         fec = forwardingGenStatus.fec.get( fecVia.fecId )
         if not fec:
            continue
         for via in fec.via.values():
            intfs.append( via.intfId )
      else:
         via = lfib.mplsVia.get( viaKey )
         if not via:
            continue
         intfs.append( via.intf )

      for intf in intfs:
         if DyTunIntfId.isDynamicTunnelIntfId( intf ):
            tid = TunnelIdType( DyTunIntfId.tunnelId( intf ) )
         else:
            continue
         tunnelName = ""
         if protocolTunnelNameStatus:
            tunnelNameStatus = protocolTunnelNameStatus.localTunnelNameStatus.get(
               tid.tunnelType() )
            if tunnelNameStatus:
               tunnelName = tunnelNameStatus.tunnelIdToName.get( tid )
         if tunnelName:
            tunnelNames.append( tunnelName )
   return tunnelNames

def showTeCbfMpls( mode, args ):
   vrfModel = CbfModel.CbfMplsVrfModel()
   cbfMplsModel = CbfModel.CbfMplsModel()
   indexId = args.get( 'INDEX_ID' )
   lfibVskOverrideConfigTable = \
         lfibVskOverrideTableMounter.getLfibVskOverrideConfigTable()
   lfibVskOverrideStatusTable = \
         lfibVskOverrideTableMounter.getLfibVskOverrideStatusTable()
   transitLfib = lfibAndTunnelNameStatusMounter.getTransitLfib()
   overrideLfib = lfibAndTunnelNameStatusMounter.getOverrideLfib()
   protocolTunnelNameStatus = \
         lfibAndTunnelNameStatusMounter.getProtocolTunnelNameStatus()
   forwardingGenStatus = \
      lfibAndTunnelNameStatusMounter.getForwardingGenStatus()

   for overrideKey in lfibVskOverrideConfigTable.override:
      if indexId and indexId != overrideKey.lfibVsk.index:
         continue
      overrideEntry = lfibVskOverrideConfigTable.override.get( overrideKey )
      if not overrideEntry:
         continue

      overrideStatus = lfibVskOverrideStatusTable.override.get( overrideKey )
      lfibVskOverrideEntry = CbfModel.CbfMplsOverride()

      defaultLfibVsk = overrideEntry.key.lfibVsk
      if defaultLfibVsk.source == LfibSource.lfibSourceLdp:
         lfibVskOverrideEntry.defaultVskSource = 'Ldp'
      elif defaultLfibVsk.source in ( LfibSource.lfibSourceIsisSrV4Prefix,
                                      LfibSource.lfibSourceIsisSrV6Prefix ):
         lfibVskOverrideEntry.defaultVskSource = 'ISIS-SR'
      else:
         continue

      viaSet = transitLfib.viaSet.get( defaultLfibVsk )
      lfibVskOverrideEntry.defaultVias = getTunnelNames( viaSet, transitLfib,
                                                         protocolTunnelNameStatus,
                                                         forwardingGenStatus )
      lfibVskOverrideEntry.defaultVskIndex = defaultLfibVsk.index

      overrideLfibVsk = overrideEntry.lfibVsk
      assert overrideLfibVsk.source is LfibSource.lfibSourceVssCbf
      lfibVskOverrideEntry.overrideVskSource = 'Cbf'
      lfibVskOverrideEntry.overrideVskIndex = overrideLfibVsk.index

      viaSet = overrideLfib.viaSet.get( overrideLfibVsk )
      lfibVskOverrideEntry.overrideVias = getTunnelNames( viaSet, overrideLfib,
                                                          protocolTunnelNameStatus,
                                                          forwardingGenStatus )

      lfibVskOverrideEntry.tc = overrideEntry.key.tc
      lfibVskOverrideEntry.indexId = bool( indexId )
      lfibVskOverrideEntry.status = overrideStatus.status \
            if overrideStatus else CbfModel.vskOverrideEntryStatus.unknownStatus
      cbfMplsModel.overrides.append( lfibVskOverrideEntry )

   vrfModel.vrfs[ DEFAULT_VRF ] = cbfMplsModel
   return vrfModel

# -x-x-x-x-x-x-x-x-x-x- Register show commands below -x-x-x-x-x-x-x-x-x-x-x-x-x-

sourceTypeMatcherForShow = CliMatcher.EnumMatcher( {
   'rsvp-ler': "RSVP-TE CBF Overrides only",
   'sr-te': "SR-TE CBF Overrides only",
} )
matcherTunnelType = CliCommand.singleNode( sourceTypeMatcherForShow )
matcherTrafficEngineering = CliMatcher.KeywordMatcher( 'traffic-engineering',
                              helpdesc='Traffic Engineering related information' )
matcherCbf = CliMatcher.KeywordMatcher( 'cbf', helpdesc='Class based forwarding' )
matcherFecId = CliMatcher.IntegerMatcher( 1, 0xFFFFFFFFFFFFFFFF,
                                          helpdesc='FEC identifier' )
matcherLfibVskIndex = CliMatcher.IntegerMatcher( 1, 0xFFFFFFFFFFFFFFFF,
                                             helpdesc='Lfib VSK Index identifier' )

#--------------------------------------------------------------------------------
# show traffic-engineering segment-routing cbf fec [ FECID ]
#-------------------------------------------------------------------------------

class ShowSrTeCbfFecCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering segment-routing cbf fec [ FECID ]'
   data = {
      'traffic-engineering': matcherTrafficEngineering,
      'segment-routing': matcherSegmentRoutingForShow,
      'cbf': matcherCbf,
      'fec': 'Show Class Based Forwarding override FEC',
      'FECID': matcherFecId
   }

   cliModel = CbfModel.SrTePolicyCbfFecVrfModel

   @staticmethod
   def handler( mode, args ):
      vrfModel = CbfModel.SrTePolicyCbfFecVrfModel()
      srTePolicyCbfFecModel = CbfModel.SrTePolicyCbfFecModel()

      fecId = args.get( 'FECID' )
      fecOverrideTable = fecOverrideTableMounter.getFecOverrideTable()
      for overrideKey in fecOverrideTable.override:
         if fecId and fecId != overrideKey.fecId:
            continue
         overrideEntry = fecOverrideTable.override.get( overrideKey )
         if not overrideEntry:
            continue
         if overrideEntry.tunnelType == TunnelType.rsvpLerTunnel:
            continue
         fecOverrideEntry = CbfModel.SrTePolicyCbfFecOverride()
         fecOverrideEntry.defaultFecId = overrideEntry.key.fecId
         fecOverrideEntry.overrideFecId = overrideEntry.fecId
         fecOverrideEntry.ipVersion = overrideEntry.key.af()
         fecOverrideEntry.dscp = overrideEntry.key.dscp
         srTePolicyCbfFecModel.overrides.append( fecOverrideEntry )

      vrfModel.vrfs[ DEFAULT_VRF ] = srTePolicyCbfFecModel
      return vrfModel

BasicCli.addShowCommandClass( ShowSrTeCbfFecCmd )

# ------------------------------------------------------------------------------
# show traffic-engineering cbf fec [ FECID | SOURCE ]
# ------------------------------------------------------------------------------

class ShowTeCbfFecCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering cbf fec [ FEC_ID | TUNNEL_TYPE ]'
   data = {
      'traffic-engineering': matcherTrafficEngineering,
      'cbf': matcherCbf,
      'fec': 'Show Class Based Forwarding override FEC',
      'FEC_ID': matcherFecId,
      'TUNNEL_TYPE': matcherTunnelType
   }

   cliModel = CbfModel.CbfFecVrfModel
   handler = showTeCbfFec

BasicCli.addShowCommandClass( ShowTeCbfFecCmd )

# ------------------------------------------------------------------------------
# show traffic-engineering cbf mpls [ index INDEX_ID ]
# ------------------------------------------------------------------------------

class ShowTeCbfMplsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering cbf mpls [ index INDEX_ID ]'
   data = {
      'traffic-engineering': matcherTrafficEngineering,
      'cbf': matcherCbf,
      'mpls': 'Show Class Based Forwarding MPLS override table',
      'index': 'CBF MPLS overrides with only given index',
      'INDEX_ID': matcherLfibVskIndex
   }

   cliModel = CbfModel.CbfMplsVrfModel
   handler = showTeCbfMpls

if toggleMplsCbfLdpOverRsvpTeEnabled():
   BasicCli.addShowCommandClass( ShowTeCbfMplsCmd )

#--------------------------------------------------------------------------------
# SrTePolicy commands in 'show tech-support'
#--------------------------------------------------------------------------------
commands = [ 'show traffic-engineering cbf fec' ]
if toggleMplsCbfLdpOverRsvpTeEnabled():
   commands.append( 'show traffic-engineering cbf mpls' )

TechSupportCli.registerShowTechSupportCmd( '2024-01-04 07:21:14', cmds=commands )

def Plugin( entityManager ):
   global entMan

   entMan = entityManager

   fecOverrideTableMounter.setEntityManager( entityManager )
   srTeColorDscpMapMounter.setEntityManager( entityManager )
   teColorDscpMapMounter.setEntityManager( entityManager )
   teColorTcMapMounter.setEntityManager( entityManager )
   lfibVskOverrideTableMounter.setEntityManager( entityManager )
   lfibAndTunnelNameStatusMounter.setEntityManager( entityManager )
