# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliGlobal
import CliMatcher
import CliParser
from CliPlugin.AleCountersCli import (
   counterFeatureSupported,
)
from CliPlugin.IraNexthopGroupCli import (
   nexthopGroupNameMatcher,
)
from CliPlugin.PbrCli import (
   matcherDscp,
   matcherSingleDscpValue,
)
import LazyMount
import ShowCommand
import SmashLazyMount
from Toggles.CbfNhgToggleLib import (
   toggleCbfNhgEcmpEnabled,
   toggleCbfNhgNonEcmpEnabled,
)

gv = CliGlobal.CliGlobal(
   cbfNhgOverrideConfig=None,
   fecOverrideConfig=None,
   fecOverrideStatus=None,
   nhgEntryStatus=None,
   nhgFecIdStatus=None,
   routingHwStatus=None )

IPv4 = 'IPv4'
IPv6 = 'IPv6'

def cbfNhgSupportedGuard( mode, token ):
   if gv.routingHwStatus.nhgCbfSupported:
      return None
   return CliParser.guardNotThisPlatform

def cbfSupportedGuard( mode, token ):
   if gv.routingHwStatus.tcBasedCbfSupported:
      return None
   return CliParser.guardNotThisPlatform

addressFamilyKwMatcher = CliMatcher.KeywordMatcher(
   'af',
   helpdesc='Address family' )

addressFamilyMatcher = CliMatcher.EnumMatcher( {
   'ipv4': 'IPv4 related',
   'ipv6': 'IPv6 related', } )

cbfKwMatcher = CliCommand.guardedKeyword(
   'cbf',
   helpdesc='Class based forwarding',
   guard=cbfSupportedGuard )

cbfNhgKwMatcher = CliCommand.guardedKeyword(
   'nexthop-group',
   helpdesc='Nexthop group class based forwarding',
   guard=cbfNhgSupportedGuard )

countersKwMatcher = CliCommand.guardedKeyword(
   'counters',
   helpdesc='Display counters for overriding FEC TCAM rules',
   guard=counterFeatureSupported( 'NexthopCbf' ) )

# If we import CliPlugin.CbfCliShow.matcherFecId then CbfNhgCliTest will fail during
# "a ws make -p CbfNhgAgent (check|product)" because CliTest.runCohabTestCli cannot
# find /src/Cbf/preinit/Cbf-Config. This is resolved when adding
# "pkgdeps: rpmwith %{_libdir}/preinit/Cbf-Config" to the test file, or alternatively
# avoiding the import altogether.
fecIdMatcher = CliMatcher.IntegerMatcher(
   1, 0xFFFFFFFFFFFFFFFF, helpdesc='FEC identifier' )

fecKwMatcher = CliMatcher.KeywordMatcher( 'fec', helpdesc='Filter by FEC ID' )

nexthopGroupNameKwMatcher = CliMatcher.KeywordMatcher(
   'name',
   helpdesc='Filter by nexthop-group name' )

overrideKwMatcher = CliMatcher.KeywordMatcher(
   'override',
   helpdesc='Nexthop group overrides' )

class DefaultNameExprFactory( CliCommand.CliExpressionFactory ):
   def __init__( self ):
      CliCommand.CliExpressionFactory.__init__( self )

   def generate( self, name ):
      nameKw = f'{ name }_name'
      defaultKw = f'{ name }_default'

      class DefaultNameExpr( CliCommand.CliExpression ):
         expression = f'( { nameKw } { defaultKw } DEFAULT_NAME )'
         data = {
            nameKw: nexthopGroupNameKwMatcher,
            defaultKw: CliMatcher.KeywordMatcher(
               'default', helpdesc='Default nexthop-group name' ),
            'DEFAULT_NAME': nexthopGroupNameMatcher,
         }
      return DefaultNameExpr

class OverrideNameExprFactory( CliCommand.CliExpressionFactory ):
   def __init__( self ):
      CliCommand.CliExpressionFactory.__init__( self )

   def generate( self, name ):
      nameKw = f'{ name }_name'
      overrideKw = f'{ name }_override'

      class OverrideNameExpr( CliCommand.CliExpression ):
         expression = f'( { nameKw } { overrideKw } OVERRIDE_NAME )'
         data = {
            nameKw: nexthopGroupNameKwMatcher,
            overrideKw: CliMatcher.KeywordMatcher(
               'override', helpdesc='Override nexthop-group name' ),
            'OVERRIDE_NAME': nexthopGroupNameMatcher,
         }
      return OverrideNameExpr

class DefaultFecExprFactory( CliCommand.CliExpressionFactory ):
   def __init__( self ):
      CliCommand.CliExpressionFactory.__init__( self )

   def generate( self, name ):
      fecKw = f'{ name }_fec'
      defaultKw = f'{ name }_default'

      class DefaultFecExpr( CliCommand.CliExpression ):
         expression = f'( { fecKw } { defaultKw } DEFAULT_ID )'
         data = {
            fecKw: fecKwMatcher,
            defaultKw: CliMatcher.KeywordMatcher(
               'default', helpdesc='Default FEC ID' ),
            'DEFAULT_ID': fecIdMatcher,
         }
      return DefaultFecExpr

class OverrideFecExprFactory( CliCommand.CliExpressionFactory ):
   def __init__( self ):
      CliCommand.CliExpressionFactory.__init__( self )

   def generate( self, name ):
      fecKw = f'{ name }_fec'
      overrideKw = f'{ name }_override'

      class OverrideFecExpr( CliCommand.CliExpression ):
         expression = f'( { fecKw } { overrideKw } OVERRIDE_ID )'
         data = {
            fecKw: fecKwMatcher,
            overrideKw: CliMatcher.KeywordMatcher(
               'override', helpdesc='Override FEC ID' ),
            'OVERRIDE_ID': fecIdMatcher,
         }
      return OverrideFecExpr

class AddressFamilyFilterExpr( CliCommand.CliExpression ):
   expression = '( af AF )'
   data = {
      'af': addressFamilyKwMatcher,
      'AF': addressFamilyMatcher,
   }

class DscpFilterExpr( CliCommand.CliExpression ):
   expression = '( dscp DSCP )'
   data = {
      'dscp': matcherDscp,
      'DSCP': matcherSingleDscpValue,
   }

class FilterExpr( CliCommand.OrExpressionFactory ):
   def __init__( self ):
      CliCommand.OrExpressionFactory.__init__( self )
      self |= ( 'DEFAULT_NAME_EXPR', DefaultNameExprFactory() )
      self |= ( 'OVERRIDE_NAME_EXPR', OverrideNameExprFactory() )
      self |= ( 'DEFAULT_FEC_EXPR', DefaultFecExprFactory() )
      self |= ( 'OVERRIDE_FEC_EXPR', OverrideFecExprFactory() )
      self |= ( 'AF_EXPR', AddressFamilyFilterExpr )
      self |= ( 'DSCP_EXPR', DscpFilterExpr )

# ---------------------------------------------------------------------------------
# show cbf nexthop-group override [ { filters } ] [ counters ]
# filters = ( name default NAME ) |
#           ( name override NAME ) |
#           ( fec default ID ) |
#           ( fec override ID ) |
#           ( af AF ) |
#           ( dscp DSCP ) |
# ---------------------------------------------------------------------------------
class ShowCbfNhgOverrideCmd( ShowCommand.ShowCliCommandClass ):
   syntax = (
      'show cbf nexthop-group override [ { FILTERS_EXPR } ] [ counters ]'
   )
   data = {
      'cbf': cbfKwMatcher,
      'nexthop-group': cbfNhgKwMatcher,
      'override': overrideKwMatcher,
      'FILTERS_EXPR': FilterExpr(),
      'counters': countersKwMatcher,
   }

   cliModel = 'CbfNhgCliModel.CbfNhgOverrideTable'
   handler = 'CbfNhgCliHandler.showCbfNhgOverrideCmdHandler'

if toggleCbfNhgEcmpEnabled() or toggleCbfNhgNonEcmpEnabled():
   BasicCli.addShowCommandClass( ShowCbfNhgOverrideCmd )

def Plugin( entityManager ):
   # Sysdb mounts.
   gv.nhgFecIdStatus = LazyMount.mount(
      entityManager,
      'te/cbf/nexthopgroup/fec/status',
      'CbfNhgExt::NhgFecIdStatus',
      'r' )
   gv.routingHwStatus = LazyMount.mount(
      entityManager,
      'routing/hardware/status',
      'Routing::Hardware::Status',
      'r' )
   # Smash mounts.
   readerMountInfo = SmashLazyMount.mountInfo( 'reader' )
   gv.cbfNhgOverrideConfig = SmashLazyMount.mount(
      entityManager,
      'te/cbf/nexthopgroup/override/config',
      'CbfNhgExt::CbfNhgOverrideConfig',
      readerMountInfo )
   gv.fecOverrideConfig = SmashLazyMount.mount(
      entityManager,
      'te/nexthopgroup/fecoverride/config',
      'Qos::Smash::FecOverrideConfig',
      readerMountInfo )
   gv.fecOverrideStatus = SmashLazyMount.mount(
      entityManager,
      'te/nexthopgroup/fecoverride/status',
      'Qos::Smash::FecOverrideStatus',
      readerMountInfo )
   gv.nhgEntryStatus = SmashLazyMount.mount(
      entityManager,
      'routing/nexthopgroup/entrystatus',
      'NexthopGroup::EntryStatus',
      readerMountInfo )
