# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import BasicCliModes
import CliCommand
import CliParser
import ConfigMount
import LazyMount

from CliMode.Cdp import CdpBaseMode

config = None
cdpHwStatus = None

def cdpRxSupportedGuard( mode, token ):
   if cdpHwStatus.cdpRxSupported:
      return None
   return CliParser.guardNotThisPlatform

class CdpMode( CdpBaseMode, BasicCli.ConfigModeBase ):
   name = "CDP configuration commands"

   def __init__( self, parent, session ):
      CdpBaseMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#------------------------------------------------------------
# [no|default] cdp 
#------------------------------------------------------------
def gotoCdpMode( mode, args ):
   childMode = mode.childMode( CdpMode )
   mode.session_.gotoChildMode( childMode )

def noCdpMode( mode, args ):
   noCdpReceive( mode, args )
   noCdpAllowTaggedPackets( mode, args )

class CdpCmd( CliCommand.CliCommandClass ):
   syntax = 'cdp'
   noOrDefaultSyntax = syntax
   data = {
      'cdp': CliCommand.guardedKeyword( 'cdp',
                                        helpdesc='CDP configuration mode',
                                        guard=cdpRxSupportedGuard )
   }
   handler = gotoCdpMode
   noOrDefaultHandler = noCdpMode

BasicCliModes.GlobalConfigMode.addCommandClass( CdpCmd )

#------------------------------------------------------------
# in cdp config mode:
# [no|default] receive
#------------------------------------------------------------

def setCdpReceive( mode, args ):
   config.rxEnabled = True

def noCdpReceive( mode, args ):
   config.rxEnabled = False

class CdpReceiveCmd( CliCommand.CliCommandClass ):
   syntax = 'receive'
   noOrDefaultSyntax = syntax
   data = {
      'receive': 'Enable CDP receive',
   }
   handler = setCdpReceive
   noOrDefaultHandler = noCdpReceive

CdpMode.addCommandClass( CdpReceiveCmd )

#------------------------------------------------------------
# hidden command in cdp config mode:
# [no|default] allow tagged-packets
#------------------------------------------------------------

def setCdpAllowTaggedPackets( mode, args ):
   config.allowTaggedPackets = True

def noCdpAllowTaggedPackets( mode, args ):
   config.allowTaggedPackets = False

class CdpAllowTaggedPacketsCmd( CliCommand.CliCommandClass ):
   syntax = 'allow tagged-packets'
   noOrDefaultSyntax = syntax
   hidden = True
   data = {
      'allow': 'Configure what packets are allowed',
      'tagged-packets': 'Allow Dot1q tagged packets',
   }
   handler = setCdpAllowTaggedPackets
   noOrDefaultHandler = noCdpAllowTaggedPackets

CdpMode.addCommandClass( CdpAllowTaggedPacketsCmd )

def Plugin( entityManager ):
   global config
   global cdpHwStatus

   config = ConfigMount.mount( entityManager, "l2discovery/cdp/config",
                               "Cdp::CdpConfig", "w" )
   cdpHwStatus = LazyMount.mount( entityManager, "l2discovery/cdp/hwstatus",
                               "Cdp::CdpHwStatus", "r" )

