#!/usr/bin/env python3
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import CliToken
import BasicCli
import CliCommand
import CliMatcher
from CliPlugin.CfmCliLib import (
   anyLmHwSupported,
   ccmAlarmSupported,
   cfmDomainMatcher,
   cfmSupported,
   defaultMipSupported,
   delayMeasurementKw,
   dmHwSupported,
   downMepSupported,
   EndPointFilter,
   lmHwSupported,
   mepSupported,
   pmPriorityHwSupported,
   pmTxIntervalEnumNeeded,
   pmTxIntervalFullRangeSupported,
   slmHwSupported,
   aisHwSupported,
   upMepSupported,
   cfmCcCountersKw,
)
from CliPlugin.CfmCliModel import (
   CfmIntfCounter,
   CfmCounter,
   CfmMaintenanceDomain,
)
from CliPlugin import EthIntfCli
from CliPlugin import IntfCli
from CliPlugin import LagIntfCli
from CliPlugin import SubIntfCli
from CliPlugin import VirtualIntfRule
from CliPlugin import MacAddr
from CliPlugin import SwitchIntfCli
import ShowCommand
import ConfigMount
import LazyMount
import Ethernet
import Arnet
import TableOutput
import MultiRangeRule
import Intf.IntfRange
import AgentCommandRequest
import CfmAgent
from Ark import timestampToStr
from CfmLib import (
   CcmDefectAlarmCliData,
   ccmTxIntervalToCliToken,
   defaultCcmTxIntervalWithUnit,
   getCcmDefectAlarmEnumFromCli,
   getCcmTxInterval,
   getDefaultCcmDefectAlarm,
)
from CfmTypes import (
   defaultMepDirection,
   defaultPrimaryVlanId,
   tacCfmConst,
   tacCos,
   tacDmMode,
   tacLmMode,
   tacMaNameWithFormat,
   tacMepDirection,
   tacPmMethod,
   tacPmTxInterval,
   tacSlmMode,
   tacMepIntfType,
)
from CliMode.Cfm import ( CfmModeBase, MaintenanceDomainModeBase, CfmProfileModeBase,
                          MaintenanceAssociationModeBase,
                          LocalMaintenanceEndPointModeBase,
                          RemoteMaintenanceEndPointModeBase )

from CfmAgentTypes import tacCounterClass
from TypeFuture import TacLazyType
from Toggles.CfmToggleLib import (
   toggleCfmLocActionEnabled,
   toggleDownMepL2EthIntfEnabled,
   toggleCfmSwUpMepCcmEnabled,
   toggleCfmUpMepL2FppEnabled,
   toggleCfmPmMiEnabled,
)

import Tracing

traceHandle = Tracing.Handle( 'CfmCli' )
t0 = traceHandle.trace0

#-----------------------------------------------------------
# The Cfm globals
#-----------------------------------------------------------
cfmConfig = None
cfmStatus = None
aleCfmConfig = None
aleCfmStatus = None
cfmHwSupportStatus = None
trapConfig = None
cfmConfigReq = None
cfmClearCounter = None

disableMIPMsg = 'Disabling the MIP for MD name %s'

TrapFeatureName = TacLazyType( 'Arnet::TrapFeatureName' )
EthAddr = TacLazyType( 'Arnet::EthAddr' )

def waitForAgent():
   return cfmConfig.mdConfig or cfmStatus.mdStatus or \
      aleCfmConfig.mdConfig or aleCfmStatus.mdStatus

def getCfmProfile( mode ):
   return cfmConfig.cfmProfile.get( mode.cfmProfileName )

def getMdConfig( mode ):
   return cfmConfig.mdConfig.get( mode.domainName )

def localMepExist( domainName ):
   mdConfig = cfmConfig.mdConfig.get( domainName )
   if mdConfig is None:
      return False
   for maNameId in mdConfig.maConfig:
      maConfig = mdConfig.maConfig.get( maNameId )
      if maConfig is None:
         continue
      if maConfig.localMepConfig:
         return True
   return False

def getMaConfig( mode ):
   mdConfig = getMdConfig( mode )
   return mdConfig.maConfig.get( mode.maNameId ) if mdConfig else None

def getLocalMepConfig( mode ):
   maConfig = getMaConfig( mode )
   return maConfig.localMepConfig.get( mode.localMepId ) if maConfig else None

def getRemoteMepConfig( mode ):
   maConfig = getMaConfig( mode )
   return maConfig.remoteMepConfig.get( mode.remoteMepId ) if maConfig else None

#-----------------------------------------------------------
# Modes
#-----------------------------------------------------------
class CfmMode( CfmModeBase, BasicCli.ConfigModeBase ):
   name = 'CFM configuration'

   def __init__( self, parent, session ):
      CfmModeBase.__init__( self, param=None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class MaintenanceDomainMode( MaintenanceDomainModeBase, BasicCli.ConfigModeBase ):
   name = 'CFM maintenance domain configuration'

   def __init__( self, parent, session, param ):
      MaintenanceDomainModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class CfmProfileMode( CfmProfileModeBase, BasicCli.ConfigModeBase ):
   name = 'CFM Profile configuration'

   def __init__( self, parent, session, param ):
      CfmProfileModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class MaintenanceAssociationMode( MaintenanceAssociationModeBase,
                                  BasicCli.ConfigModeBase ):
   name = 'CFM maintenance association configuration'

   def __init__( self, parent, session, param ):
      MaintenanceAssociationModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class LocalMaintenanceEndPointMode( LocalMaintenanceEndPointModeBase,
                                    BasicCli.ConfigModeBase ):
   name = 'CFM local maintenance end point configuration'

   def __init__( self, parent, session, param ):
      LocalMaintenanceEndPointModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RemoteMaintenanceEndPointMode( RemoteMaintenanceEndPointModeBase,
                                    BasicCli.ConfigModeBase ):
   name = 'CFM remote maintenance end point configuration'

   def __init__( self, parent, session, param ):
      RemoteMaintenanceEndPointModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#-----------------------------------------------------------
# (config)# cfm
#-----------------------------------------------------------
class CfmCommand( CliCommand.CliCommandClass ):
   syntax = 'cfm'
   noOrDefaultSyntax = syntax

   data = { 'cfm':
               CliCommand.Node(
                  CliMatcher.KeywordMatcher( 'cfm',
                     helpdesc='Configure connectivity fault management parameters' ),
                  guard=cfmSupported )
          }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( CfmMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfmConfig.cfmLmEnabled = False
      cfmConfig.cfmProfile.clear()
      cfmConfig.cfmSlmEnabled = False
      cfmConfig.locActionRoutingDisabled = False
      cfmConfig.mdConfig.clear()
      trapConfig.features.remove( TrapFeatureName.cfm )

BasicCli.GlobalConfigMode.addCommandClass( CfmCommand )

# ----------------------------------------------------------------
# (config-cfm)# [no|default] measurement loss (inband|synthethic)
# ----------------------------------------------------------------
class MeasurementEnableCmd( CliCommand.CliCommandClass ):
   syntax = 'measurement loss ( inband | synthetic )'
   noOrDefaultSyntax = syntax

   data = {
      'measurement' : CliCommand.guardedKeyword(
         'measurement',
         helpdesc='Ethernet OAM performance monitoring functions',
         guard=anyLmHwSupported,
      ),
      'loss' : 'Ethernet OAM loss measurement functions',
      'inband' : CliCommand.guardedKeyword(
         'inband',
         helpdesc='Enable hardware support for OAM loss measurement',
         guard=lmHwSupported,
      ),
      'synthetic' : CliCommand.guardedKeyword(
         'synthetic',
         helpdesc='Enable hardware support for OAM synthetic loss measurement',
         guard=slmHwSupported,
      ),
   }

   @staticmethod
   def handler( mode, args ):
      if 'inband' in args:
         cfmConfig.cfmLmEnabled = True
      elif 'synthetic' in args:
         cfmConfig.cfmSlmEnabled = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'inband' in args:
         cfmConfig.cfmLmEnabled = False
      elif 'synthetic' in args:
         cfmConfig.cfmSlmEnabled = False

CfmMode.addCommandClass( MeasurementEnableCmd )

# --------------------------------------------------------------------------
# (config-cfm)# [no|default]
#                continuity-check loc-state action disable interface routing
# --------------------------------------------------------------------------
class LocActionCmd( CliCommand.CliCommandClass ):
   syntax = 'continuity-check loc-state action disable interface routing'
   noOrDefaultSyntax = syntax

   data = { 'continuity-check' : 'Configure continuity check global settings',
            'loc-state' : 'Configure settings when LOC is detected',
            'action' : 'Configure action based on LOC state',
            'disable' : 'Toggle the interface status based on LOC state',
            'interface' : 'Apply the action for interfaces',
            'routing' : 'Disable routing for interfaces',
         }

   @staticmethod
   def handler( mode, args ):
      cfmConfig.locActionRoutingDisabled = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfmConfig.locActionRoutingDisabled = False

if toggleCfmLocActionEnabled():
   CfmMode.addCommandClass( LocActionCmd )

mdLevelMatcher = CliMatcher.IntegerMatcher(
   0, 7, helpdesc='Maintenance domain level' )
# ------------------------------------------------------------
# (config-cfm)# [no|default] domain <domain-name> level <0-7>
# ------------------------------------------------------------
class DomainCommand( CliCommand.CliCommandClass ):
   syntax = 'domain NAME level LEVEL'
   noOrDefaultSyntax = 'domain NAME ...'

   data = { 'domain':
               'Configure a maintenance domain',
            'NAME' : cfmDomainMatcher,
            'level' :
               'Configure a maintenance domain level',
            'LEVEL' : mdLevelMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      domainName = args[ 'NAME' ]
      if len( domainName ) > tacCfmConst.mdNameMaxLen:
         mode.addError( 'Domain name is too long (maximum {})'.
                                               format( tacCfmConst.mdNameMaxLen ) )
         return
      mdLevel = args[ 'LEVEL' ]
      mdConfig = cfmConfig.mdConfig.get( domainName )
      if mdConfig:
         currMdLevel = mdConfig.mdLevel
         if mdLevel != currMdLevel:
            mode.addError( 'Domain name {} already exists with domain level {}'.
                           format( domainName, currMdLevel ) )
            return
      mdConfig = cfmConfig.mdConfig.newMember( domainName, mdLevel )
      mdConfig.config = cfmConfig
      trapConfig.features.add( TrapFeatureName.cfm )
      childMode = mode.childMode( MaintenanceDomainMode,
                                  param=( domainName, mdLevel ) )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      domainName = args[ 'NAME' ]
      del cfmConfig.mdConfig[ domainName ]
      if not cfmConfig.mdConfig:
         trapConfig.features.remove( TrapFeatureName.cfm )

CfmMode.addCommandClass( DomainCommand )

#------------------------------------------------------------------------------------
# (config-cfm)# [no|default] profile <cfmProfileName>
#------------------------------------------------------------------------------------
cfmProfileMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: cfmConfig.cfmProfile,
   'Connectivity fault management profile name' )

class ProfileCommand( CliCommand.CliCommandClass ):
   syntax = 'profile PROFILE'
   noOrDefaultSyntax = 'profile PROFILE'

   data = { 'profile':
               CliCommand.Node(
                  CliMatcher.KeywordMatcher( 'profile',
                     helpdesc='Configure connectivity fault management profile' ),
                  guard=mepSupported ),
            'PROFILE' : cfmProfileMatcher
          }

   @staticmethod
   def handler( mode, args ):
      cfmProfileName = args[ 'PROFILE' ]
      cfmConfig.cfmProfile.newMember( cfmProfileName )
      childMode = mode.childMode( CfmProfileMode, param=cfmProfileName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfmProfileName = args[ 'PROFILE' ]
      del cfmConfig.cfmProfile[ cfmProfileName ]

CfmMode.addCommandClass( ProfileCommand )

continuityCheckMatcher = CliMatcher.KeywordMatcher( 'continuity-check',
                          helpdesc='Configure Continuity Check protocol parameters' )
qosMatcher = CliCommand.guardedKeyword( 'qos',
                        helpdesc='Configure QoS parameters',
                        guard=pmPriorityHwSupported )
cosMatcher = CliMatcher.KeywordMatcher( 'cos',
                        helpdesc='Set CoS value for CFM frames' )
cosValueMatcher = CliMatcher.IntegerMatcher( tacCos.min, tacCos.max,
                                             helpdesc='CoS value' )

#------------------------------------------------------------------------------------
# (config-cfm-profile-<cfmProfileName>)# [no|default] continuity-check
#------------------------------------------------------------------------------------
class ContinuityCheckCommand( CliCommand.CliCommandClass ):
   syntax = 'continuity-check'
   noOrDefaultSyntax = 'continuity-check'

   data = { 'continuity-check' : continuityCheckMatcher }

   @staticmethod
   def handler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      assert cfmProfile
      cfmProfile.ccmEnable = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      assert cfmProfile
      cfmProfile.ccmEnable = False

CfmProfileMode.addCommandClass( ContinuityCheckCommand )

#------------------------------------------------------------------------------------
# (config-cfm-profile-<cfmProfileName>)#
# [no|default] continuity-check tx-interval <INTERVAL>
#------------------------------------------------------------------------------------
def intervalSyntax():
   choices = []
   for ccmTxIntervalWithUnit in ccmTxIntervalToCliToken.values():
      if ccmTxIntervalWithUnit.ccmTxInterval == 0:
         continue
      choices.append( '( {} {} )'.format( ccmTxIntervalWithUnit.ccmTxInterval,
                                      ccmTxIntervalWithUnit.unit ) )
   return ' | '.join( choices )

def intervalData():
   data = {}
   for interval in intervals():
      data[ interval ] = 'Set Continuity Check transmission interval to ' + interval
   for unit in units():
      data[ unit ] = 'Continuity Check transmission interval in ' + unit
   return data

def intervals():
   result = set()
   for ccmTxIntervalWithUnit in ccmTxIntervalToCliToken.values():
      if ccmTxIntervalWithUnit.ccmTxInterval == 0:
         continue
      result.add( str( ccmTxIntervalWithUnit.ccmTxInterval ) )
   return result

def units():
   result = set()
   for ccmTxIntervalWithUnit in ccmTxIntervalToCliToken.values():
      if ccmTxIntervalWithUnit.ccmTxInterval == 0:
         continue
      result.add( ccmTxIntervalWithUnit.unit )
   return result

class ContinuityCheckTxIntervalCommand( CliCommand.CliCommandClass ):
   syntax = 'continuity-check tx-interval ( %s )' % intervalSyntax()
   noOrDefaultSyntax = 'continuity-check tx-interval ...'

   data = { 'continuity-check' : continuityCheckMatcher,
            'tx-interval':
               'Set Continuity Check transmission interval'
          }
   data.update( intervalData() )

   @staticmethod
   def handler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      assert cfmProfile
      interval = set( args ) & intervals()
      assert len( interval ) == 1
      interval = interval.pop()
      unit = set( args ) & units()
      assert len( unit ) == 1
      unit = unit.pop()
      ccmConfig = Tac.nonConst( cfmProfile.ccmConfig )
      ccmConfig.ccmTxInterval = getCcmTxInterval( interval, unit )
      cfmProfile.ccmConfig = ccmConfig

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      assert cfmProfile
      ccmConfig = Tac.nonConst( cfmProfile.ccmConfig )
      ccmConfig.ccmTxInterval = getCcmTxInterval(
                                    defaultCcmTxIntervalWithUnit.ccmTxInterval,
                                    defaultCcmTxIntervalWithUnit.unit )
      cfmProfile.ccmConfig = ccmConfig

CfmProfileMode.addCommandClass( ContinuityCheckTxIntervalCommand )

# ---------------------------------------------------------------------------------
# (config-cfm-profile-<cfmProfileName>)#
# [no|default] continuity-check alarm defect DEFECTS
# ---------------------------------------------------------------------------------
continuityCheckAlarmMatcher = CliCommand.guardedKeyword( 'alarm',
                          helpdesc='Defects that can raise alarms',
                          guard=ccmAlarmSupported )

def ccmDefectAlarmData( alarms ):
   data = []
   for defectName in alarms:
      data.append( getCcmDefectAlarmEnumFromCli( defectName ) )
   return data

class ContinuityCheckAlarmCommand( CliCommand.CliCommandClass ):
   syntax = 'continuity-check alarm defect DEFECTS'
   noOrDefaultSyntax = 'continuity-check alarm defect ...'

   data = { 'continuity-check' : continuityCheckMatcher,
            'alarm' : continuityCheckAlarmMatcher,
            'defect' : 'Defects that can raise alarms',
            'DEFECTS' : CliCommand.SetEnumMatcher( CcmDefectAlarmCliData )
         }

   @staticmethod
   def handler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      assert cfmProfile
      alarms = args[ 'DEFECTS' ]
      ccmConfig = Tac.nonConst( cfmProfile.ccmConfig )
      enabledAlarms = ccmDefectAlarmData( alarms )
      ccmConfig.ccmDefectAlarm.clear()
      for alarm in enabledAlarms:
         ccmConfig.ccmDefectAlarm.add( alarm )
      cfmProfile.ccmConfig = ccmConfig

   @staticmethod
   def noHandler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      assert cfmProfile
      ccmConfig = Tac.nonConst( cfmProfile.ccmConfig )
      ccmConfig.ccmDefectAlarm.clear()
      cfmProfile.ccmConfig = ccmConfig

   @staticmethod
   def defaultHandler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      assert cfmProfile
      ccmConfig = Tac.nonConst( cfmProfile.ccmConfig )
      enabledAlarms = getDefaultCcmDefectAlarm()
      ccmConfig.ccmDefectAlarm.clear()
      for alarm in enabledAlarms:
         ccmConfig.ccmDefectAlarm.add( alarm )
      cfmProfile.ccmConfig = ccmConfig

CfmProfileMode.addCommandClass( ContinuityCheckAlarmCommand )

#------------------------------------------------------------------------------------
# (config-cfm-profile-<cfmProfileName>)#
# [no|default] continuity-check qos cos <COS>
#------------------------------------------------------------------------------------

class ContinuityCheckQosCommand( CliCommand.CliCommandClass ):
   syntax = '''continuity-check qos cos COS'''
   noOrDefaultSyntax = '''continuity-check qos cos ...'''

   data = { 'continuity-check' : continuityCheckMatcher,
            'qos' : qosMatcher,
            'cos' : cosMatcher,
            'COS' : cosValueMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      assert cfmProfile
      cos = args.get( 'COS', tacCos.max )
      ccmConfig = Tac.nonConst( cfmProfile.ccmConfig )
      ccmConfig.priority = cos
      cfmProfile.ccmConfig = ccmConfig

   noOrDefaultHandler = handler

CfmProfileMode.addCommandClass( ContinuityCheckQosCommand )

alarmMatcher = CliCommand.guardedKeyword(
   'alarm', helpdesc='Configure alarm indication signal protocol parameters',
   guard=aisHwSupported )
alarmIndicationMatcher = CliMatcher.KeywordMatcher(
   'indication',
   helpdesc='Configure alarm indication signal protocol parameters' )

# ---------------------------------------------------------------------------------
# (config-cfm-profile-<cfmProfileName>)# [no|default] alarm indication
# ---------------------------------------------------------------------------------
class AlarmIndicationCommand( CliCommand.CliCommandClass ):
   syntax = 'alarm indication'
   noOrDefaultSyntax = 'alarm indication'

   data = { 'alarm' : alarmMatcher,
            'indication' : alarmIndicationMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      assert cfmProfile
      cfmProfile.aisEnable = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      assert cfmProfile
      cfmProfile.aisEnable = False

CfmProfileMode.addCommandClass( AlarmIndicationCommand )

aisClientMdLevelHelpDesc = 'Configure client maintenance domain level for '\
   'AIS transmission'
clientMatcher = CliMatcher.KeywordMatcher(
   'client', helpdesc=aisClientMdLevelHelpDesc )
clientDomainMatcher = CliMatcher.KeywordMatcher(
   'domain', helpdesc=aisClientMdLevelHelpDesc )
aisTxIntervalMatcher = CliMatcher.KeywordMatcher(
   'tx-interval', helpdesc='Configure alarm indication transmission interval' )
aisTxIntervalValueMatcher = CliMatcher.IntegerMatcher(
   1, 1, helpdesc='Set alarm indication transmission interval' )
aisTxIntervalUnits = {
   'seconds' : 'Set alarm indication transmission interval in seconds',
   'minutes' : 'Set alarm indication transmission interval in minutes',
}
aisTxIntervalUnitMatcher = CliMatcher.EnumMatcher( aisTxIntervalUnits )
# ----------------------------------------------------------------------------------
# (config-cfm-profile-<cfmProfileName>)#
# [ no|default ] alarm indication client domain level <LEVEL> [tx-interval 1 <UNIT>]
# ----------------------------------------------------------------------------------

class AlarmIndicationClientDomainCommand( CliCommand.CliCommandClass ):
   syntax = 'alarm indication client domain level LEVEL '\
      '[ tx-interval INTERVAL UNIT ]'
   noOrDefaultSyntax = 'alarm indication client ...'

   data = { 'alarm' : alarmMatcher,
            'indication' : alarmIndicationMatcher,
            'client' : clientMatcher,
            'domain' : clientDomainMatcher,
            'level' : aisClientMdLevelHelpDesc,
            'LEVEL' : mdLevelMatcher,
            'tx-interval' : aisTxIntervalMatcher,
            'INTERVAL' : aisTxIntervalValueMatcher,
            'UNIT' : aisTxIntervalUnitMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      assert cfmProfile
      clientDomainLevel = args[ 'LEVEL' ]
      aisConfig = Tac.Value( 'Cfm::AisConfig' )
      txInterval = args.get( 'INTERVAL', aisConfig.txIntervalValue )
      txIntervalUnit = args.get( 'UNIT' )
      if not txIntervalUnit:
         txIntervalUnit = aisConfig.txIntervalUnit
      aisConfig.clientDomainLevel = clientDomainLevel
      aisConfig.clientDomainState = 'configured'
      aisConfig.txIntervalValue = txInterval
      aisConfig.txIntervalUnit = txIntervalUnit
      cfmProfile.aisConfig = aisConfig

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      assert cfmProfile
      cfmProfile.aisConfig = Tac.Value( 'Cfm::AisConfig' )

CfmProfileMode.addCommandClass( AlarmIndicationClientDomainCommand )

measurementMatcher = CliMatcher.KeywordMatcher( 'measurement',
                        helpdesc='Configure peformance monitoring function' )
delayMeasurementMatcher = CliCommand.guardedKeyword( 'delay',
                           helpdesc='Configure delay measurement',
                           guard=dmHwSupported )
lossMeasurementMatcher = CliCommand.guardedKeyword( 'loss',
                           helpdesc='Configure loss measurement',
                           guard=lmHwSupported )
anyLossMeasurementMatcher = CliCommand.guardedKeyword( 'loss',
                           helpdesc='Configure loss measurement',
                           guard=anyLmHwSupported )
syntheticLossMeasurementMatcher = CliCommand.guardedKeyword( 'synthetic',
                                    helpdesc='Configure synthetic loss measurement',
                                    guard=slmHwSupported )
txIntervalMatcher = CliMatcher.KeywordMatcher( 'tx-interval',
                        helpdesc='Configure interval between successive frames' )
endModeOptions = {
   'single-ended' : 'Configure single-ended performance measurement',
}
endModeMatcher = CliCommand.SetEnumMatcher( endModeOptions )
pmTxIntervalUnits = {
   'milliseconds' : 'Performance measurement transmission interval in milliseconds',
}
pmTxIntervalUnitsMatcher = CliCommand.SetEnumMatcher( pmTxIntervalUnits )

def getTxIntervalEnumOptions( mode ):
   options = {}
   if cfmHwSupportStatus:
      for txInterval in cfmHwSupportStatus.pmTxIntervalsSupported:
         # avoid the trailing ".0" for whole numbers
         txInterval = int( txInterval ) if txInterval.is_integer() else txInterval
         intervalStr = str( txInterval )
         options[ intervalStr ] = 'Set measurement frame transmission interval to ' \
                                 + intervalStr + ' milliseconds'
   return options

txIntEnumNode = CliCommand.Node(
                     matcher=CliMatcher.DynamicKeywordMatcher(
                        getTxIntervalEnumOptions ), guard=pmTxIntervalEnumNeeded )
txIntRangeNode = CliCommand.Node(
                     matcher=CliMatcher.FloatMatcher(
                        tacPmTxInterval.minInterval,
                        tacPmTxInterval.maxInterval,
                        helpdesc='Interval between successive measurement frames',
                        precisionString='%.2f' ),
                     guard=pmTxIntervalFullRangeSupported )
cosRangeMatcher = MultiRangeRule.MultiRangeMatcher(
                        rangeFn=lambda: ( tacCos.min, tacCos.max ),
                        noSingletons=False,
                        helpdesc='Set of CoS values' )

def validatePmTxInterval( mode, txInterval ):
   if cfmHwSupportStatus and cfmHwSupportStatus.pmTxIntervalEnumNeeded and \
      txInterval not in cfmHwSupportStatus.pmTxIntervalsSupported:
      supportedTxIntervalStr = ', '.join( str( k )
                  for k in sorted( cfmHwSupportStatus.pmTxIntervalsSupported ) )
      mode.addErrorAndStop( 'Supported tx-interval values on this platform: ' +
                            f'{supportedTxIntervalStr}' )

#------------------------------------------------------------------------------------
# (config-cfm-profile-<cfmProfileName>)# [no|default] measurement delay
#------------------------------------------------------------------------------------
class DelayMeasurementCommand( CliCommand.CliCommandClass ):
   syntax = 'measurement delay ENDMODE'
   noOrDefaultSyntax = '''measurement delay ...'''

   data = { 'measurement' : measurementMatcher,
            'delay' : delayMeasurementMatcher,
            'ENDMODE' : endModeMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      assert cfmProfile
      cfmProfile.dmEnable = True
      dmConfig = Tac.nonConst( cfmProfile.dmConfig )
      dmConfig.endMode = tacDmMode.dmModeSingle
      cfmProfile.dmConfig = dmConfig

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      assert cfmProfile
      cfmProfile.dmEnable = False

CfmProfileMode.addCommandClass( DelayMeasurementCommand )

#------------------------------------------------------------------------------------
# (config-cfm-profile-<cfmProfileName>)#
# [no|default] measurement delay tx-interval <INTERVAL> <UNITS>
#------------------------------------------------------------------------------------

class DelayMeasurementTxIntervalCommand( CliCommand.CliCommandClass ):
   syntax = '''measurement delay tx-interval ( RANGE | DISCRETE ) UNITS'''
   noOrDefaultSyntax = '''measurement delay tx-interval ...'''

   data = { 'measurement' : measurementMatcher,
            'delay' : delayMeasurementMatcher,
            'tx-interval' : txIntervalMatcher,
            'RANGE' : txIntRangeNode,
            'DISCRETE' : txIntEnumNode,
            'UNITS' : pmTxIntervalUnitsMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      assert cfmProfile
      if cfmHwSupportStatus.pmTxIntervalEnumNeeded:
         txInterval = float( args.get( 'DISCRETE' ) )
      else:
         txInterval = args.get( 'RANGE' )
      if not txInterval:
         txInterval = tacPmTxInterval.intervalDefault
      validatePmTxInterval( mode, txInterval )
      dmConfig = Tac.nonConst( cfmProfile.dmConfig )
      dmConfig.txInterval = txInterval
      cfmProfile.dmConfig = dmConfig

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      assert cfmProfile
      dmConfig = Tac.nonConst( cfmProfile.dmConfig )
      dmConfig.txInterval = tacPmTxInterval.intervalDefault
      cfmProfile.dmConfig = dmConfig

CfmProfileMode.addCommandClass( DelayMeasurementTxIntervalCommand )

#------------------------------------------------------------------------------------
# (config-cfm-profile-<cfmProfileName>)#
# [no|default] measurement delay qos cos <COS>
#------------------------------------------------------------------------------------

class DelayMeasurementQosCommand( CliCommand.CliCommandClass ):
   syntax = '''measurement delay qos cos COS'''
   noOrDefaultSyntax = '''measurement delay qos cos ...'''

   data = { 'measurement' : measurementMatcher,
            'delay' : delayMeasurementMatcher,
            'qos' : qosMatcher,
            'cos' : cosMatcher,
            'COS' : cosValueMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      assert cfmProfile
      cos = args.get( 'COS', tacCos.max )
      dmConfig = Tac.nonConst( cfmProfile.dmConfig )
      dmConfig.priority = cos
      cfmProfile.dmConfig = dmConfig

   noOrDefaultHandler = handler

CfmProfileMode.addCommandClass( DelayMeasurementQosCommand )

#------------------------------------------------------------------------------------
# (config-cfm-profile-<cfmProfileName>)# [no|default] measurement loss
#------------------------------------------------------------------------------------
class LossMeasurementCommand( CliCommand.CliCommandClass ):
   syntax = 'measurement loss ENDMODE'
   noOrDefaultSyntax = '''measurement loss ...'''

   data = { 'measurement' : measurementMatcher,
            'loss' : lossMeasurementMatcher,
            'ENDMODE' : endModeMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      assert cfmProfile
      cfmProfile.lmEnable = True
      lmConfig = Tac.nonConst( cfmProfile.lmConfig )
      lmConfig.endMode = tacLmMode.lmModeSingle
      cfmProfile.lmConfig = lmConfig

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      assert cfmProfile
      cfmProfile.lmEnable = False

CfmProfileMode.addCommandClass( LossMeasurementCommand )

#------------------------------------------------------------------------------------
# (config-cfm-profile-<cfmProfileName>)#
# [no|default] measurement loss tx-interval <INTERVAL> <UNITS>
#------------------------------------------------------------------------------------
class LossMeasurementTxIntervalCommand( CliCommand.CliCommandClass ):
   syntax = '''measurement loss tx-interval ( RANGE | DISCRETE ) UNITS'''
   noOrDefaultSyntax = '''measurement loss tx-interval ...'''

   data = { 'measurement' : measurementMatcher,
            'loss' : lossMeasurementMatcher,
            'tx-interval' : txIntervalMatcher,
            'RANGE' : txIntRangeNode,
            'DISCRETE' : txIntEnumNode,
            'UNITS' : pmTxIntervalUnitsMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      assert cfmProfile
      if cfmHwSupportStatus.pmTxIntervalEnumNeeded:
         txInterval = float( args.get( 'DISCRETE' ) )
      else:
         txInterval = args.get( 'RANGE', tacPmTxInterval.intervalDefault )
      if not txInterval:
         txInterval = tacPmTxInterval.intervalDefault
      validatePmTxInterval( mode, txInterval )
      lmConfig = Tac.nonConst( cfmProfile.lmConfig )
      lmConfig.txInterval = txInterval
      cfmProfile.lmConfig = lmConfig

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      assert cfmProfile
      lmConfig = Tac.nonConst( cfmProfile.lmConfig )
      lmConfig.txInterval = tacPmTxInterval.intervalDefault
      cfmProfile.lmConfig = lmConfig

CfmProfileMode.addCommandClass( LossMeasurementTxIntervalCommand )

#------------------------------------------------------------------------------------
# (config-cfm-profile-<cfmProfileName>)#
# [no|default] measurement loss qos cos <COS>
#------------------------------------------------------------------------------------

class LossMeasurementQosCommand( CliCommand.CliCommandClass ):
   syntax = '''measurement loss qos cos COS'''
   noOrDefaultSyntax = '''measurement loss qos cos ...'''

   data = { 'measurement' : measurementMatcher,
            'loss' : lossMeasurementMatcher,
            'qos' : qosMatcher,
            'cos' : cosMatcher,
            'COS' : cosValueMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      cos = args.get( 'COS', tacCos.max )
      lmConfig = Tac.nonConst( cfmProfile.lmConfig )
      lmConfig.priority = cos
      cfmProfile.lmConfig = lmConfig

   noOrDefaultHandler = handler

CfmProfileMode.addCommandClass( LossMeasurementQosCommand )

#------------------------------------------------------------------------------------
# (config-cfm-profile-<cfmProfileName>)# [no|default] measurement loss synthetic
#------------------------------------------------------------------------------------
class SyntheticLossMeasurementCommand( CliCommand.CliCommandClass ):
   syntax = 'measurement loss synthetic ENDMODE'
   noOrDefaultSyntax = '''measurement loss synthetic ...'''

   data = { 'measurement' : measurementMatcher,
            'loss' : anyLossMeasurementMatcher,
            'synthetic' : syntheticLossMeasurementMatcher,
            'ENDMODE' : endModeMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      t0( 'Slm handler:', mode )
      cfmProfile = getCfmProfile( mode )
      cfmProfile.slmEnable = True
      slmConfig = Tac.nonConst( cfmProfile.slmConfig )
      slmConfig.endMode = tacSlmMode.slmModeSingle
      if not slmConfig.priority:
         slmConfig.priority[ tacCos.max ] = True
      cfmProfile.slmConfig = slmConfig

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      cfmProfile.slmEnable = False

CfmProfileMode.addCommandClass( SyntheticLossMeasurementCommand )

slmPeriodMatcher = CliCommand.guardedKeyword( 'period',
                              helpdesc='Configure synthetic loss measurement period',
                              guard=slmHwSupported )
slmPeriodValueMatcher = CliMatcher.IntegerMatcher( 1, 65535,
                        helpdesc='Synthetic loss measurement transmission frames' )
#------------------------------------------------------------------------------------
# (config-cfm-profile-<cfmProfileName>)#
# [no|default] measurement loss synthetic tx-interval <INTERVAL> <UNITS>
#                                                         [ period <PERIOD> frames ]
#------------------------------------------------------------------------------------
class SyntheticLossMeasurementTxIntervalCommand( CliCommand.CliCommandClass ):
   syntax = 'measurement loss synthetic tx-interval ( RANGE | DISCRETE ) UNITS '\
            '[ period PERIOD frames ]'
   noOrDefaultSyntax = 'measurement loss synthetic tx-interval ...'

   data = { 'measurement' : measurementMatcher,
            'loss' : anyLossMeasurementMatcher,
            'synthetic' : syntheticLossMeasurementMatcher,
            'tx-interval' : txIntervalMatcher,
            'RANGE' : txIntRangeNode,
            'DISCRETE' : txIntEnumNode,
            'UNITS' : pmTxIntervalUnitsMatcher,
            'period' : slmPeriodMatcher,
            'PERIOD' : slmPeriodValueMatcher,
            'frames' : 'Synthetic loss measurement frames'
          }

   @staticmethod
   def handler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      if cfmHwSupportStatus.pmTxIntervalEnumNeeded:
         txInterval = float( args.get( 'DISCRETE' ) )
      else:
         txInterval = args.get( 'RANGE', tacPmTxInterval.intervalDefault )
      if not txInterval:
         txInterval = tacPmTxInterval.intervalDefault
      validatePmTxInterval( mode, txInterval )
      slmConfig = Tac.nonConst( cfmProfile.slmConfig )
      slmConfig.txInterval = txInterval
      txFrames = args.get( 'PERIOD', tacCfmConst.defaultSlmTxFrames )
      slmConfig.measurementTxFrames = txFrames
      cfmProfile.slmConfig = slmConfig

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      assert cfmProfile
      slmConfig = Tac.nonConst( cfmProfile.slmConfig )
      slmConfig.txInterval = tacPmTxInterval.intervalDefault
      slmConfig.measurementTxFrames = tacCfmConst.defaultSlmTxFrames
      cfmProfile.slmConfig = slmConfig

CfmProfileMode.addCommandClass( SyntheticLossMeasurementTxIntervalCommand )

#------------------------------------------------------------------------------------
# (config-cfm-profile-<cfmProfileName>)#
# [no|default] measurement loss synthetic qos cos <COS>
#------------------------------------------------------------------------------------

class SyntheticLossMeasurementQosCommand( CliCommand.CliCommandClass ):
   syntax = '''measurement loss synthetic qos cos COS_RANGE'''
   noOrDefaultSyntax = '''measurement loss synthetic qos cos ...'''

   data = { 'measurement' : measurementMatcher,
            'loss' : anyLossMeasurementMatcher,
            'synthetic' : syntheticLossMeasurementMatcher,
            'qos' : qosMatcher,
            'cos' : cosMatcher,
            'COS_RANGE' : cosRangeMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      cosRange = args.get( 'COS_RANGE' )
      cosRange = cosRange.values() if cosRange else [ tacCos.max ]
      slmConfig = Tac.nonConst( cfmProfile.slmConfig )
      # reconcile slmconfig with new cos values
      for cos in cosRange:
         slmConfig.priority[ cos ] = True
      for cos in slmConfig.priority:
         if cos not in cosRange:
            slmConfig.priority.remove( cos )
      cfmProfile.slmConfig = slmConfig

   noOrDefaultHandler = handler

CfmProfileMode.addCommandClass( SyntheticLossMeasurementQosCommand )

measurementIntervalMatcher = CliMatcher.KeywordMatcher( 'interval',
                                helpdesc='Configure measurement interval' )
measurementIntervalValueMatcher = CliMatcher.IntegerMatcher( 1, 60,
   helpdesc='Measurement interval for performance monitoring history' )
# -----------------------------------------------------------------------------------
# (config-cfm-profile-<cfmProfileName>)#] [no|default] measurement delay interval
# <INTERVAL> minutes
# -----------------------------------------------------------------------------------

class DelayMeasurementIntervalCommand( CliCommand.CliCommandClass ):
   syntax = 'measurement delay interval INTERVAL minutes'
   noOrDefaultSyntax = 'measurement delay interval ...'

   data = { 'measurement' : measurementMatcher,
            'delay' : delayMeasurementMatcher,
            'interval' : measurementIntervalMatcher,
            'INTERVAL' : measurementIntervalValueMatcher,
            'minutes' : 'Measurement interval in minutes',
          }

   @staticmethod
   def handler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      assert cfmProfile
      interval = args.get( 'INTERVAL' ) * 60
      dmConfig = Tac.nonConst( cfmProfile.dmConfig )
      miConfig = Tac.nonConst( dmConfig.miConfig )
      miConfig.interval = interval
      dmConfig.miConfig = miConfig
      cfmProfile.dmConfig = dmConfig

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      miConfig = Tac.Value( 'Cfm::MeasurementIntervalConfig' )
      dmConfig = Tac.nonConst( cfmProfile.dmConfig )
      dmConfig.miConfig = miConfig
      cfmProfile.dmConfig = dmConfig

if toggleCfmPmMiEnabled():
   CfmProfileMode.addCommandClass( DelayMeasurementIntervalCommand )

# -----------------------------------------------------------------------------------
# (config-cfm-profile-<cfmProfileName>)#] [no|default] measurement loss synthetic
# interval <INTERVAL> minutes
# -----------------------------------------------------------------------------------

class SyntheticLossMeasurementIntervalCommand( CliCommand.CliCommandClass ):
   syntax = 'measurement loss synthetic interval INTERVAL minutes'
   noOrDefaultSyntax = 'measurement loss synthetic interval ...'

   data = { 'measurement' : measurementMatcher,
            'loss' : anyLossMeasurementMatcher,
            'synthetic' : syntheticLossMeasurementMatcher,
            'interval' : measurementIntervalMatcher,
            'INTERVAL' : measurementIntervalValueMatcher,
            'minutes' : 'Measurement interval in minutes',
          }

   @staticmethod
   def handler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      assert cfmProfile
      interval = args.get( 'INTERVAL' ) * 60
      slmConfig = Tac.nonConst( cfmProfile.slmConfig )
      miConfig = Tac.nonConst( slmConfig.miConfig )
      miConfig.interval = interval
      slmConfig.miConfig = miConfig
      cfmProfile.slmConfig = slmConfig

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cfmProfile = getCfmProfile( mode )
      miConfig = Tac.Value( 'Cfm::MeasurementIntervalConfig' )
      slmConfig = Tac.nonConst( cfmProfile.slmConfig )
      slmConfig.miConfig = miConfig
      cfmProfile.slmConfig = slmConfig

if toggleCfmPmMiEnabled():
   CfmProfileMode.addCommandClass( SyntheticLossMeasurementIntervalCommand )

#----------------------------------------------------------------------
#  (config-cfm-md-X)# [no|default] intermediate-point
#----------------------------------------------------------------------
class IntermediatePointCommand( CliCommand.CliCommandClass ):
   syntax = 'intermediate-point'
   noOrDefaultSyntax = 'intermediate-point'

   data = { 'intermediate-point':
               CliCommand.Node(
                  CliMatcher.KeywordMatcher( 'intermediate-point',
                     helpdesc='Configure the switch as '
                              'Maintenance intermediate point in this domain' ),
                  guard=defaultMipSupported )
          }

   @staticmethod
   def handler( mode, args ):
      mdConfig = getMdConfig( mode )
      assert mdConfig
      if localMepExist( mode.domainName ):
         mode.addWarning( disableMIPMsg % mode.domainName )
      mdConfig.defaultMip = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mdConfig = getMdConfig( mode )
      assert mdConfig
      mdConfig.defaultMip = False

MaintenanceDomainMode.addCommandClass( IntermediatePointCommand )

#------------------------------------------------------------------------------------
# (config-md-<domainName>)# [no|default] association <maName>
#------------------------------------------------------------------------------------
class AssociationCommand( CliCommand.CliCommandClass ):
   syntax = 'association NAME'
   noOrDefaultSyntax = 'association NAME'

   data = { 'association':
               CliCommand.Node(
                  CliMatcher.KeywordMatcher( 'association',
                     helpdesc='Configure maintenance association' ),
                  guard=mepSupported ),
            'NAME':
               CliMatcher.IntegerMatcher( 1, 65535,
                  helpdesc='Maintenance association name' )
          }

   @staticmethod
   def handler( mode, args ):
      maName = str( args[ 'NAME' ] )
      mdConfig = getMdConfig( mode )
      assert mdConfig
      maConfig = mdConfig.maConfig.newMember( maName )
      maConfig.maName = tacMaNameWithFormat( maName, 'maNameFormatShortInt' )
      maConfig.mdConfig = mdConfig
      childMode = mode.childMode( MaintenanceAssociationMode,
                                  param=( mode.domainName, maName ) )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      maName = str( args[ 'NAME' ] )
      mdConfig = getMdConfig( mode )
      assert mdConfig
      del mdConfig.maConfig[ maName ]

MaintenanceDomainMode.addCommandClass( AssociationCommand )

#------------------------------------------------------------------------------------
# (config-md-<domainName>-ma-<maNameId>)# [no|default] direction <up|down>
# Default direction is 'down'
#------------------------------------------------------------------------------------
class DirectionCommand( CliCommand.CliCommandClass ):
   syntax = 'direction ( up | down )'

   noOrDefaultSyntax = 'direction ...'

   data = { 'direction':
               'Set local maintenance end point direction',
            'up':
               CliCommand.Node(
                  CliMatcher.KeywordMatcher( 'up',
                     helpdesc='Set direction as Up' ),
                  guard=upMepSupported ),
            'down':
               CliCommand.Node(
                  CliMatcher.KeywordMatcher( 'down',
                     helpdesc='Set direction as Down' ),
                  guard=downMepSupported ),
          }

   @staticmethod
   def handler( mode, args ):
      maConfig = getMaConfig( mode )
      assert maConfig
      if 'down' in args:
         maConfig.direction = tacMepDirection.MepDirectionDown
      elif 'up' in args:
         maConfig.direction = tacMepDirection.MepDirectionUp
      else:
         mode.addWarning( 'Invalid direction' )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      maConfig = getMaConfig( mode )
      assert maConfig
      maConfig.direction = defaultMepDirection

MaintenanceAssociationMode.addCommandClass( DirectionCommand )

#------------------------------------------------------------------------------------
# (config-md-<domainName>-ma-<maNameId>)#[ no|default ] profile <cfmProfileName>
#------------------------------------------------------------------------------------
class MaProfileCommand( CliCommand.CliCommandClass ):
   syntax = 'profile PROFILE'
   noOrDefaultSyntax = 'profile ...'

   data = { 'profile':
               CliCommand.Node(
                  CliMatcher.KeywordMatcher( 'profile',
                     helpdesc='Apply connectivity fault management profile' ),
                  guard=mepSupported ),
            'PROFILE' : cfmProfileMatcher
          }

   @staticmethod
   def handler( mode, args ):
      cfmProfileName = args[ 'PROFILE' ]
      maConfig = getMaConfig( mode )
      assert maConfig
      cfmProfile = cfmConfig.cfmProfile.get( cfmProfileName )
      if not cfmProfile:
         mode.addWarning( 'Connectivity fault management profile %s doesn\'t exist' %
                          cfmProfileName )
      maConfig.cfmProfileName = cfmProfileName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      maConfig = getMaConfig( mode )
      assert maConfig
      maConfig.cfmProfileName = ''

MaintenanceAssociationMode.addCommandClass( MaProfileCommand )

#------------------------------------------------------------------------------------
# (config-md-<domainName>-ma-<maNameId>)# [ no|default ] vlan <1-4094>
# By default, vlanId will be 0 which means MA is not associated with any vlan
#------------------------------------------------------------------------------------
class VlanCommand( CliCommand.CliCommandClass ):
   syntax = 'vlan VLAN'
   noOrDefaultSyntax = 'vlan ...'

   data = { 'vlan' : CliCommand.Node(
               matcher=CliMatcher.KeywordMatcher(
                  'vlan', helpdesc='Set VLAN in the maintenance association' ),
               hidden=not ( toggleDownMepL2EthIntfEnabled() or
                            toggleCfmUpMepL2FppEnabled() ) ),
            'VLAN':
               CliMatcher.IntegerMatcher( 1, 4094,
                  helpdesc='Set VLAN ID in the given range' )
          }

   @staticmethod
   def handler( mode, args ):
      vlanId = args[ 'VLAN' ]
      maConfig = getMaConfig( mode )
      assert maConfig
      maConfig.primaryVlanId = vlanId

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      maConfig = getMaConfig( mode )
      assert maConfig
      maConfig.primaryVlanId = defaultPrimaryVlanId

MaintenanceAssociationMode.addCommandClass( VlanCommand )

#------------------------------------------------------------------------------------
# (config-md-<domainName>-ma-<maNameId>)# [no|default] remote end-point <1-8191>
#-----------------------------------------------------------------------------------
class RemoteEndpointModeCommand( CliCommand.CliCommandClass ):
   syntax = 'remote end-point ENDPOINT'
   noOrDefaultSyntax = 'remote end-point ENDPOINT'

   data = { 'remote':
               'Configure remote parameters',
            'end-point':
               'Configure remote maintenance end point',
            'ENDPOINT':
               CliMatcher.IntegerMatcher( 1, 8191,
                  helpdesc='Configure remote maintenance end point ID' )
          }

   @staticmethod
   def handler( mode, args ):
      rMepId = args[ 'ENDPOINT' ]
      maConfig = getMaConfig( mode )
      assert maConfig
      if getMdConfig( mode ).defaultMip:
         mode.addWarning( disableMIPMsg % mode.domainName )
      maConfig.remoteMepConfig.newMember( rMepId )
      childMode = mode.childMode( RemoteMaintenanceEndPointMode,
                                  param=( mode.domainName, mode.maNameId, rMepId ) )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      rMepId = args[ 'ENDPOINT' ]
      maConfig = getMaConfig( mode )
      assert maConfig
      del maConfig.remoteMepConfig[ rMepId ]

MaintenanceAssociationMode.addCommandClass( RemoteEndpointModeCommand )

#------------------------------------------------------------------------------------
# (config-md-<domainName>-ma-<maNameId>)# [no|default] end-point <1-8191>
#-----------------------------------------------------------------------------------
class EndpointCommand( CliCommand.CliCommandClass ):
   syntax = 'end-point ENDPOINT'
   noOrDefaultSyntax = 'end-point ENDPOINT'

   data = { 'end-point':
               'Configure local maintenance end point',
            'ENDPOINT':
               CliMatcher.IntegerMatcher( 1, 8191,
                  helpdesc='Set local maintenance end point ID' )
          }

   @staticmethod
   def handler( mode, args ):
      mepId = args[ 'ENDPOINT' ]
      maConfig = getMaConfig( mode )
      assert maConfig
      if getMdConfig( mode ).defaultMip:
         mode.addWarning( disableMIPMsg % mode.domainName )
      localMepConfig = maConfig.localMepConfig.newMember( mepId )
      localMepConfig.maConfig = maConfig
      childMode = mode.childMode( LocalMaintenanceEndPointMode,
                                  param=( mode.domainName, mode.maNameId, mepId ) )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mepId = args[ 'ENDPOINT' ]
      maConfig = getMaConfig( mode )
      assert maConfig
      del maConfig.localMepConfig[ mepId ]

MaintenanceAssociationMode.addCommandClass( EndpointCommand )

#--------------------------------------------------------------------------------
# [ no | default ] interface INTF
#--------------------------------------------------------------------------------
intfMatcher = VirtualIntfRule.IntfMatcher()
intfMatcher |= EthIntfCli.EthPhyIntf.ethMatcher
intfMatcher |= LagIntfCli.EthLagIntf.matcher
intfMatcher |= SubIntfCli.subMatcher
intfMatcher |= LagIntfCli.subMatcher
intfMatcher |= SwitchIntfCli.SwitchIntf.matcher

def mepIntfSupported( mode, mepIntf ):
   t0( 'mepIntfSupported for', mepIntf )
   intfSupported = True
   intf = mepIntf.lower()
   # Originally it was a CliGuard, and guard checks are skipped
   # during startup. So if cfmHwSupportStatus.mepIntfSupported
   # was not populated, we would let the config go through.
   # CfmAgent has the reactors to cfmHwSupportStatus.mepIntfSupported
   # which do the right thing.
   # Since complicated CliGuard was causing errors during parsing,
   # we decided to go with CliError. To gracefully handle
   # the startup case and preserve the original functionality,
   # we need to return true if we are in startup config session
   # because cfmHwSupportStatus.mepIntfSupported may not be populated yet.
   # The reactors to cfmHwSupportStatus.mepIntfSupported in CfmAgent
   # will do the right thing when it gets populated.
   if mode.session_.startupConfig():
      return intfSupported
   ethIntfShortLowerCasePrefixes = ( 'et', 'sw' )
   if '.' in intf:
      if intf.startswith( ethIntfShortLowerCasePrefixes ):
         if not cfmHwSupportStatus.isMepIntfSupported( tacMepIntfType.ethSubIntf ):
            intfSupported = False
      elif intf.startswith( 'po' ):
         if not cfmHwSupportStatus.isMepIntfSupported( tacMepIntfType.lagSubIntf ):
            intfSupported = False
      else:
         intfSupported = False
   elif intf.startswith( ethIntfShortLowerCasePrefixes ):
      if not cfmHwSupportStatus.isMepIntfSupported( tacMepIntfType.ethIntf ):
         intfSupported = False
   elif intf.startswith( 'po' ):
      if not cfmHwSupportStatus.isMepIntfSupported( tacMepIntfType.lagIntf ):
         intfSupported = False
   else:
      intfSupported = False
   return intfSupported

class InterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'interface INTF'
   noOrDefaultSyntax = 'interface ...'
   data = {
      'interface' : 'Configure local maintenance end point on interface',
      'INTF' : CliCommand.Node( intfMatcher ),
   }

   @staticmethod
   def handler( mode, args ):
      localMepConfig = getLocalMepConfig( mode )
      assert localMepConfig
      intf = args.get( 'INTF' ).name
      if not mepIntfSupported( mode, intf ):
         mode.addErrorAndStop(
            'Interface type not supported on this hardware platform' )
      localMepConfig.intfId = Tac.newInstance( 'Arnet::IntfId', intf )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      localMepConfig = getLocalMepConfig( mode )
      assert localMepConfig
      localMepConfig.intfId = Tac.newInstance( 'Arnet::IntfId', '' )

LocalMaintenanceEndPointMode.addCommandClass( InterfaceCmd )

#--------------------------------------------------------------------------------
# [ no | default ] remote end-point [ ACTION ] REMOTE_MEP_IDS
#--------------------------------------------------------------------------------
class RemoteMaintenanceEndPointCmd( CliCommand.CliCommandClass ):
   syntax = 'remote end-point [ ACTION ] REMOTE_MEP_IDS'
   noOrDefaultSyntax = 'remote end-point ...'
   data = {
      'remote' : 'Configure remote maintenance end point',
      'end-point' : 'Configure remote bridge parameters',
      'ACTION' : CliMatcher.EnumMatcher( {
         'add' : 'Add remote end point(s) to the current list',
         'remove' : 'Remove remote end point(s) from the current list',
      } ),
      'REMOTE_MEP_IDS' : MultiRangeRule.MultiRangeMatcher(
         rangeFn=lambda: ( 1, 8191 ),
         noSingletons=False,
         helpdesc=( 'Remote maintenance end point ID(s) or range(s) of '
                    'remote maintenance end point ID(s)' ) )
   }

   @staticmethod
   def handler( mode, args ):
      remoteMepIds = list( args[ 'REMOTE_MEP_IDS' ].values() )
      action = args.get( 'ACTION' )
      localMepConfig = getLocalMepConfig( mode )
      if localMepConfig.localMepId in remoteMepIds:
         mode.addError( 'Remote MEP ID must be different from Local MEP ID' )
         return
      if action == 'add': # Addition of remote MEP IDs
         for remoteMepId in remoteMepIds:
            localMepConfig.remoteMep[ remoteMepId ] = True
      elif action == 'remove': # Deletion of remote MEP IDs
         for remoteMepId in remoteMepIds:
            del localMepConfig.remoteMep[ remoteMepId ]
      else: # Overwrite the existing remote MEP IDs with the new ones
         for remoteMepId in localMepConfig.remoteMep:
            if remoteMepId not in remoteMepIds:
               del localMepConfig.remoteMep[ remoteMepId ]
         for remoteMepId in remoteMepIds:
            localMepConfig.remoteMep[ remoteMepId ] = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      localMepConfig = getLocalMepConfig( mode )
      localMepConfig.remoteMep.clear()

LocalMaintenanceEndPointMode.addCommandClass( RemoteMaintenanceEndPointCmd )

class RemoteMaintenanceEndPointMacCmd( CliCommand.CliCommandClass ):
   syntax = 'mac address MAC_ADDR'
   noOrDefaultSyntax = 'mac address ...'
   data = {
      'mac' : 'Configure mac address',
      'address' : 'Configure mac address',
      'MAC_ADDR' : MacAddr.MacAddrMatcher()
   }

   @staticmethod
   def handler( mode, args ):
      macAddr = args[ 'MAC_ADDR' ]
      remoteMepConfig = getRemoteMepConfig( mode )
      remoteMepConfig.macAddr = macAddr

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      remoteMepConfig = getRemoteMepConfig( mode )
      remoteMepConfig.macAddr = EthAddr.ethAddrZero

RemoteMaintenanceEndPointMode.addCommandClass( RemoteMaintenanceEndPointMacCmd )

# Cfm global tokens
nodeCfm = CliCommand.guardedKeyword( 'cfm',
      helpdesc='Connectivity fault management information', guard=cfmSupported )
nodeCfmMipGuard = CliCommand.guardedKeyword( 'cfm',
      helpdesc='Connectivity fault management information',
      guard=defaultMipSupported )
intfTypes = ( LagIntfCli.LagAutoIntfType, EthIntfCli.EthPhyAutoIntfType )
intfRangeMatcher = Intf.IntfRange.IntfRangeMatcher( explicitIntfTypes=intfTypes )
matcherCfmInterface = CliMatcher.KeywordMatcher( 'interface',
      helpdesc='CFM interface level information' )
matcherCfmCounters = CliMatcher.KeywordMatcher( 'counters',
                                          helpdesc='CFM counters' )

#--------------------------------------------------------------------------------
# show cfm counters [ interface INTFS ]
#--------------------------------------------------------------------------------
def showInterfaceCounters( mode, args ):
   isEthIntfId = Tac.Type( 'Arnet::EthIntfId' ).isEthIntfId
   isPortChannelIntfId = Tac.Type( 'Arnet::PortChannelIntfId' ).isPortChannelIntfId
   model = CfmCounter()

   intfs = args.get( 'INTFS' )
   if not intfs:
      TYPES = ( LagIntfCli.EthLagIntf, EthIntfCli.EthPhyIntf )
      intfs = ( intf.name for intf in IntfCli.Intf.getAll( mode, intfType=TYPES ) )

   for intf in intfs:
      if isEthIntfId( intf ) or isPortChannelIntfId( intf ):
         model.interfaces[ intf ] = getCfmIntfCounter( intf )

   return model

def getCfmIntfCounter( intfId ):
   cfmIntfCounter = CfmIntfCounter()
   counterHelper = Tac.newInstance( "CfmAgent::CounterHelper", cfmStatus.counter )
   counterFilter = Tac.newInstance( "CfmAgent::CounterFilter" )
   counterFilter.intf[ intfId ] = True
   counterFilter.counterClass[ tacCounterClass.counterClassIn ] = True
   counterFilter.counterClass[ tacCounterClass.counterClassInError ] = True
   counterFilter.counterClass[ tacCounterClass.counterClassOut ] = True

   aggregateCounterClass = counterHelper.aggregateCounterClass( counterFilter )
   cfmIntfCounter.inPkts = aggregateCounterClass.counter.get(
      tacCounterClass.counterClassIn, 0 )
   cfmIntfCounter.errorPkts = aggregateCounterClass.counter.get(
      tacCounterClass.counterClassInError, 0 )
   cfmIntfCounter.outPkts = aggregateCounterClass.counter.get(
      tacCounterClass.counterClassOut, 0 )
   return cfmIntfCounter

#--------------------------------------------------------------------------------
# clear cfm measurement ( delay | loss [ synthetic ] )
#  [ domain < Domain ID > [ association < MA name > [ end-point < MEP ID > ] ] ]
#--------------------------------------------------------------------------------
class ClearPmStats( CliCommand.CliCommandClass ):
   syntax = 'clear cfm measurement ( delay | ( loss [ synthetic ] )  )  [ FILTER ]'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'cfm' : nodeCfm,
      'measurement' : measurementMatcher,
      'delay' : delayMeasurementKw,
      'loss' : anyLossMeasurementMatcher,
      'synthetic' : syntheticLossMeasurementMatcher,
      'FILTER' : EndPointFilter,
   }

   @staticmethod
   def handler( mode, args ):
      domainNameFilter = args.get( 'DOMAIN_NAME' )
      maNameFilter = args.get( 'MA_NAME' )
      localMepIdFilter = args.get( 'MEP_ID' )

      clearPmStatsReq = Tac.Value( "Cfm::ClearPmStatsReq" )
      if domainNameFilter:
         clearPmStatsReq.domainName = domainNameFilter
         if maNameFilter:
            clearPmStatsReq.maNameId = maNameFilter
            if localMepIdFilter:
               clearPmStatsReq.localMepId = localMepIdFilter
      if 'delay' in args:
         clearPmStatsReq.pmMethod = tacPmMethod.DelayMeasurement
      elif 'loss' in args:
         clearPmStatsReq.pmMethod = tacPmMethod.LossMeasurement
         if 'synthetic' in args:
            clearPmStatsReq.pmMethod = tacPmMethod.SyntheticLossMeasurement
      clearPmStatsReq.reqTime = Tac.now()
      cfmConfigReq.clearPmStats = clearPmStatsReq

BasicCli.EnableMode.addCommandClass( ClearPmStats )

class CfmCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cfm counters [ interface INTFS ]'
   data = {
      'cfm' : nodeCfm,
      'counters' : CliCommand.guardedKeyword( 'counters',
                   helpdesc='CFM counters',
                   guard=defaultMipSupported ),
      'interface' : matcherCfmInterface,
      'INTFS' : intfRangeMatcher,
   }

   cliModel = CfmCounter
   handler = showInterfaceCounters

BasicCli.addShowCommandClass( CfmCountersCmd )

#--------------------------------------------------------------------------------
# clear cfm counters [ interface INTFS ]
#--------------------------------------------------------------------------------
class ClearCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear cfm counters [ interface INTFS ]'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'cfm' : nodeCfmMipGuard,
      'counters' : matcherCfmCounters,
      'interface' : matcherCfmInterface,
      'INTFS' : intfRangeMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      if not waitForAgent():
         return

      # command format : protocol, domain, interface1, .., interfaceN
      # value of protocol, domain, list of interfaces are used by counterFilter to
      # clear the corresponding protocol, mdLevel and interfaces counters
      # empty value for each field implies to clear counters over all possible
      # values for the field
      command = ','.join( (
         '', # Protocol.
         '', # MD level.
         ','.join( args.get( 'INTFS', () ) ), # All spec'd intfs or an empty string.
      ) )
      AgentCommandRequest.runSocketCommand( mode.entityManager, CfmAgent.name,
                                            "CfmCliCallback",
                                            command, keepalive=False )

BasicCli.EnableMode.addCommandClass( ClearCountersCmd )

#--------------------------------------------------------------------------------
# show cfm continuity-check intermediate-point database [ count ]
#--------------------------------------------------------------------------------
nodeIntermediatePoint = CliCommand.guardedKeyword( 'intermediate-point',
      helpdesc='Maintenance intermediate point information',
      guard=defaultMipSupported )
matcherDatabase = CliMatcher.KeywordMatcher( 'database',
      helpdesc='Database entries' )
matcherContinuityCheck = CliMatcher.KeywordMatcher( 'continuity-check',
      helpdesc='Continuity check information' )

def intfThenMacKey( keyValuePair ):
   '''We sort by interface, then MAC address.
   '''
   mac, intf = keyValuePair
   return ( Arnet.intfNameKey( intf ), mac )

class ShowMipCcmDbCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cfm continuity-check intermediate-point database [ count ]'
   data = {
      'cfm' : nodeCfm,
      'continuity-check' : matcherContinuityCheck,
      'intermediate-point' : nodeIntermediatePoint,
      'database' : matcherDatabase,
      'count' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'count', helpdesc='Count keyword' ),
         hidden=True ),
   }

   @staticmethod
   def handler( mode, args ):
      count = 'count' in args
      mipCcmDb = cfmStatus.mipCcmDb

      if count:
         count = 0
         if mipCcmDb:
            count = sum( len( mipCcmDbVlan.entry )
                         for mipCcmDbVlan in mipCcmDb.vlan.values() )
         print( 'Total entries in database:', count )
         return
      print( '%-4s    %-14s    %-8s' % ( 'Vlan', 'Mac Address', 'Interface' ) )
      print( '%-4s    %-14s    %-8s' % ( '----', '-----------', '---------' ) )

      if not mipCcmDb:
         return
      for vlanId, mipCcmDbVlan in sorted( mipCcmDb.vlan.items() ):
         for mac, intf in sorted( mipCcmDbVlan.entry.items(), key=intfThenMacKey ):
            print( '%4d    %-14s    %-8s' % ( vlanId,
                  Ethernet.convertMacAddrCanonicalToDisplay( mac ),
                  IntfCli.Intf.getShortname( intf ) ) )

BasicCli.addShowCommandClass( ShowMipCcmDbCmd )

#--------------------------------------------------------------------------------
# show cfm continuity-check database [ domain DOMAIN ]
#  Database for maintenance domain level: <mdLevel>
#  Maintenance association name: <maName1>
#  MEP Id  Mac Address     Interface  Vlan  Last Received
#  ------  -------------   ---------  ----  -------------
#  666     0012.2222.1212  Et1/3      300  0:00:02 ago
#  777     00aa.aaaa.aaaa  Et1/1      100  0:00:02 ago
#
#  Database for maintenance domain level: <mdLevel>
#  Maintenance association name: <maName2>
#  MEP Id  Mac Address     Interface  Vlan Last Received
#  ------  -------------   ---------  ---- -------------
#--------------------------------------------------------------------------------
class CfmContinuityCheckDatabaseCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cfm continuity-check database [ domain DOMAIN ]'
   data = {
      'cfm' : nodeCfm,
      'continuity-check' : matcherContinuityCheck,
      'database' : CliCommand.guardedKeyword( 'database',
                   helpdesc='Database entries',
                   guard=defaultMipSupported ),
      'domain' : 'Identify the database by its maintenance domain level',
      'DOMAIN' : CliMatcher.IntegerMatcher( 0, 7,
         helpdesc='Maintenance domain level' ),
   }

   @staticmethod
   def handler( mode, args ):
      domain = args.get( 'DOMAIN' )
      ccmDb = cfmStatus.ccmDb
      hdr = ' Database for maintenance domain level:'
      hdrMa = ' Maintenance association name:'
      hdrTable = [ 'MEP Id', 'Mac Address', 'Interface', 'VLAN',
                   'Last Received' ]

      fl = TableOutput.Format( justify='left' )
      fl.padLimitIs( True )

      def printHeaderAndCreateTable( domain, maName ):
         if domain is not None:
            print( hdr, domain )
         else:
            print( hdr )
         if maName is not None:
            print( hdrMa, maName )
         else:
            print( hdrMa )
         table = TableOutput.createTable( hdrTable )
         table.formatColumns( fl, fl, fl, fl, fl )
         return table

      printed = False
      if ccmDb and ccmDb.ccmMd:
         for mdLevel, ccmMd in sorted( ccmDb.ccmMd.items() ):
            if domain is None or domain == mdLevel:
               for maName, ccmMa in sorted( ccmMd.ccmMa.items() ):
                  table = printHeaderAndCreateTable( mdLevel, maName )
                  for mepId, info in sorted( ccmMa.ccmMepInfo.items() ):
                     table.newRow( mepId,
                         Ethernet.convertMacAddrCanonicalToDisplay( info.srcMac ),
                         IntfCli.Intf.getShortname( info.intf ),
                         info.vlanId, timestampToStr( info.lastCcmReceivedTime ) )
                  print( table.output() )
                  printed = True
      if not printed:
         table = printHeaderAndCreateTable( domain, None )
         print( table.output() )

BasicCli.addShowCommandClass( CfmContinuityCheckDatabaseCmd )

#--------------------------------------------------------------------------------
# clear cfm continuity-check database
#--------------------------------------------------------------------------------
class ClearCcmDbCmd( CliCommand.CliCommandClass ):
   syntax = 'clear cfm continuity-check database'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'cfm' : nodeCfmMipGuard,
      'continuity-check' : 'Continuity check information',
      'database' : 'Database entries',
   }

   @staticmethod
   def handler( mode, args ):
      if not waitForAgent():
         return
      command = 'clear continuity-check database'
      AgentCommandRequest.runSocketCommand( mode.entityManager, CfmAgent.name,
                                            "CfmCliCallback",
                                            command, keepalive=False )

BasicCli.EnableMode.addCommandClass( ClearCcmDbCmd )

#--------------------------------------------------------------------------------
# clear cfm continuity-check intermediate-point database
#--------------------------------------------------------------------------------
class ClearMipCcmDbCmd( CliCommand.CliCommandClass ):
   syntax = 'clear cfm continuity-check intermediate-point database'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'cfm' : nodeCfmMipGuard,
      'continuity-check' : 'Continuity check information',
      'intermediate-point' : CliCommand.guardedKeyword( 'intermediate-point',
         helpdesc='Maintenance intermediate point information',
         guard=defaultMipSupported ),
      'database' : 'Database entries',
   }

   @staticmethod
   def handler( mode, args ):
      if not waitForAgent():
         return
      command = 'clear intermediate-point database'
      AgentCommandRequest.runSocketCommand( mode.entityManager, CfmAgent.name,
                                            "CfmCliCallback",
                                            command, keepalive=False )

BasicCli.EnableMode.addCommandClass( ClearMipCcmDbCmd )

#--------------------------------------------------------------------------------
# show cfm intermediate-point
#--------------------------------------------------------------------------------
class CfmIntermediatePointCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cfm intermediate-point'
   data = {
      'cfm' : nodeCfm,
      'intermediate-point' : nodeIntermediatePoint,
   }
   cliModel = CfmMaintenanceDomain

   @staticmethod
   def handler( mode, args ):
      cfmMaintenanceDomain = CfmMaintenanceDomain()
      for ( k, v ) in cfmStatus.mdStatus.items():
         cfmMaintenanceDomain.intermediatePoint[ k ] = v.defaultMip
      return cfmMaintenanceDomain

BasicCli.addShowCommandClass( CfmIntermediatePointCmd )

#--------------------------------------------------------------------------------
# clear cfm continuity-check counters
#  [ domain < Domain ID > [ association < MA name > [ end-point < MEP ID > ] ] ]
#--------------------------------------------------------------------------------
class ClearCcCounters( CliCommand.CliCommandClass ):
   syntax = 'clear cfm continuity-check counters [ FILTER ]'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'cfm' : nodeCfm,
      'continuity-check' : continuityCheckMatcher,
      'counters' : cfmCcCountersKw,
      'FILTER' : EndPointFilter,
   }

   @staticmethod
   def handler( mode, args ):
      domainNameFilter = args.get( 'DOMAIN_NAME' )
      maNameFilter = args.get( 'MA_NAME' )
      localMepIdFilter = args.get( 'MEP_ID' )

      domainName = ""
      maName = Tac.Value( "Cfm::MaName" )
      localMepId = Tac.Value( "Cfm::MepId" )

      if domainNameFilter:
         domainName = domainNameFilter
         if maNameFilter:
            maName = maNameFilter
            if localMepIdFilter:
               localMepId = localMepIdFilter

      localMepKey = Tac.const( Tac.Value( 'Cfm::LocalMepKey',
                                          domainName,
                                          Tac.Value( "Cfm::MdLevel" ),
                                          maName,
                                          localMepId ) )
      ccCheckCtr = Tac.Value( "CfmAgent::ContinuityCheckCounter",
                              localMepKey,
                              Tac.now() )
      cfmClearCounter.continuityCheckCounter = ccCheckCtr

if toggleCfmSwUpMepCcmEnabled():
   BasicCli.EnableMode.addCommandClass( ClearCcCounters )

#-----------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-----------------------------------------------------------
def Plugin( entityManager ):
   global cfmConfig
   global cfmStatus
   global aleCfmConfig
   global aleCfmStatus
   global cfmHwSupportStatus
   global trapConfig
   global cfmConfigReq
   global cfmClearCounter

   cfmConfig = ConfigMount.mount( entityManager, "cfm/config",
                                  "CfmAgent::Config", "w" )
   trapConfig = ConfigMount.mount( entityManager, "hardware/trap/config/trapConfig",
                                   "Arnet::TrapConfig", 'w' )
   cfmStatus = LazyMount.mount( entityManager, "cfm/status",
                                   "CfmAgent::Status", "r" )
   cfmHwSupportStatus = LazyMount.mount( entityManager, "cfm/hwSupportStatus",
                                         "Cfm::HwSupportStatus", "r" )
   aleCfmConfig = LazyMount.mount( entityManager, "cfm/aleConfig",
                                   "Cfm::AleConfig", "r" )
   aleCfmStatus = LazyMount.mount( entityManager, "cfm/aleStatus",
                                   "Cfm::AleStatus", "r" )
   cfmConfigReq = LazyMount.mount( entityManager, "cfm/configReq",
                                   "Cfm::ConfigReq", "w" )
   cfmClearCounter = LazyMount.mount( entityManager, "cfm/clearCounter",
                                      "CfmAgent::CfmClearCounter", "w" )
